/*//////////////////////////////////////////////////////////////////////////

	flaXen's Colorspace Conversion for VirtualDub
	(c) Copyright 2000 Dan Flower (flaXen)

	Convert between three different colorspaces

*//////////////////////////////////////////////////////////////////////////

#include <windows.h>
#include <stdio.h>
#include <math.h>

#include "resource.h"
#include "filter.h"
#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#define	FORMAT_YUV	0
#define	FORMAT_YIQ	1

typedef struct MyFilterData {
	long	*yr, *yg, *yb;
	long	*ir, *ig, *ib;
	long	*qr, *qg, *qb;
	long	*ri, *rq;
	long	*gi, *gq;
	long	*bi, *bq;
	int		ColorSpace;
} MyFilterData;

///////////////////////////////////////////////////////////////////////////

int		fxColorSpaceRunProc(const FilterActivation *fa, const FilterFunctions *ff);
void	fxColorSpace_RGB2YUV(Pixel32 *, Pixel32 *, long, long, long, MyFilterData *);
void	fxColorSpace_RGB2YIQ(Pixel32 *, Pixel32 *, long, long, long, MyFilterData *);
void	fxColorSpace_YUV2RGB(Pixel32 *, Pixel32 *, long, long, long, MyFilterData *);
void	fxColorSpace_YIQ2RGB(Pixel32 *, Pixel32 *, long, long, long, MyFilterData *);
void	fxColorSpace_YUV2YIQ(Pixel32 *, Pixel32 *, long, long, long, MyFilterData *);
void	fxColorSpace_YIQ2YUV(Pixel32 *, Pixel32 *, long, long, long, MyFilterData *);
void	fxColorSpace_GenTabs(long, MyFilterData *);
int		fxColorSpaceStartProc(FilterActivation *fa, const FilterFunctions *ff);
int		fxColorSpaceEndProc(FilterActivation *fa, const FilterFunctions *ff); 
int		fxColorSpaceInitProc(FilterActivation *fa, const FilterFunctions *ff);
int		fxColorSpaceConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void	fxColorSpaceStringProc(const FilterActivation *fa, const FilterFunctions *ff, char *buf);
void	fxColorSpaceScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool	fxColorSpaceFssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

ScriptFunctionDef fxColorSpace_func_defs[]={
    { (ScriptFunctionPtr)fxColorSpaceScriptConfig, "Config", "0i" },
    { NULL },
};

CScriptObject fxColorSpaceobj={
    NULL, fxColorSpace_func_defs
};

struct FilterDefinition filterDef_fxColorSpace = {
	NULL, NULL, NULL,		// next, prev, module

	"Colorspace Conversion",			// name
	"flaXen's Colorspace Conversion Tool",	// desc
	"flaXen",				// maker

	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size
	fxColorSpaceInitProc,	// initProc
	NULL,					// deinitProc
	fxColorSpaceRunProc,	// runProc
	NULL,					// paramProc
	fxColorSpaceConfigProc,	// configProc
	fxColorSpaceStringProc,	// stringProc
	fxColorSpaceStartProc,	// startProc
	fxColorSpaceEndProc,	// endProc
	&fxColorSpaceobj,		// script_obj
	fxColorSpaceFssProc,	// fssProc
};

///////////////////////////////////////////////////////////////////////////

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_fxColorSpace;

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
	if (!(fd_fxColorSpace = ff->addFilter(fm, &filterDef_fxColorSpace, sizeof(FilterDefinition))))
		return 1;

	vdfd_ver    = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;
	return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
	ff->removeFilter(fd_fxColorSpace);
}

///////////////////////////////////////////////////////////////////////////

int fxColorSpaceRunProc(const FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData	*mfd = (MyFilterData *)fa->filter_data;
	Pixel32	*isrc, *idst;
	const long	swid = fa->src.w, shei = fa->src.h;
	const long	spit = fa->src.pitch >> 2;
	const long	ssize = fa->src.pitch * fa->src.h;

	isrc = (Pixel32 *)fa->src.data;
	idst = (Pixel32 *)fa->dst.data;

	switch (mfd->ColorSpace) {
	case 0:	// RGB -> YUV
		fxColorSpace_RGB2YUV(isrc, idst, swid, shei, spit, mfd);
		break;
	case 1:	// RGB -> YIQ
		fxColorSpace_RGB2YIQ(isrc, idst, swid, shei, spit, mfd);
		break;
	case 2:	// YUV -> RGB
		fxColorSpace_YUV2RGB(isrc, idst, swid, shei, spit, mfd);
		break;
	case 3:	// YIQ -> RGB
		fxColorSpace_YIQ2RGB(isrc, idst, swid, shei, spit, mfd);
		break;
	case 4:	// YUV -> YIQ
		fxColorSpace_YUV2YIQ(isrc, idst, swid, shei, spit, mfd);
		break;
	case 5:	// YIQ -> YUV
		fxColorSpace_YIQ2YUV(isrc, idst, swid, shei, spit, mfd);
		break;
	}

	return 0;
}

///////////////////////////////////////////////////////////////////////////

void	fxColorSpace_RGB2YUV(Pixel32 *isrc, Pixel32 *idst, long swid, long shei, long spit, MyFilterData *mfd) {
	Pixel32	*src, *dst;
	long	x, y, c, c1, tr, tg, tb;
	long	*yr = (long *)mfd->yr, *yg = (long *)mfd->yg, *yb = (long *)mfd->yb;
	long	*ir = (long *)mfd->ir, *ig = (long *)mfd->ig, *ib = (long *)mfd->ib;
	long	*qr = (long *)mfd->qr, *qg = (long *)mfd->qg, *qb = (long *)mfd->qb;

	// Convert RGB -> YUV
	fxColorSpace_GenTabs(FORMAT_YUV, mfd);
	for (y = 0; y < shei; y++) {
		src = isrc + y * spit;
		dst = idst + y * spit;
		for (x = 0; x < swid; x++) {
			c = *src++;
			tr = (c >> 16) & 255; tg = (c >> 8) & 255; tb = c & 255;
			c1 = (mfd->yr[tr] + mfd->yg[tg] + mfd->yb[tb] + 1048576)>>16;
			if (c1<0) c1=0; else if (c1>255) c1=255;
			c = (mfd->ir[tr] + mfd->ig[tg] + mfd->ib[tb] + 8388608)>>16;
			if (c<0) c=0; else if (c>255) c=255;
			c1 = (c1 << 8) | c;
			c = (mfd->qr[tr] + mfd->qg[tg] + mfd->qb[tb] + 8388608)>>16;
			if (c<0) c=0; else if (c>255) c=255;
			c1 = (c1 << 8) | c;
			*dst++ = c1;
		}
	}
}

///////////////////////////////////////////////////////////////////////////

void	fxColorSpace_RGB2YIQ(Pixel32 *isrc, Pixel32 *idst, long swid, long shei, long spit, MyFilterData *mfd) {
	Pixel32	*src, *dst;
	long	x, y, c, c1, tr, tg, tb;
	long	*yr = (long *)mfd->yr, *yg = (long *)mfd->yg, *yb = (long *)mfd->yb;
	long	*ir = (long *)mfd->ir, *ig = (long *)mfd->ig, *ib = (long *)mfd->ib;
	long	*qr = (long *)mfd->qr, *qg = (long *)mfd->qg, *qb = (long *)mfd->qb;

	// Convert RGB -> YIQ
	fxColorSpace_GenTabs(FORMAT_YIQ, mfd);
	for (y = 0; y < shei; y++) {
		src = isrc + y * spit;
		dst = idst + y * spit;
		for (x = 0; x < swid; x++) {
			c = *src++;
			tr = (c >> 16) & 255; tg = (c >> 8) & 255; tb = c & 255;
			c1 = (mfd->yr[tr] + mfd->yg[tg] + mfd->yb[tb])>>16;
			if (c1<0) c1=0; else if (c1>255) c1=255;
			c = (mfd->ir[tr] + mfd->ig[tg] + mfd->ib[tb] + 8388608)>>16;
			if (c<0) c=0; else if (c>255) c=255;
			c1 = (c1 << 8) | c;
			c = (mfd->qr[tr] + mfd->qg[tg] + mfd->qb[tb] + 8388608)>>16;
			if (c<0) c=0; else if (c>255) c=255;
			c1 = (c1 << 8) | c;
			*dst++ = c1;
		}
	}
}

///////////////////////////////////////////////////////////////////////////

void	fxColorSpace_YUV2RGB(Pixel32 *isrc, Pixel32 *idst, long swid, long shei, long spit, MyFilterData *mfd) {
	Pixel32	*src, *dst;
	long	x, y, c, tr, tg, tb, cy, ci, cq;
	long	*ri = (long *)mfd->ri, *rq = (long *)mfd->rq;
	long	*gi = (long *)mfd->gi, *gq = (long *)mfd->gq;
	long	*bi = (long *)mfd->bi, *bq = (long *)mfd->bq;

	// Convert YUV -> RGB
	fxColorSpace_GenTabs(FORMAT_YUV, mfd);
	for (y = 0; y < shei; y++) {
		src = isrc + y * spit;
		dst = idst + y * spit;
		for	(x = 0; x < swid; x++) {
			c = *src++;
			cy = mfd->rq[(c >> 16) & 255]; ci = (c >> 8) & 255; cq = c & 255;
			tr = (cy + mfd->ri[ci])>>16;
			if (tr<0) tr=0; else if (tr>255) tr=255;
			tg = (cy + mfd->gi[ci] + mfd->gq[cq])>>16;
			if (tg<0) tg=0; else if (tg>255) tg=255;
			tb = (cy + mfd->bq[cq])>>16;
			if (tb<0) tb=0; else if (tb>255) tb=255;

			*dst++ = (tr << 16) | (tg << 8) | tb;
		}
	}
}

///////////////////////////////////////////////////////////////////////////

void	fxColorSpace_YIQ2RGB(Pixel32 *isrc, Pixel32 *idst, long swid, long shei, long spit, MyFilterData *mfd) {
	Pixel32	*src, *dst;
	long	x, y, c, tr, tg, tb, cy, ci, cq;
	long	*ri = (long *)mfd->ri, *rq = (long *)mfd->rq;
	long	*gi = (long *)mfd->gi, *gq = (long *)mfd->gq;
	long	*bi = (long *)mfd->bi, *bq = (long *)mfd->bq;

	// Convert YIQ -> RGB
	fxColorSpace_GenTabs(FORMAT_YIQ, mfd);
	for (y = 0; y < shei; y++) {
		src = isrc + y * spit;
		dst = idst + y * spit;
		for	(x = 0; x < swid; x++) {
			c = *src++;
			cy = c & 0x00FF0000L; ci = (c >> 8) & 255; cq = c & 255;
			tr = (cy + mfd->ri[ci] + mfd->rq[cq])>>16;
			if (tr<0) tr=0; else if (tr>255) tr=255;
			tg = (cy + mfd->gi[ci] + mfd->gq[cq])>>16;
			if (tg<0) tg=0; else if (tg>255) tg=255;
			tb = (cy + mfd->bi[ci] + mfd->bq[cq])>>16;
			if (tb<0) tb=0; else if (tb>255) tb=255;
			*dst++ = (tr << 16) | (tg << 8) | tb;
		}
	}
}

///////////////////////////////////////////////////////////////////////////

void	fxColorSpace_YUV2YIQ(Pixel32 *isrc, Pixel32 *idst, long swid, long shei, long spit, MyFilterData *mfd) {
	// Convert YUV -> YIQ
	fxColorSpace_GenTabs(FORMAT_YUV, mfd);
	fxColorSpace_YUV2RGB(isrc, idst, swid, shei, spit, mfd);
	fxColorSpace_GenTabs(FORMAT_YIQ, mfd);
	fxColorSpace_RGB2YIQ(idst, idst, swid, shei, spit, mfd);
}

///////////////////////////////////////////////////////////////////////////

void	fxColorSpace_YIQ2YUV(Pixel32 *isrc, Pixel32 *idst, long swid, long shei, long spit, MyFilterData *mfd) {
	// Convert YIQ -> YUV
	fxColorSpace_GenTabs(FORMAT_YIQ, mfd);
	fxColorSpace_YIQ2RGB(isrc, idst, swid, shei, spit, mfd);
	fxColorSpace_GenTabs(FORMAT_YUV, mfd);
	fxColorSpace_RGB2YUV(idst, idst, swid, shei, spit, mfd);
}

///////////////////////////////////////////////////////////////////////////

int fxColorSpaceStartProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	// RGB -> YUV/YIQ conversion tables
	if (!(mfd->yr = new long[256])) return 1;
	if (!(mfd->yg = new long[256])) return 1;
	if (!(mfd->yb = new long[256])) return 1;

	if (!(mfd->ir = new long[256])) return 1;
	if (!(mfd->ig = new long[256])) return 1;
	if (!(mfd->ib = new long[256])) return 1;

	if (!(mfd->qr = new long[256])) return 1;
	if (!(mfd->qg = new long[256])) return 1;
	if (!(mfd->qb = new long[256])) return 1;

	// YUV/YIQ) -> RGB conversion tables
	if (!(mfd->ri = new long[256])) return 1;
	if (!(mfd->rq = new long[256])) return 1;
	if (!(mfd->gi = new long[256])) return 1;
	if (!(mfd->gq = new long[256])) return 1;
	if (!(mfd->bi = new long[256])) return 1;
	if (!(mfd->bq = new long[256])) return 1;

	return 0;
}

///////////////////////////////////////////////////////////////////////////

void	fxColorSpace_GenTabs(long format, MyFilterData *mfd) {
	long	d;

	// Calculate data for RGB -> YUV/YIQ conversion tables
	for (d = 0; d < 256; d++) {
		if (format == FORMAT_YIQ) {
			// RGB -> YIQ transformation matrix
			mfd->yr[d] = (long)(+0.299 * d * 65536 + 0.5);
			mfd->yg[d] = (long)(+0.587 * d * 65536 + 0.5);
			mfd->yb[d] = (long)(+0.114 * d * 65536 + 0.5);
			mfd->ir[d] = (long)(+0.596 * d * 65536 + 0.5);
			mfd->ig[d] = (long)(-0.275 * d * 65536 + 0.5);
			mfd->ib[d] = (long)(-0.321 * d * 65536 + 0.5);
			mfd->qr[d] = (long)(+0.212 * d * 65536 + 0.5);
			mfd->qg[d] = (long)(-0.523 * d * 65536 + 0.5);
			mfd->qb[d] = (long)(+0.311 * d * 65536 + 0.5);
			// YIQ -> RGB transformation matrix
			mfd->ri[d] = (long)(+0.956 * (d-128) * 65536 + 0.5);
			mfd->rq[d] = (long)(+0.621 * (d-128) * 65536 + 0.5);
			mfd->gi[d] = (long)(-0.272 * (d-128) * 65536 + 0.5);
			mfd->gq[d] = (long)(-0.647 * (d-128) * 65536 + 0.5);
			mfd->bi[d] = (long)(-1.105 * (d-128) * 65536 + 0.5);
			mfd->bq[d] = (long)(+1.702 * (d-128) * 65536 + 0.5);
		} else {
			// RGB -> YUV transformation matrix
			mfd->yr[d] = (long)(+0.257 * d * 65536 + 0.5);
			mfd->yg[d] = (long)(+0.504 * d * 65536 + 0.5);
			mfd->yb[d] = (long)(+0.098 * d * 65536 + 0.5);
			mfd->ir[d] = (long)(+0.439 * d * 65536 + 0.5);
			mfd->ig[d] = (long)(-0.368 * d * 65536 + 0.5);
			mfd->ib[d] = (long)(-0.071 * d * 65536 + 0.5);
			mfd->qr[d] = (long)(-0.148 * d * 65536 + 0.5);
			mfd->qg[d] = (long)(-0.291 * d * 65536 + 0.5);
			mfd->qb[d] = (long)(+0.439 * d * 65536 + 0.5);
			// YUV -> RGB transformation matrix
			mfd->ri[d] = (long)(+1.596 * (d-128) * 65536 + 0.5);
			mfd->rq[d] = (long)((d - 16.0) * 1.164 * 65536.0 + 0.5);	// Actually for CY transform
			mfd->gi[d] = (long)(-0.813 * (d-128) * 65536 + 0.5);
			mfd->gq[d] = (long)(-0.391 * (d-128) * 65536 + 0.5);
			mfd->bi[d] = 0;
			mfd->bq[d] = (long)(+2.018 * (d-128) * 65536 + 0.5);
		}
	}
}

///////////////////////////////////////////////////////////////////////////

int fxColorSpaceEndProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	// Delete RGB -> YIQ/YUV conversion tables
	delete[] mfd->yr; mfd->yr = NULL;
	delete[] mfd->yg; mfd->yg = NULL;
	delete[] mfd->yb; mfd->yb = NULL;

	delete[] mfd->ir; mfd->ir = NULL;
	delete[] mfd->ig; mfd->ig = NULL;
	delete[] mfd->ib; mfd->ib = NULL;

	delete[] mfd->qr; mfd->qr = NULL;
	delete[] mfd->qg; mfd->qg = NULL;
	delete[] mfd->qb; mfd->qb = NULL;

	// Delete YUV/YIQ -> RGB conversion tables
	delete[] mfd->ri; mfd->ri = NULL;
	delete[] mfd->rq; mfd->rq = NULL;
	delete[] mfd->gi; mfd->gi = NULL;
	delete[] mfd->gq; mfd->gq = NULL;
	delete[] mfd->bi; mfd->bi = NULL;
	delete[] mfd->bq; mfd->bq = NULL;

	return 0;
}

///////////////////////////////////////////////////////////////////////////

int fxColorSpaceInitProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->ColorSpace = 0;

	return 0;
}

///////////////////////////////////////////////////////////////////////////

BOOL CALLBACK fxColorSpaceConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
	MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);
	long	d;

	switch(msg) {
	case WM_INITDIALOG:
		SetWindowLong(hdlg, DWL_USER, lParam);
		mfd = (MyFilterData *)lParam;
		for (d = 1100; d < 1106; d++) CheckDlgButton(hdlg, d, BST_UNCHECKED);
		CheckDlgButton(hdlg, 1100 + mfd->ColorSpace, BST_CHECKED);
		return TRUE;

	case WM_COMMAND:
		switch(LOWORD(wParam)) {
		case IDOK:
			d = 1099;
			do {
				d++;
			} while (!!IsDlgButtonChecked(hdlg, d) == BST_UNCHECKED);
			mfd->ColorSpace = d - 1100;
			EndDialog(hdlg, 0);

			return TRUE;
		case IDCANCEL:
			EndDialog(hdlg, 1);
			return FALSE;
		}
		break;
	}

	return FALSE;
}

///////////////////////////////////////////////////////////////////////////

int fxColorSpaceConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd) {
	return DialogBoxParam(fa->filter->module->hInstModule,
	       MAKEINTRESOURCE(IDD_FXCOLORSPACE_CONFIG), hwnd,
		   fxColorSpaceConfigDlgProc, (LPARAM)fa->filter_data);
}

///////////////////////////////////////////////////////////////////////////

void fxColorSpaceStringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	char	fstr[20];

	sprintf(fstr, "");
	switch (mfd->ColorSpace) {
	case 0:
		sprintf(fstr, "RGB -> YUV");
		break;
	case 1:
		sprintf(fstr, "RGB -> YIQ");
		break;
	case 2:
		sprintf(fstr, "YUV -> RGB");
		break;
	case 3:
		sprintf(fstr, "YIQ -> RGB");
		break;
	case 4:
		sprintf(fstr, "YUV -> YIQ");
		break;
	case 5:
		sprintf(fstr, "YIQ -> YUV");
		break;
	}

	sprintf(str, " Convert from %s", fstr);
}

///////////////////////////////////////////////////////////////////////////

void fxColorSpaceScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
    FilterActivation *fa = (FilterActivation *)lpVoid;
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->ColorSpace = argv[1].asInt();
}

///////////////////////////////////////////////////////////////////////////

bool fxColorSpaceFssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

    _snprintf(buf, buflen, "Config(%d)", mfd->ColorSpace);

    return true;
}