/*
    Hue Cycle Filter for VirtualDub -- Rotate the image hue with
      increasing frames. Copyright (C) 2001 Tom Ford

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    The author can be contacted at:
      Tom Ford
      flend@compsoc.net

    This work includes code from:
      Colorize Filter for VirtualDub -- Change a video to a monochrome video
        with selectable base color. Copyright (C) 1999-2000 Donald A. Graft

      By 
        Donald Graft
        neuron2@home.com.
*/

#include <windows.h>
#include <commctrl.h>
#include <stdio.h>
#include <crtdbg.h>
#include <math.h>
#include <stdlib.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

#define CYCLE_FRAMES_MAX 1440

int RunProc(const FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int InitProc(FilterActivation *fa, const FilterFunctions *ff);
int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);
void RGBtoHSL(Pixel32 src, double *H, double *S, double *L);
void HLStoRGB(double H, double L, double S, int *r, int *g, int *b);

typedef struct FilterDataStruct {
  IFilterPreview *ifp;
  int hue_displacement;
  int cycle_frames;
} FilterData;


bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
  FilterData *filter_data = (FilterData *)fa->filter_data;

  _snprintf(buf, buflen, "Config(%d, %d)", filter_data->cycle_frames, filter_data->hue_displacement);
  return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {

  FilterActivation *fa = (FilterActivation *)lpVoid;
  FilterData *filter_data = (FilterData *)fa->filter_data;

  filter_data->cycle_frames = argv[0].asInt();
  filter_data->hue_displacement = argv[1].asInt();
}

ScriptFunctionDef func_defs[]={
  { (ScriptFunctionPtr)ScriptConfig, "Config", "0ii" },
  { NULL },
};

CScriptObject script_obj={
  NULL, func_defs
};

struct FilterDefinition filterdef = {

  NULL, NULL, NULL,     // next, prev, module
  "hue cycle",  // name
  "Cycle hues.",        // desc
  "Tom Ford",           // maker
  NULL,                 // private_data
  sizeof(FilterData),   // inst_data_size

  InitProc,             // initProc
  NULL,                 // deinitProc
  RunProc,              // runProc
  NULL,                 // paramProc
  ConfigProc,           // configProc
  StringProc,           // stringProc
  NULL,            // startProc
  NULL,              // endProc

  &script_obj,          // script_obj
  FssProc,              // fssProc
};

int RunProc(const FilterActivation *fa, const FilterFunctions *ff) {

  FilterData *filter_data = (FilterData *)fa->filter_data;
  FilterStateInfo *info = fa->pfsi;
  const PixDim width = fa->src.w;
  const PixDim height = fa->src.h;
  Pixel32 *src, *dst;
  int x, y;
  int r, g, b;
  double H, S, L;
  int hue;
  double output_hue;
  int hue_frame_offset = 0;

  if(filter_data->cycle_frames) {
    hue_frame_offset = (int)((240.0 / filter_data->cycle_frames) * info->lCurrentFrame);
  }

  src = fa->src.data;
  dst = fa->dst.data;

  for (y = 0; y < height; y++)
    {
      for (x = 0; x < width; x++)
        {
          //Generate HLS of input pixel
          RGBtoHSL(src[x], &H, &S, &L);
          hue = (int)(H * 239);

          //Displace hue by initial offset
          hue += filter_data->hue_displacement;

          //Displace hue by frame dependant offset
          hue += hue_frame_offset;

          hue %= 240;

          //Write output pixel
          output_hue = (double)hue / 240.0;
          HLStoRGB(output_hue, L, S, &r, &g, &b);
          dst[x] = (r << 16) | (g << 8) | (b);                   
        }

      src = (Pixel *)((char *)src + fa->src.pitch);
      dst = (Pixel *)((char *)dst + fa->dst.pitch);
    }
  return 0;
}

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_huecycle;

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
  if (!(fd_huecycle = ff->addFilter(fm, &filterdef, sizeof(FilterDefinition))))
    return 1;

  vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
  vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

  return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
  ff->removeFilter(fd_huecycle);
}

int InitProc(FilterActivation *fa, const FilterFunctions *ff)
{
  FilterData *filter_data = (FilterData *)fa->filter_data;

  filter_data->cycle_frames = 240;
  filter_data->hue_displacement = 0;

  return 0;
}

BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
  FilterData *filter_data = (FilterData *)GetWindowLong(hdlg, DWL_USER);

  switch(msg) {
    HWND hWnd;

  case WM_PAINT:
    return 0;

  case WM_INITDIALOG:
    SetWindowLong(hdlg, DWL_USER, lParam);
    filter_data = (FilterData *)lParam;

    hWnd = GetDlgItem(hdlg, IDC_CYCLE);
    SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, CYCLE_FRAMES_MAX));
    SendMessage(hWnd, TBM_SETTICFREQ, 10 , 0);
    SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, filter_data->cycle_frames);
    SetDlgItemInt(hdlg, IDC_ECYCLE, filter_data->cycle_frames, FALSE);
    hWnd = GetDlgItem(hdlg, IDC_DISP);
    SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 239));
    SendMessage(hWnd, TBM_SETTICFREQ, 10 , 0);
    SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, filter_data->hue_displacement);
    SetDlgItemInt(hdlg, IDC_EDISP, filter_data->hue_displacement, FALSE);

    filter_data->ifp->InitButton(GetDlgItem(hdlg, IDPREVIEW));
    return TRUE;

  case WM_HSCROLL:
    {
      int cycle, disp;
      HWND hwnd = GetDlgItem(hdlg, IDC_CYCLE);
      cycle = SendMessage(hwnd, TBM_GETPOS, 0, 0);
      SetDlgItemInt(hdlg, IDC_ECYCLE, cycle, FALSE);
      hwnd = GetDlgItem(hdlg, IDC_DISP);
      disp = SendMessage(hwnd, TBM_GETPOS, 0, 0);
      SetDlgItemInt(hdlg, IDC_EDISP, disp, FALSE);

      if (cycle != filter_data->cycle_frames || disp != filter_data->hue_displacement) {
        filter_data->cycle_frames = cycle;
        filter_data->hue_displacement = disp;
      }

      filter_data->ifp->RedoFrame();
      break;
    }

  case WM_COMMAND:
    switch(LOWORD(wParam)) {
    case IDPREVIEW:
      filter_data->ifp->Toggle(hdlg);
      break;
    case IDOK:
      EndDialog(hdlg, 0);
      return TRUE;
    case IDHELP:
      {
        MessageBox(NULL, "This filter should be pretty straightforward.", "HueCycle 'help'.", MB_OK | MB_ICONINFORMATION);
        /*char prog[256];
          char path[256];
          LPTSTR ptr;

          GetModuleFileName(NULL, prog, 255);
          GetFullPathName(prog, 255, path, &ptr);
          *ptr = 0;
          strcat(path, "plugins\\Colorize.html");
          ShellExecute(hdlg, "open", path, NULL, NULL, SW_SHOWNORMAL);*/
        return TRUE;
      }
    case IDCANCEL:
      EndDialog(hdlg, 1);
      return TRUE;
    case IDC_CYCLEPLUS:
   
      if (filter_data->cycle_frames < CYCLE_FRAMES_MAX) {

        filter_data->cycle_frames++;
        SetDlgItemInt(hdlg, IDC_ECYCLE, filter_data->cycle_frames, FALSE);
        SendMessage(GetDlgItem(hdlg, IDC_CYCLE), TBM_SETPOS, (WPARAM)TRUE, filter_data->cycle_frames);

        filter_data->ifp->RedoFrame();
      }
      break;
    case IDC_CYCLEMINUS:

      if (filter_data->cycle_frames > 0) {

        filter_data->cycle_frames--;
        SetDlgItemInt(hdlg, IDC_ECYCLE, filter_data->cycle_frames, FALSE);
        SendMessage(GetDlgItem(hdlg, IDC_CYCLE), TBM_SETPOS, (WPARAM)TRUE, filter_data->cycle_frames);

        filter_data->ifp->RedoFrame();
      }
      break;
    case IDC_DISPPLUS:
      if (filter_data->hue_displacement < 239) {

        filter_data->hue_displacement++;
        SetDlgItemInt(hdlg, IDC_EDISP, filter_data->hue_displacement, FALSE);
        SendMessage(GetDlgItem(hdlg, IDC_DISP), TBM_SETPOS, (WPARAM)TRUE, filter_data->hue_displacement);

        filter_data->ifp->RedoFrame();
      }
      break;
    case IDC_DISPMINUS:
      if (filter_data->hue_displacement > 0) {

        filter_data->hue_displacement--;
        SetDlgItemInt(hdlg, IDC_EDISP, filter_data->hue_displacement, FALSE);
        SendMessage(GetDlgItem(hdlg, IDC_DISP), TBM_SETPOS, (WPARAM)TRUE, filter_data->hue_displacement);

        filter_data->ifp->RedoFrame();
      }
      break;

    }
    break;
  }

  return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd)
{
  FilterData *filter_data = (FilterData *)fa->filter_data;
  FilterData filter_data_old = *filter_data;
  int ret;

  filter_data->ifp = fa->ifp;

  ret = DialogBoxParam(fa->filter->module->hInstModule, MAKEINTRESOURCE(IDD_FILTER),
                       hwnd, ConfigDlgProc, (LPARAM) filter_data);
  if (ret)
    {
      *filter_data = filter_data_old;
      ret = TRUE;
    }
  else
    {
      ret = FALSE;
    }
  return(ret);
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str)
{
  FilterData *filter_data = (FilterData *)fa->filter_data;

  sprintf(str, " (cframes %d, disp %d)", filter_data->cycle_frames, filter_data->hue_displacement);
}

void RGBtoHSL(Pixel32 src, double *H, double *S, double *L)
{
  double delta;
  double r = (double)((src & 0xff0000) >> 16)/255;
  double g = (double)((src & 0xff00) >> 8)/255;
  double b = (double)(src & 0xff)/255;
  double cmax = max(r,max(g,b));
  double cmin = min(r,min(g,b));

  *L=(cmax+cmin)/2.0;
  if(cmax==cmin) {
    *S = 0;
    *H = 0; // it's really undefined
  } else {
    if(*L < 0.5) 
      *S = (cmax-cmin)/(cmax+cmin);
    else
      *S = (cmax-cmin)/(2.0-cmax-cmin);
    delta = cmax - cmin;
    if(r==cmax)
      *H = (g-b)/delta;
    else if(g==cmax)
      *H = 2.0 +(b-r)/delta;
    else
      *H=4.0+(r-g)/delta;
    *H /= 6.0;
    if(*H < 0.0)
      *H += 1;
  }
}

int HuetoRGB(double m1, double m2, double h)
{
  if( h < 0 ) h += 1.0;
  if( h > 1 ) h -= 1.0;
  if( 6.0*h < 1 )
    return ((int) (255 * (m1+(m2-m1)*h*6.0)));
  if( 2.0*h < 1 )
    return ((int) (255 * m2));
  if( 3.0*h < 2.0 )
    return ((int) (255 * (m1+(m2-m1)*((2.0/3.0)-h)*6.0)));
  return ((int) (255 * m1));
}

void HLStoRGB(double H, double L, double S, int *r, int *g, int *b)
{
  double m1, m2;

  if(S==0) {
    *r = *g = *b = (int) (255 * L);
  } else {
    if(L <=0.5)
      m2 = L*(1.0+S);
    else
      m2 = L+S-L*S;
    m1 = 2.0*L-m2;
    *r = HuetoRGB(m1,m2,H+1.0/3.0);
    *g = HuetoRGB(m1,m2,H);
    *b = HuetoRGB(m1,m2,H-1.0/3.0);
  }
}


