/*//////////////////////////////////////////////////////////////////////////

	flaXen's Chroma Shifting Tool for VirtualDub
	(c) Copyright 2000 Dan Flower (flaXen)

	Apply field shifting for each of the channels (luma / chroma x / choma y)

*///////////////////////////////////////////////////////////////////////////

#include <windows.h>
#include <stdio.h>
#include <math.h>

#include "resource.h"
#include "filter.h"
#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#define	FORMAT_YUV	0
#define	FORMAT_YIQ	1

typedef struct MyFilterData {
	Pixel32	*TmpBuf1, *TmpBuf2;
	long	*ytab, *itab, *qtab;
	long	*rtab, *gtab, *btab;
	long	*cytf, mmx;
	int		fsHOff, fsHNeg, fsVOff, fsVNeg;
	int		fsFld1, fsFld2, fsFld3;
	int		fsLeave, fsYccIn, fsYccOut, fsYIQ;
} MyFilterData;

///////////////////////////////////////////////////////////////////////////

int		fxShiftRunProc(const FilterActivation *fa, const FilterFunctions *ff);
void	fxShift_RGB2YCC(Pixel32 *, Pixel32 *, long, long, long, MyFilterData *);
void	fxShift_YCC2RGB(Pixel32 *, Pixel32 *, long, long, long, MyFilterData *);
void	fxShift_Shift(Pixel32 *, Pixel32 *, long, long, long, MyFilterData *);
void	fxShift_ShiftLeave(Pixel32 *, Pixel32 *, long, long, long, MyFilterData *);
void	fxShift_GenTabs(long, MyFilterData *);
int		fxShiftStartProc(FilterActivation *fa, const FilterFunctions *ff);
int		fxShiftEndProc(FilterActivation *fa, const FilterFunctions *ff); 
int		fxShiftInitProc(FilterActivation *fa, const FilterFunctions *ff);
int		fxShiftConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void	fxShiftStringProc(const FilterActivation *fa, const FilterFunctions *ff, char *buf);
void	fxShiftScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool	fxShiftFssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

ScriptFunctionDef fxShift_func_defs[]={
    { (ScriptFunctionPtr)fxShiftScriptConfig, "Config", "0iiiiiiiii" },
    { NULL },
};

CScriptObject fxShiftobj={
    NULL, fxShift_func_defs
};

struct FilterDefinition filterDef_fxShift = {
	NULL, NULL, NULL,		// next, prev, module

	"Field Shift",		// name
	"flaXen's Field Shifting Tool",	// desc
	"flaXen",				// maker

	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size
	fxShiftInitProc,	// initProc
	NULL,					// deinitProc
	fxShiftRunProc,	// runProc
	NULL,					// paramProc
	fxShiftConfigProc,	// configProc
	fxShiftStringProc,	// stringProc
	fxShiftStartProc,	// startProc
	fxShiftEndProc,	// endProc
	&fxShiftobj,		// script_obj
	fxShiftFssProc,	// fssProc
};

///////////////////////////////////////////////////////////////////////////

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_fxShift;

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
	if (!(fd_fxShift = ff->addFilter(fm, &filterDef_fxShift, sizeof(FilterDefinition))))
		return 1;

	vdfd_ver    = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;
	return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
	ff->removeFilter(fd_fxShift);
}

///////////////////////////////////////////////////////////////////////////

int fxShiftRunProc(const FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData	*mfd = (MyFilterData *)fa->filter_data;
	Pixel32		*isrc, *idst, *itm1, *itm2;
	const long	swid = fa->src.w, shei = fa->src.h;
	const long	spit = fa->src.pitch >> 2;
	const long	ssize = fa->src.pitch * fa->src.h;

	isrc = (Pixel32 *)fa->src.data;
	idst = (Pixel32 *)fa->dst.data;
	itm1 = (Pixel32 *)mfd->TmpBuf1;
	itm2 = (Pixel32 *)mfd->TmpBuf2;

	if ((mfd->fsHOff == 0 && mfd->fsVOff == 0) ||
	((mfd->fsFld1 | mfd->fsFld2 | mfd->fsFld3) == BST_UNCHECKED) ||
	(mfd->fsHOff >= swid || mfd->fsVOff >= shei)) {
		memcpy(idst, isrc, spit * shei * 4);
		return 0;
	}

	if (mfd->fsYIQ == BST_CHECKED)
		fxShift_GenTabs(FORMAT_YIQ, mfd);
	else
		fxShift_GenTabs(FORMAT_YUV, mfd);

	if (mfd->fsYccIn == BST_UNCHECKED) {
		// RGB input
		fxShift_RGB2YCC(isrc, itm1, swid, shei, spit, mfd);
		// If output is Ycc, change pointer from temporary to destination
		if (mfd->fsYccOut == BST_CHECKED) itm2 = idst;
		if (mfd->fsLeave == BST_CHECKED)
			fxShift_ShiftLeave(itm1, itm2, swid, shei, spit, mfd);
		else
			fxShift_Shift(itm1, itm2, swid, shei, spit, mfd);
		// If output is RGB, convert Ycc to RGB
		if (mfd->fsYccOut == BST_UNCHECKED)
			fxShift_YCC2RGB(itm2, idst, swid, shei, spit, mfd);
	} else {
		// Ycc input
		// If output is Ycc, change pointer from temporary to destination
		if (mfd->fsYccOut == BST_CHECKED) itm1 = idst;
		if (mfd->fsLeave == BST_CHECKED)
			fxShift_ShiftLeave(isrc, itm1, swid, shei, spit, mfd);
		else
			fxShift_Shift(isrc, itm1, swid, shei, spit, mfd);
		// If output is RGB, convert Ycc to RGB
		if (mfd->fsYccOut == BST_UNCHECKED)
			fxShift_YCC2RGB(itm1, idst, swid, shei, spit, mfd);
	}

//	fxShift_Shift(isrc, idst, swid, shei, spit, mfd);
	return 0;
}

///////////////////////////////////////////////////////////////////////////

void	fxShift_Shift(Pixel32 *isrc, Pixel32 *idst, long swid, long shei, long spit, MyFilterData *mfd) {
	Pixel32		*src1, *src2, *dst;
	long		x, y, yy, xx;
	long		lf, rt, to, bt, ho, vo, pho;
	long		rmask, wmask;
	_int64		rmmask, wmmask;
	const long	sw1 = swid - 1, sh1 = shei - 1;

	pho = mfd->fsHOff;
	ho = mfd->fsHOff * (mfd->fsHNeg == BST_CHECKED ? 1 : -1);
	vo = mfd->fsVOff * (mfd->fsVNeg == BST_CHECKED ? -1 : 1);

	lf = ho; to = vo;
	rt = sw1 + ho; bt = sh1 + vo;
	if (lf < 0) lf = 0;
	else if (lf > sw1) lf = sw1;
	if (rt < 0) rt = 0;
	else if (rt > sw1) rt = sw1;
	if (to < 0) to = 0;
	else if (to > sh1) to = sh1;
	if (bt < 0) bt = 0;
	else if (bt > sh1) bt = sh1;
	yy = 0;

	if (mfd->mmx) {
		// Create 64bit I/O masks for MMX operations
		rmmask = 0x0000000000000000i64; wmmask = 0x00FFFFFF00FFFFFFi64;
		if (mfd->fsFld1 == BST_CHECKED) { rmmask |= 0x00FF000000FF0000i64; wmmask &= 0x0000FFFF0000FFFFi64; }
		if (mfd->fsFld2 == BST_CHECKED) { rmmask |= 0x0000FF000000FF00i64; wmmask &= 0x00FF00FF00FF00FFi64; }
		if (mfd->fsFld3 == BST_CHECKED) { rmmask |= 0x000000FF000000FFi64; wmmask &= 0x00FFFF0000FFFF00i64; }
		__asm movq	mm2, [rmmask];
		__asm movq	mm3, [wmmask];

		for (y = 0; y < shei; y++) {
			yy = y + vo;
			if (yy < 0) yy = 0;
			else if (yy > sh1) yy = sh1;
			x = ho; xx = 0;
			if (x < 0) { x = 0; xx = pho; }
			else if (x > sw1) x = sw1;
			src1 = isrc + yy * spit + x;
			src2 = isrc + y * spit + xx;
			dst = idst + y * spit + xx;
			x = swid - pho;
			__asm {
				mov		ecx, [x]
				shr		ecx, 1
				mov		eax, [src1]
				mov		ebx, [src2]
				mov		edx, [dst]

sm10:			movq	mm0, [eax]
				add		eax, 8
				pand	mm0, mm2
				movq	mm1, [ebx]
				add		ebx, 8
				pand	mm1, mm3
				por		mm0, mm1
				movq	[edx], mm0
				add		edx, 8

				dec		ecx
				jnz		sm10
			}
			if (pho) {
				if (mfd->fsHNeg == BST_CHECKED) { x = swid - 2; xx = swid - pho - 2; }
				else {x = 0; xx = 0; }
				src1 = isrc + yy * spit + x;
				src2 = isrc + y * spit + xx;
				dst = idst + y * spit + xx;
				x = pho;
				__asm {
					mov		ecx, [x]
					shr		ecx, 1
					inc		ecx
					mov		ebx, [src1]
					movq	mm4, [ebx]
					mov		ebx, [src2]
					mov		edx, [dst]

sm20:				movq	mm0, mm4
					pand	mm0, mm2
					movq	mm1, [ebx]
					add		ebx, 8
					pand	mm1, mm3
					por		mm0, mm1
					movq	[edx], mm0
					add		edx, 8

					dec		ecx
					jnz		sm20
				}
			}
		}
		__asm emms;
	} else {
		// Create 32bit I/O masks for non-MMX operations
		rmask = 0x00000000L; wmask = 0x00FFFFFFL;
		if (mfd->fsFld1 == BST_CHECKED) { rmask |= 0x00FF0000L; wmask &= 0x0000FFFFL; }
		if (mfd->fsFld2 == BST_CHECKED) { rmask |= 0x0000FF00L; wmask &= 0x00FF00FFL; }
		if (mfd->fsFld3 == BST_CHECKED) { rmask |= 0x000000FFL; wmask &= 0x00FFFF00L; }

		for (y = 0; y < shei; y++) {
			yy = y + vo;
			if (yy < 0) yy = 0;
			else if (yy > sh1) yy = sh1;
			x = ho; xx = 0;
			if (x < 0) { x = 0; xx = pho; }
			else if (x > sw1) x = sw1;
			src1 = isrc + yy * spit + x;
			src2 = isrc + y * spit + xx;
			dst = idst + y * spit + xx;
			x = swid - pho;
			__asm {
				push	esi
				push	edi
				mov		ecx, [x]
				mov		esi, [rmask]
				mov		edi, [wmask]

sh10:			mov		ebx, [src1]
				add		[src1], 4
				mov		eax, [ebx]
				and		eax, esi
				mov		ebx, [src2]
				add		[src2], 4
				mov		edx, [ebx]
				and		edx, edi
				or		eax, edx
				mov		ebx, [dst]
				add		[dst], 4
				mov		[ebx], eax

				dec		ecx
				jnz		sh10
				pop		edi
				pop		esi
			}
			if (pho) {
				if (mfd->fsHNeg == BST_CHECKED) { x = swid - 1; xx = swid - pho; }
				else {x = 0; xx = 0; }
				src1 = isrc + yy * spit + x;
				src2 = isrc + y * spit + xx;
				dst = idst + y * spit + xx;
				x = pho;
				__asm {
					push	esi
					push	edi
					mov		ecx, [x]
					mov		esi, [rmask]
					mov		edi, [wmask]

sh20:				mov		ebx, [src1]
					mov		eax, [ebx]
					and		eax, esi
					mov		ebx, [src2]
					add		[src2], 4
					mov		edx, [ebx]
					and		edx, edi
					or		eax, edx
					mov		ebx, [dst]
					add		[dst], 4
					mov		[ebx], eax

					dec		ecx
					jnz		sh20
					pop		edi
					pop		esi
				}
			}
		}
	}
}

///////////////////////////////////////////////////////////////////////////

void	fxShift_ShiftLeave(Pixel32 *isrc, Pixel32 *idst, long swid, long shei, long spit, MyFilterData *mfd) {
}

///////////////////////////////////////////////////////////////////////////

void	fxShift_RGB2YCC(Pixel32 *isrc, Pixel32 *idst, long swid, long shei, long spit, MyFilterData *mfd) {
	Pixel32	*src, *dst;
	long	y, c;
	long	*ytab = (long *)mfd->ytab;
	long	*itab = (long *)mfd->itab;
	long	*qtab = (long *)mfd->qtab;

	if (mfd->fsYIQ == BST_CHECKED) {
		// Convert RGB -> YIQ
		for (y = 0; y < shei; y++) {
			src = isrc + y * spit;
			dst = idst + y * spit;
			__asm {
				push	esi
				push	edi
				mov		ecx, [swid]

cty10:			mov		esi, [src]
				add		[src], 4
				mov		edx, [esi]
				mov		ebx, edx
				shr		ebx, 6
				and		ebx, 0x0003FCL
				mov		eax, edx
				shr		eax, 14
				and		eax, 0x0003FCL
				shl		edx, 2
				and		edx, 0x0003FCL
				add		ebx, 1024
				add		edx, 2048

				mov		esi, [ytab]
				mov		edi, [esi + eax]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cty20
				test	edi, 0xFF000000L
				jz		cty21
				mov		edi, 0x00FF0000L
				jmp		cty21
cty20:			xor		edi, edi
cty21:			mov		[c], edi

				mov		esi, [itab]
				mov		edi, [esi + eax]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cty30
				test	edi, 0xFF000000L
				jz		cty31
				mov		edi, 0x00FF0000L
				jmp		cty31
cty30:			xor		edi, edi
cty31:			shr		edi, 8
				or		[c], edi

				mov		esi, [qtab]
				mov		edi, [esi + eax]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cty40
				test	edi, 0xFF000000L
				jz		cty41
				mov		edi, 0x00FF0000L
cty40:			xor		edi, edi
cty41:			shr		edi, 16
				or		[c], edi

				mov		edi, [dst]
				add		[dst], 4
				mov		eax, [c]
				mov		[edi], eax

				dec		ecx
				jnz		cty10
				pop		edi
				pop		esi
			}
		}
	} else {
		// Convert RGB -> YUV
		for (y = 0; y < shei; y++) {
			src = isrc + y * spit;
			dst = idst + y * spit;
			__asm {
				push	esi
				push	edi
				mov		ecx, [swid]

ctu10:			mov		esi, [src]
				add		[src], 4
				mov		edx, [esi]
				mov		ebx, edx
				shr		ebx, 6
				and		ebx, 0x0003FCL
				mov		eax, edx
				shr		eax, 14
				and		eax, 0x0003FCL
				shl		edx, 2
				and		edx, 0x0003FCL
				add		ebx, 1024
				add		edx, 2048

				mov		esi, [ytab]
				mov		edi, [esi + eax]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		ctu20
				test	edi, 0xFF000000L
				jz		ctu21
				mov		edi, 0x00FF0000L
				jmp		ctu21
ctu20:			xor		edi, edi
ctu21:			mov		[c], edi

				mov		esi, [itab]
				mov		edi, [esi + eax]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		ctu30
				test	edi, 0xFF000000L
				jz		ctu31
				mov		edi, 0x00FF0000L
				jmp		ctu31
ctu30:			xor		edi, edi
ctu31:			shr		edi, 8
				or		[c], edi

				mov		esi, [qtab]
				mov		edi, [esi + eax]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		ctu40
				test	edi, 0xFF000000L
				jz		ctu41
				mov		edi, 0x00FF0000L
				jmp		ctu41
ctu40:			xor		edi, edi
ctu41:			shr		edi, 16
				or		[c], edi

				mov		edi, [dst]
				add		[dst], 4
				mov		eax, [c]
				mov		[edi], eax

				dec		ecx
				jnz		ctu10
				pop		edi
				pop		esi
			}
		}
	}
}

///////////////////////////////////////////////////////////////////////////

void	fxShift_YCC2RGB(Pixel32 *isrc, Pixel32 *idst, long swid, long shei, long spit, MyFilterData *mfd) {
	Pixel32	*src, *dst;
	long	y, c;
	long	*rtab = (long *)mfd->rtab;
	long	*gtab = (long *)mfd->gtab;
	long	*btab = (long *)mfd->btab;
	long	*cytf = (long *)mfd->cytf;

	if (mfd->fsYIQ == BST_CHECKED) {
		// Convert YIQ -> RGB
		for (y = 0; y < shei; y++) {
			src = isrc + y * spit;
			dst = idst + y * spit;
			__asm {
				push	esi
				push	edi
				mov		ecx, [swid]

cfy10:			mov		esi, [src]
				add		[src], 4
				mov		edx, [esi]
				mov		ebx, edx
				shr		ebx, 6
				and		ebx, 0x0003FCL
				mov		eax, edx
				and		eax, 0x00FF0000L
				shl		edx, 2
				and		edx, 0x0003FCL
				add		edx, 1024

				mov		edi, eax
				mov		esi, [rtab]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cfy20
				test	edi, 0xFF000000L
				jz		cfy21
				mov		edi, 0x00FF0000L
				jmp		cfy21
cfy20:			xor		edi, edi
cfy21:			mov		[c], edi

				mov		edi, eax
				mov		esi, [gtab]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cfy30
				test	edi, 0xFF000000L
				jz		cfy31
				mov		edi, 0x00FF0000L
				jmp		cfy31
cfy30:			xor		edi, edi
cfy31:			shr		edi, 8
				or		[c], edi

				mov		edi, eax
				mov		esi, [btab]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cfy40
				test	edi, 0xFF000000L
				jz		cfy41
				mov		edi, 0x00FF0000L
				jmp		cfy41
cfy40:			xor		edi, edi
cfy41:			shr		edi, 16
				or		[c], edi

				mov		edi, [dst]
				add		[dst], 4
				mov		eax, [c]
				mov		[edi], eax

				dec		ecx
				jnz		cfy10
				pop		edi
				pop		esi
			}
		}
	} else {
		// Convert YUV -> RGB
		for (y = 0; y < shei; y++) {
			src = isrc + y * spit;
			dst = idst + y * spit;
			__asm {
				push	esi
				push	edi
				mov		ecx, [swid]

cfu10:			mov		esi, [src]
				add		[src], 4
				mov		edx, [esi]
				mov		ebx, edx
				shr		ebx, 6
				and		ebx, 0x0003FCL
				mov		eax, edx
				shr		eax, 14
				and		eax, 0x0003FCL
				shl		edx, 2
				and		edx, 0x0003FCL
				add		edx, 1024

				mov		esi, [cytf]
				mov		eax, [esi + eax]

				mov		edi, eax
				mov		esi, [rtab]
				add		edi, [esi + ebx]
				and		edi, 0xFFFF0000L
				js		cfu20
				test	edi, 0xFF000000L
				jz		cfu21
				mov		edi, 0x00FF0000L
				jmp		cfu21
cfu20:			xor		edi, edi
cfu21:			mov		[c], edi

				mov		edi, eax
				mov		esi, [gtab]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cfu30
				test	edi, 0xFF000000L
				jz		cfu31
				mov		edi, 0x00FF0000L
				jmp		cfu31
cfu30:			xor		edi, edi
cfu31:			shr		edi, 8
				or		[c], edi

				mov		edi, eax
				mov		esi, [btab]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cfu40
				test	edi, 0xFF000000L
				jz		cfu41
				mov		edi, 0x00FF0000L
				jmp		cfu41
cfu40:			xor		edi, edi
cfu41:			shr		edi, 16
				or		[c], edi

				mov		edi, [dst]
				add		[dst], 4
				mov		eax, [c]
				mov		[edi], eax

				dec		ecx
				jnz		cfu10
				pop		edi
				pop		esi
			}
		}
	}
}

///////////////////////////////////////////////////////////////////////////

void	fxShift_GenTabs(long format, MyFilterData *mfd) {
	long	d;

	// Calculate data for RGB -> Ycc conversion tables
	for (d = 0; d < 256; d++) {
		if (mfd->fsYIQ == BST_CHECKED) {
			// RGB -> YIQ transformation matrix
			mfd->ytab[d      ] = (long)(+0.299 * d * 65536 + 0.5);
			mfd->ytab[d + 256] = (long)(+0.587 * d * 65536 + 0.5);
			mfd->ytab[d + 512] = (long)(+0.114 * d * 65536 + 0.5);
			mfd->itab[d      ] = (long)(+0.596 * d * 65536 + 0.5) + 8388608;
			mfd->itab[d + 256] = (long)(-0.275 * d * 65536 + 0.5);
			mfd->itab[d + 512] = (long)(-0.321 * d * 65536 + 0.5);
			mfd->qtab[d      ] = (long)(+0.212 * d * 65536 + 0.5) + 8388608;
			mfd->qtab[d + 256] = (long)(-0.523 * d * 65536 + 0.5);
			mfd->qtab[d + 512] = (long)(+0.311 * d * 65536 + 0.5);
			// YIQ -> RGB transformation matrix
			mfd->rtab[d      ] = (long)(+0.956 * (d-128) * 65536 + 0.5);
			mfd->rtab[d + 256] = (long)(+0.621 * (d-128) * 65536 + 0.5);
			mfd->gtab[d      ] = (long)(-0.272 * (d-128) * 65536 + 0.5);
			mfd->gtab[d + 256] = (long)(-0.647 * (d-128) * 65536 + 0.5);
			mfd->btab[d      ] = (long)(-1.105 * (d-128) * 65536 + 0.5);
			mfd->btab[d + 256] = (long)(+1.702 * (d-128) * 65536 + 0.5);
		} else {
			// RGB -> YUV transformation matrix
			mfd->ytab[d      ] = (long)(+0.257 * d * 65536 + 0.5) + 1048576;
			mfd->ytab[d + 256] = (long)(+0.504 * d * 65536 + 0.5);
			mfd->ytab[d + 512] = (long)(+0.098 * d * 65536 + 0.5);
			mfd->itab[d      ] = (long)(+0.439 * d * 65536 + 0.5) + 8388608;
			mfd->itab[d + 256] = (long)(-0.368 * d * 65536 + 0.5);
			mfd->itab[d + 512] = (long)(-0.071 * d * 65536 + 0.5);
			mfd->qtab[d      ] = (long)(-0.148 * d * 65536 + 0.5) + 8388608;
			mfd->qtab[d + 256] = (long)(-0.291 * d * 65536 + 0.5);
			mfd->qtab[d + 512] = (long)(+0.439 * d * 65536 + 0.5);
			// YUV -> RGB transformation matrix
			mfd->rtab[d      ] = (long)(+1.596 * (d-128) * 65536 + 0.5);
			mfd->rtab[d + 256] = 0;
			mfd->gtab[d      ] = (long)(-0.813 * (d-128) * 65536 + 0.5);
			mfd->gtab[d + 256] = (long)(-0.391 * (d-128) * 65536 + 0.5);
			mfd->btab[d      ] = 0;
			mfd->btab[d + 256] = (long)(+2.018 * (d-128) * 65536 + 0.5);
			mfd->cytf[d] = (long)(((d - 16) * 1.164) * 65536.0 + 0.5);	// Y transform
		}
	}
}

///////////////////////////////////////////////////////////////////////////

int fxShiftStartProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	long	d;

	mfd->mmx = ff->isMMXEnabled();
	if (fa->src.w & 1) mfd->mmx = 0;
	//mfd->mmx = 0;

	// Temporary frame buffers
	d = (fa->src.pitch >> 2) * fa->src.h;
	if (!(mfd->TmpBuf1 = new Pixel32[d])) return 1;
	if (!(mfd->TmpBuf2 = new Pixel32[d])) return 1;

	// Y transform for YUV -> RGB conversion
	if (!(mfd->cytf = new long[256])) return 1;

	// RGB -> Ycc conversion tables
	if (!(mfd->ytab = new long[768])) return 1;
	if (!(mfd->itab = new long[768])) return 1;
	if (!(mfd->qtab = new long[768])) return 1;

	// Ycc -> RGB conversion tables
	if (!(mfd->rtab = new long[512])) return 1;
	if (!(mfd->gtab = new long[512])) return 1;
	if (!(mfd->btab = new long[512])) return 1;

	return 0;
}

///////////////////////////////////////////////////////////////////////////

int fxShiftEndProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	// Delete the temporary frame buffers
	delete[] mfd->TmpBuf1; mfd->TmpBuf1 = NULL;
	delete[] mfd->TmpBuf2; mfd->TmpBuf2 = NULL;

	// Delete RGB -> YIQ/YUV conversion tables
	delete[] mfd->ytab; mfd->ytab = NULL;
	delete[] mfd->itab; mfd->itab = NULL;
	delete[] mfd->qtab; mfd->qtab = NULL;
	
	delete[] mfd->rtab; mfd->rtab = NULL;
	delete[] mfd->gtab; mfd->gtab = NULL;
	delete[] mfd->btab; mfd->btab = NULL;

	return 0;
}

///////////////////////////////////////////////////////////////////////////

int fxShiftInitProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->fsHOff = 2;
	mfd->fsHNeg = BST_CHECKED;
	mfd->fsVOff = 0;
	mfd->fsVNeg = BST_UNCHECKED;
	mfd->fsLeave = BST_UNCHECKED;
	mfd->fsFld1 = BST_UNCHECKED;
	mfd->fsFld2 = BST_CHECKED;
	mfd->fsFld3 = BST_CHECKED;
	mfd->fsYccIn = BST_UNCHECKED;
	mfd->fsYccOut = BST_UNCHECKED;
	mfd->fsYIQ = BST_UNCHECKED;

	return 0;
}

///////////////////////////////////////////////////////////////////////////

BOOL CALLBACK fxShiftConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
	MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

	switch(msg) {
	case WM_INITDIALOG:
		SetWindowLong(hdlg, DWL_USER, lParam);
		mfd = (MyFilterData *)lParam;
		SetDlgItemInt(hdlg, IDC_FS_HOFF, mfd->fsHOff, FALSE);
		SetDlgItemInt(hdlg, IDC_FS_VOFF, mfd->fsVOff, FALSE);
		CheckDlgButton(hdlg, IDC_FS_HNEG, mfd->fsHNeg ? BST_CHECKED : BST_UNCHECKED);
		CheckDlgButton(hdlg, IDC_FS_VNEG, mfd->fsVNeg ? BST_CHECKED : BST_UNCHECKED);
		CheckDlgButton(hdlg, IDC_FS_LEAVE, mfd->fsLeave ? BST_CHECKED : BST_UNCHECKED);
		CheckDlgButton(hdlg, IDC_FS_FLD1, mfd->fsFld1 ? BST_CHECKED : BST_UNCHECKED);
		CheckDlgButton(hdlg, IDC_FS_FLD2, mfd->fsFld2 ? BST_CHECKED : BST_UNCHECKED);
		CheckDlgButton(hdlg, IDC_FS_FLD3, mfd->fsFld3 ? BST_CHECKED : BST_UNCHECKED);
		CheckDlgButton(hdlg, IDC_FS_YCCIN, mfd->fsYccIn ? BST_CHECKED : BST_UNCHECKED);
		CheckDlgButton(hdlg, IDC_FS_YCCOUT, mfd->fsYccOut ? BST_CHECKED : BST_UNCHECKED);
		CheckDlgButton(hdlg, IDC_FS_YIQ, mfd->fsYIQ ? BST_CHECKED : BST_UNCHECKED);
		return TRUE;

	case WM_COMMAND:
		switch(LOWORD(wParam)) {
		case IDOK:
			mfd->fsHOff = GetDlgItemInt(hdlg, IDC_FS_HOFF, &mfd->fsHOff, FALSE);
			mfd->fsHNeg = !!IsDlgButtonChecked(hdlg, IDC_FS_HNEG);
			mfd->fsVOff = GetDlgItemInt(hdlg, IDC_FS_VOFF, &mfd->fsVOff, FALSE);
			mfd->fsVNeg = !!IsDlgButtonChecked(hdlg, IDC_FS_VNEG);
			mfd->fsLeave = !!IsDlgButtonChecked(hdlg, IDC_FS_LEAVE);
			mfd->fsFld1 = !!IsDlgButtonChecked(hdlg, IDC_FS_FLD1);
			mfd->fsFld2 = !!IsDlgButtonChecked(hdlg, IDC_FS_FLD2);
			mfd->fsFld3 = !!IsDlgButtonChecked(hdlg, IDC_FS_FLD3);
			mfd->fsYccIn = !!IsDlgButtonChecked(hdlg, IDC_FS_YCCIN);
			mfd->fsYccOut = !!IsDlgButtonChecked(hdlg, IDC_FS_YCCOUT);
			mfd->fsYIQ = !!IsDlgButtonChecked(hdlg, IDC_FS_YIQ);
			EndDialog(hdlg, 0);

			return TRUE;
		case IDCANCEL:
			EndDialog(hdlg, 1);
			return FALSE;
		}
		break;
	}

	return FALSE;
}

///////////////////////////////////////////////////////////////////////////

int fxShiftConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd) {
	return DialogBoxParam(fa->filter->module->hInstModule,
	       MAKEINTRESOURCE(IDD_FXSHIFT_CONFIG), hwnd,
		   fxShiftConfigDlgProc, (LPARAM)fa->filter_data);
}

///////////////////////////////////////////////////////////////////////////

void fxShiftStringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	char	fstr[10];

	if (mfd->fsYIQ == BST_CHECKED)
		sprintf(fstr, "YIQ");
	else
		sprintf(fstr, "YUV");

	sprintf(str, ": H-Shift:%s%d V-Shift:%s%d Fields:%s%s%s %s -> %s", mfd->fsHNeg == BST_CHECKED ? "-" : "", mfd->fsHOff, 
		mfd->fsVNeg == BST_CHECKED ? "-" : "", mfd->fsVOff, mfd->fsFld1 == BST_CHECKED ? "1" : "", 
		mfd->fsFld2 == BST_CHECKED ? "2" : "", mfd->fsFld3 == BST_CHECKED ? "3" : "", mfd->fsYccIn == BST_CHECKED ? fstr : "RGB",
		mfd->fsYccOut == BST_CHECKED ? fstr : "RGB");
}

///////////////////////////////////////////////////////////////////////////

void fxShiftScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
    FilterActivation *fa = (FilterActivation *)lpVoid;
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->fsHOff = argv[0].asInt();
	mfd->fsHNeg = mfd->fsHOff < 0 ? BST_CHECKED: BST_UNCHECKED;
	mfd->fsHOff = abs(mfd->fsHOff);
	mfd->fsVOff = argv[1].asInt();
	mfd->fsVNeg = mfd->fsVOff < 0 ? BST_CHECKED: BST_UNCHECKED;
	mfd->fsVOff = abs(mfd->fsVOff);
	mfd->fsLeave = argv[2].asInt() == 1 ? BST_CHECKED : BST_UNCHECKED;
	mfd->fsFld1 = argv[3].asInt() == 1 ? BST_CHECKED : BST_UNCHECKED;
	mfd->fsFld2 = argv[4].asInt() == 1 ? BST_CHECKED : BST_UNCHECKED;
	mfd->fsFld3 = argv[5].asInt() == 1 ? BST_CHECKED : BST_UNCHECKED;
	mfd->fsYccIn = argv[6].asInt() == 1 ? BST_CHECKED : BST_UNCHECKED;
	mfd->fsYccOut = argv[7].asInt() == 1 ? BST_CHECKED : BST_UNCHECKED;
	mfd->fsYIQ = argv[8].asInt() == 1 ? BST_CHECKED : BST_UNCHECKED;
}

///////////////////////////////////////////////////////////////////////////

bool fxShiftFssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

    _snprintf(buf, buflen, "Config(%d, %d, %d, %d, %d, %d, %d, %d, %d)", mfd->fsHOff * (mfd->fsHNeg == BST_CHECKED ? -1 : 1),
		mfd->fsVOff * (mfd->fsVNeg == BST_CHECKED ? -1 : 1), mfd->fsLeave == BST_CHECKED ? 1 : 0, mfd->fsFld1 == BST_CHECKED ? 1 : 0,
		mfd->fsFld2 == BST_CHECKED ? 1 : 0, mfd->fsFld3 == BST_CHECKED ? 1 : 0, mfd->fsYccIn == BST_CHECKED ? 1 : 0,
		mfd->fsYccOut == BST_CHECKED ? 1 : 0, mfd->fsYIQ == BST_CHECKED ? 1 : 0);

    return true;
}