/*
    Unsharp Mask Filter for VirtualDub -- performs contrast enhancement or "sharpening".
	Developed by Donald Graft and reworked by Antonio Foranna.

	Copyright (C) 1999-2000 Donald A. Graft (neuron2@home.com)
	Copyright (C) 2003 Donald A. Graft/Antonio Foranna (kreel@tiscali.it)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <windows.h>
#include <commctrl.h>
#include <stdio.h>
#include <crtdbg.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"
#include "filter.h"

#include "resource.h"

///////////////////////////////////////////////////////////////////////////

int RunProc(const FilterActivation *fa, const FilterFunctions *ff);
int StartProc(FilterActivation *fa, const FilterFunctions *ff);
int EndProc(FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int InitProc(FilterActivation *fa, const FilterFunctions *ff);
int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

typedef struct MyFilterData
{
	IFilterPreview		*ifp;
	int		diameter;
	int		strength;
	int		threshold;
	int		interlaced;
	int		bypass;
	int		xloClip, xhiClip, yloClip, yhiClip;
	DWORD	DivTab[256];
} MyFilterData;

bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	_snprintf(buf, buflen, "Config(%d, %d, %d, %d, %d, %d, %d, %d)",
		mfd->diameter,
		mfd->strength,
		mfd->threshold,
		mfd->xloClip, mfd->xhiClip, mfd->yloClip, mfd->yhiClip,
		mfd->interlaced);

	return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->diameter		= argv[0].asInt();
	mfd->strength		= argv[1].asInt();
	mfd->threshold		= argv[2].asInt();
	mfd->xloClip		= argv[3].asInt();
	mfd->xhiClip		= argv[4].asInt();
	mfd->yloClip		= argv[5].asInt();
	mfd->yhiClip		= argv[6].asInt();
	mfd->interlaced		= !!argv[7].asInt();
}

ScriptFunctionDef func_defs[]={
	{ (ScriptFunctionPtr)ScriptConfig, "Config", "0iiiiiiii" },
	{ NULL },
};

CScriptObject script_obj={
	NULL, func_defs
};

struct FilterDefinition filterDef_tutorial = {

	NULL, NULL, NULL,		// next, prev, module
	"Unsharp mask",	// name
	"Enhance edges using unsharp mask.",
							// desc
	"Antonio Foranna/Donald Graft", 		// maker
	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size

	InitProc,				// initProc
	NULL,					// deinitProc
	RunProc,				// runProc
	NULL,					// paramProc
	ConfigProc, 			// configProc
	StringProc, 			// stringProc
	StartProc,				// startProc
	EndProc,				// endProc

	&script_obj,			// script_obj
	FssProc,				// fssProc

};

int StartProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	return 0;
}

int EndProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	return 0;
}

///////////////////////////////////////////////////////////////////////////

#define CLIP(a) ((a)<0 ? 0 : (a)>255 ? 255 : (a))

inline void SharpenPixel(BYTE *bsrc, BYTE *bdst, DWORD SumR, DWORD SumG, DWORD SumB, DWORD Mul, WORD Strength, DWORD Threshold)
{
	// blurred pixel
	SumR=(SumR*Mul)>>16;
	SumG=(SumG*Mul)>>16;
	SumB=(SumB*Mul)>>16;

	if(	(DWORD)((bsrc[0]-(long)SumR)*(bsrc[0]-(long)SumR)+
				(bsrc[1]-(long)SumG)*(bsrc[1]-(long)SumG)+
				(bsrc[2]-(long)SumB)*(bsrc[2]-(long)SumB)
				)>Threshold)
	{
	short R,G,B;

		// sharpen amount
		R=((bsrc[0]-(long)SumR)*Strength)>>8;
		G=((bsrc[1]-(long)SumG)*Strength)>>8;
		B=((bsrc[2]-(long)SumB)*Strength)>>8;
	
		// clipped output
		bdst[0]=CLIP(bsrc[0]+R);
		bdst[1]=CLIP(bsrc[1]+G);
		bdst[2]=CLIP(bsrc[2]+B);
	}
	else
		*(DWORD *)bdst=*(DWORD *)bsrc;
}
// -----------------------------------------------------------------------------------------------

void SharpenRow(Pixel32 *src, Pixel32 *dst, WORD yOffset, WORD w, WORD h, WORD SharpenStrength, MyFilterData *mfd)
{
BYTE	*bsrc,
		*bdst=(BYTE *)dst;
const DWORD	Threshold=mfd->threshold*mfd->threshold*3;
DWORD	SumR=0,SumG=0,SumB=0;
DWORD	diameter=mfd->diameter;
const DWORD
		Interlaced=mfd->interlaced,
		ImageWidth=(WORD)w<<Interlaced,
		ImageWidthx4=ImageWidth<<2,
		radius=diameter>>1,
		radiusx4=radius<<2,
		Top=ImageWidth*(yOffset>(WORD)(radius<<Interlaced) ? radius : yOffset>>Interlaced),
		Bot=ImageWidth*((yOffset+(radius<<Interlaced)+1)<(WORD)h ? radius : (h-yOffset)>>Interlaced);
WORD	in_width=(WORD)(w-diameter-1);
WORD	cnt=0;
WORD	i;

	if(w<diameter)
		diameter=(BYTE)w;

	// Init sum
	for(bsrc=(BYTE *)(src-Top); bsrc<=(BYTE *)(src+Bot); bsrc+=ImageWidthx4)
		for(i=0; i<radiusx4; i+=4)
		{
			SumR+=*(bsrc+i);
			SumG+=*(bsrc+i+1);
			SumB+=*(bsrc+i+2);
			cnt++;
		}

	// left column -------------------------------------------------------------------------------
	for(i=0; i<=radius; i++)
	{
		for(bsrc=(BYTE *)(src-Top); bsrc<=(BYTE *)(src+Bot); bsrc+=ImageWidthx4)
		{
			SumR+=*(bsrc+radiusx4);
			SumG+=*(bsrc+radiusx4+1);
			SumB+=*(bsrc+radiusx4+2);
			cnt++;
		}
		SharpenPixel((BYTE *)src,bdst,SumR,SumG,SumB,mfd->DivTab[cnt],SharpenStrength,Threshold);
		src++;
		bdst+=4;
	}
	in_width--;

	// mid pixels --------------------------------------------------------------------------------
	for(i=0; i<in_width; i++)
	{
		for(bsrc=(BYTE *)(src-Top); bsrc<=(BYTE *)(src+Bot); bsrc+=ImageWidthx4)
		{
			SumR+=*(bsrc+radiusx4)  -*(bsrc-radiusx4-4);
			SumG+=*(bsrc+radiusx4+1)-*(bsrc-radiusx4-3);
			SumB+=*(bsrc+radiusx4+2)-*(bsrc-radiusx4-2);
		}
		SharpenPixel((BYTE *)src,bdst,SumR,SumG,SumB,mfd->DivTab[cnt],SharpenStrength,Threshold);
		src++;
		bdst+=4;
	}

	// right column ------------------------------------------------------------------------------
	for(i=0; i<radius; i++)
	{
		for(bsrc=(BYTE *)(src-Top); bsrc<=(BYTE *)(src+Bot); bsrc+=ImageWidthx4)
		{
			SumR-=*(bsrc-radiusx4-4);
			SumG-=*(bsrc-radiusx4-3);
			SumB-=*(bsrc-radiusx4-2);
			cnt--;
		}
		SharpenPixel((BYTE *)src,bdst,SumR,SumG,SumB,mfd->DivTab[cnt],SharpenStrength,Threshold);
		src++;
		bdst+=4;
	}
}
// -----------------------------------------------------------------------------------------------

int RunProc(const FilterActivation *fa, const FilterFunctions *ff)
{
MyFilterData *mfd = (MyFilterData *)fa->filter_data;
Pixel32 *src=fa->src.data, *dst=fa->dst.data;
const PixDim	w = fa->src.w;
const PixDim	h = fa->src.h;

WORD SharpenStrength;
	if(mfd->strength>=0)
		SharpenStrength=255*(mfd->strength+1);
	else
		SharpenStrength=255/(-mfd->strength+1);

	if(mfd->bypass)
	{
		memcpy(dst,src,w*h*sizeof(Pixel32));
		return 0;
	}
	else
		for(WORD yOffset=0; yOffset<h; yOffset++, src+=w, dst+=w)
			SharpenRow(src,dst,yOffset,(WORD)w,(WORD)h,SharpenStrength,mfd);

	// Pass through the areas we clipped.

	int x, y;
	int xloClip = mfd->xloClip;
	if (xloClip >= w) xloClip = w;
	int xhiClip = mfd->xhiClip;
	if (xhiClip >= w) xhiClip = w;
	int yloClip = mfd->yloClip;
	if (yloClip >= h) yloClip = h;
	int yhiClip = mfd->yhiClip;
	if (yhiClip >= h) yhiClip = h;

	src	= fa->src.data;
	dst = fa->dst.data;
	for (y = 0; y < yloClip; y++)
	{
		for (x = 0; x < w; x++)
		{
			dst[x] = src[x];
		}
		src	= (Pixel *)((char *)src + fa->src.pitch);
		dst = (Pixel *)((char *)dst + fa->dst.pitch);
	}
	src	= fa->src.data;
	dst = fa->dst.data;
	for (y = 0; y < h; y++)
	{
		for (x = 0; x < xloClip; x++)
		{
			dst[x] = src[x];
		}
		src	= (Pixel *)((char *)src + fa->src.pitch);
		dst = (Pixel *)((char *)dst + fa->dst.pitch);
	}
	src	= fa->src.data;
	dst = fa->dst.data;
	for (y = 0; y < h; y++)
	{
		for (x = w - xhiClip; x < w; x++)
		{
			dst[x] = src[x];
		}
		src	= (Pixel *)((char *)src + fa->src.pitch);
		dst = (Pixel *)((char *)dst + fa->dst.pitch);
	}
	src	= (Pixel *)((char *)fa->src.data + fa->src.pitch * (h - yhiClip));
	dst = (Pixel *)((char *)fa->dst.data + fa->dst.pitch * (h - yhiClip));
	for (y = h - yhiClip; y < h; y++)
	{
		for (x = 0; x < w; x++)
		{
			dst[x] = src[x];
		}
		src	= (Pixel *)((char *)src + fa->src.pitch);
		dst = (Pixel *)((char *)dst + fa->dst.pitch);
	}

	return 0;
}

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_tutorial;

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
	if (!(fd_tutorial = ff->addFilter(fm, &filterDef_tutorial, sizeof(FilterDefinition))))
		return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

	return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
	ff->removeFilter(fd_tutorial);
}

int InitProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->diameter = 5;
	mfd->strength = 0;
	mfd->threshold = 10;
	mfd->xloClip = 0;
	mfd->xhiClip = 0;
	mfd->yloClip = 0;
	mfd->yhiClip = 0;
	mfd->interlaced = FALSE;

	for(int i=1; i<256; i++)
		mfd->DivTab[i]=0x10000/i;
	mfd->DivTab[0]=0;

	return 0;
}

BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
	MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

	switch(msg) {
		case WM_INITDIALOG:
			SetWindowLong(hdlg, DWL_USER, lParam);
			mfd = (MyFilterData *)lParam;
			HWND hWnd;

			SetDlgItemInt(hdlg, IDC_X2CLIP, mfd->xhiClip, FALSE);
			SetDlgItemInt(hdlg, IDC_Y1CLIP, mfd->yloClip, FALSE);
			SetDlgItemInt(hdlg, IDC_Y2CLIP, mfd->yhiClip, FALSE);
			CheckDlgButton(hdlg, IDC_INTERLACED, mfd->interlaced ? BST_CHECKED : BST_UNCHECKED);

			hWnd = GetDlgItem(hdlg, IDC_SDIAMETER);
			SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(1, 5));
//			SendMessage(hWnd, TBM_SETTICFREQ, 1 , 0);
			SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->diameter/2);
			SetDlgItemInt(hdlg, IDC_DIAMETER, mfd->diameter, FALSE);

			hWnd = GetDlgItem(hdlg, IDC_SSTRENGTH);
			SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(-8, 8));
//			SendMessage(hWnd, TBM_SETTICFREQ, 10 , 0);
			SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->strength);
			SetDlgItemInt(hdlg, IDC_STRENGTH, mfd->strength, TRUE);

			hWnd = GetDlgItem(hdlg, IDC_STHRESHOLD);
			SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 255));
//			SendMessage(hWnd, TBM_SETTICFREQ, 5 , 0);
			SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->threshold);
			SetDlgItemInt(hdlg, IDC_THRESHOLD, mfd->threshold, FALSE);

			hWnd = GetDlgItem(hdlg, IDC_SX1CLIP);
			SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 255));
//			SendMessage(hWnd, TBM_SETTICFREQ, 5 , 0);
			SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->xloClip);
			SetDlgItemInt(hdlg, IDC_X1CLIP, mfd->xloClip, FALSE);

			hWnd = GetDlgItem(hdlg, IDC_SX2CLIP);
			SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 255));
//			SendMessage(hWnd, TBM_SETTICFREQ, 5 , 0);
			SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->xhiClip);
			SetDlgItemInt(hdlg, IDC_X2CLIP, mfd->xhiClip, FALSE);

			hWnd = GetDlgItem(hdlg, IDC_SY1CLIP);
			SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 255));
//			SendMessage(hWnd, TBM_SETTICFREQ, 5 , 0);
			SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->yloClip);
			SetDlgItemInt(hdlg, IDC_Y1CLIP, mfd->yloClip, FALSE);

			hWnd = GetDlgItem(hdlg, IDC_SY2CLIP);
			SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 255));
//			SendMessage(hWnd, TBM_SETTICFREQ, 5 , 0);
			SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->yhiClip);
			SetDlgItemInt(hdlg, IDC_Y2CLIP, mfd->yhiClip, FALSE);

			mfd->ifp->InitButton(GetDlgItem(hdlg, IDPREVIEW));

			return TRUE;
		case WM_HSCROLL:
			if ((HWND) lParam == GetDlgItem(hdlg, IDC_SDIAMETER))
			{
				int diameter = SendMessage(GetDlgItem(hdlg, IDC_SDIAMETER), TBM_GETPOS, 0, 0) * 2 + 1;
				if (diameter != mfd->diameter)
				{
					mfd->diameter = diameter;
					SetDlgItemInt(hdlg, IDC_DIAMETER, mfd->diameter, FALSE);
					mfd->ifp->RedoFrame();
				}
			}
			else if ((HWND) lParam == GetDlgItem(hdlg, IDC_SSTRENGTH))
			{
				int strength = SendMessage(GetDlgItem(hdlg, IDC_SSTRENGTH), TBM_GETPOS, 0, 0);
				if (strength != mfd->strength)
				{
					mfd->strength = strength;
					SetDlgItemInt(hdlg, IDC_STRENGTH, mfd->strength, TRUE);
					mfd->ifp->RedoFrame();
				}
			}
			else if ((HWND) lParam == GetDlgItem(hdlg, IDC_STHRESHOLD))
			{
				int threshold = SendMessage(GetDlgItem(hdlg, IDC_STHRESHOLD), TBM_GETPOS, 0, 0);
				if (threshold != mfd->threshold)
				{
					mfd->threshold = threshold;
					SetDlgItemInt(hdlg, IDC_THRESHOLD, mfd->threshold, FALSE);
					mfd->ifp->RedoFrame();
				}
			}
			else if ((HWND) lParam == GetDlgItem(hdlg, IDC_SX1CLIP))
			{
				int xloClip = SendMessage(GetDlgItem(hdlg, IDC_SX1CLIP), TBM_GETPOS, 0, 0);
				if (xloClip != mfd->xloClip)
				{
					mfd->xloClip = xloClip;
					SetDlgItemInt(hdlg, IDC_X1CLIP, mfd->xloClip, FALSE);
					mfd->ifp->RedoFrame();
				}
			}
			else if ((HWND) lParam == GetDlgItem(hdlg, IDC_SX2CLIP))
			{
				int xhiClip = SendMessage(GetDlgItem(hdlg, IDC_SX2CLIP), TBM_GETPOS, 0, 0);
				if (xhiClip != mfd->xhiClip)
				{
					mfd->xhiClip = xhiClip;
					SetDlgItemInt(hdlg, IDC_X2CLIP, mfd->xhiClip, FALSE);
					mfd->ifp->RedoFrame();
				}
			}
			else if ((HWND) lParam == GetDlgItem(hdlg, IDC_SY1CLIP))
			{
				int yloClip = SendMessage(GetDlgItem(hdlg, IDC_SY1CLIP), TBM_GETPOS, 0, 0);
				if (yloClip != mfd->yloClip)
				{
					mfd->yloClip = yloClip;
					SetDlgItemInt(hdlg, IDC_Y1CLIP, mfd->yloClip, FALSE);
					mfd->ifp->RedoFrame();
				}
			}
			else if ((HWND) lParam == GetDlgItem(hdlg, IDC_SY2CLIP))
			{
				int yhiClip = SendMessage(GetDlgItem(hdlg, IDC_SY2CLIP), TBM_GETPOS, 0, 0);
				if (yhiClip != mfd->yhiClip)
				{
					mfd->yhiClip = yhiClip;
					SetDlgItemInt(hdlg, IDC_Y2CLIP, mfd->yhiClip, FALSE);
					mfd->ifp->RedoFrame();
				}
			}
			break;
		case WM_COMMAND:
			switch(LOWORD(wParam))
			{
			case IDPREVIEW:
				mfd->ifp->Toggle(hdlg);
				break;
			case IDOK:
				EndDialog(hdlg, 0);
				return TRUE;
			case IDHELP:
				{
				char prog[256];
				char path[256];
				LPTSTR ptr;
				GetModuleFileName(NULL, prog, 255);
				GetFullPathName(prog, 255, path, &ptr);
				*ptr = 0;
				strcat(path, "plugins\\Unsharp.txt");
				OutputDebugString(path);
				OutputDebugString("\n");
				strcpy(prog, "Notepad ");
				strcat(prog, path);
				WinExec(prog, SW_SHOW);
				return TRUE;
				}
			case IDCANCEL:
				EndDialog(hdlg, 1);
				return TRUE;
			case IDC_INTERLACED:
				mfd->interlaced = !!IsDlgButtonChecked(hdlg, IDC_INTERLACED);
				mfd->ifp->RedoFrame();
				break;
			case IDC_BYPASS:
				mfd->bypass = !!IsDlgButtonChecked(hdlg, IDC_BYPASS);
				mfd->ifp->RedoFrame();
				break;
			case IDC_DIAMETERPLUS:
				if (mfd->diameter < 11)
				{
					mfd->diameter += 2;
					SetDlgItemInt(hdlg, IDC_DIAMETER, mfd->diameter, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_SDIAMETER), TBM_SETPOS, (WPARAM)TRUE, mfd->diameter/2);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_DIAMETERMINUS:
				if (mfd->diameter > 3)
				{
					mfd->diameter -= 2;
					SetDlgItemInt(hdlg, IDC_DIAMETER, mfd->diameter, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_SDIAMETER), TBM_SETPOS, (WPARAM)TRUE, mfd->diameter/2);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_STRENGTHPLUS:
				if (mfd->strength < 8)
				{
					mfd->strength += 1;
					SetDlgItemInt(hdlg, IDC_STRENGTH, mfd->strength, TRUE);
					SendMessage(GetDlgItem(hdlg, IDC_SSTRENGTH), TBM_SETPOS, (WPARAM)TRUE, mfd->strength);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_STRENGTHMINUS:
				if (mfd->strength > -8)
				{
					mfd->strength -= 1;
					SetDlgItemInt(hdlg, IDC_STRENGTH, mfd->strength, TRUE);
					SendMessage(GetDlgItem(hdlg, IDC_SSTRENGTH), TBM_SETPOS, (WPARAM)TRUE, mfd->strength);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_THRESHOLDPLUS:
				if (mfd->threshold < 100)
				{
					mfd->threshold += 1;
					SetDlgItemInt(hdlg, IDC_THRESHOLD, mfd->threshold, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_STHRESHOLD), TBM_SETPOS, (WPARAM)TRUE, mfd->threshold);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_THRESHOLDMINUS:
				if (mfd->threshold > 0)
				{
					mfd->threshold -= 1;
					SetDlgItemInt(hdlg, IDC_THRESHOLD, mfd->threshold, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_STHRESHOLD), TBM_SETPOS, (WPARAM)TRUE, mfd->threshold);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_LEFTPLUS:
				if (mfd->xloClip < 255)
				{
					mfd->xloClip += 1;
					SetDlgItemInt(hdlg, IDC_X1CLIP, mfd->xloClip, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_SX1CLIP), TBM_SETPOS, (WPARAM)TRUE, mfd->xloClip);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_LEFTMINUS:
				if (mfd->xloClip > 0)
				{
					mfd->xloClip -= 1;
					SetDlgItemInt(hdlg, IDC_X1CLIP, mfd->xloClip, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_SX1CLIP), TBM_SETPOS, (WPARAM)TRUE, mfd->xloClip);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_RIGHTPLUS:
				if (mfd->xhiClip < 255)
				{
					mfd->xhiClip += 1;
					SetDlgItemInt(hdlg, IDC_X2CLIP, mfd->xhiClip, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_SX2CLIP), TBM_SETPOS, (WPARAM)TRUE, mfd->xhiClip);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_RIGHTMINUS:
				if (mfd->xhiClip > 0)
				{
					mfd->xhiClip -= 1;
					SetDlgItemInt(hdlg, IDC_X2CLIP, mfd->xhiClip, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_SX2CLIP), TBM_SETPOS, (WPARAM)TRUE, mfd->xhiClip);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_BOTTOMPLUS:
				if (mfd->yloClip < 255)
				{
					mfd->yloClip += 1;
					SetDlgItemInt(hdlg, IDC_Y1CLIP, mfd->yloClip, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_SY1CLIP), TBM_SETPOS, (WPARAM)TRUE, mfd->yloClip);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_BOTTOMMINUS:
				if (mfd->yloClip > 0)
				{
					mfd->yloClip -= 1;
					SetDlgItemInt(hdlg, IDC_Y1CLIP, mfd->yloClip, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_SY1CLIP), TBM_SETPOS, (WPARAM)TRUE, mfd->yloClip);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_TOPPLUS:
				if (mfd->yhiClip < 255)
				{
					mfd->yhiClip += 1;
					SetDlgItemInt(hdlg, IDC_Y2CLIP, mfd->yhiClip, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_SY2CLIP), TBM_SETPOS, (WPARAM)TRUE, mfd->yhiClip);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_TOPMINUS:
				if (mfd->yhiClip > 0)
				{
					mfd->yhiClip -= 1;
					SetDlgItemInt(hdlg, IDC_Y2CLIP, mfd->yhiClip, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_SY2CLIP), TBM_SETPOS, (WPARAM)TRUE, mfd->yhiClip);
					mfd->ifp->RedoFrame();
				}
				break;
			}
			break;
	}

	return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd)
{
	MyFilterData *mfd = (MyFilterData *) fa->filter_data;
	MyFilterData mfd_old = *mfd;
	int ret;

	mfd->ifp = fa->ifp;
	if (DialogBoxParam(fa->filter->module->hInstModule,
			MAKEINTRESOURCE(IDD_FILTER), hwnd,
			ConfigDlgProc, (LPARAM) mfd))
	{
		*mfd = mfd_old;
		ret = TRUE;
	}
	else
	{
		ret = FALSE;
	}
	return(ret);
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	sprintf(str, " (diam %d, str %d, thr %d, clip %d %d %d %d, inter %s)",
				mfd->diameter, mfd->strength, mfd->threshold,
				mfd->xloClip, mfd->xhiClip, mfd->yloClip, mfd->yhiClip,
				mfd->interlaced ? "yes" : "no");
}
