/*//////////////////////////////////////////////////////////////////////////

	flaXen Overlay Tool for VirtualDub
	(c) Copyright 2000 Dan Flower (flaXen)

	Overlay a title or banner on a video source

*//////////////////////////////////////////////////////////////////////////

#include <windows.h>
#include <stdio.h>
#include <math.h>

#include "resource.h"
#include "filter.h"
#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#define		MAX_FNAME	1024

typedef struct MyFilterData {
	Pixel32	*Bitmap;
	long	*VertTab;
	long	tgaWid, tgaHei;
	char	ooSrcTGA[MAX_FNAME + 1];
	int		ooMethod, ooIntensity, ooTime;
	int		ooXOff, ooYOff;
} MyFilterData;

///////////////////////////////////////////////////////////////////////////

int		fxOvlRunProc(const FilterActivation *fa, const FilterFunctions *ff);
int		fxOvlStartProc(FilterActivation *fa, const FilterFunctions *ff);
int		fxOvlEndProc(FilterActivation *fa, const FilterFunctions *ff); 
int		fxOvlInitProc(FilterActivation *fa, const FilterFunctions *ff);
int		fxOvlConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void	fxOvlConfigDlgBrowse(HWND);
void	fxOvlStringProc(const FilterActivation *fa, const FilterFunctions *ff, char *buf);
void	fxOvlScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool	fxOvlFssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

ScriptFunctionDef fxOvl_func_defs[]={
    { (ScriptFunctionPtr)fxOvlScriptConfig, "Config", "0siiiii" },
    { NULL },
};

CScriptObject fxOvlobj={
    NULL, fxOvl_func_defs
};

struct FilterDefinition filterDef_fxOvl = {
	NULL, NULL, NULL,		// next, prev, module

	"Overlay",				// name
	"flaXen's Image Overlay Tool",	// desc
	"flaXen",				// maker

	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size
	fxOvlInitProc,			// initProc
	NULL,					// deinitProc
	fxOvlRunProc,			// runProc
	NULL,					// paramProc
	fxOvlConfigProc,		// configProc
	fxOvlStringProc,		// stringProc
	fxOvlStartProc,			// startProc
	fxOvlEndProc,			// endProc
	&fxOvlobj,				// script_obj
	fxOvlFssProc,			// fssProc
};

///////////////////////////////////////////////////////////////////////////

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_fxOvl;

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
	if (!(fd_fxOvl = ff->addFilter(fm, &filterDef_fxOvl, sizeof(FilterDefinition))))
		return 1;

	vdfd_ver    = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;
	return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
	ff->removeFilter(fd_fxOvl);
}

///////////////////////////////////////////////////////////////////////////

int fxOvlRunProc(const FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData	*mfd = (MyFilterData *)fa->filter_data;
	Pixel32		*isrc, *idst, *src, *dst, *misc;
	long		x, y, w, h, c, i, d, o;
	long		MMX;
	const long	swid = fa->src.w, shei = fa->src.h;
	const long	spit = fa->src.pitch >> 2;
	const __int64	AvgAND = 0x00FEFEFE00FEFEFEi64;
	//char	msg[100];

	isrc = (Pixel32 *)fa->src.data;
	idst = (Pixel32 *)fa->dst.data;

	MMX = ff->isMMXEnabled();
	if (swid & 1) MMX = 0;

	memcpy(idst, isrc, spit * shei * 4);

	if (mfd->ooTime > 0)
		if (fa->pfsi->lCurrentFrame * fa->pfsi->lMicrosecsPerFrame / 1000000 >= mfd->ooTime) return 0;

	w = mfd->tgaWid;
	if (mfd->ooXOff + w >= swid) w = swid - mfd->ooXOff;
	h = mfd->tgaHei;
	if (mfd->ooYOff + h >= shei) h = shei - mfd->ooYOff;
/*
	for (y = 0; y < mfd->tgaHei; y++) {
		src = (Pixel32 *)mfd->Bitmap + (y * mfd->tgaWid);
		dst = (Pixel32 *)idst + (y * swid);
		for(x = 0; x < mfd->tgaWid; x++) {
			*dst++ = *src++;
		}
	}
*/
	//sprintf(msg, "%d %d", MMX, (swid | shei) & 1);
	//MessageBox(NULL, msg, "Notice", NULL);

	for (y = 0; y < h; y++) {
		c = (y + shei - mfd->ooYOff - mfd->tgaHei) * spit + mfd->ooXOff;
		src = isrc + c;
		dst = idst + c;
		switch (mfd->ooMethod) {
		case 0:		// Copy
			memcpy(dst, (Pixel32 *)mfd->Bitmap + mfd->VertTab[y], mfd->tgaWid * 4);
			break;
		case 1:		// Average
			if (MMX) {
				misc = (Pixel32 *)mfd->Bitmap + mfd->VertTab[y];
				__asm {
					mov		ecx, [w]
					shr		ecx, 1
					movq	mm1, [AvgAND]
					mov		eax, [src]
					mov		ebx, [misc]
					mov		edx, [dst]

avg10:				movq	mm0, [eax]
					paddusb	mm0, [ebx]
					pand	mm0, mm1
					psrl	mm0, 1
					movq	[edx], mm0
					add		eax, 8
					add		ebx, 8
					add		edx, 8

					dec		ecx
					jnz		avg10
				}
			} else {
				for (x = 0; x < w; x++) {
					i = *src++;
					i = (i & 0x00FEFEFE) >> 1;
					c = *((Pixel32 *)mfd->Bitmap + mfd->VertTab[y] + x);
					c = (c & 0x00FEFEFE) >> 1;
					*dst++ = c + i;
				}
			}
			break;
		case 2:		// Additive
			if (MMX) {
				misc = (Pixel32 *)mfd->Bitmap + mfd->VertTab[y];
				__asm {
					mov		ecx, [w]
					shr		ecx, 1
					mov		eax, [src]
					mov		ebx, [misc]
					mov		edx, [dst]

add10:				movq	mm0, [eax]
					paddusb	mm0, [ebx]
					movq	[edx], mm0
					add		ebx, 8
					add		eax, 8
					add		edx, 8

					dec		ecx
					jnz		add10
				}
			} else {
				for (x = 0; x < w; x++) {
					i = *src++;
					d = *((Pixel32 *)mfd->Bitmap + mfd->VertTab[y] + x);
					c = (i & 0xFF) + (d & 0xFF);
					if (c > 255) c = 255;
					o = (c << 16);
					c = ((i >> 8) & 0xFF) + ((d >> 8) & 0xFF);
					if (c > 255) c = 255;
					o = o | (c << 8);
					c = ((i >> 16) & 0xFF) + ((d >> 16) & 0xFF);
					if (c > 255) c = 255;
					o = o | c;
					*dst++ = o;
				}
			}
			break;
		case 3:		// Subtractive
			if (MMX) {
				misc = (Pixel32 *)mfd->Bitmap + mfd->VertTab[y];
				__asm {
					mov		ecx, [w]
					shr		ecx, 1
					mov		eax, [src]
					mov		ebx, [misc]
					mov		edx, [dst]

sub10:				movq	mm0, [eax]
					psubusb	mm0, [ebx]
					movq	[edx], mm0
					add		eax, 8
					add		ebx, 8
					add		edx, 8

					dec		ecx
					jnz		sub10
				}
			} else {
				for (x = 0; x < w; x++) {
					i = *src++;
					d = *((Pixel32 *)mfd->Bitmap + mfd->VertTab[y] + x);
					c = (i & 0xFF) - (d & 0xFF);
					if (c < 0) c = 0;
					o = (c << 16);
					c = ((i >> 8) & 0xFF) - ((d >> 8) & 0xFF);
					if (c < 0) c = 0;
					o = o | (c << 8);
					c = ((i >> 16) & 0xFF) - ((d >> 16) & 0xFF);
					if (c < 0) c = 0;
					o = o | c;
					*dst++ = o;
				}
			}
			break;
		case 4:		// Mask AND
			if (MMX) {
				misc = (Pixel32 *)mfd->Bitmap + mfd->VertTab[y];
				__asm {
					mov		ecx, [w]
					shr		ecx, 1
					mov		eax, [src]
					mov		ebx, [misc]
					mov		edx, [dst]

and10:				movq	mm0, [eax]
					pand	mm0, [ebx]
					movq	[edx], mm0
					add		eax, 8
					add		ebx, 8
					add		edx, 8

					dec		ecx
					jnz		and10
				}
			} else {
				for (x = 0; x < w; x++) {
					i = *src++;
					c = *((Pixel32 *)mfd->Bitmap + mfd->VertTab[y] + x);
					*dst++ = c & i;
				}
			}
			break;
		case 5:		// Mask OR
			if (MMX) {
				misc = (Pixel32 *)mfd->Bitmap + mfd->VertTab[y];
				__asm {
					mov		ecx, [w]
					shr		ecx, 1
					mov		eax, [src]
					mov		ebx, [misc]
					mov		edx, [dst]

or10:				movq	mm0, [eax]
					por		mm0, [ebx]
					movq	[edx], mm0
					add		eax, 8
					add		ebx, 8
					add		edx, 8

					dec		ecx
					jnz		or10
				}
			} else {
				for (x = 0; x < w; x++) {
					i = *src++;
					c = *((Pixel32 *)mfd->Bitmap + mfd->VertTab[y] + x);
					*dst++ = c | i;
				}
			}
			break;
		}
	}

	if (MMX) __asm emms;

	return 0;
}

///////////////////////////////////////////////////////////////////////////

int fxOvlStartProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	FILE	*fh = NULL;
	Pixel32	*src, p;
	int		fReturn = 0;
	long	d, x, y, ScaleTab[256];
	long	tgaType, tgaBPP, tgaImg;
	//char	str[100];
	
	// Generate scalar table
	for (d = 0; d < 256; d++) {
		ScaleTab[d] = (long)(mfd->ooIntensity / 100.0 * d + 0.5);
		if (ScaleTab[d] > 255) ScaleTab[d] = 255;
	}

	// Open input file
	try {
		if (!(fh = fopen(mfd->ooSrcTGA, "rb"))) throw errno;

		fseek(fh, 2, SEEK_SET);
		tgaType = _getw(fh) & 0xFFFF;
		fseek(fh, 10, SEEK_SET);
		tgaImg = 18 + _getw(fh) & 0xFFFF;
		fseek(fh, 12, SEEK_SET);
		mfd->tgaWid = _getw(fh) & 0xFFFF;
		fseek(fh, 14, SEEK_SET);
		mfd->tgaHei = _getw(fh) & 0xFFFF;
		fseek(fh, 16, SEEK_SET);
		tgaBPP = _getw(fh) & 0xFFFF;
		fseek(fh, tgaImg, SEEK_SET);

		if (!(mfd->VertTab = new long[mfd->tgaHei])) return 1;
		for (d = 0; d < mfd->tgaHei; d++)
			mfd->VertTab[d] = (mfd->tgaHei - d - 1) * mfd->tgaWid;

		if (!(mfd->Bitmap = new Pixel32[mfd->tgaHei * mfd->tgaWid])) return 1;
		//sprintf(str, "%d %d %d %d", mfd->tgaWid, mfd->tgaHei, tgaBPP, tgaImg);
		//MessageBox(NULL, str, "Blah", NULL);
		switch (tgaBPP) {
		case 24:
			for (y = 0; y < mfd->tgaHei; y++) {
				src = (Pixel32 *)(mfd->Bitmap + mfd->VertTab[y]);
				for (x = 0; x < mfd->tgaWid; x++) {
					d = ScaleTab[(unsigned char)fgetc(fh)];
					d |= ScaleTab[(unsigned char)fgetc(fh)] << 8;
					d |= ScaleTab[(unsigned char)fgetc(fh)] << 16;
					*src++ = d;
				}
			}
			break;
		case 32:
			for (y = 0; y < mfd->tgaHei; y++) {
				src = (Pixel32 *)mfd->Bitmap + (mfd->tgaHei - y - 1) * mfd->tgaWid;
				for (x = 0; x < mfd->tgaWid; x++) {
					fgetc(fh);
					p = ScaleTab[(unsigned char)fgetc(fh)]; d = p;
					p = ScaleTab[(unsigned char)fgetc(fh)]; d = d | (p << 8);
					p = ScaleTab[(unsigned char)fgetc(fh)]; d = d | (p << 16);
					*src++ = d;
				}
			}
			break;
		default:
			throw 0x0001;
		}
	} catch(int) {
		fReturn = 1;
	}

	if (fh) fclose(fh);

	return fReturn;
}

///////////////////////////////////////////////////////////////////////////

int fxOvlEndProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	// Free previously allocated memory
	delete[] mfd->Bitmap; mfd->Bitmap = NULL;
	delete[] mfd->VertTab; mfd->VertTab = NULL;

	return 0;
}

///////////////////////////////////////////////////////////////////////////

int fxOvlInitProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	sprintf(mfd->ooSrcTGA, "overlay.tga");
	mfd->ooIntensity = 100;
	mfd->ooMethod = 0;
	mfd->ooTime = 0;

	return 0;
}

///////////////////////////////////////////////////////////////////////////

BOOL CALLBACK fxOvlConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
	MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);
	HWND	hwndItem;
	int		i;
	static const char * const ooMeths[]={
		"Copy", "Average", "Additive",
		"Subtractive", "Mask AND", "Mask OR"
	};

	switch(msg) {
	case WM_INITDIALOG:
		SetWindowLong(hdlg, DWL_USER, lParam);
		mfd = (MyFilterData *)lParam;

		hwndItem = GetDlgItem(hdlg, IDC_OO_METHOD);
		for (i = 0; i < 6; i++)
			SendMessage(hwndItem, CB_ADDSTRING, 0, (LPARAM)ooMeths[i]);

		SetDlgItemText(hdlg, IDC_OO_SRC, mfd->ooSrcTGA);
		SetDlgItemInt(hdlg, IDC_OO_INTENSITY, mfd->ooIntensity, FALSE);
		SendMessage(hwndItem, CB_SETCURSEL, mfd->ooMethod, 0);
		SetDlgItemInt(hdlg, IDC_OO_ENDFRAME, mfd->ooTime, FALSE);
		SetDlgItemInt(hdlg, IDC_OO_XOFF, mfd->ooXOff, FALSE);
		SetDlgItemInt(hdlg, IDC_OO_YOFF, mfd->ooYOff, FALSE);
		return TRUE;
	case WM_COMMAND:
		switch(LOWORD(wParam)) {
		case IDOK:
			GetDlgItemText(hdlg, IDC_OO_SRC, mfd->ooSrcTGA, MAX_FNAME);
			mfd->ooMethod = SendDlgItemMessage(hdlg, IDC_OO_METHOD, CB_GETCURSEL, 0, 0);
			mfd->ooIntensity = GetDlgItemInt(hdlg, IDC_OO_INTENSITY, &mfd->ooIntensity, FALSE);
			mfd->ooTime = GetDlgItemInt(hdlg, IDC_OO_ENDFRAME, &mfd->ooTime, FALSE);
			mfd->ooXOff = GetDlgItemInt(hdlg, IDC_OO_XOFF, &mfd->ooXOff, FALSE);
			mfd->ooYOff = GetDlgItemInt(hdlg, IDC_OO_YOFF, &mfd->ooYOff, FALSE);
			EndDialog(hdlg, 0);

			if (mfd->ooIntensity > 200) mfd->ooIntensity = 200;
			if (mfd->ooTime > 32000) mfd->ooTime = 32000;
			return TRUE;
		case IDBROWSE:
			fxOvlConfigDlgBrowse(hdlg);
			return TRUE;
		case IDCANCEL:
			EndDialog(hdlg, 1);
			return FALSE;
		}
		break;
	}

	return FALSE;
}

///////////////////////////////////////////////////////////////////////////

void fxOvlConfigDlgBrowse(HWND hWnd) {
	OPENFILENAME ofn;
	char szFile[MAX_PATH];

	SendMessage(GetDlgItem(hWnd, IDC_OO_SRC), WM_GETTEXT, sizeof szFile, (LPARAM)szFile);

	ofn.lStructSize			= sizeof(OPENFILENAME);
	ofn.hwndOwner			= hWnd;
	ofn.lpstrFilter			= "24bit Targa File (*.tga)\0*.tga\0All files (*.*)\0*.*\0";
	ofn.lpstrCustomFilter	= NULL;
	ofn.nFilterIndex		= 1;
	ofn.lpstrFile			= szFile;
	ofn.nMaxFile			= sizeof szFile;
	ofn.lpstrFileTitle		= NULL;
	ofn.nMaxFileTitle		= 0;
	ofn.lpstrInitialDir		= NULL;
	ofn.lpstrTitle			= "Select Overlay File";
	ofn.Flags				= OFN_EXPLORER | OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST | OFN_HIDEREADONLY | OFN_ENABLESIZING;
	ofn.lpstrDefExt			= NULL;

	if (GetOpenFileName(&ofn))
		SendMessage(GetDlgItem(hWnd, IDC_OO_SRC), WM_SETTEXT, 0, (LPARAM)szFile);
}

///////////////////////////////////////////////////////////////////////////

int fxOvlConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd) {
	return DialogBoxParam(fa->filter->module->hInstModule,
	       MAKEINTRESOURCE(IDD_FXOVL_CONFIG), hwnd,
		   fxOvlConfigDlgProc, (LPARAM)fa->filter_data);
}

///////////////////////////////////////////////////////////////////////////

void fxOvlStringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	char	*meth;

	meth = "???";
	switch (mfd->ooMethod) {
	case 0:
		meth = "Copy";
		break;
	case 1:
		meth = "Average";
		break;
	case 2:
		meth = "Additive";
		break;
	case 3:
		meth = "Subtractive";
		break;
	case 4:
		meth = "Mask AND";
		break;
	case 5:
		meth = "Mask OR";
		break;
	}
	sprintf(str, " %s %d%% Time:%ds Pos:%d,%d File:%s ", meth, mfd->ooIntensity, 
		mfd->ooTime, mfd->ooXOff, mfd->ooYOff, mfd->ooSrcTGA);
}

///////////////////////////////////////////////////////////////////////////

void fxOvlScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
    FilterActivation *fa = (FilterActivation *)lpVoid;
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	strcpy(mfd->ooSrcTGA, *argv[0].asString());
	mfd->ooMethod = argv[1].asInt();
	mfd->ooIntensity = argv[2].asInt();
	mfd->ooTime = argv[3].asInt();
	mfd->ooXOff = argv[4].asInt();
	mfd->ooYOff = argv[5].asInt();
}

///////////////////////////////////////////////////////////////////////////

bool fxOvlFssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

    _snprintf(buf, buflen, "Config(%s, %d, %d, %d, %d, %d)", mfd->ooSrcTGA, 
		mfd->ooMethod, mfd->ooIntensity, mfd->ooTime, mfd->ooXOff, mfd->ooYOff);

    return true;
}