/*
    Zoom Filter for VirtualDub.
    Copyright (C) 2001 Donald A. Graft

    Based on VirtualDub's internal resampling engine by Avery Lee.
    This filter is co-authored by Donald Graft and Avery Lee.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    The author can be contacted at:

	Donald Graft
	neuron2@home.com.
*/

#define VDEXT_VIDEO_FILTER
#define VDEXT_MAIN
#define USE_INTERLACED

#include <windows.h>
#include <commctrl.h>
#include <crtdbg.h>
#include <assert.h>
#include <stdio.h>
#include <math.h>

#include "ScriptInterpreter.h"
#include "ScriptValue.h"
#include "ScriptError.h"

#include "misc.h"
#include "cpuaccel.h"
#include "resource.h"
#include "gui.h"
#include "filter.h"
#include "resample.h"
#include "vbitmap.h"

///////////////////////

enum {
	FILTER_NONE				= 0,
	FILTER_BILINEAR			= 1,
	FILTER_BICUBIC			= 2,
	FILTER_TABLEBILINEAR	= 3,
	FILTER_TABLEBICUBIC		= 4
};

static char *filter_names[]={
	"Nearest neighbor",
	"Bilinear",
	"Bicubic",
	"Precise bilinear",
	"Precise bicubic",
};

typedef struct MyFilterData {
	long new_x, new_y;
	int filter_mode;
	int focusx, focusy;
	int startpercent;
	int endpercent;
	int start, end;
	int startpercent2;
	int endpercent2;
	int start2, end2;
	int red, green, blue;
	bool fInterlaced;

	IFilterPreview *ifp;
	Resampler *resampler;
	Resampler::eFilter fmode;
} MyFilterData;

////////////////////

static int resize_run(const FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	double dstw, dsth;
	double posx, posy;
	int frame = fa->pfsi->lCurrentSourceFrame;
	int start = mfd->start;
	int end = mfd->end;
	int startpercent = mfd->startpercent;
	int endpercent = mfd->endpercent;
	int start2 = mfd->start2;
	int end2 = mfd->end2;
	int startpercent2 = mfd->startpercent2;
	int endpercent2 = mfd->endpercent2;
	double percent;
	Pixel *dst, *src;
	unsigned long color;
	int x, y;

	/* Paint the destination frame with the color fill. */
	dst = fa->dst.data;
	color = (mfd->red << 16) | (mfd->green << 8) | mfd->blue;
	for (y = 0; y < fa->dst.h; y++)
	{
		for (x = 0; x < fa->dst.w; x++)
			dst[x] = color;
		dst	= (Pixel *)((char *)dst + fa->dst.pitch);
	}

	/* Set up the resampling engine for the right scaling for the current frame. */
	if (frame <= mfd->start)
	{
		/* Before the starting frame for zoom. Use the configured starting scale. */
		percent = startpercent;
		dstw = (percent * fa->src.w) / 100.0;
		dsth = (percent * fa->src.h) / 100.0;
	}
	else if (frame <= end)
	{
		/* Zooming. Implement perspective zoom between starting and ending scales. */
		percent = 1.0 / (1.0/startpercent +
			            ((1.0/endpercent - 1.0/startpercent) * (frame - start))/(end - start));
		dstw = (percent * fa->src.w) / 100.0;
		dsth = (percent * fa->src.h) / 100.0;
	}
	else
	{
		/* After the ending frame for zoom. Use the ending scale. */
		percent = endpercent;
		dstw = (percent * fa->src.w) / 100.0;
		dsth = (percent * fa->src.h) / 100.0;
	}

	if (mfd->startpercent2 != 0 && frame >= mfd->start2)
	{
		if (frame <= end2)
		{
			/* Zooming. Implement perspective zoom between starting and ending scales. */
			percent = 1.0 / (1.0/startpercent2 +
							((1.0/endpercent2 - 1.0/startpercent2) * (frame - start2))/(end2 - start2));
			dstw = (percent * fa->src.w) / 100.0;
			dsth = (percent * fa->src.h) / 100.0;
		}
		else
		{
			/* After the ending frame for zoom. Use the ending scale. */
			percent = endpercent2;
			dstw = (percent * fa->src.w) / 100.0;
			dsth = (percent * fa->src.h) / 100.0;
		}
	}
	
	/* Now calculate the correct offset in the source frame to implement the
       configured focus point. */
	posx = mfd->focusx - fa->src.w / 2.0;
	posy = mfd->focusy - fa->src.h / 2.0;

#if 0
	if (posx < 0  || posy < 0)
	{
		MessageBox(NULL,
			"The parameters you have set cause attempted display of "
			"an area off the input frame. Please correct this and try again."
			,
			"Invalid display parameter",
			MB_ICONEXCLAMATION);
		return 1;
	}
#endif

#ifdef USE_INTERLACED
	if (mfd->fInterlaced)
		mfd->resampler->Init(mfd->fmode, mfd->fmode, dstw, dsth/2, fa->src.w, fa->src.h/2);
	else
#endif
		mfd->resampler->Init(mfd->fmode, mfd->fmode, dstw, dsth, fa->src.w, fa->src.h);

	dst = fa->dst.data;
	src = fa->src.data;

#ifdef USE_INTERLACED
	if (mfd->fInterlaced) {
		VBitmap vbHalfSrc, vbHalfDst;

		vbHalfSrc = fa->src;
		vbHalfSrc.modulo += vbHalfSrc.pitch;
		vbHalfSrc.pitch *= 2;
		vbHalfSrc.data = fa->src.Address32i(0, fa->src.h&1);
		vbHalfSrc.h >>= 1;

		vbHalfDst = fa->dst;
		vbHalfDst.modulo += vbHalfDst.pitch;
		vbHalfDst.pitch *= 2;
		vbHalfDst.h >>= 1;

		double dy = 0.25 * (1.0 - (double)vbHalfSrc.h / (double)vbHalfDst.h);

		mfd->resampler->Process(&vbHalfDst, mfd->new_x / 2.0 - dstw / 2.0, mfd->new_y / 4.0 - dsth / 4.0,
			&vbHalfSrc, posx, posy/2-dy, false);

		vbHalfSrc.data = fa->src.Address32i(0, 1+(fa->src.h&1));
		vbHalfDst.data = fa->dst.Address32i(0, 1);

		mfd->resampler->Process(&vbHalfDst, mfd->new_x / 2.0 - dstw / 2.0, mfd->new_y / 4.0 - dsth / 4.0,
			&vbHalfSrc, posx, posy/2+dy, false);
	} else
#endif
		mfd->resampler->Process(&fa->dst, mfd->new_x / 2.0 - dstw / 2.0, mfd->new_y / 2.0 - dsth / 2.0,
		                        &fa->src, posx, posy, false);

	return 0;
}

static long resize_param(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	if (!mfd) return 0;
	fa->dst.w		= mfd->new_x;
	fa->dst.h		= mfd->new_y;

#ifdef USE_INTERLACED
	if (mfd->fInterlaced)
		fa->dst.h = (fa->dst.h+1)&~1;
#endif

	fa->dst.AlignTo8();

	return FILTERPARAM_SWAP_BUFFERS;
}

static BOOL APIENTRY resizeDlgProc( HWND hDlg, UINT message, UINT wParam, LONG lParam) {
	MyFilterData *mfd = (struct MyFilterData *)GetWindowLong(hDlg, DWL_USER);

    switch (message)
    {
        case WM_INITDIALOG:
			{
				HWND hwndItem;
				int i;

				mfd = (MyFilterData *)lParam;
				SetDlgItemInt(hDlg, IDC_WIDTH, mfd->new_x, FALSE);
				SetDlgItemInt(hDlg, IDC_HEIGHT, mfd->new_y, FALSE);
				SetDlgItemInt(hDlg, IDC_FOCUSX, mfd->focusx, FALSE);
				SetDlgItemInt(hDlg, IDC_FOCUSY, mfd->focusy, FALSE);
				SetDlgItemInt(hDlg, IDC_STARTPERCENT, mfd->startpercent, FALSE);
				SetDlgItemInt(hDlg, IDC_ENDPERCENT, mfd->endpercent, FALSE);
				SetDlgItemInt(hDlg, IDC_START, mfd->start, FALSE);
				SetDlgItemInt(hDlg, IDC_END, mfd->end, FALSE);
				SetDlgItemInt(hDlg, IDC_STARTPERCENT2, mfd->startpercent2, FALSE);
				SetDlgItemInt(hDlg, IDC_ENDPERCENT2, mfd->endpercent2, FALSE);
				SetDlgItemInt(hDlg, IDC_START2, mfd->start2, FALSE);
				SetDlgItemInt(hDlg, IDC_END2, mfd->end2, FALSE);
				SetDlgItemInt(hDlg, IDC_FILLRED, mfd->red, FALSE);
				SetDlgItemInt(hDlg, IDC_FILLGREEN, mfd->green, FALSE);
				SetDlgItemInt(hDlg, IDC_FILLBLUE, mfd->blue, FALSE);
				CheckDlgButton(hDlg, IDC_INTERLACED, mfd->fInterlaced ? BST_CHECKED : BST_UNCHECKED);
				hwndItem = GetDlgItem(hDlg, IDC_FILTER_MODE);
				for(i=0; i<(sizeof filter_names/sizeof filter_names[0]); i++)
					SendMessage(hwndItem, CB_ADDSTRING, 0, (LPARAM)filter_names[i]);
				SendMessage(hwndItem, CB_SETCURSEL, mfd->filter_mode, 0);
				SetWindowLong(hDlg, DWL_USER, (LONG)mfd);
				mfd->ifp->InitButton(GetDlgItem(hDlg, IDC_PREVIEW));
			}
            return (TRUE);

        case WM_COMMAND:                      
			switch(LOWORD(wParam)) {
			case IDOK:
				mfd->ifp->Close();
				EndDialog(hDlg, 0);
				return TRUE;

			case IDCANCEL:
				mfd->ifp->Close();
                EndDialog(hDlg, 1);
                return TRUE;

			case IDC_WIDTH:
				if (HIWORD(wParam) == EN_UPDATE) {
					long new_x;
					BOOL success;

					new_x = GetDlgItemInt(hDlg, IDC_WIDTH, &success, FALSE);
					if (!success || new_x < 1 || new_x > 1024) {
						SetDlgItemInt(hDlg, IDC_WIDTH, mfd->new_x, FALSE);
						return TRUE;
					}
					if (mfd != NULL)
					{
						mfd->ifp->UndoSystem();
						mfd->new_x = new_x;
						mfd->ifp->RedoSystem();
					}
				}
				return TRUE;

			case IDC_HEIGHT:
				if (HIWORD(wParam) == EN_UPDATE) {
					long new_y;
					BOOL success;

					new_y = GetDlgItemInt(hDlg, IDC_HEIGHT, &success, FALSE);
					if (!success || new_y < 1 || new_y > 1024) {
						SetDlgItemInt(hDlg, IDC_HEIGHT, mfd->new_y, FALSE);
						return TRUE;
					}
					if (mfd != NULL)
					{
						mfd->ifp->UndoSystem();
						mfd->new_y = new_y;
						mfd->ifp->RedoSystem();
					}
				}
				return TRUE;

			case IDC_FOCUSX:
				if (HIWORD(wParam) == EN_UPDATE) {
					long focusx;
					BOOL success;

					focusx = GetDlgItemInt(hDlg, IDC_FOCUSX, &success, FALSE);
					if (!success || focusx > 1024) {
						SetDlgItemInt(hDlg, IDC_FOCUSX, mfd->focusx, FALSE);
						return TRUE;
					}
					if (mfd != NULL)
					{
						mfd->ifp->UndoSystem();
						mfd->focusx = focusx;
						mfd->ifp->RedoSystem();
					}
				}
				return TRUE;

			case IDC_FOCUSY:
				if (HIWORD(wParam) == EN_UPDATE) {
					long focusy;
					BOOL success;

					focusy = GetDlgItemInt(hDlg, IDC_FOCUSY, &success, FALSE);
					if (!success || focusy > 1024) {
						SetDlgItemInt(hDlg, IDC_FOCUSY, mfd->focusy, FALSE);
						return TRUE;
					}
					if (mfd != NULL)
					{
						mfd->ifp->UndoSystem();
						mfd->focusy = focusy;
						mfd->ifp->RedoSystem();
					}
				}
				return TRUE;

			case IDC_STARTPERCENT:
				if (HIWORD(wParam) == EN_UPDATE) {
					long startpercent;
					BOOL success;

					startpercent = GetDlgItemInt(hDlg, IDC_STARTPERCENT, &success, FALSE);
					if (!success || startpercent < 1) {
						SetDlgItemInt(hDlg, IDC_STARTPERCENT, mfd->startpercent, FALSE);
						return TRUE;
					}
					if (mfd != NULL)
					{
						mfd->ifp->UndoSystem();
						mfd->startpercent = startpercent;
						mfd->ifp->RedoSystem();
					}
				}
				return TRUE;

			case IDC_ENDPERCENT:
				if (HIWORD(wParam) == EN_UPDATE) {
					long endpercent;
					BOOL success;

					endpercent = GetDlgItemInt(hDlg, IDC_ENDPERCENT, &success, FALSE);
					if (!success || endpercent < 1) {
						SetDlgItemInt(hDlg, IDC_ENDPERCENT, mfd->endpercent, FALSE);
						return TRUE;
					}
					if (mfd != NULL)
					{
						mfd->ifp->UndoSystem();
						mfd->endpercent = endpercent;
						mfd->ifp->RedoSystem();
					}
				}
				return TRUE;

			case IDC_START:
				if (HIWORD(wParam) == EN_UPDATE) {
					long start;
					BOOL success;

					start = GetDlgItemInt(hDlg, IDC_START, &success, FALSE);
					if (!success) {
						SetDlgItemInt(hDlg, IDC_START, mfd->start, FALSE);
						return TRUE;
					}
					if (mfd != NULL)
					{
						mfd->ifp->UndoSystem();
						mfd->start = start;
						mfd->ifp->RedoSystem();
					}
				}
				return TRUE;

			case IDC_END:
				if (HIWORD(wParam) == EN_UPDATE) {
					long end;
					BOOL success;

					end = GetDlgItemInt(hDlg, IDC_END, &success, FALSE);
					if (!success) {
						SetDlgItemInt(hDlg, IDC_END, mfd->end, FALSE);
						return TRUE;
					}
					if (mfd != NULL)
					{
						mfd->ifp->UndoSystem();
						mfd->end = end;
						mfd->ifp->RedoSystem();
					}
				}
				return TRUE;

			case IDC_STARTPERCENT2:
				if (HIWORD(wParam) == EN_UPDATE) {
					long startpercent2;
					BOOL success;

					startpercent2 = GetDlgItemInt(hDlg, IDC_STARTPERCENT2, &success, FALSE);
					if (!success) {
						SetDlgItemInt(hDlg, IDC_STARTPERCENT2, mfd->startpercent2, FALSE);
						return TRUE;
					}
					if (mfd != NULL)
					{
						mfd->ifp->UndoSystem();
						mfd->startpercent2 = startpercent2;
						mfd->ifp->RedoSystem();
					}
				}
				return TRUE;

			case IDC_ENDPERCENT2:
				if (HIWORD(wParam) == EN_UPDATE) {
					long endpercent2;
					BOOL success;

					endpercent2 = GetDlgItemInt(hDlg, IDC_ENDPERCENT2, &success, FALSE);
					if (!success) {
						SetDlgItemInt(hDlg, IDC_ENDPERCENT2, mfd->endpercent2, FALSE);
						return TRUE;
					}
					if (mfd != NULL)
					{
						mfd->ifp->UndoSystem();
						mfd->endpercent2 = endpercent2;
						mfd->ifp->RedoSystem();
					}
				}
				return TRUE;

			case IDC_START2:
				if (HIWORD(wParam) == EN_UPDATE) {
					long start2;
					BOOL success;

					start2 = GetDlgItemInt(hDlg, IDC_START2, &success, FALSE);
					if (!success) {
						SetDlgItemInt(hDlg, IDC_START2, mfd->start2, FALSE);
						return TRUE;
					}
					if (mfd != NULL)
					{
						mfd->ifp->UndoSystem();
						mfd->start2 = start2;
						mfd->ifp->RedoSystem();
					}
				}
				return TRUE;

			case IDC_END2:
				if (HIWORD(wParam) == EN_UPDATE) {
					long end2;
					BOOL success;

					end2 = GetDlgItemInt(hDlg, IDC_END2, &success, FALSE);
					if (!success) {
						SetDlgItemInt(hDlg, IDC_END2, mfd->end2, FALSE);
						return TRUE;
					}
					if (mfd != NULL)
					{
						mfd->ifp->UndoSystem();
						mfd->end2 = end2;
						mfd->ifp->RedoSystem();
					}
				}
				return TRUE;

			case IDC_FILLRED:
				if (HIWORD(wParam) == EN_UPDATE) {
					long red;
					BOOL success;

					red = GetDlgItemInt(hDlg, IDC_FILLRED, &success, FALSE);
					if (!success || red > 255) {
						SetDlgItemInt(hDlg, IDC_FILLRED, mfd->red, FALSE);
						return TRUE;
					}
					if (mfd != NULL)
					{
						mfd->ifp->UndoSystem();
						mfd->red = red;
						mfd->ifp->RedoSystem();
					}
				}
				return TRUE;

			case IDC_FILLGREEN:
				if (HIWORD(wParam) == EN_UPDATE) {
					long green;
					BOOL success;

					green = GetDlgItemInt(hDlg, IDC_FILLGREEN, &success, FALSE);
					if (!success || green > 255) {
						SetDlgItemInt(hDlg, IDC_FILLGREEN, mfd->green, FALSE);
						return TRUE;
					}
					if (mfd != NULL)
					{
						mfd->ifp->UndoSystem();
						mfd->green = green;
						mfd->ifp->RedoSystem();
					}
				}
				return TRUE;

			case IDC_FILLBLUE:
				if (HIWORD(wParam) == EN_UPDATE) {
					long blue;
					BOOL success;

					blue = GetDlgItemInt(hDlg, IDC_FILLBLUE, &success, FALSE);
					if (!success || blue > 255) {
						SetDlgItemInt(hDlg, IDC_FILLBLUE, mfd->blue, FALSE);
						return TRUE;
					}
					if (mfd != NULL)
					{
						mfd->ifp->UndoSystem();
						mfd->blue = blue;
						mfd->ifp->RedoSystem();
					}
				}
				return TRUE;

			case IDC_PREVIEW:
				mfd->ifp->Toggle(hDlg);
				return TRUE;

			case IDC_FILTER_MODE:
				if (HIWORD(wParam) == CBN_SELCHANGE) {
					mfd->ifp->UndoSystem();
					mfd->filter_mode = SendDlgItemMessage(hDlg, IDC_FILTER_MODE, CB_GETCURSEL, 0, 0);
					mfd->ifp->RedoSystem();
				}
				return TRUE;

			case IDC_INTERLACED:
				if (HIWORD(wParam) == BN_CLICKED) {
					BOOL f = IsDlgButtonChecked(hDlg, IDC_INTERLACED);

					mfd->ifp->UndoSystem();
					mfd->fInterlaced = !!f;

					mfd->ifp->RedoSystem();
				}
				return TRUE;

 			case IDHELP:
				{
				char prog[256];
				char path[256];
				LPTSTR ptr;

				GetModuleFileName(NULL, prog, 255);
				GetFullPathName(prog, 255, path, &ptr);
				*ptr = 0;
				strcat(path, "plugins\\Zoom.html");
				ShellExecute(hDlg, "open", path, NULL, NULL, SW_SHOWNORMAL);
				return TRUE;
				}
            }
            break;
    }
    return FALSE;
}

static int resize_config(FilterActivation *fa, const FilterFunctions *ff, HWND hWnd) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	MyFilterData mfd2 = *mfd;
	int ret;

	mfd->ifp = fa->ifp;

	if (mfd->new_x < 16)
		mfd->new_x = 320;
	if (mfd->new_y < 16)
		mfd->new_y = 240;

	ret = DialogBoxParam(fa->filter->module->hInstModule, MAKEINTRESOURCE(IDD_FILTER_ZOOM),
						 hWnd, resizeDlgProc, (LONG)mfd);

	if (ret)
		*mfd = mfd2;

	return ret;
}

static void resize_string(const FilterActivation *fa, const FilterFunctions *ff, char *buf) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	wsprintf(buf, " (%s)", filter_names[mfd->filter_mode]);
}

static int resize_start(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	double dstw = (mfd->startpercent * fa->src.w) / 100.0;
	double dsth = (mfd->startpercent * fa->src.h) / 100.0;

	if (mfd->new_x > fa->src.w || mfd->new_y > fa->src.h)
	{
		MessageBox(NULL,
			"Dimensions of the output frame cannot exceed those of "
			"the input frame. Please correct this and try again."
			,
			"Invalid output window size parameter",
			MB_ICONEXCLAMATION);
		return 1;
	}
	if (mfd->end < mfd->start)
	{
		MessageBox(NULL,
			"The specified ending frame cannot be before the specified "
			"starting frame. Please correct this and try again."
			,
			"Invalid ending frame parameter",
			MB_ICONEXCLAMATION);
		return 1;
	}
	if (mfd->startpercent2 != 0 && mfd->start2 < mfd->end)
	{
		MessageBox(NULL,
			"The specified phase 2 starting frame cannot be before the "
			"specified phase 1 ending frame. Please correct this and try again."
			,
			"Invalid starting frame parameter",
			MB_ICONEXCLAMATION);
		return 1;
	}
	if (mfd->startpercent2 != 0 && mfd->end2 < mfd->start2)
	{
		MessageBox(NULL,
			"The specified phase 2 ending frame cannot be before the "
			"specified phase 2 starting frame. Please correct this and try again."
			,
			"Invalid ending frame parameter",
			MB_ICONEXCLAMATION);
		return 1;
	}
	if (mfd->focusx > fa->src.w - 1 || mfd->focusy > fa->src.h - 1)
	{
		MessageBox(NULL,
			"The focus point must be within the dimensions of "
			"the input frame. Please correct this and try again."
			,
			"Invalid focus parameter",
			MB_ICONEXCLAMATION);
		return 1;
	}
	if (dstw<16 || dsth<16)
	{
		MessageBox(NULL,
			"The resampler requires the scaled output area to be at least "
			"16 x 16 pixels. Please correct this and try again."
			,
			"Invalid resampler configuration",
			MB_ICONEXCLAMATION);
		return 1;
	}

	switch(mfd->filter_mode) {
	case FILTER_NONE:			mfd->fmode = Resampler::eFilter::kPoint; break;
	case FILTER_BILINEAR:		mfd->fmode = Resampler::eFilter::kLinearInterp; break;
	case FILTER_BICUBIC:		mfd->fmode = Resampler::eFilter::kCubicInterp; break;
	case FILTER_TABLEBILINEAR:	mfd->fmode = Resampler::eFilter::kLinearDecimate; break;
	case FILTER_TABLEBICUBIC:	mfd->fmode = Resampler::eFilter::kCubicDecimate; break;
	}

	mfd->resampler = new Resampler();

#ifdef USE_INTERLACED
	if (mfd->fInterlaced)
		mfd->resampler->Init(mfd->fmode, mfd->fmode, dstw, dsth/2, fa->src.w, fa->src.h/2);
	else
#endif
		mfd->resampler->Init(mfd->fmode, mfd->fmode, dstw, dsth, fa->src.w, fa->src.h);

	return 0;
}

static int resize_stop(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	delete mfd->resampler;	mfd->resampler = NULL;

	return 0;
}

static void resize_script_config(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc)
{
	FilterActivation *fa = (FilterActivation *)lpVoid;

	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->new_x			= argv[0].asInt();
	mfd->new_y			= argv[1].asInt();
	mfd->focusx			= argv[2].asInt();
	mfd->focusy			= argv[3].asInt();
	mfd->startpercent	= argv[4].asInt();
	mfd->endpercent		= argv[5].asInt();
	mfd->start			= argv[6].asInt();
	mfd->end			= argv[7].asInt();
	mfd->startpercent2	= argv[8].asInt();
	mfd->endpercent2	= argv[9].asInt();
	mfd->start2			= argv[10].asInt();
	mfd->end2			= argv[11].asInt();
	mfd->red			= argv[12].asInt();
	mfd->green			= argv[13].asInt();
	mfd->blue			= argv[14].asInt();
	mfd->filter_mode	= argv[15].asInt();
	mfd->fInterlaced = false;
	if (mfd->filter_mode & 128) {
		mfd->fInterlaced = true;
		mfd->filter_mode &= 127;
	}
}

static ScriptFunctionDef resize_func_defs[]={
	{ (ScriptFunctionPtr)resize_script_config, "Config", "0iiiiiiiiiiiiiiii" },
	{ NULL },
};

static CScriptObject resize_obj={
	NULL, resize_func_defs
};

static bool resize_script_line(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	int filtmode = mfd->filter_mode + (mfd->fInterlaced ? 128 : 0);

	_snprintf(buf, buflen, "Config(%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d)",
		      mfd->new_x, mfd->new_y, mfd->focusx, mfd->focusy,
			  mfd->startpercent, mfd->endpercent, mfd->start, mfd->end,
			  mfd->startpercent2, mfd->endpercent2, mfd->start2, mfd->end2,
			  mfd->red, mfd->green, mfd->blue, filtmode);
	return true;
}

int resize_init(FilterActivation *fa, const FilterFunctions *ff);

FilterDefinition filterDef_resize={
	0,0,NULL,
	"zoom (1.2)",
	"Image zoom (based on internal resize filter by Avery Lee)."
#ifdef USE_ASM
			"\n[Assembly optimized] [FPU optimized] [MMX optimized]"
#endif
			,
	"Donald Graft/Avery Lee",
	NULL,
	sizeof(MyFilterData),
	resize_init,
	NULL,
	resize_run,
	resize_param,
	resize_config,
	resize_string,
	resize_start,
	resize_stop,

	&resize_obj,
	resize_script_line,
};

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_tutorial;

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
	if (!(fd_tutorial = ff->addFilter(fm, &filterDef_resize, sizeof(FilterDefinition))))
		return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

	return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
	ff->removeFilter(fd_tutorial);
}

int resize_init(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	INITIALIZE_VTBLS;
	mfd->new_x = 320;
	mfd->new_y = 240;
	mfd->focusx = 160;
	mfd->focusy = 120;
	mfd->startpercent = 100;
	mfd->endpercent = 500;
	mfd->start = 0;
	mfd->end = 500;
	mfd->startpercent2 = 0;
	mfd->endpercent2 = 0;
	mfd->start2 = 0;
	mfd->end2 = 0;
	mfd->red = 0;
	mfd->green = 140;
	mfd->blue = 180;
	mfd->filter_mode = FILTER_TABLEBICUBIC;
	mfd->fInterlaced = false;
	return 0;
}
