#include <windows.h>
#include <stdio.h>
#include <math.h>
#include <commctrl.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"
/*
*******************************************************************************

	Virtualdub filter to radial correct the luminance
	-------------------------------------------------


		Copyright (C) 2002-2004 Alessandro Malanca, 

		http://web.tiscali.it/minomala/
 
	--------------------------------------------------

	This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  


	UPDATES:
	10.12.02 -	1) duplicated code for speed when no color correction is applied
				2) disabled colours correction on titles
				3) default title thresold set to 255 (disabled)

	16.12.02 -  1) version 2.1 independex center and border corrections
				2) black correction back to original algo
				3) added bright radial control
				4) added invert parameter
				5) updated help

*******************************************************************************
*/

int radiaLumRunProc(const FilterActivation *fa, const FilterFunctions *ff);
int radiaLumStartProc(FilterActivation *fa, const FilterFunctions *ff);
int radiaLumEndProc(FilterActivation *fa, const FilterFunctions *ff);
long radiaLumParamProc(FilterActivation *fa, const FilterFunctions *ff);
int radiaLumConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void radiaLumStringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void radiaLumScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool radiaLumFssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

int radiaLumInitProc(FilterActivation *fa, const FilterFunctions *ff);


///////////////////////////////////////////////////////////////////////////


#define RED_SCALE 0.299
#define GRE_SCALE 0.587
#define BLU_SCALE 0.114


#define MAX_CONTRAST  100
#define MAX_BRIGHTNESS 50
#define DEF_TILTLE_THRES 255


typedef struct FilterParameters {

	bool	invert_correction;

	int		center_correct;
	int		border_correct;
	int		bright_correct;

//	int		radius;
	int		titleThreshold;
	bool	interlaced;

	int		brightness;
	int		contrast;
	int		greenRed;
	int		yellowBlue;

	bool	keep_chroma;



} FilterParameters;

const FilterParameters 
	  nullPreviewParameters = {0,0,0,0,false /*,false*/ };

class FrameHistogram {

#define NUM_SAMPLES 80

public:

FrameHistogram() {
	for (int i=0; i<NUM_SAMPLES; i++) histogram[i]=1;
}

void setScaleSliderValue(int v) {
	sliderScale = 5-v;
	sliderPosition = v;
}

int getScaleSliderValue() {
	return sliderPosition;
}

void  paintHisto (HWND hdlg) {

	RECT	rHistoCtrl;

	HDC hdc;
	PAINTSTRUCT ps;

	HPEN whitePen = (HPEN)GetStockObject(WHITE_PEN);
	HPEN blackPen = (HPEN)GetStockObject(BLACK_PEN);

	hdc = BeginPaint(GetDlgItem(hdlg,IDC_HISTO), &ps);

	GetClientRect(GetDlgItem(hdlg, IDC_HISTO), &rHistoCtrl);
	paintBackGr(hdc,rHistoCtrl);

	RECT rHisto;
	int lineSize = calcPaintArea (rHisto, rHistoCtrl );
	FillRect(hdc, &rHisto, (HBRUSH)GetStockObject(LTGRAY_BRUSH));

	long media;
	HPEN pen;
	pen = CreatePen( PS_SOLID,1,RGB(13,176,255));

	if ( sliderScale!=0) {
		media = rHisto.bottom - (rHisto.bottom+rHisto.top)/2;
	} else {
		media =  scaleValue((double)mediaLum,rHisto.bottom-rHisto.top);
	}


	int xPos = rHisto.left;
	for (int j=0;j<NUM_SAMPLES;j++) {
		int h = scaleValue((double)histogram[j],rHisto.bottom-rHisto.top);
		for ( int l=0; l<lineSize; l++) {
			SelectObject( hdc, pen);
			MoveToEx(hdc, xPos+l, rHisto.bottom, NULL);
			if (h<media) {			
				LineTo(hdc,	xPos+l, rHisto.bottom-h);
				SelectObject( hdc, blackPen);
				LineTo(hdc,	xPos+l, rHisto.bottom-media);
			} else {
				LineTo(hdc,	xPos+l, rHisto.bottom-media);
				SelectObject( hdc, whitePen);
				LineTo(hdc,	xPos+l, rHisto.bottom-h);
			}
		}
		xPos += lineSize;
	}

	// draw media line

	{
		HPEN rPen = CreatePen( PS_SOLID,2,RGB(255,50,70));
		SelectObject( hdc, rPen);
		MoveToEx(hdc, rHisto.left, rHisto.bottom-media, NULL);
		LineTo(hdc,	rHisto.right, rHisto.bottom-media);
		DeleteObject(rPen); 
	}

	DeleteObject(pen); 
	EndPaint(hdlg, &ps);
	return;
}


void invalidate(HWND hdlg)
{
	InvalidateRgn(hdlg,NULL,FALSE);
}

void calculateHisto(const FilterActivation *fa) {

	int cw = fa->dst.w/2;
	int ch = fa->dst.h/2;
	double max = sqrt(cw*cw+ch*ch);

	max *= 0.95; // some more pixels in boder ...

	for ( int i=0; i<NUM_SAMPLES; i++ ) totLum[i]=numVal[i]=0;

	Pixel32 *dst =  (Pixel32 *)fa->dst.data;

    int y = fa->dst.h;
	do {int x = fa->dst.w;
		do {
			Pixel32 pixel = *dst++;

			int r = (pixel & 0xFF0000)>>16;
			int g = (pixel & 0x00FF00)>>8;
			int b = (pixel & 0x0000FF);
		
			long lum = (long)((RED_SCALE * r) + (GRE_SCALE * g) + (BLU_SCALE * b)); 
			double d = sqrt((x-cw)*(x-cw)+(y-ch)*(y-ch)) +1 ; // +1 ????
			d = d/max*(double)(NUM_SAMPLES-1);

			if (d>= (double)NUM_SAMPLES) d= (double)(NUM_SAMPLES-1);
			totLum[(int) d] += lum;
			numVal[(int) d]++;

		} while(--x);
		dst = (Pixel32 *)((char *)dst + fa->dst.modulo);
    } while(--y);

	mediaLum=0;
	long numMedia=0;

	for( int j=0; j<(double)NUM_SAMPLES;j++) {
		int t = (numVal[j]>0) ? numVal[j] : 1;
		histogram[j] = totLum[j] / t;
		mediaLum += totLum[j];
		numMedia += numVal[j];
	}
	mediaLum /= numMedia;
}

private:

	long	totLum[NUM_SAMPLES];
	long	numVal[NUM_SAMPLES];
	int		histogram[NUM_SAMPLES];
	long	mediaLum;
	int		sliderScale; // 0 .. 5
	int		sliderPosition;

void paintBackGr( HDC hdc, RECT r ) {

	FillRect(hdc, &r, (HBRUSH)GetStockObject(LTGRAY_BRUSH));

	HPEN blackPen = (HPEN)GetStockObject(BLACK_PEN);
	SelectObject( hdc, blackPen);

	MoveToEx(hdc, r.left, r.top, NULL);
	LineTo(hdc,	r.right-1,r.top);
	LineTo(hdc,	r.right-1,r.bottom-1);
	LineTo(hdc,	r.left,r.bottom-1);
	LineTo(hdc,	r.left,r.top);
}

int calcPaintArea ( RECT & res, RECT rCtrl ) {
	int step = (rCtrl.right-rCtrl.left) / NUM_SAMPLES;
	int left = (rCtrl.right - rCtrl.left - (step * NUM_SAMPLES)) / 2;
	res.left = rCtrl.left+left;
	res.right = rCtrl.left+left+step*NUM_SAMPLES;
	res.bottom = rCtrl.bottom-4;
	res.top = rCtrl.top+4;
	return step;
}

int scaleValue(double histo, int max) {

#define FIT(v)  ( (double)v / 255.0 * (double)max )

	int res;
	res = (int)FIT(histo);
	if ( sliderScale == 0) {
		res = (int)FIT(histo);
	} else {
		double value = histo - (double)mediaLum;		
		double scale = 1.0 + ((double)(sliderScale-1) * 0.25);
		res = (int)(((double)max/2.0)+FIT(value*scale)) ;
	}
	if (res<0) res = 0;
	if (res>=max) res = max-1;
	return res;
#undef FIT
}

}; // FrameHistogram

typedef struct PixelAdjust {

	Pixel32 red[256];
	Pixel32 green[256];
	Pixel32 blue[256];

} PixelAdjust;


typedef struct MyFilterData {

	int					**luma_change;	
	bool				disableFilter;
	FilterParameters	par,save;
	IFilterPreview		*ifp;
	FrameHistogram      frameHistogram;
	bool				displayingPanel;
	PixelAdjust			*pixelAdjust; 
	
	bool				synchronized;


} MyFilterData;


ScriptFunctionDef radiaLum_func_defs[]={
    { (ScriptFunctionPtr)radiaLumScriptConfig, "Config", "0iiiiiiiiiii" },
    { NULL },
};

CScriptObject radiaLum_obj={
    NULL, radiaLum_func_defs
};

struct FilterDefinition filterDef_radiaLum = {

    NULL, NULL, NULL,			// next, prev, module
    "radial luminance [2.1]",	// name
    "Correct radial luminance for 8mm or Super8 films.\nUses the tstudent distribution and allows rgb color correction.",
								// desc
    "Alessandro Malanca",		// maker
    NULL,						// private_data
    sizeof(MyFilterData),		// inst_data_size
    radiaLumInitProc,			// initProc
    NULL,						// deinitProc
    radiaLumRunProc,			// runProc
    NULL,						// ParamProc
    radiaLumConfigProc,			// ConfigProc
    radiaLumStringProc,			// stringProc
    radiaLumStartProc,			// startProc
    radiaLumEndProc,			// endProc
    &radiaLum_obj,				// script_obj
    radiaLumFssProc,			// fssProc

};

///////////////////////////////////////////////////////////////////////////

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_radiaLum;

///////////////////////////////////////////////////////////////////////////

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
    if (!(fd_radiaLum = ff->addFilter(fm, &filterDef_radiaLum, sizeof(FilterDefinition))))
        return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

    return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
    ff->removeFilter(fd_radiaLum);
}

///////////////////////////////////////////////////////////////////////////


int radiaLumInitProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	mfd->displayingPanel = false;
	mfd->par.titleThreshold = DEF_TILTLE_THRES;
//	mfd->par.keep_chroma = true;
	mfd->synchronized = true;

	return 0;
}


PixelAdjust * prepareToAdjust(FilterParameters par) {

#define P_SCALE 2

	long dr_adjust =  par.greenRed - par.yellowBlue/2;
	long dg_adjust =  - par.greenRed - par.yellowBlue/2;
	long db_adjust =  par.yellowBlue;


	if ( dr_adjust<0 ) dr_adjust /= P_SCALE;
	if ( dg_adjust<0 ) dg_adjust /= P_SCALE;
	if ( db_adjust<0 ) db_adjust /= P_SCALE;

    dr_adjust += 256;
	dg_adjust += 256;
	db_adjust += 256;


#undef P_SCALE 


//	long scale_r, scale_g, scale_b;

	long brightness	= par.brightness;
	long contrast	= par.contrast + 256;

	long r,g,b;
	
	PixelAdjust * res;
	res = new PixelAdjust();
	for (int i=0; i<256; i++) {

		r = g = b = i;

		// adjust brightness
		r += brightness;
		g += brightness;
		b += brightness;

		// adjust contrast

		r = ( ( r * contrast ) / 256) ;
		g = ( ( g * contrast ) / 256) ;
		b = ( ( b * contrast ) / 256) ;

		// adjust colors 

		r = (r * dr_adjust) / 256; 
		g = (g * dg_adjust) / 256; 
		b = (b * db_adjust) / 256;

		if (r < 0) r=0; else if (r > 255) r=255;
		if (g < 0) g=0; else if (g > 255) g=255;
		if (b < 0) b=0; else if (b > 255) b=255;

		res->red[i]		= r<<16;
		res->green[i]	= g<<8;
		res->blue[i]	= b;		
	}
	return res;

}


double tStudent3 (double x, double y ) {
	x *= 2; y *= 2;
	return  1.0 / ( 1.0 + ((x*x)+(y*y)) );
}


int radiaLumStartProc(FilterActivation *fa, const FilterFunctions *ff) {

    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	int cols = fa->src.w+1;
	int rows = fa->src.h+1;

	if (mfd->par.interlaced) rows = rows/2;

	int cx = cols/2; cols = cx*2;
	int cy = rows/2; rows = cy*2;

	double fInScale = sqrt ( ((double)cx*(double)cx) + ((double)cy*(double)cy)) ;

	// ----------------------------------
	//  Allocate param tables
	// ----------------------------------	
	int  * temp;
	if (!(temp = new int[cols*rows])) return 1;

	if (mfd->par.interlaced) {
		if (!(mfd->luma_change = new int * [rows*2])) return 1;
	} else {
		if (!(mfd->luma_change = new int * [rows])) return 1;	
	}

	for ( int i =0; i<rows; i++) {
		mfd->luma_change[i]= &temp[i*cols];
	}

	int *row, *row2;

	// ----------------------------------
	// Fill radial correction table	
	// ----------------------------------

	int x,y;

	for ( y=0; y<cy; y++ ) {

		row = mfd->luma_change[cy-y-1];
		row2 = mfd->luma_change[cy+y];

		for ( x=0; x<cx; x++ ) {

			double xx = (double) x / fInScale;
			double yy = (double) y / fInScale;
			double fun = 0.5 - tStudent3 (xx,yy) ;

			if (mfd->par.invert_correction) {
				fun -= (double)(mfd->par.bright_correct)/200.0;
			} else {
				fun += (double)(mfd->par.bright_correct)/200.0;
			}

			if ( fun > 0.0 ) {			
				fun *= mfd->par.border_correct * 3; 
			} else {
				fun *= mfd->par.center_correct * 3; 
			}

			if (mfd->par.invert_correction) {
				fun = -fun;
			}
			if ( fun > 0.0 ) fun *=2;


			if (fun < -255.0) fun = -255.0;

			row[cx+x]	=  row[cx-x-1]	=
			row2[cx+x]	=  row2[cx-x-1] =   (int) fun;
		}	
	}

	if (mfd->par.interlaced) {
		int k;
		for (i =0, k=0; i<rows; i++, k+=2) {
			mfd->luma_change[k] = mfd->luma_change[k+1]= &temp[i*cols];
		}
	}

	mfd->pixelAdjust = prepareToAdjust(mfd->par);

    return 0;

}

bool testIsTitle(const long soglia, Pixel * src, int width) {

	if (soglia >= DEF_TILTLE_THRES) return false;

	Pixel32 * p = src;
	long media=0;
	for (int i=0; i<5; i++) {
		Pixel32 pix = *p++;
		media +=	((pix & 0xFF0000)>>16)
					+ ((pix & 0x00FF00)>>8)
					+ (pix & 0x0000FF);
	}
	p += width-11;
	for ( i=0; i<5; i++) {
		Pixel32 pix = *p++;
		media +=	((pix & 0xFF0000)>>16)
					+ ((pix & 0x00FF00)>>8)
					+ (pix & 0x0000FF);
	}
	media /= 10*3;
	return  media > soglia;
}

#define _LOOP_FOR_ALL_FRAMES \
		h = fa->src.h; \
		do { \
			w = fa->src.w; \
			rr = mfd->luma_change[h-1];	\
			do { scale = rr[w-1];	\
				old_pixel = *src++;	\
				r = (old_pixel & 0xFF0000)>>16;\
				g = (old_pixel & 0x00FF00)>>8;\
				b =  old_pixel & 0x0000FF;	
// _LOOP_FOR_ALL_FRAMES

#define _END_FRAME_LOOP	\
  			} while(--w);		\
			src = (Pixel32 *)((char *)src + fa->src.modulo);		\
			dst = (Pixel32 *)((char *)dst + fa->dst.modulo);		\
		} while(--h); 
// _END_FRAME_LOOP

int radiaLumRunProc(const FilterActivation *fa, const FilterFunctions *ff) {

    MyFilterData *mfd = (MyFilterData *)fa->filter_data;
    PixDim w, h;
    Pixel32 *src, *dst;

	boolean isTitle = testIsTitle(mfd->par.titleThreshold,(Pixel32 *)fa->src.data,fa->src.w);

	if ( mfd->disableFilter || isTitle ) {
		fa->dst.data = fa->src.data;
		mfd->frameHistogram.calculateHisto(fa);
		return 0;
	}

    src = (Pixel32 *)fa->src.data;
    dst = (Pixel32 *)fa->dst.data;

	long r, g, b; 
	int *rr;
	Pixel32 old_pixel;
	int scale;

	// check to skip bright contrast color correction

	if ((mfd->par.brightness!=0) || (mfd->par.contrast!=0) ||
		(mfd->par.greenRed!=0) || (mfd->par.yellowBlue!=0) ) 
	{	
		
		// apply color correction

		Pixel32 * red_pix = mfd->pixelAdjust->red;
		Pixel32 * gre_pix = mfd->pixelAdjust->green;
		Pixel32 * blu_pix = mfd->pixelAdjust->blue;

		if (mfd->par.keep_chroma) {

			_LOOP_FOR_ALL_FRAMES {
				// adjust radial luminance keep chroma
				if (scale>0) {
					int max = r;
					if (g>max) max = g;
					if (b>max) max = b;
					if ((max+scale) >255) scale = (255-max);
				}
				scale += 256;
				r = (r * scale) >> 8;
				g = (g * scale) >> 8;
				b = (b * scale) >> 8;
				// adjust brightness, contrast and colors
				*dst++ = red_pix[r] | gre_pix[g] | blu_pix[b];
			} _END_FRAME_LOOP;

		} else {
			// do not keep chroma

			_LOOP_FOR_ALL_FRAMES {
				// adjust radial luminance
				scale += 256;
				r = (r * scale) >> 8 ;
				g = (g * scale) >> 8;
				b = (b * scale) >> 8;
				if ( r > 0xFF ) r = 0xFF;
				if ( g > 0xFF ) g = 0xFF;
				if ( b > 0xFF ) b = 0xFF;
				// adjust brightness, contrast and colors
				*dst++ = red_pix[r] | gre_pix[g] | blu_pix[b];
			} _END_FRAME_LOOP;

		} //	if (mfd->par.keep_chroma)

	} else { // do NOT correct colors

		if (mfd->par.keep_chroma) {

			_LOOP_FOR_ALL_FRAMES {			
				// adjust radial luminance keeping chroma
				if (scale>0) {
					int max = r;
					if (g>max) max = g;
					if (b>max) max = b;
					if ((max+scale) >255) scale = (255-max);
				}
				scale += 256;
				*dst++= ( ((r * scale) << 8) & 0xFF0000) |
						( (g * scale) & 0xFF00 ) |
						( (b * scale) >> 8 );
			} _END_FRAME_LOOP;
			
		} else {

			_LOOP_FOR_ALL_FRAMES {			
				// adjust radial luminance				
				scale += 256;
				r = (r * scale) << 8 ;
				g = (g * scale);
				b = (b * scale) >> 8;
				if ( r > 0xFF0000 ) r = 0xFF0000;
				if ( g > 0xFF00 ) g = 0xFF00;
				if ( b > 0xFF ) b = 0xFF;
				*dst++= (r & 0xFF0000) | (g & 0xFF00) | b ;
			} _END_FRAME_LOOP;	

		} //if (mfd->par.keep_chroma)

	} // if correction

	if (mfd->displayingPanel) {
		mfd->frameHistogram.calculateHisto(fa);
	}
    return 0;
}


int radiaLumEndProc(FilterActivation *fa, const FilterFunctions *ff) {

    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	delete[] mfd->luma_change[0];
	delete[] mfd->luma_change;
	mfd->luma_change = NULL;

	delete mfd->pixelAdjust;
	mfd->pixelAdjust=NULL;

    return 0;
}


void setDialogFeedbackValues (HWND hdlg, MyFilterData *mfd) {
		
	SetDlgItemInt(hdlg, IDC_CENTER_CORR_VALUE, mfd->par.center_correct, TRUE);
	SetDlgItemInt(hdlg, IDC_BORDER_CORR_VALUE, mfd->par.border_correct, TRUE);
	SetDlgItemInt(hdlg, IDC_BRIGHT_CORR_VALUE, mfd->par.bright_correct, TRUE);

	SetDlgItemInt(hdlg, IDC_CONTRAST_VALUE, mfd->par.contrast, TRUE);
	SetDlgItemInt(hdlg, IDC_BRIGHT_VALUE, mfd->par.brightness, TRUE);
	SetDlgItemInt(hdlg, IDC_GREENRED_VALUE, mfd->par.greenRed, TRUE);
	SetDlgItemInt(hdlg, IDC_YELLOWBLUE_VALUE, mfd->par.yellowBlue, TRUE);

}


void localRedoSystem( MyFilterData * mfd, HWND hdlg) {		

	mfd->ifp->UndoSystem();
	mfd->ifp->RedoSystem();
	setDialogFeedbackValues(hdlg,mfd);
	mfd->frameHistogram.invalidate(hdlg);
}


void localRedoFrame ( MyFilterData * mfd, HWND hdlg ) {

//	redoSystem( mfd, hdlg);

	mfd->ifp->RedoFrame();
	setDialogFeedbackValues(hdlg,mfd);
	mfd->frameHistogram.invalidate(hdlg);

}

void getHorSliderValues(MyFilterData *mfd,HWND hdlg) {

	int center_corr = SendMessage(GetDlgItem(hdlg, IDC_CENTER_CORR_SLIDER), TBM_GETPOS, 0, 0);
	if ( center_corr != mfd->par.center_correct ) {
		if (mfd->synchronized) {
			int delta = center_corr - mfd->par.center_correct;
			mfd->par.border_correct = mfd->par.border_correct + delta;
			if (mfd->par.border_correct>100) mfd->par.border_correct = 100;
			if (mfd->par.border_correct<0) mfd->par.border_correct = 0;
			SendMessage(GetDlgItem(hdlg,IDC_BORDER_CORR_SLIDER), TBM_SETPOS, (WPARAM)TRUE, mfd->par.border_correct);
		}
		mfd->par.center_correct = center_corr;		
		localRedoSystem( mfd, hdlg );
		return ;
	}

	int border_corr = SendMessage(GetDlgItem(hdlg, IDC_BORDER_CORR_SLIDER), TBM_GETPOS, 0, 0);
	if ( border_corr != mfd->par.border_correct ) {
		if (mfd->synchronized) {
			int delta =  border_corr - mfd->par.border_correct;
			mfd->par.center_correct = mfd->par.center_correct + delta;
			if (mfd->par.center_correct>100) mfd->par.center_correct = 100;
			if (mfd->par.center_correct<0) mfd->par.center_correct = 0;
			SendMessage(GetDlgItem(hdlg,IDC_CENTER_CORR_SLIDER), TBM_SETPOS, (WPARAM)TRUE, mfd->par.center_correct);
		}
		mfd->par.border_correct = border_corr;
		localRedoSystem( mfd, hdlg );
		return ;
	}

	int bright_corr = SendMessage(GetDlgItem(hdlg, IDC_GLOBAL_CORR_SLIDER), TBM_GETPOS, 0, 0);
	if ( bright_corr != mfd->par.bright_correct ) {
		mfd->par.bright_correct = bright_corr;
		localRedoSystem( mfd, hdlg );
		return ;
	}

	int contrast = SendMessage(GetDlgItem(hdlg, IDC_CONTRAST_SLIDER), TBM_GETPOS, 0, 0);
	if ( contrast != mfd->par.contrast ) {
		mfd->par.contrast = contrast;
		localRedoSystem( mfd, hdlg );
		return;
	}

	int brightness = SendMessage(GetDlgItem(hdlg, IDC_BRIGHT_SLIDER), TBM_GETPOS, 0, 0);
	if ( brightness != mfd->par.brightness ) {
		mfd->par.brightness = brightness;
		localRedoSystem( mfd, hdlg );
		return;
	}
		
	int greenRed = SendMessage(GetDlgItem(hdlg, IDC_GREENRED_SLIDER), TBM_GETPOS, 0, 0);
	if ( greenRed != mfd->par.greenRed ) {
		mfd->par.greenRed = greenRed;
		localRedoSystem( mfd, hdlg );
		return; 
	}
	
	int yellowBlue = SendMessage(GetDlgItem(hdlg, IDC_YELLOWBLUE_SLIDER), TBM_GETPOS, 0, 0);
	if ( yellowBlue != mfd->par.yellowBlue ) {
		mfd->par.yellowBlue = yellowBlue;
		localRedoSystem( mfd, hdlg );
		return;
	}

}

void getVerSliderValues(MyFilterData *mfd,HWND hdlg) {

	int histoSlider = SendMessage(GetDlgItem(hdlg, IDC_HISTO_SLIDER), TBM_GETPOS, 0, 0);
	if ( histoSlider != mfd->frameHistogram.getScaleSliderValue() ) {
		 mfd->frameHistogram.setScaleSliderValue(histoSlider);
		localRedoFrame( mfd, hdlg );
		return; 
	}

}


static void levelsButtonCallback(bool fNewState, void *pvData)
{
	HWND hdlg = (HWND)pvData;
	EnableWindow(GetDlgItem(hdlg, IDSAMPLE), fNewState);
	EnableWindow(GetDlgItem(hdlg, IDC_HISTO_SLIDER), fNewState);
	EnableWindow(GetDlgItem(hdlg, IDC_DISABLE_CHECK), fNewState);
	HWND wHist = GetDlgItem(hdlg, IDC_HISTO);
	if (fNewState == TRUE) {
		ShowWindow(wHist, SW_SHOWNORMAL);
	} else {
		ShowWindow(wHist, SW_HIDE);
	}

}

void setPanelDefaults(HWND hdlg, MyFilterData *mfd) {

	HWND hWnd = GetDlgItem(hdlg,IDC_CENTER_CORR_SLIDER);
	SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 100));
	SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->par.center_correct);

	hWnd = GetDlgItem(hdlg,IDC_BORDER_CORR_SLIDER);
	SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 100));
	SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->par.border_correct);

	hWnd = GetDlgItem(hdlg,IDC_GLOBAL_CORR_SLIDER);
	SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(-50, 50));
	SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->par.bright_correct);

	hWnd = GetDlgItem(hdlg,IDC_CONTRAST_SLIDER);
	SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(-MAX_CONTRAST, MAX_CONTRAST));
	SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->par.contrast );

	hWnd = GetDlgItem(hdlg,IDC_BRIGHT_SLIDER);
	SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(-MAX_BRIGHTNESS, MAX_BRIGHTNESS));
	SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->par.brightness);

	hWnd = GetDlgItem(hdlg,IDC_GREENRED_SLIDER);
	SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(-100, 100));
	SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->par.greenRed);

	hWnd = GetDlgItem(hdlg,IDC_YELLOWBLUE_SLIDER);
	SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(-100, 100));
	SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->par.yellowBlue);

	hWnd = GetDlgItem(hdlg,IDC_HISTO_SLIDER);
	SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 5));
	SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, 7);

	CheckDlgButton(hdlg, IDC_INTERLACED_CHECK, mfd->par.interlaced?BST_CHECKED:BST_UNCHECKED);
	CheckDlgButton(hdlg, IDC_INVERT_CHECK, mfd->par.invert_correction?BST_CHECKED:BST_UNCHECKED);
	CheckDlgButton(hdlg, IDC_KEEPCHROMA_CHECK, mfd->par.keep_chroma?BST_CHECKED:BST_UNCHECKED);

	CheckDlgButton(hdlg, IDC_SYNCHRO_CHECK, mfd->synchronized?BST_CHECKED:BST_UNCHECKED);

	SetDlgItemInt(hdlg, IDC_TITLE_SAMPLE, mfd->par.titleThreshold, FALSE);

}



BOOL CALLBACK radiaLumConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
    MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

    switch(msg) {

        case WM_INITDIALOG:
            SetWindowLong(hdlg, DWL_USER, lParam);
            mfd = (MyFilterData *)lParam;

			mfd->displayingPanel = true;
			EnableWindow(GetDlgItem(hdlg, IDSAMPLE), false);

			levelsButtonCallback(false, (void *)hdlg);
			mfd->ifp->SetButtonCallback(levelsButtonCallback, (void *)hdlg);

			// save for cancel ...
			mfd->save = mfd->par;
			HWND hWnd;

			hWnd = GetDlgItem(hdlg,IDPREVIEW);
			mfd->ifp->InitButton(hWnd);
			setPanelDefaults(hdlg,mfd);
			// update vales in boxes
			setDialogFeedbackValues(hdlg,mfd);

			return TRUE;


        case WM_COMMAND:

            switch(LOWORD(wParam)) {

            case IDOK:

				mfd->disableFilter = false;
				mfd->displayingPanel = false;
                EndDialog(hdlg, 0);
                return TRUE;

			case IDHELP:	{
				char prog[256];
				char path[256];
				LPTSTR ptr;

				GetModuleFileName(NULL, prog, 255);
				GetFullPathName(prog, 255, path, &ptr);
				*ptr = 0;
				strcat(path, "plugins\\RadiaLum21.htm");
				ShellExecute(hdlg, "open", path, NULL, NULL, SW_SHOWNORMAL);
				return TRUE;
				}

            case IDCANCEL:
				mfd->par = mfd->save;
				mfd->disableFilter = false;
				mfd->displayingPanel = false;
                EndDialog(hdlg, 1);
                return FALSE;

			case IDPREVIEW: {
				HWND hWnd = GetDlgItem(hdlg,IDPREVIEW);
				mfd->ifp->RedoFrame();
				mfd->ifp->Toggle(hWnd);
				break;
			}
			
			case IDSAMPLE:
				mfd->frameHistogram.invalidate(hdlg);
				break;

			case IDC_COLOR_DEFAULT :
				mfd->par.greenRed = mfd->par.yellowBlue = 0;
				SendMessage(GetDlgItem(hdlg,IDC_GREENRED_SLIDER), 
					TBM_SETPOS, (WPARAM)TRUE, mfd->par.greenRed);
				SendMessage(GetDlgItem(hdlg,IDC_YELLOWBLUE_SLIDER), 
					TBM_SETPOS, (WPARAM)TRUE, mfd->par.yellowBlue);
				localRedoSystem( mfd, hdlg );
				break;

			case IDC_LUMACORR_DEFAULT:
				mfd->par.contrast = mfd->par.brightness = 0;
				SendMessage(GetDlgItem(hdlg,IDC_CONTRAST_SLIDER), 
					TBM_SETPOS, (WPARAM)TRUE, mfd->par.contrast );
				SendMessage(GetDlgItem(hdlg,IDC_BRIGHT_SLIDER), 
					TBM_SETPOS, (WPARAM)TRUE, mfd->par.brightness);
				localRedoSystem( mfd, hdlg );
				break;

			case IDC_BRIGHT_CORR_DEFAULT:
				mfd->par.bright_correct = 0;
				SendMessage(GetDlgItem(hdlg,IDC_GLOBAL_CORR_SLIDER), 
					TBM_SETPOS, (WPARAM)TRUE, mfd->par.bright_correct);
				localRedoSystem( mfd, hdlg );
				break;

			case IDC_DISABLE_CHECK:
				mfd->disableFilter = ! mfd->disableFilter;
				localRedoSystem( mfd, hdlg );
		//		mfd->ifp->RedoSystem();
		//		mfd->frameHistogram.invalidate(hdlg);
				break;


			case IDC_SYNCHRO_CHECK:
				mfd->synchronized = ! mfd->synchronized;
				break;

			case IDC_INVERT_CHECK:
				mfd->par.invert_correction = ! mfd->par.invert_correction;
				localRedoSystem( mfd, hdlg );
				break;

			case IDC_KEEPCHROMA_CHECK:
				mfd->par.keep_chroma = ! mfd->par.keep_chroma;
				localRedoSystem( mfd, hdlg );
				break;

			case IDC_INTERLACED_CHECK:
				mfd->par.interlaced = ! mfd->par.interlaced;
				mfd->ifp->UndoSystem();
				mfd->ifp->RedoSystem();
				//setDialogFeedbackValues(hdlg,mfd);
				mfd->frameHistogram.invalidate(hdlg);
				break;

			case IDC_TITLE_SAMPLE: {
				int thresold = GetDlgItemInt(hdlg, IDC_TITLE_SAMPLE, NULL, FALSE);
				if (thresold != mfd->par.titleThreshold) {
					if ( thresold > 255 )  {
						thresold=255;
						SetDlgItemInt(hdlg, IDC_TITLE_SAMPLE, thresold, FALSE);
					}
					mfd->par.titleThreshold=thresold;
					mfd->ifp->RedoFrame();
					mfd->frameHistogram.invalidate(hdlg);
				}
				} break;

		}	// case WM_COMMAND:

		case WM_HSCROLL: 
			
			if ( //(LOWORD(wParam) == TB_THUMBTRACK) ||
				 (LOWORD(wParam) == TB_ENDTRACK) ) 
				getHorSliderValues(mfd,hdlg);
			break;	
			

		case WM_VSCROLL: 
			if (LOWORD(wParam) == TB_ENDTRACK)
				getVerSliderValues(mfd,hdlg);
			break;	


		case WM_PAINT:
			mfd->frameHistogram.paintHisto(hdlg);
			break;
    }

    return FALSE;
}

int radiaLumConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd) {

	MyFilterData * mfd = (MyFilterData *)fa->filter_data;
	mfd->ifp = fa->ifp;
	mfd->displayingPanel = false;
    return DialogBoxParam(fa->filter->module->hInstModule,
            MAKEINTRESOURCE(IDD_FILTER_LUM8MM), hwnd,
            radiaLumConfigDlgProc, (LPARAM)fa->filter_data
	);
}


void radiaLumStringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str) {

    MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	sprintf(str,"(bor=%d,cen=%d,bric=%d,inter=%s,bri=%d,con=%d,g-r=%d,y-b=%d,tit=%d,inv=%s,kc=%s)",

		mfd->par.border_correct, mfd->par.center_correct,mfd->par.bright_correct,
		mfd->par.interlaced ? "y": "n",
		mfd->par.brightness,  mfd->par.contrast,
		mfd->par.greenRed, mfd->par.yellowBlue,
		mfd->par.titleThreshold,
		mfd->par.invert_correction ? "y": "n",
		mfd->par.keep_chroma ? "y": "n"

	);

}

void radiaLumScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {

    FilterActivation *fa = (FilterActivation *)lpVoid;
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	int seq=0;

	mfd->par.border_correct = argv[seq++].asInt();
	mfd->par.center_correct = argv[seq++].asInt();
	mfd->par.bright_correct = argv[seq++].asInt();
	mfd->par.interlaced = ((argv[seq++].asInt()==0) ? false : true);
	mfd->par.brightness= argv[seq++].asInt();
	mfd->par.contrast = argv[seq++].asInt();	
	mfd->par.greenRed = argv[seq++].asInt();
	mfd->par.yellowBlue = argv[seq++].asInt();
	mfd->par.titleThreshold = argv[seq++].asInt();
	mfd->par.invert_correction = ((argv[seq++].asInt()==0) ? false : true);
	mfd->par.keep_chroma = ((argv[seq++].asInt()==0) ? false : true);

}


bool radiaLumFssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

    _snprintf(buf, buflen, "Config(%d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d)",

		mfd->par.border_correct,  mfd->par.center_correct, mfd->par.bright_correct,
		(mfd->par.interlaced ? 1: 0),
		mfd->par.brightness,mfd->par.contrast, 
		mfd->par.greenRed ,	mfd->par.yellowBlue,
		mfd->par.titleThreshold,
		(mfd->par.invert_correction ? 1: 0),
		(mfd->par.keep_chroma ? 1: 0)
	);

    return true;
}
