/*//////////////////////////////////////////////////////////////////////////

	flaXen's Noise Filter for VirtualDub
	(c) Copyright 2001 Dan Flower (flaXen)

	Add quasi-analog-style noise to video

*//////////////////////////////////////////////////////////////////////////

#include <windows.h>
#include <stdio.h>

#include "resource.h"
#include "filter.h"
#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

typedef struct MyFilterData {
	Pixel32	*Tmp1, *Tmp2, *Tmp3;
	long	*ytab, *itab, *qtab;
	long	*rtab, *gtab, *btab;
	long	*cytf;
	int		*lNoise, *cNoise, *plNoise, *pcNoise;
	int		lHB, lVB, lEI, lHP;
	int		cHB, cVB, cEI, cHP;
	int		dFreq, dHalo, dTBC;
	int		dGhost, dCBand, dDrop;
	int		YIQin, YIQout, YUVtog;
} MyFilterData;

///////////////////////////////////////////////////////////////////////////

int		fxNoiseRunProc(const FilterActivation *fa, const FilterFunctions *ff);
void	fxNoise_RGB2YIQ(Pixel32 *, Pixel32 *, long, long, long, MyFilterData *);
void	fxNoise_YIQ2RGB(Pixel32 *, Pixel32 *, long, long, long, MyFilterData *);
void	fxNoise_LumaNoise(Pixel32 *, Pixel32 *, long, long, long, MyFilterData *);
void	fxNoise_ChromaNoise(Pixel32 *, Pixel32 *, long, long, long, long, MyFilterData *);
void	fxNoise_Effects(Pixel32 *, Pixel32 *, long, long, long, MyFilterData *);
void	fxNoise_eHighFreq(Pixel32 *, Pixel32 *, long, long, long, MyFilterData *);
void	fxNoise_eHalo(Pixel32 *, Pixel32 *, long, long, long, MyFilterData *);
void	fxNoise_eTBC(Pixel32 *, Pixel32 *, long, long, long, MyFilterData *);
void	fxNoise_eGhost(Pixel32 *, Pixel32 *, long, long, long, MyFilterData *);
void	fxNoise_eCBands(Pixel32 *, Pixel32 *, long, long, long, MyFilterData *);
void	fxNoise_eDropouts(Pixel32 *, Pixel32 *, long, long, long, MyFilterData *);
int		fxNoiseStartProc(FilterActivation *fa, const FilterFunctions *ff);
int		fxNoiseEndProc(FilterActivation *fa, const FilterFunctions *ff); 
int		fxNoiseInitProc(FilterActivation *fa, const FilterFunctions *ff);
int		fxNoiseConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void	fxNoiseStringProc(const FilterActivation *fa, const FilterFunctions *ff, char *buf);
void	fxNoiseScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool	fxNoiseFssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);
long	rnd(long);

long	seed = 63281735;

///////////////////////////////////////////////////////////////////////////

ScriptFunctionDef fxNoise_func_defs[]={
    { (ScriptFunctionPtr)fxNoiseScriptConfig, "Config", "0iiii" },
    { NULL },
};

CScriptObject fxNoiseobj={
    NULL, fxNoise_func_defs
};

struct FilterDefinition filterDef_fxNoise = {
	NULL, NULL, NULL,		// next, prev, module

	"Analog Noise",	// name
	"flaXen's Analog Noise Tool",	// desc
	"flaXen",				// maker

	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size
	fxNoiseInitProc,			// initProc
	NULL,					// deinitProc
	fxNoiseRunProc,			// runProc
	NULL,					// paramProc
	fxNoiseConfigProc,		// configProc
	fxNoiseStringProc,		// stringProc
	fxNoiseStartProc,			// startProc
	fxNoiseEndProc,			// endProc
	&fxNoiseobj,				// script_obj
	fxNoiseFssProc,			// fssProc
};

///////////////////////////////////////////////////////////////////////////

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_fxNoise;

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
	if (!(fd_fxNoise = ff->addFilter(fm, &filterDef_fxNoise, sizeof(FilterDefinition))))
		return 1;

	vdfd_ver    = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;
	return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
	ff->removeFilter(fd_fxNoise);
}

///////////////////////////////////////////////////////////////////////////

int fxNoiseRunProc(const FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData	*mfd = (MyFilterData *)fa->filter_data;
	Pixel32	*isrc, *idst, *itmp1, *itmp2;
	const long	swid = fa->src.w, shei = fa->src.h;
	const long	spit = fa->src.pitch >> 2;
	const long	ssize = fa->src.pitch * fa->src.h;

	isrc = (Pixel32 *)fa->src.data;
	idst = (Pixel32 *)fa->dst.data;
	itmp1 = (Pixel32 *)mfd->Tmp1;
	itmp2 = (Pixel32 *)mfd->Tmp2;

	if (mfd->YIQin == BST_CHECKED) {
		// YUV/YIQ colorspace input
		if (mfd->YIQout == BST_UNCHECKED) {
			// RGB colorspace output
			//memcpy(itmp, isrc, ssize);
			fxNoise_LumaNoise(isrc, itmp1, swid, shei, spit, mfd);
			fxNoise_Effects(itmp1, itmp2, swid, shei, spit, mfd);
			fxNoise_ChromaNoise(itmp2, itmp1, swid, shei, spit, 0, mfd);
			fxNoise_ChromaNoise(itmp1, itmp2, swid, shei, spit, 1, mfd);
			fxNoise_YIQ2RGB(itmp2, idst, swid, shei, spit, mfd);
		} else {
			// YIQ/YUV colorspace output
			//memcpy(idst, isrc, ssize);
			fxNoise_LumaNoise(isrc, itmp1, swid, shei, spit, mfd);
			fxNoise_Effects(itmp1, itmp2, swid, shei, spit, mfd);
			fxNoise_ChromaNoise(itmp2, itmp1, swid, shei, spit, 0, mfd);
			fxNoise_ChromaNoise(itmp1, idst, swid, shei, spit, 1, mfd);
		}
	} else {
		// RGB colorspace input
		if (mfd->YIQout == BST_UNCHECKED) {
			// RGB colorspace output
			fxNoise_RGB2YIQ(isrc, itmp1, swid, shei, spit, mfd);
			fxNoise_LumaNoise(itmp1, itmp2, swid, shei, spit, mfd);
			fxNoise_Effects(itmp2, itmp1, swid, shei, spit, mfd);
			fxNoise_ChromaNoise(itmp1, itmp2, swid, shei, spit, 0, mfd);
			fxNoise_ChromaNoise(itmp2, itmp1, swid, shei, spit, 1, mfd);
			fxNoise_YIQ2RGB(itmp1, idst, swid, shei, spit, mfd);
			
		} else {
			// YIQ colorspace output
			fxNoise_RGB2YIQ(isrc, itmp1, swid, shei, spit, mfd);
			fxNoise_LumaNoise(itmp1, itmp2, swid, shei, spit, mfd);
			fxNoise_Effects(itmp2, itmp1, swid, shei, spit, mfd);
			fxNoise_ChromaNoise(itmp1, itmp2, swid, shei, spit, 0, mfd);
			fxNoise_ChromaNoise(itmp2, idst, swid, shei, spit, 1, mfd);
			//memcpy(idst, itmp, ssize);
		}
	}

	return 0;
}

///////////////////////////////////////////////////////////////////////////

void	fxNoise_LumaNoise(Pixel32 *isrc, Pixel32 *idst, long swid, long shei, long spit, MyFilterData *mfd) {
	Pixel32	*src, *dst;
	int		*lns = mfd->lNoise, *cns = mfd->cNoise;
	int		*rns = mfd->plNoise;
	int		*lnp, *cnp, *rnp;
	double	p;
	long	x, y, c, d, n, cy, ciq;

	for (y = 0; y < shei; y++) {
		lnp = (int *)lns + y * spit;
		rnp = (int *)rns + rnd(shei) * spit;
		memcpy(lnp, rnp, spit * 4);
	}

	if (mfd->lVB > 0) {
		n = mfd->lVB * 2 + 1;
		for (y = mfd->lVB; y < swid - mfd->lVB; y++) {
			cnp = (int *)cns + y * spit;
			for (x = 0; x < swid; x++) {
				c = 0;
				lnp = (int *)lns + (y - mfd->lVB) * spit + x;
				for (d = 0; d <= mfd->lVB * 2; d++) {
					c += *lnp;
					lnp += spit;
				}
				*cnp++ = c / n;
			}
		}
		cnp = lns; lns = cns; cns = cnp;
	}

	for (y = 0; y < shei; y++) {
		src = isrc + y * spit;
		dst = idst + y * spit;
		lnp = (int *)lns + y * spit;
		for (x = 1; x < swid - 1; x++) {
			c = *src++;
			cy = ((c >> 16) & 0x00FFL);
			ciq = c & 0xFF00FFFFL;

			if (mfd->lHP > 0) {
				p = (double)(rnd(mfd->lHP) % 100) / 100.0;
				c = rnd(10);
				if (c == 0) {
					c = ((*(src - 2)) >> 16) & 0x00FFL;
					cy = (long)((double)cy * (1.0 - p) + c * p);
				} else if (c == 1) {
					c = ((*src) >> 16) & 0x00FFL;
					cy = (long)((double)cy * (1.0 - p) + c * p);
				}
				
			}

			cy += *lnp++;
			if (cy < 0) cy = 0;
			else if (cy > 255) cy = 255;
			*dst++ = (cy << 16) | ciq;
		}
	}
}

///////////////////////////////////////////////////////////////////////////

void	fxNoise_ChromaNoise(Pixel32 *isrc, Pixel32 *idst, long swid, long shei, long spit, long chan, MyFilterData *mfd) {
	Pixel32	*src, *dst;
	int		*lns = mfd->lNoise, *cns = mfd->cNoise;
	int		*rns = mfd->pcNoise;
	int		*lnp, *cnp, *rnp;
	double	p;
	long	x, y, c, d, n, cy, ciq;
	long	csh, cmk;

	if (chan == 0) {
		csh = 8; cmk = 0x000000FFL;
	} else {
		csh = 0; cmk = 0x0000FF00L;
	}

	for (y = 0; y < shei; y++) {
		lnp = (int *)lns + y * spit;
		rnp = (int *)rns + rnd(shei) * spit;
		memcpy(lnp, rnp, spit * 4);
	}

	if (mfd->cVB > 0) {
		n = mfd->cVB * 2 + 1;
		for (y = mfd->cVB; y < swid - mfd->cVB; y++) {
			cnp = (int *)cns + y * spit;
			for (x = 0; x < swid; x++) {
				c = 0;
				lnp = (int *)lns + (y - mfd->cVB) * spit + x;
				for (d = 0; d <= mfd->cVB * 2; d++) {
					c += *lnp;
					lnp += spit;
				}
				*cnp++ = c / n;
			}
		}
		cnp = lns; lns = cns; cns = cnp;
	}

	for (y = 0; y < shei; y++) {
		src = isrc + y * spit;
		dst = idst + y * spit;
		lnp = (int *)lns + y * spit;
		for (x = 1; x < swid - 1; x++) {
			c = *src++;
			cy = (c & (0xFFFF0000L | cmk));
			ciq = (c >> csh) & 0x00FFL;

			if (mfd->lHP > 0) {
				p = (double)(rnd(mfd->lHP) % 100) / 100.0;
				c = rnd(10);
				if (c == 0) {
					c = ((*(src - 2)) >> csh) & 0x00FFL;
					ciq = (long)((double)ciq * (1.0 - p) + c * p);
				} else if (c == 1) {
					c = ((*src) >> csh) & 0x00FFL;
					ciq = (long)((double)ciq * (1.0 - p) + c * p);
				}
			}

			ciq += *lnp++;
			if (ciq < 0) ciq = 0;
			else if (ciq > 255) ciq = 255;
			*dst++ = cy | (ciq << csh);
		}
	}
}

///////////////////////////////////////////////////////////////////////////

void	fxNoise_Effects(Pixel32 *isrc, Pixel32 *idst, long swid, long shei, long spit, MyFilterData *mfd) {
	Pixel32		*esrc = isrc, *edst = mfd->Tmp3;
	const long	ssize = spit * shei * 4;

	if (mfd->dHalo > 0) {
		fxNoise_eHalo(esrc, edst, swid, shei, spit, mfd);
		memcpy(esrc, edst, ssize);
	}
	if (mfd->dFreq > 0) {
		fxNoise_eHighFreq(esrc, edst, swid, shei, spit, mfd);
		memcpy(esrc, edst, ssize);
	}
	if (mfd->dGhost > 0) {
		fxNoise_eGhost(esrc, edst, swid, shei, spit, mfd);
		memcpy(esrc, edst, ssize);
	}

	if (mfd->dTBC > 0) {
		fxNoise_eTBC(esrc, edst, swid, shei, spit, mfd);
		memcpy(esrc, edst, ssize);
	}

	memcpy(idst, edst, ssize);
}

///////////////////////////////////////////////////////////////////////////

void	fxNoise_eHighFreq(Pixel32 *isrc, Pixel32 *idst, long swid, long shei, long spit, MyFilterData *mfd) {
	Pixel32		*src, *dst;
	long		x, y, c, l, ll, r, rr, a, t;
	long		pcalc1[256], pcalc2[256];

	for (x = 0; x < 256; x++) {
		pcalc1[x] = (long)(double)((double)x / 100.0 * (double)(100 - mfd->dFreq));
		pcalc2[x] = (long)(double)((double)x / 100.0 * (double)mfd->dFreq);
	}

	for (y = 0; y < shei; y++) {
		src = isrc + y * spit + 2;
		dst = idst + y * spit + 2;
		ll = (*(src - 2) >> 16) & 0x00FFL;
		l = (*(src - 1) >> 16) & 0x00FFL;
		c = (*src >> 16) & 0x00FFL;
		r = (*(src + 1) >> 16) & 0x00FFL;
		rr = (*(src + 2) >> 16) & 0x00FFL;
		for (x = 2; x < swid - 2; x++) {
			a = ((l + c + r + ((ll + rr) >> 1)) >> 2) & 0x00FFL;
			t = *src;
			*dst++ = (t & 0xFF00FFFFL) | ((pcalc2[a] + pcalc1[(t >> 16) & 0x00FFL]) << 16);
			ll = l; l = c; c = r; r = rr; rr = (*++src >> 16) & 0x00FFL;
		}
	}
}

///////////////////////////////////////////////////////////////////////////

void	fxNoise_eHalo(Pixel32 *isrc, Pixel32 *idst, long swid, long shei, long spit, MyFilterData *mfd) {
	Pixel32		*src, *dst;
	long		x, y, c, a, t, s, n;

	s = spit * 2;
	n = mfd->dHalo;
	if (n == 100) n = 99;
	n = (100 - n) << 16;

	for (y = 2; y < shei - 2; y++) {
		src = isrc + y * spit + 2;
		dst = idst + y * spit + 2;
		for (x = 2; x < swid - 2; x++) {
			c = (*(src - s - 1) >> 16) + (*(src - s) >> 16) + (*(src - s + 1) >> 16);
			c+= (*(src - spit - 2) >> 16) + (*(src - spit - 1) >> 16) + (*(src - spit) >> 16) + (*(src - spit + 1) >> 16) + (*(src - spit + 2) >> 16);
			c+= (*(src - 2) >> 16) + (*(src - 1) >> 16) + (*(src + 1) >> 16) + (*(src + 2) >> 16);
			c+= (*(src + spit - 2) >> 16) + (*(src + spit - 1) >> 16) + (*(src + spit) >> 16) + (*(src + spit + 1) >> 16) + (*(src + spit + 2) >> 16);
			c+= (*(src + s - 1) >> 16) + (*(src + s) >> 16) + (*(src + s + 1) >> 16);
			a = ((c / 20) & 0x00FFL);
			c = *src++;
			t = (c >> 16) & 0x00FFL;
			a = (((t - a) * 100) << 16) / n - 24;
			if (a > 0) {
				a += t;
				if (a < 0) a = 0;
				else if (a > 255) a = 255;
				*dst++ = (c & 0xFF00FFFFL) | (a << 16);
			} else {
				*dst++ = c;
			}
		}
	}
}

///////////////////////////////////////////////////////////////////////////

void	fxNoise_eTBC(Pixel32 *isrc, Pixel32 *idst, long swid, long shei, long spit, MyFilterData *mfd) {
	Pixel32		*src, *dst;
	long		x, y, i, t1, t2, r, g, b;
	double		o, il, ir;

	for (y = 2; y < shei - 2; y++) {
		o = (double)rnd(100);
		if (o > 25.0) o = 25.0;
		o = o / 20.0 / 100.0 * (double)mfd->dTBC;
		i = (long)o;
		ir = o - (double)i; il = 1.0 - ir;
		dst = idst + y * spit;
		for (x = 0; x < i; x++)
			*dst++ = 0x00007F7F;
		dst = idst + y * spit + i;
		src = isrc + y * spit;
		for (x = i; x < swid - 1; x++) {
			t1 = *src++ & 0x00FFFFFFL;
			t2 = *src & 0x00FFFFFFL;
			r = (long)(double)((double)(t1 >> 16) * il + (double)(t2 >> 16) * ir);
			g = (long)(double)((double)((t1 >> 8) & 0x00FFL) * il + (double)((t2 >> 8) & 0x00FFL) * ir);
			b = (long)(double)((double)(t1 & 0x00FFL) * il + (double)(t2 & 0x00FFL) * ir);
			if (r < 0) r = 0;
			else if (r > 255) r = 255;
			if (g < 0) g = 0;
			else if (g > 255) g = 255;
			if (b < 0) b = 0;
			else if (b > 255) b = 255;
			//r = 0; g = 127; b = 127;
			*dst++ = (r << 16) | (g << 8) | b;
		}
	}
}

///////////////////////////////////////////////////////////////////////////

void	fxNoise_eGhost(Pixel32 *isrc, Pixel32 *idst, long swid, long shei, long spit, MyFilterData *mfd) {
	Pixel32		*src, *dst;
	long		x, y, c, a, t, i, d, n;
//	long		pcalc1[256];

	n = mfd->dGhost;
	if (n == 100) n = 99;
	n = (100 - n) << 16;

//	for (d = 0; d < 256; d++)
//		pcalc1[d] = (long)(double)((double)d / 100.0 * (double)mfd->dGhost);

	for (y = 0; y < shei; y++) {
		src = isrc + y * spit + 3;
		dst = idst + y * spit + 3;
		//c = (*src >> 16) & 0x00FFL; a = 1;
		c = 0; a = 0;
		for (x = 3; x < swid; x++) {
			t = *src++;
			i = (t >> 16) & 0x00FFL;
			c += i; a++;
			c -= c >> 6;
			d = c / a;
			i -= (((d / 2 - d) * 100) << 16) / n;
			//i += pcalc1[d - (d >> 1)];
			//i += (long)(double)(((double)i / 256.0 * (double)c) / 100.0 * (double)mfd->dGhost);
			//i = (long)(double)((double)i / 100.0 * (double)(100 - mfd->dGhost) + (double)d / 100.0 * (double)mfd->dGhost);
			if (i < 0) i = 0;
			else if (i > 255) i = 255;
			*dst++ = (t & 0xFF00FFFFL) | (i << 16);
		}
	}
}

///////////////////////////////////////////////////////////////////////////

void	fxNoise_eCBands(Pixel32 *isrc, Pixel32 *idst, long swid, long shei, long spit, MyFilterData *mfd) {

}

///////////////////////////////////////////////////////////////////////////

void	fxNoise_eDropouts(Pixel32 *isrc, Pixel32 *idst, long swid, long shei, long spit, MyFilterData *mfd) {

}

///////////////////////////////////////////////////////////////////////////

void	fxNoise_RGB2YIQ(Pixel32 *isrc, Pixel32 *idst, long swid, long shei, long spit, MyFilterData *mfd) {
	Pixel32	*src, *dst;
	long	y, c;
	long	*ytab = (long *)mfd->ytab;
	long	*itab = (long *)mfd->itab;
	long	*qtab = (long *)mfd->qtab;

	if (mfd->YUVtog == BST_CHECKED) {
		// Convert RGB -> YIQ
		for (y = 0; y < shei; y++) {
			src = isrc + y * spit;
			dst = idst + y * spit;
			__asm {
				push	esi
				push	edi
				mov		ecx, [swid]

cty10:			mov		esi, [src]
				add		[src], 4
				mov		edx, [esi]
				mov		ebx, edx
				shr		ebx, 6
				and		ebx, 0x0003FCL
				mov		eax, edx
				shr		eax, 14
				and		eax, 0x0003FCL
				shl		edx, 2
				and		edx, 0x0003FCL
				add		ebx, 1024
				add		edx, 2048

				mov		esi, [ytab]
				mov		edi, [esi + eax]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cty20
				test	edi, 0xFF000000L
				jz		cty21
				mov		edi, 0x00FF0000L
				jmp		cty21
cty20:			xor		edi, edi
cty21:			mov		[c], edi

				mov		esi, [itab]
				mov		edi, [esi + eax]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cty30
				test	edi, 0xFF000000L
				jz		cty31
				mov		edi, 0x00FF0000L
				jmp		cty31
cty30:			xor		edi, edi
cty31:			shr		edi, 8
				or		[c], edi

				mov		esi, [qtab]
				mov		edi, [esi + eax]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cty40
				test	edi, 0xFF000000L
				jz		cty41
				mov		edi, 0x00FF0000L
cty40:			xor		edi, edi
cty41:			shr		edi, 16
				or		[c], edi

				mov		edi, [dst]
				add		[dst], 4
				mov		eax, [c]
				mov		[edi], eax

				dec		ecx
				jnz		cty10
				pop		edi
				pop		esi
			}
		}
	} else {
		// Convert RGB -> YUV
		for (y = 0; y < shei; y++) {
			src = isrc + y * spit;
			dst = idst + y * spit;
			__asm {
				push	esi
				push	edi
				mov		ecx, [swid]

ctu10:			mov		esi, [src]
				add		[src], 4
				mov		edx, [esi]
				mov		ebx, edx
				shr		ebx, 6
				and		ebx, 0x0003FCL
				mov		eax, edx
				shr		eax, 14
				and		eax, 0x0003FCL
				shl		edx, 2
				and		edx, 0x0003FCL
				add		ebx, 1024
				add		edx, 2048

				mov		esi, [ytab]
				mov		edi, [esi + eax]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		ctu20
				test	edi, 0xFF000000L
				jz		ctu21
				mov		edi, 0x00FF0000L
				jmp		ctu21
ctu20:			xor		edi, edi
ctu21:			mov		[c], edi

				mov		esi, [itab]
				mov		edi, [esi + eax]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		ctu30
				test	edi, 0xFF000000L
				jz		ctu31
				mov		edi, 0x00FF0000L
				jmp		ctu31
ctu30:			xor		edi, edi
ctu31:			shr		edi, 8
				or		[c], edi

				mov		esi, [qtab]
				mov		edi, [esi + eax]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		ctu40
				test	edi, 0xFF000000L
				jz		ctu41
				mov		edi, 0x00FF0000L
				jmp		ctu41
ctu40:			xor		edi, edi
ctu41:			shr		edi, 16
				or		[c], edi

				mov		edi, [dst]
				add		[dst], 4
				mov		eax, [c]
				mov		[edi], eax

				dec		ecx
				jnz		ctu10
				pop		edi
				pop		esi
			}
		}
	}
}

///////////////////////////////////////////////////////////////////////////

void	fxNoise_YIQ2RGB(Pixel32 *isrc, Pixel32 *idst, long swid, long shei, long spit, MyFilterData *mfd) {
	Pixel32	*src, *dst;
	long	y, c;
	long	*rtab = (long *)mfd->rtab;
	long	*gtab = (long *)mfd->gtab;
	long	*btab = (long *)mfd->btab;
	long	*cytf = (long *)mfd->cytf;

	if (mfd->YUVtog == BST_CHECKED) {
		for (y = 0; y < shei; y++) {
			src = isrc + y * spit;
			dst = idst + y * spit;
			__asm {
				push	esi
				push	edi
				mov		ecx, [swid]

cfy10:			mov		esi, [src]
				add		[src], 4
				mov		edx, [esi]
				mov		ebx, edx
				shr		ebx, 6
				and		ebx, 0x0003FCL
				mov		eax, edx
				and		eax, 0x00FF0000L
				shl		edx, 2
				and		edx, 0x0003FCL
				add		edx, 1024

				mov		edi, eax
				mov		esi, [rtab]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cfy20
				test	edi, 0xFF000000L
				jz		cfy21
				mov		edi, 0x00FF0000L
				jmp		cfy21
cfy20:			xor		edi, edi
cfy21:			mov		[c], edi

				mov		edi, eax
				mov		esi, [gtab]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cfy30
				test	edi, 0xFF000000L
				jz		cfy31
				mov		edi, 0x00FF0000L
				jmp		cfy31
cfy30:			xor		edi, edi
cfy31:			shr		edi, 8
				or		[c], edi

				mov		edi, eax
				mov		esi, [btab]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cfy40
				test	edi, 0xFF000000L
				jz		cfy41
				mov		edi, 0x00FF0000L
				jmp		cfy41
cfy40:			xor		edi, edi
cfy41:			shr		edi, 16
				or		[c], edi

				mov		edi, [dst]
				add		[dst], 4
				mov		eax, [c]
				mov		[edi], eax

				dec		ecx
				jnz		cfy10
				pop		edi
				pop		esi
			}
		}
	} else {
		for (y = 0; y < shei; y++) {
			src = isrc + y * spit;
			dst = idst + y * spit;
			__asm {
				push	esi
				push	edi
				mov		ecx, [swid]

cfu10:			mov		esi, [src]
				add		[src], 4
				mov		edx, [esi]
				mov		ebx, edx
				shr		ebx, 6
				and		ebx, 0x0003FCL
				mov		eax, edx
				shr		eax, 14
				and		eax, 0x0003FCL
				shl		edx, 2
				and		edx, 0x0003FCL
				add		edx, 1024

				mov		esi, [cytf]
				mov		eax, [esi + eax]

				mov		edi, eax
				mov		esi, [rtab]
				add		edi, [esi + ebx]
				and		edi, 0xFFFF0000L
				js		cfu20
				test	edi, 0xFF000000L
				jz		cfu21
				mov		edi, 0x00FF0000L
				jmp		cfu21
cfu20:			xor		edi, edi
cfu21:			mov		[c], edi

				mov		edi, eax
				mov		esi, [gtab]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cfu30
				test	edi, 0xFF000000L
				jz		cfu31
				mov		edi, 0x00FF0000L
				jmp		cfu31
cfu30:			xor		edi, edi
cfu31:			shr		edi, 8
				or		[c], edi

				mov		edi, eax
				mov		esi, [btab]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cfu40
				test	edi, 0xFF000000L
				jz		cfu41
				mov		edi, 0x00FF0000L
				jmp		cfu41
cfu40:			xor		edi, edi
cfu41:			shr		edi, 16
				or		[c], edi

				mov		edi, [dst]
				add		[dst], 4
				mov		eax, [c]
				mov		[edi], eax

				dec		ecx
				jnz		cfu10
				pop		edi
				pop		esi
			}
		}
	}
}

///////////////////////////////////////////////////////////////////////////

int fxNoiseStartProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	int		*lns, *cns, *lnp, *cnp;
	long	d, i, c, n, x;

	// Temporary frame buffer
	d = fa->src.pitch * fa->src.h;
	if (!(mfd->Tmp1 = new Pixel32[d])) return 1;
	if (!(mfd->Tmp2 = new Pixel32[d])) return 1;
	if (!(mfd->Tmp3 = new Pixel32[d])) return 1;

	if (!(mfd->lNoise = new int[d])) return 1;
	if (!(mfd->cNoise = new int[d])) return 1;
	if (!(mfd->plNoise = new int[d])) return 1;
	if (!(mfd->pcNoise = new int[d])) return 1;

	for (i = 0; i < d; i++) {
		mfd->lNoise[i] = (int)(double)((double)(rnd(512) - 256) / 100.0 * (double)mfd->lEI + 0.5);
		mfd->cNoise[i] = (int)(double)((double)(rnd(512) - 256) / 100.0 * (double)mfd->cEI + 0.5);
	}

	lns = mfd->lNoise; cns = mfd->plNoise;
	memcpy(cns, lns, d);
	if (mfd->lHB > 0) {
		n = mfd->lHB * 2 + 1;
		cnp = (int *)mfd->plNoise + mfd->lHB;
		for (i = mfd->lHB; i < d - mfd->lHB; i++) {
			c = 0;
			lnp = (int *)lns + i - mfd->lHB;
			for (x = 0; x <= mfd->lHB * 2; x++) 
				c += *lnp++;
			*cnp++ = c / n;
		}
	}

	lns = mfd->cNoise; cns = mfd->pcNoise;
	memcpy(cns, lns, d);
	if (mfd->cHB > 0) {
		n = mfd->cHB * 2 + 1;
		cnp = (int *)mfd->pcNoise + mfd->cHB;
		for (i = mfd->cHB; i < d - mfd->cHB; i++) {
			c = 0;
			lnp = (int *)lns + i - mfd->cHB;
			for (x = 0; x <= mfd->cHB * 2; x++) 
				c += *lnp++;
			*cnp++ = c / n;
		}
	}

	// Precalc tables
	//if (!(mfd->pcalc1 = new long[768])) return 1;
	//if (!(mfd->pcalc2 = new long[768])) return 1;

	// Y transform for YUV -> RGB conversion
	if (!(mfd->cytf = new long[256])) return 1;

	// RGB -> Ycc conversion tables
	if (!(mfd->ytab = new long[768])) return 1;
	if (!(mfd->itab = new long[768])) return 1;
	if (!(mfd->qtab = new long[768])) return 1;

	// Ycc -> RGB conversion tables
	if (!(mfd->rtab = new long[512])) return 1;
	if (!(mfd->gtab = new long[512])) return 1;
	if (!(mfd->btab = new long[512])) return 1;

	// Calculate data for RGB -> Ycc conversion tables
	for (d = 0; d < 256; d++) {
		if (mfd->YUVtog == BST_CHECKED) {
			// RGB -> YIQ transformation matrix
			mfd->ytab[d      ] = (long)(+0.299 * d * 65536 + 0.5);
			mfd->ytab[d + 256] = (long)(+0.587 * d * 65536 + 0.5);
			mfd->ytab[d + 512] = (long)(+0.114 * d * 65536 + 0.5);
			mfd->itab[d      ] = (long)(+0.596 * d * 65536 + 0.5) + 8388608;
			mfd->itab[d + 256] = (long)(-0.275 * d * 65536 + 0.5);
			mfd->itab[d + 512] = (long)(-0.321 * d * 65536 + 0.5);
			mfd->qtab[d      ] = (long)(+0.212 * d * 65536 + 0.5) + 8388608;
			mfd->qtab[d + 256] = (long)(-0.523 * d * 65536 + 0.5);
			mfd->qtab[d + 512] = (long)(+0.311 * d * 65536 + 0.5);
			// YIQ -> RGB transformation matrix
			mfd->rtab[d      ] = (long)(+0.956 * (d-128) * 65536 + 0.5);
			mfd->rtab[d + 256] = (long)(+0.621 * (d-128) * 65536 + 0.5);
			mfd->gtab[d      ] = (long)(-0.272 * (d-128) * 65536 + 0.5);
			mfd->gtab[d + 256] = (long)(-0.647 * (d-128) * 65536 + 0.5);
			mfd->btab[d      ] = (long)(-1.105 * (d-128) * 65536 + 0.5);
			mfd->btab[d + 256] = (long)(+1.702 * (d-128) * 65536 + 0.5);
		} else {
			// RGB -> YUV transformation matrix
			mfd->ytab[d      ] = (long)(+0.257 * d * 65536 + 0.5) + 1048576;
			mfd->ytab[d + 256] = (long)(+0.504 * d * 65536 + 0.5);
			mfd->ytab[d + 512] = (long)(+0.098 * d * 65536 + 0.5);
			mfd->itab[d      ] = (long)(+0.439 * d * 65536 + 0.5) + 8388608;
			mfd->itab[d + 256] = (long)(-0.368 * d * 65536 + 0.5);
			mfd->itab[d + 512] = (long)(-0.071 * d * 65536 + 0.5);
			mfd->qtab[d      ] = (long)(-0.148 * d * 65536 + 0.5) + 8388608;
			mfd->qtab[d + 256] = (long)(-0.291 * d * 65536 + 0.5);
			mfd->qtab[d + 512] = (long)(+0.439 * d * 65536 + 0.5);
			// YUV -> RGB transformation matrix
			mfd->rtab[d      ] = (long)(+1.596 * (d-128) * 65536 + 0.5);
			mfd->rtab[d + 256] = 0;
			mfd->gtab[d      ] = (long)(-0.813 * (d-128) * 65536 + 0.5);
			mfd->gtab[d + 256] = (long)(-0.391 * (d-128) * 65536 + 0.5);
			mfd->btab[d      ] = 0;
			mfd->btab[d + 256] = (long)(+2.018 * (d-128) * 65536 + 0.5);
			mfd->cytf[d] = (long)(((d - 16) * 1.164) * 65536.0 + 0.5);	// Y transform
		}
	}

	return 0;
}

///////////////////////////////////////////////////////////////////////////

int fxNoiseEndProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	// Delete temporary frame buffer
	delete[] mfd->Tmp1; mfd->Tmp1 = NULL;
	delete[] mfd->Tmp2; mfd->Tmp2 = NULL;
	delete[] mfd->Tmp3; mfd->Tmp3 = NULL;
	delete[] mfd->lNoise; mfd->lNoise = NULL;
	delete[] mfd->cNoise; mfd->cNoise = NULL;
	delete[] mfd->plNoise; mfd->plNoise = NULL;
	delete[] mfd->pcNoise; mfd->pcNoise = NULL;

	// Delete precalc tables
	//delete[] mfd->pcalc1; mfd->pcalc1 = NULL;
	//delete[] mfd->pcalc2; mfd->pcalc2 = NULL;

	// Delete RGB -> YIQ/YUV conversion tables
	delete[] mfd->ytab; mfd->ytab = NULL;
	delete[] mfd->itab; mfd->itab = NULL;
	delete[] mfd->qtab; mfd->qtab = NULL;
	delete[] mfd->cytf; mfd->cytf = NULL;
	delete[] mfd->rtab; mfd->rtab = NULL;
	delete[] mfd->gtab; mfd->gtab = NULL;
	delete[] mfd->btab; mfd->btab = NULL;

	return 0;
}

///////////////////////////////////////////////////////////////////////////

int fxNoiseInitProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->lHB	= 2;
	mfd->lVB	= 0;
	mfd->lEI	= 4;
	mfd->lHP	= 25;

	mfd->cHB	= 9;
	mfd->cVB	= 2;
	mfd->cEI	= 20;
	mfd->cHP	= 100;

	mfd->dFreq	= 75;
	mfd->dHalo	= 60;
	mfd->dTBC	= 0;
	mfd->dGhost	= 0;
	mfd->dCBand	= 0;
	mfd->dDrop	= 0;

	mfd->YIQin	= BST_UNCHECKED;
	mfd->YIQout	= BST_UNCHECKED;
	mfd->YUVtog	= BST_CHECKED;

	return 0;
}

///////////////////////////////////////////////////////////////////////////

BOOL CALLBACK fxNoiseConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
	MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

	switch(msg) {
	case WM_INITDIALOG:
		SetWindowLong(hdlg, DWL_USER, lParam);
		mfd = (MyFilterData *)lParam;
		SetDlgItemInt(hdlg, IDC_LHB, mfd->lHB, FALSE);
		SetDlgItemInt(hdlg, IDC_LVB, mfd->lVB, FALSE);
		SetDlgItemInt(hdlg, IDC_LEI, mfd->lEI, FALSE);
		SetDlgItemInt(hdlg, IDC_LHP, mfd->lHP, FALSE);
		SetDlgItemInt(hdlg, IDC_CHB, mfd->cHB, FALSE);
		SetDlgItemInt(hdlg, IDC_CVB, mfd->cVB, FALSE);
		SetDlgItemInt(hdlg, IDC_CEI, mfd->cEI, FALSE);
		SetDlgItemInt(hdlg, IDC_CHP, mfd->cHP, FALSE);
		SetDlgItemInt(hdlg, IDC_DHFL, mfd->dFreq, FALSE);
		SetDlgItemInt(hdlg, IDC_DHALO, mfd->dHalo, FALSE);
		SetDlgItemInt(hdlg, IDC_DTBC, mfd->dTBC, FALSE);
		SetDlgItemInt(hdlg, IDC_DGHOST, mfd->dGhost, FALSE);
		SetDlgItemInt(hdlg, IDC_DCB, mfd->dCBand, FALSE);
		SetDlgItemInt(hdlg, IDC_DDROP, mfd->dDrop, FALSE);
		CheckDlgButton(hdlg, IDC_YIQIN, mfd->YIQin ? BST_CHECKED : BST_UNCHECKED);
		CheckDlgButton(hdlg, IDC_YIQOUT, mfd->YIQout ? BST_CHECKED : BST_UNCHECKED);
		CheckDlgButton(hdlg, IDC_YUVTOG, mfd->YUVtog ? BST_CHECKED : BST_UNCHECKED);
		return TRUE;

	case WM_COMMAND:
		switch(LOWORD(wParam)) {
		case IDOK:
			mfd->lHB = GetDlgItemInt(hdlg, IDC_LHB, &mfd->lHB, FALSE);
			mfd->lVB = GetDlgItemInt(hdlg, IDC_LVB, &mfd->lVB, FALSE);
			mfd->lEI = GetDlgItemInt(hdlg, IDC_LEI, &mfd->lEI, FALSE);
			mfd->lHP = GetDlgItemInt(hdlg, IDC_LHP, &mfd->lHP, FALSE);
			mfd->cHB = GetDlgItemInt(hdlg, IDC_CHB, &mfd->cHB, FALSE);
			mfd->cVB = GetDlgItemInt(hdlg, IDC_CVB, &mfd->cVB, FALSE);
			mfd->cEI = GetDlgItemInt(hdlg, IDC_CEI, &mfd->cEI, FALSE);
			mfd->cHP = GetDlgItemInt(hdlg, IDC_CHP, &mfd->cHP, FALSE);
			mfd->dFreq = GetDlgItemInt(hdlg, IDC_DHFL, &mfd->dFreq, FALSE);
			mfd->dHalo = GetDlgItemInt(hdlg, IDC_DHALO, &mfd->dHalo, FALSE);
			mfd->dTBC = GetDlgItemInt(hdlg, IDC_DTBC, &mfd->dTBC, FALSE);
			mfd->dGhost = GetDlgItemInt(hdlg, IDC_DGHOST, &mfd->dGhost, FALSE);
			mfd->dCBand = GetDlgItemInt(hdlg, IDC_DCB, &mfd->dCBand, FALSE);
			mfd->dDrop = GetDlgItemInt(hdlg, IDC_DDROP, &mfd->dDrop, FALSE);
			mfd->YIQin = !!IsDlgButtonChecked(hdlg, IDC_YIQIN);
			mfd->YIQout = !!IsDlgButtonChecked(hdlg, IDC_YIQOUT);
			mfd->YUVtog = !!IsDlgButtonChecked(hdlg, IDC_YUVTOG);

			if (mfd->lHB > 19) mfd->lHB = 19;
			if (mfd->lVB > 19) mfd->lVB = 19;
			if (mfd->lEI > 100) mfd->lHB = 100;
			if (mfd->lHP > 100) mfd->lHP = 100;
			if (mfd->cHB > 19) mfd->cHB = 19;
			if (mfd->cVB > 19) mfd->cVB = 19;
			if (mfd->cEI > 100) mfd->cHB = 100;
			if (mfd->cHP > 100) mfd->cHP = 100;
			if (mfd->dFreq > 100) mfd->dFreq = 100;
			if (mfd->dHalo > 100) mfd->dHalo = 100;
			if (mfd->dTBC > 100) mfd->dTBC = 100;
			if (mfd->dGhost > 100) mfd->dGhost = 100;
			if (mfd->dCBand > 100) mfd->dCBand = 100;
			if (mfd->dDrop > 100) mfd->dDrop = 100;

			EndDialog(hdlg, 0);

			return TRUE;
		case IDCANCEL:
			EndDialog(hdlg, 1);
			return FALSE;
		}
		break;
	}

	return FALSE;
}

///////////////////////////////////////////////////////////////////////////

int fxNoiseConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd) {
	return DialogBoxParam(fa->filter->module->hInstModule,
	       MAKEINTRESOURCE(IDD_FXN_CONFIG), hwnd,
		   fxNoiseConfigDlgProc, (LPARAM)fa->filter_data);
}

///////////////////////////////////////////////////////////////////////////

void fxNoiseStringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	char	fstr[10];

	if (mfd->YUVtog == BST_CHECKED)
		sprintf(fstr, "YIQ");
	else
		sprintf(fstr, "YUV");


	sprintf(str, "  %s -> %s", mfd->YIQin == BST_CHECKED ? fstr : "RGB", mfd->YIQout == BST_CHECKED ? fstr : "RGB");
}

///////////////////////////////////////////////////////////////////////////

void fxNoiseScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
    FilterActivation *fa = (FilterActivation *)lpVoid;
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	long	i;

	i = argv[0].asInt();
	mfd->YIQin = (i & 4) == 4 ? BST_CHECKED : BST_UNCHECKED;
	mfd->YIQout = (i & 2) == 2 ? BST_CHECKED : BST_UNCHECKED;
	mfd->YUVtog = (i & 1) == 1 ? BST_CHECKED : BST_UNCHECKED;
	i = argv[1].asInt();
	mfd->lHB = i >> 8; mfd->lVB = i & 0xFF;
	i = argv[2].asInt();
	mfd->lEI = i >> 8; mfd->lHB = i & 0xFF;
	i = argv[3].asInt();
	//mfd->c14 = (i & 2) == 2 ? BST_CHECKED : BST_UNCHECKED;
	//mfd->cBM = (i & 1) == 1 ? BST_CHECKED : BST_UNCHECKED;
}

///////////////////////////////////////////////////////////////////////////

bool fxNoiseFssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	//(mfd->c14 == BST_CHECKED ? 2 : 0) | (mfd->cBM == BST_CHECKED ? 1 : 0)
    _snprintf(buf, buflen, "Config(%d,%d,%d,%d)", 
		(mfd->YIQin == BST_CHECKED ? 4 : 0) | (mfd->YIQout == BST_CHECKED ? 2 : 0) | (mfd->YUVtog == BST_CHECKED ? 1 : 0),
		(mfd->lHB << 8) | mfd->lVB, (mfd->lEI << 8) | mfd->lHP,
		(mfd->cHB << 8) | mfd->cVB, (mfd->cEI << 8) | mfd->cHP,
		0);

    return true;
}

long rnd(long range) {
	long	rndret;

	__asm {
		mov		eax, [range]
		mov		ecx, eax
		imul    eax, [seed], 041C64E6Dh
		add     eax, 03039h
		mov     [seed], eax
		mov		ebx, eax
		shr		ebx, 16
		mov		eax, ecx
		mul     bx
		and		edx, 0x0000FFFFL
		mov		[rndret], edx
	}
	return rndret;
}