/*
    Red/Green/Ble (RGB) Adjustment Filter for VirtualDub.
    Copyright (C) 1999-2000 Donald A. Graft

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

	The author can be contacted at:
	Donald Graft
	neuron2@home.com.
*/

#include <windows.h>
#include <commctrl.h>
#include <stdio.h>
#include <crtdbg.h>
#include <math.h>
#include <stdlib.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

///////////////////////////////////////////////////////////////////////////

int RunProc(const FilterActivation *fa, const FilterFunctions *ff);
int StartProc(FilterActivation *fa, const FilterFunctions *ff);
int EndProc(FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int InitProc(FilterActivation *fa, const FilterFunctions *ff);
int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

typedef struct MyFilterData {
	IFilterPreview		*ifp;
	int					red;
	int					green;
	int					blue;
} MyFilterData;

bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	_snprintf(buf, buflen, "Config(%d, %d, %d)",
		mfd->red,
		mfd->green,
		mfd->blue);

	return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->red			= argv[0].asInt();
	mfd->green			= argv[1].asInt();
	mfd->blue			= argv[2].asInt();
}

ScriptFunctionDef func_defs[]={
	{ (ScriptFunctionPtr)ScriptConfig, "Config", "0iii" },
	{ NULL },
};

CScriptObject script_obj={
	NULL, func_defs
};

struct FilterDefinition filterDef_tutorial = {

	NULL, NULL, NULL,		// next, prev, module
	"red/green/blue adjustment (1.0)",	// name
	"Adjust red, green, blue.",
							// desc
	"Donald Graft", 		// maker
	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size

	InitProc,				// initProc
	NULL,					// deinitProc
	RunProc,				// runProc
	NULL,					// paramProc
	ConfigProc, 			// configProc
	StringProc, 			// stringProc
	StartProc,				// startProc
	EndProc,				// endProc

	&script_obj,			// script_obj
	FssProc,				// fssProc

};

int StartProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	return 0;
}

int EndProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	return 0;
}

///////////////////////////////////////////////////////////////////////////

int RunProc(const FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	const PixDim	width = fa->src.w;
	const PixDim	height = fa->src.h;
	Pixel32 *src, *dst;
	int x, y;
	long r, g, b;

	src = fa->src.data;
	dst = fa->dst.data;
	for (y = 0; y < height; y++)
	{
		for (x = 0; x < width; x++)
		{
			r = ((src[x] & 0xff0000) >> 16);
			g = ((src[x] & 0xff00) >> 8);
			b = (src[x] & 0xff);
			r = ((mfd->red + 100) * r) / 100;
			if (r > 255) r = 255;
			g = ((mfd->green + 100) * g) / 100;
			if (g > 255) g = 255;
			b = ((mfd->blue + 100) * b) / 100;
			if (b > 255) b = 255;
			dst[x] = ((int) r << 16) | ((int) g << 8) | ((int) b);			 
		}
		src	= (Pixel *)((char *)src + fa->src.pitch);
		dst	= (Pixel *)((char *)dst + fa->dst.pitch);
	}
	return 0;
}

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_tutorial;

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
	if (!(fd_tutorial = ff->addFilter(fm, &filterDef_tutorial, sizeof(FilterDefinition))))
		return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

	return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
	ff->removeFilter(fd_tutorial);
}

int InitProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->red = 0;
	mfd->green = 0;
	mfd->blue = 0;
	return 0;
}

BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
	MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

	switch(msg) {
		case WM_INITDIALOG:
			SetWindowLong(hdlg, DWL_USER, lParam);
			mfd = (MyFilterData *)lParam;
			HWND hWnd;

			hWnd = GetDlgItem(hdlg, IDC_RED);
			SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 300));
			SendMessage(hWnd, TBM_SETTICFREQ, 10 , 0);
			SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->red+100);
			SetDlgItemInt(hdlg, IDC_REDVAL, mfd->red+100, FALSE);
			hWnd = GetDlgItem(hdlg, IDC_GREEN);
			SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 300));
			SendMessage(hWnd, TBM_SETTICFREQ, 10, 0);
			SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->green+100);
			SetDlgItemInt(hdlg, IDC_GREENVAL, mfd->green+100, FALSE);
			hWnd = GetDlgItem(hdlg, IDC_BLUE);
			SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 300));
			SendMessage(hWnd, TBM_SETTICFREQ, 10, 0);
			SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->blue+100);
			SetDlgItemInt(hdlg, IDC_BLUEVAL, mfd->blue+100, FALSE);
			mfd->ifp->InitButton(GetDlgItem(hdlg, IDPREVIEW));
			return TRUE;

		case WM_COMMAND:
			switch(LOWORD(wParam)) {
			case IDPREVIEW:
				mfd->ifp->Toggle(hdlg);
				break;
			case IDOK:
				EndDialog(hdlg, 0);
				return TRUE;
			case IDHELP:
				{
				char prog[256];
				char path[256];
				LPTSTR ptr;
				GetModuleFileName(NULL, prog, 255);
				GetFullPathName(prog, 255, path, &ptr);
				*ptr = 0;
				strcat(path, "plugins\\Rgb.txt");
				OutputDebugString(path);
				OutputDebugString("\n");
				strcpy(prog, "Notepad ");
				strcat(prog, path);
				WinExec(prog, SW_SHOW);
				return TRUE;
				}
			case IDCANCEL:
				EndDialog(hdlg, 1);
				return TRUE;
			case IDC_ZEROR:
				SendMessage(GetDlgItem(hdlg, IDC_RED), TBM_SETPOS, (WPARAM)TRUE, 100);
				SetDlgItemInt(hdlg, IDC_REDVAL, 100, FALSE);
				mfd->red = 0;
				mfd->ifp->RedoFrame();
				return TRUE;
			case IDC_ZEROG:
				SendMessage(GetDlgItem(hdlg, IDC_GREEN), TBM_SETPOS, (WPARAM)TRUE, 100);
				SetDlgItemInt(hdlg, IDC_GREENVAL, 100, FALSE);
				mfd->green = 0;
				mfd->ifp->RedoFrame();
				return TRUE;
			case IDC_ZEROB:
				SendMessage(GetDlgItem(hdlg, IDC_BLUE), TBM_SETPOS, (WPARAM)TRUE, 100);
				SetDlgItemInt(hdlg, IDC_BLUEVAL, 100, FALSE);
				mfd->blue = 0;
				mfd->ifp->RedoFrame();
				return TRUE;
			}
			break;
		case WM_HSCROLL:
			{
				int red, green, blue;
				red = SendMessage(GetDlgItem(hdlg, IDC_RED), TBM_GETPOS, 0, 0)-100;
				SetDlgItemInt(hdlg, IDC_REDVAL, red+100, FALSE);
				green = SendMessage(GetDlgItem(hdlg, IDC_GREEN), TBM_GETPOS, 0, 0)-100;
				SetDlgItemInt(hdlg, IDC_GREENVAL, green+100, FALSE);
				blue = SendMessage(GetDlgItem(hdlg, IDC_BLUE), TBM_GETPOS, 0, 0)-100;
				SetDlgItemInt(hdlg, IDC_BLUEVAL, blue+100, FALSE);
			if (red != mfd->red || green != mfd->green || blue != mfd->blue)
				{
					mfd->red = red;
					mfd->green = green;
					mfd->blue = blue;
					mfd->ifp->RedoFrame();
				}
				break;
			}
	}

	return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	MyFilterData mfd_old = *mfd;
	int ret;

	mfd->ifp = fa->ifp;

	if (DialogBoxParam(fa->filter->module->hInstModule, MAKEINTRESOURCE(IDD_FILTER),
		hwnd, ConfigDlgProc, (LPARAM)mfd))
	{
		*mfd = mfd_old;
		ret = TRUE;
	}
    else
	{
		ret = FALSE;
	}
	return(ret);
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
}

