/*
    Smart Bob Filter for VirtualDub -- Break fields into frames using
    a motion-adaptive algorithm. Copyright (C) 1999-2001 Donald A. Graft

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

	The author can be contacted at:
	Donald Graft
	neuron2@home.com.
*/

#include <windows.h>
#include <stdio.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

#define DENOISE_DIAMETER 5
#define DENOISE_THRESH 7

///////////////////////////////////////////////////////////////////////////

int  InitProc(FilterActivation *fa, const FilterFunctions *ff);
int  StartProc(FilterActivation *fa, const FilterFunctions *ff);
int  EndProc(FilterActivation *fa, const FilterFunctions *ff);
int  RunProc(const FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int  ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

typedef struct MyFilterData
{
    bool			bShiftEven;
	bool			bMotionOnly;
	bool			bDenoise;
	int 			threshold;
	int				*prevFrame;
	unsigned char	*moving;
	unsigned char	*fmoving;
} MyFilterData;

bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	_snprintf(buf, buflen, "Config(%i, %i, %i, %i)",
		      mfd->bShiftEven, mfd->bMotionOnly, mfd->threshold, mfd->bDenoise);
	return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc)
{
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	mfd->bShiftEven = !!argv[0].asInt();
	mfd->bMotionOnly = !!argv[1].asInt();
	mfd->threshold = argv[2].asInt();
	mfd->bDenoise = !!argv[3].asInt();
}

ScriptFunctionDef func_defs[] =
{
	{ (ScriptFunctionPtr)ScriptConfig, "Config", "0iiii" },
	{ NULL },
};

CScriptObject script_obj=
{
	NULL, func_defs
};

struct FilterDefinition filterDef =
{

	NULL, NULL, NULL,	// next, prev, module
	"smart bob (1.1 beta 2)",	// name
	"Motion-adaptive deinterlacing for double-frame-rate output.",	// desc
	"Donald Graft",		// maker
	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size

	InitProc,		// initProc
	NULL,			// deinitProc
	RunProc,		// runProc
	ParamProc,		// paramProc
	ConfigProc,		// configProc
	StringProc,		// stringProc
	StartProc,		// startProc
	EndProc,		// endProc

	&script_obj,	// script_obj
	FssProc,		// fssProc

};

///////////////////////////////////////////////////////////////////////////

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd;

///////////////////////////////////////////////////////////////////////////

int __declspec(dllexport) __cdecl
VirtualdubFilterModuleInit2(FilterModule *fm,
						  const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat)
{
    if (!(fd = ff->addFilter(fm, &filterDef, sizeof(FilterDefinition))))
        return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

    return 0;
}

void __declspec(dllexport) __cdecl
VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff)
{
    ff->removeFilter(fd);
}

///////////////////////////////////////////////////////////////////////////

int InitProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->bShiftEven = true;
	mfd->bMotionOnly = false;
	mfd->bDenoise = true;
	mfd->threshold = 12;
	return 0;
}

int StartProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *) fa->filter_data;

	mfd->prevFrame = new int[fa->src.w*fa->src.h];
	memset(mfd->prevFrame, 0, fa->src.w*fa->src.h*sizeof(int));
	mfd->moving	= new unsigned char[fa->src.w*fa->src.h];
	memset(mfd->moving, 0, fa->src.w*fa->src.h*sizeof(unsigned char));
	mfd->fmoving = new unsigned char[fa->src.w*fa->src.h];
	memset(mfd->fmoving, 0, fa->src.w*fa->src.h*sizeof(unsigned char));

	return 0;
}

int EndProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	delete[] mfd->prevFrame;
	mfd->prevFrame = NULL;
	delete[] mfd->moving;
	mfd->moving = NULL;
	delete[] mfd->fmoving;
	mfd->fmoving = NULL;

	return 0;
}

int RunProc(const FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	int iOddEven = ((MyFilterData *)(fa->filter_data))->bShiftEven ? 0 : 1;
	Pixel32 *src, *dst, *srcn, *srcnn, *srcp;
	unsigned char *moving, *fmoving;
	int x, y, *prev;
	long currValue, prevValue, nextValue, nextnextValue, luma, lumap, luman;
	int r, g, b, rp, gp, bp, rn, gn, bn, rnn, gnn, bnn, R, G, B, T = mfd->threshold * mfd->threshold;
	int h = fa->src.h;
	int w = fa->src.w;
	int hminus = fa->src.h - 1;
	int hminus2 = fa->src.h - 2;
	int wminus = fa->src.w - 1;

	/* Calculate the motion map. */
	moving = mfd->moving;
	/* Threshold 0 means treat all areas as moving, i.e., dumb bob. */
	if (mfd->threshold == 0)
	{
		memset(moving, 1, fa->src.h * fa->src.w);
	}
	else
	{
		memset(moving, 0, fa->src.h * fa->src.w);
		src = (Pixel32 *)fa->src.data;
		srcn = (Pixel32 *)((char *)src + fa->src.pitch);
		prev = mfd->prevFrame;
		if(fa->pfsi->lCurrentSourceFrame % 2 == iOddEven)
			prev += fa->src.w;
		for (y = 0; y < hminus; y++)
		{
			for (x = 0; x < w; x++)
			{
				currValue = prev[x];
				nextValue = srcn[x];
				prevValue = src[x];
				r = (currValue >> 16) & 0xff;
				rp = (prevValue >> 16) & 0xff;
				rn = (nextValue >> 16) & 0xff;
				g = (currValue >> 8) & 0xff;
				gp = (prevValue >> 8) & 0xff;
				gn = (nextValue >> 8) & 0xff;
				b = currValue & 0xff;
				bp = prevValue & 0xff;
				bn = nextValue & 0xff;
				luma = (55 * r + 182 * g + 19 * b) >> 8;
				lumap = (55 * rp + 182 * gp + 19 * bp) >> 8;
				luman = (55 * rn + 182 * gn + 19 * bn) >> 8;
				if ((lumap - luma) * (luman - luma) >= T)
					moving[x] = 1;
			}
			src = (Pixel32 *)((char *)src + fa->src.pitch);
			srcn = (Pixel32 *)((char *)srcn + fa->src.pitch);
			moving += w;
			prev += w;
		}
		/* Can't diff the last line. */
		memset(moving, 0, fa->src.w);

		/* Motion map denoising. */
		if (mfd->bDenoise)
		{
			int xlo, xhi, ylo, yhi, xsize;
			int u, v;
			int N = DENOISE_DIAMETER;
			int Nover2 = N/2;
			int sum;
			unsigned char *m;


			// Erode.
			moving = mfd->moving;
			fmoving = mfd->fmoving;
			for (y = 0; y < h; y++)
			{
				for (x = 0; x < w; x++)
				{
					if (moving[x] == 0)
					{
						fmoving[x] = 0;	
						continue;
					}
					xlo = x - Nover2; if (xlo < 0) xlo = 0;
					xhi = x + Nover2; if (xhi >= w) xhi = wminus;
					ylo = y - Nover2; if (ylo < 0) ylo = 0;
					yhi = y + Nover2; if (yhi >= h) yhi = hminus;
					for (u = ylo, sum = 0, m = mfd->moving + ylo * w; u <= yhi; u++)
					{
						for (v = xlo; v <= xhi; v++)
						{
							sum += m[v];
						}
						m += w;
					}
					if (sum > DENOISE_THRESH)
						fmoving[x] = 1;
					else
						fmoving[x] = 0;
				}
				moving += w;
				fmoving += w;
			}

			// Dilate.
			moving = mfd->moving;
			fmoving = mfd->fmoving;
			for (y = 0; y < h; y++)
			{
				for (x = 0; x < w; x++)
				{
					if (fmoving[x] == 0)
					{
						moving[x] = 0;	
						continue;
					}
					xlo = x - Nover2;
					if (xlo < 0) xlo = 0;
					xhi = x + Nover2;
					/* Use w here instead of wminus so we don't have to add 1 in the
					   the assignment of xsize. */
					if (xhi >= w) xhi = w;
					xsize = xhi - xlo;
					ylo = y - Nover2;
					if (ylo < 0) ylo = 0;
					yhi = y + Nover2;
					if (yhi >= h) yhi = hminus;
					m = mfd->moving + ylo * w;
					for (u = ylo; u <= yhi; u++)
					{
						memset(&m[xlo], 1, xsize); 
						m += w;
					}
				}
				moving += w;
				fmoving += w;
			}
		}
	}

	/* Output the destination frame. */
	if (!mfd->bMotionOnly)
	{
		/* Output the destination frame. */
		src = (Pixel32 *)fa->src.data;
		srcn = (Pixel32 *)((char *)fa->src.data + fa->src.pitch);
		srcnn = (Pixel32 *)((char *)fa->src.data + 2 * fa->src.pitch);
		srcp = (Pixel32 *)((char *)fa->src.data - fa->src.pitch);
		dst = (Pixel32 *)fa->dst.data;
		if(fa->pfsi->lCurrentSourceFrame % 2 == iOddEven)
		{
			/* Shift this frame's output up by one line. */
			memcpy(dst, src, fa->src.w * sizeof(Pixel32));
			dst = (Pixel32 *)((char *)dst + fa->dst.pitch);
			prev = mfd->prevFrame + w;
		}
		else
		{
			prev = mfd->prevFrame;
		}
		moving = mfd->moving;
		for (y = 0; y < hminus; y++)
		{
			/* Even output line. Pass it through. */
			memcpy(dst, src, fa->src.w * sizeof(Pixel32));
			dst = (Pixel32 *)((char *)dst + fa->dst.pitch);
			/* Odd output line. Synthesize it. */
			for (x = 0; x < w; x++)
			{
				if (moving[x] == 1)
				{
					/* Make up a new line. Use cubic interpolation where there
					   are enough samples and linear where there are not enough. */
					nextValue = srcn[x];
					r = (src[x] >> 16) & 0xff;
					rn = (nextValue >> 16) & 0xff;
					g = (src[x] >> 8) & 0xff;
					gn = (nextValue >>8) & 0xff;
					b = src[x] & 0xff;
					bn = nextValue & 0xff;
					if (y == 0 || y == hminus2)
					{	/* Not enough samples; use linear. */
						R = (r + rn) >> 1;
						G = (g + gn) >> 1;
						B = (b + bn) >> 1;
					}
					else
					{
						/* Enough samples; use cubic. */
						prevValue = srcp[x];
						nextnextValue = srcnn[x];
						rp = (prevValue >> 16) & 0xff;
						rnn = (nextnextValue >>16) & 0xff;
						gp = (prevValue >> 8) & 0xff;
						gnn = (nextnextValue >> 8) & 0xff;
						bp = prevValue & 0xff;
						bnn = nextnextValue & 0xff;
						R = (5 * (r + rn) - (rp + rnn)) >> 3;
						if (R > 255) R = 255;
						else if (R < 0) R = 0;
						G = (5 * (g + gn) - (gp + gnn)) >> 3;
						if (G > 255) G = 255;
						else if (G < 0) G = 0;
						B = (5 * (b + bn) - (bp + bnn)) >> 3;
						if (B > 255) B = 255;
						else if (B < 0) B = 0;
					}
					dst[x] = (R << 16) | (G << 8) | B;  
				}
				else
				{
					/* Use line from previous field. */
					dst[x] = prev[x];
				}
			}
			src = (Pixel32 *)((char *)src + fa->src.pitch);
			srcn = (Pixel32 *)((char *)srcn + fa->src.pitch);
			srcnn = (Pixel32 *)((char *)srcnn + fa->src.pitch);
			srcp = (Pixel32 *)((char *)srcp + fa->src.pitch);
			dst = (Pixel32 *)((char *)dst + fa->dst.pitch);
			moving += w;
			prev += w;
		}
		/* Copy through the last source line. */

		memcpy(dst, src, fa->src.w * sizeof(Pixel32));
		if(fa->pfsi->lCurrentSourceFrame % 2 != iOddEven)
		{
			dst = (Pixel32 *)((char *)dst + fa->dst.pitch);
			memcpy(dst, src, fa->src.w * sizeof(Pixel32));
		}
	}
	else
	{
		/* Show motion only. */
		moving = mfd->moving;
		src = (Pixel32 *)fa->src.data;
		dst = (Pixel32 *)fa->dst.data;
		for (y = 0; y < hminus; y++)
		{
			for (x = 0; x < w; x++)
			{
				if (moving[x])
				{
					dst[x] = ((Pixel32 *)((char *)dst + fa->dst.pitch))[x] = src[x];
				}
				else
				{
					dst[x] = ((Pixel32 *)((char *)dst + fa->dst.pitch))[x] = 0;
				}
			}
			src = (Pixel32 *)((char *)src + fa->src.pitch);
			dst = (Pixel32 *)((char *)dst + fa->dst.pitch);
			dst = (Pixel32 *)((char *)dst + fa->dst.pitch);
			moving += w;
		}
	}

	/* Buffer the input frame (aka field). */
	src = (Pixel32 *)fa->src.data;
	prev = mfd->prevFrame;
	for (y = 0; y < h; y++)
	{
		memcpy(prev, src, w * sizeof(Pixel32));
		src = (Pixel32 *)((char *)src + fa->src.pitch);
		prev += w;
	}

    return 0;
}

long ParamProc(FilterActivation *fa, const FilterFunctions *ff)
{
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

    fa->dst.h *= 2;
    return FILTERPARAM_SWAP_BUFFERS;
}

BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
    MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

    switch(msg) {
        case WM_INITDIALOG:
            SetWindowLong(hdlg, DWL_USER, lParam);
            mfd = (MyFilterData *)lParam;
            CheckDlgButton(hdlg, IDC_SHIFT_EVEN, mfd->bShiftEven?BST_CHECKED:BST_UNCHECKED);
            CheckDlgButton(hdlg, IDC_MOTION_ONLY, mfd->bMotionOnly?BST_CHECKED:BST_UNCHECKED);
            CheckDlgButton(hdlg, IDC_DENOISE, mfd->bDenoise?BST_CHECKED:BST_UNCHECKED);
 			SetDlgItemInt(hdlg, IDC_THRESHOLD, mfd->threshold, FALSE);
			return TRUE;

        case WM_COMMAND:
            switch(LOWORD(wParam)) {
            case IDOK:
                mfd->bShiftEven = !!IsDlgButtonChecked(hdlg, IDC_SHIFT_EVEN);
                mfd->bMotionOnly = !!IsDlgButtonChecked(hdlg, IDC_MOTION_ONLY);
                mfd->bDenoise = !!IsDlgButtonChecked(hdlg, IDC_DENOISE);
 				mfd->threshold = GetDlgItemInt(hdlg, IDC_THRESHOLD, &mfd->threshold, FALSE);
                EndDialog(hdlg, 0);
                return TRUE;
 			case IDHELP:
				{
				char prog[256];
				char path[256];
				LPTSTR ptr;

				GetModuleFileName(NULL, prog, 255);
				GetFullPathName(prog, 255, path, &ptr);
				*ptr = 0;
				strcat(path, "plugins\\Bob.html");
				ShellExecute(hdlg, "open", path, NULL, NULL, SW_SHOWNORMAL);
				return TRUE;
				}

           case IDCANCEL:
                EndDialog(hdlg, 1);
                return FALSE;
            }
            break;
    }

    return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd)
{
    return DialogBoxParam(fa->filter->module->hInstModule,
            MAKEINTRESOURCE(IDD_FILTER_FIELD_SHIFT), hwnd,
            (DLGPROC)ConfigDlgProc, (LPARAM)fa->filter_data);
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str)
{

}
