// VirtualDub filter: general convolution 3d
// by Gunnar Thalin (guth@home.se)

#include <windows.h>
#include <stdio.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

///////////////////////////////////////////////////////////////////////////

int InitProc(FilterActivation *fa, const FilterFunctions *ff);
int StartProc(FilterActivation *fa, const FilterFunctions *ff);
int EndProc(FilterActivation *fa, const FilterFunctions *ff);
int RunProc(const FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);
void ShortToHex(short n, char *sz);
void HexToShort(short &n, char *sz);

///////////////////////////////////////////////////////////////////////////

typedef struct MyFilterData {
	BYTE *pbyTR, *pbyTG, *pbyTB, *pbyPR, *pbyPG, *pbyPB, *pbyOR, *pbyOG, *pbyOB;
	short anThis[25];
	short anPrev[25];
	short anPrevPrev[25];
	short nBias;
	bool bUseProcPrev;
} MyFilterData;


bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	char szMatrixValues[12 * 25 + 1];
	for(int i = 0; i < 25; i++)
	{
		ShortToHex(mfd->anThis[i], szMatrixValues + 12 * i);
		ShortToHex(mfd->anPrev[i], szMatrixValues + 12 * i + 4);
		ShortToHex(mfd->anPrevPrev[i], szMatrixValues + 12 * i + 8);
	}
	szMatrixValues[12 * 25] = '\0';

	_snprintf(buf, buflen, "Config(%i, %i, \"%s\")", mfd->nBias, mfd->bUseProcPrev ? 1 : 0, szMatrixValues);

	return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->nBias = argv[0].asInt();
	mfd->bUseProcPrev = !!argv[1].asInt();
	for(int i = 0; i < 25; i++)
	{
		HexToShort(mfd->anThis[i], *(argv[2].asString()) + 12 * i);
		HexToShort(mfd->anPrev[i], *(argv[2].asString()) + 12 * i + 4);
		HexToShort(mfd->anPrevPrev[i], *(argv[2].asString()) + 12 * i + 8);
	}
}

void ShortToHex(short n, char *sz)
{
	char szTmp[5];
	sprintf(szTmp, "%04x", n + 0x8000);
	strncpy(sz, szTmp, 4);
}

void HexToShort(short &n, char *sz)
{
	char szTmp[5];
	strncpy(szTmp, sz, 4);
	szTmp[4] = '\0';
	sscanf(szTmp, "%04x", &n);
	n += -0x8000;
}

ScriptFunctionDef func_defs[]={
	{ (ScriptFunctionPtr)ScriptConfig, "Config", "0iis" },
	{ NULL },
};

CScriptObject script_obj={
	NULL, func_defs
};


struct FilterDefinition filterDef = {

	NULL, NULL, NULL,	// next, prev, module
	"general convolution 3d v1.1",	// name
	"Applies different general 5x5 convolution matrixes to up to 3 consecutive frames.",	// desc
	"Gunnar Thalin",		// maker
	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size

	InitProc,	// initProc
	NULL,			// deinitProc
	RunProc,		// runProc
	ParamProc,		// paramProc
	ConfigProc,		// configProc
	StringProc,		// stringProc
	StartProc,		// startProc
	EndProc,			// endProc

	&script_obj,	// script_obj
	FssProc,		// fssProc

};


///////////////////////////////////////////////////////////////////////////

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd;

///////////////////////////////////////////////////////////////////////////

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
    if (!(fd = ff->addFilter(fm, &filterDef, sizeof(FilterDefinition))))
        return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

    return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
    ff->removeFilter(fd);
}

///////////////////////////////////////////////////////////////////////////

int InitProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	
	for(int i = 0; i < 25; i++)
	{
		mfd->anPrevPrev[i] = 0;
		mfd->anPrev[i] = 0;
		mfd->anThis[i] = 0;
	}

	mfd->anThis[12] = 100;
	mfd->nBias = 0;
	mfd->bUseProcPrev = false;

	return 0;
}
	
int StartProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->pbyTR = new BYTE[fa->src.w * fa->src.h];
	mfd->pbyTG = new BYTE[fa->src.w * fa->src.h];
	mfd->pbyTB = new BYTE[fa->src.w * fa->src.h];
	mfd->pbyPR = new BYTE[fa->src.w * fa->src.h];
	mfd->pbyPG = new BYTE[fa->src.w * fa->src.h];
	mfd->pbyPB = new BYTE[fa->src.w * fa->src.h];
	mfd->pbyOR = new BYTE[fa->src.w * fa->src.h];
	mfd->pbyOG = new BYTE[fa->src.w * fa->src.h];
	mfd->pbyOB = new BYTE[fa->src.w * fa->src.h];
	memset(mfd->pbyPR, 0, fa->src.w * fa->src.h);
	memset(mfd->pbyPG, 0, fa->src.w * fa->src.h);
	memset(mfd->pbyPB, 0, fa->src.w * fa->src.h);
	memset(mfd->pbyOR, 0, fa->src.w * fa->src.h);
	memset(mfd->pbyOG, 0, fa->src.w * fa->src.h);
	memset(mfd->pbyOB, 0, fa->src.w * fa->src.h);

	return 0;
}

int EndProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	delete[] mfd->pbyTR;
	mfd->pbyTR = NULL;
	delete[] mfd->pbyTG;
	mfd->pbyTG = NULL;
	delete[] mfd->pbyTB;
	mfd->pbyTB = NULL;

	delete[] mfd->pbyPR;
	mfd->pbyPR = NULL;
	delete[] mfd->pbyPG;
	mfd->pbyPG = NULL;
	delete[] mfd->pbyPB;
	mfd->pbyPB = NULL;
	
	delete[] mfd->pbyOR;
	mfd->pbyOR = NULL;
	delete[] mfd->pbyOG;
	mfd->pbyOG = NULL;
	delete[] mfd->pbyOB;
	mfd->pbyOB = NULL;

	return 0;
}


int RunProc(const FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	Pixel32 *src, *dst;
	BYTE *pbyTR0, *pbyTG0, *pbyTB0, *pbyTR1, *pbyTG1, *pbyTB1, *pbyTR2, *pbyTG2, *pbyTB2, *pbyTR3, *pbyTG3, *pbyTB3, *pbyTR4, *pbyTG4, *pbyTB4;
	BYTE *pbyPR0, *pbyPG0, *pbyPB0, *pbyPR1, *pbyPG1, *pbyPB1, *pbyPR2, *pbyPG2, *pbyPB2, *pbyPR3, *pbyPG3, *pbyPB3, *pbyPR4, *pbyPG4, *pbyPB4;
	BYTE *pbyOR0, *pbyOG0, *pbyOB0, *pbyOR1, *pbyOG1, *pbyOB1, *pbyOR2, *pbyOG2, *pbyOB2, *pbyOR3, *pbyOG3, *pbyOB3, *pbyOR4, *pbyOG4, *pbyOB4;

	int h = fa->src.h;
	int w = fa->src.w;

	short nBias = mfd->nBias;

	src = fa->src.data;
	pbyTR0 = mfd->pbyTR;
	pbyTG0 = mfd->pbyTG;
	pbyTB0 = mfd->pbyTB;
	for(int y = 0; y < h; y++)
	{
		for(int x = 0; x < w; x++)
		{
			*pbyTR0++ = (BYTE)((*src & 0xff0000) >> 16);
			*pbyTG0++ = (BYTE)((*src & 0xff00) >> 8);
			*pbyTB0++ = (BYTE)(*src++ & 0xff);
		}
		src = (Pixel32 *)((char *)src + fa->src.modulo);
	}

	int iR, iG, iB, x0, x1, x2, x3, x4;
	
	int iO00 = mfd->anPrevPrev[0];
	int iO10 = mfd->anPrevPrev[1];
	int iO20 = mfd->anPrevPrev[2];
	int iO30 = mfd->anPrevPrev[3];
	int iO40 = mfd->anPrevPrev[4];
	int iO01 = mfd->anPrevPrev[5];
	int iO11 = mfd->anPrevPrev[6];
	int iO21 = mfd->anPrevPrev[7];
	int iO31 = mfd->anPrevPrev[8];
	int iO41 = mfd->anPrevPrev[9];
	int iO02 = mfd->anPrevPrev[10];
	int iO12 = mfd->anPrevPrev[11];
	int iO22 = mfd->anPrevPrev[12];
	int iO32 = mfd->anPrevPrev[13];
	int iO42 = mfd->anPrevPrev[14];
	int iO03 = mfd->anPrevPrev[15];
	int iO13 = mfd->anPrevPrev[16];
	int iO23 = mfd->anPrevPrev[17];
	int iO33 = mfd->anPrevPrev[18];
	int iO43 = mfd->anPrevPrev[19];
	int iO04 = mfd->anPrevPrev[20];
	int iO14 = mfd->anPrevPrev[21];
	int iO24 = mfd->anPrevPrev[22];
	int iO34 = mfd->anPrevPrev[23];
	int iO44 = mfd->anPrevPrev[24];
	
	int iP00 = mfd->anPrev[0];
	int iP10 = mfd->anPrev[1];
	int iP20 = mfd->anPrev[2];
	int iP30 = mfd->anPrev[3];
	int iP40 = mfd->anPrev[4];
	int iP01 = mfd->anPrev[5];
	int iP11 = mfd->anPrev[6];
	int iP21 = mfd->anPrev[7];
	int iP31 = mfd->anPrev[8];
	int iP41 = mfd->anPrev[9];
	int iP02 = mfd->anPrev[10];
	int iP12 = mfd->anPrev[11];
	int iP22 = mfd->anPrev[12];
	int iP32 = mfd->anPrev[13];
	int iP42 = mfd->anPrev[14];
	int iP03 = mfd->anPrev[15];
	int iP13 = mfd->anPrev[16];
	int iP23 = mfd->anPrev[17];
	int iP33 = mfd->anPrev[18];
	int iP43 = mfd->anPrev[19];
	int iP04 = mfd->anPrev[20];
	int iP14 = mfd->anPrev[21];
	int iP24 = mfd->anPrev[22];
	int iP34 = mfd->anPrev[23];
	int iP44 = mfd->anPrev[24];
	
	int iT00 = mfd->anThis[0];
	int iT10 = mfd->anThis[1];
	int iT20 = mfd->anThis[2];
	int iT30 = mfd->anThis[3];
	int iT40 = mfd->anThis[4];
	int iT01 = mfd->anThis[5];
	int iT11 = mfd->anThis[6];
	int iT21 = mfd->anThis[7];
	int iT31 = mfd->anThis[8];
	int iT41 = mfd->anThis[9];
	int iT02 = mfd->anThis[10];
	int iT12 = mfd->anThis[11];
	int iT22 = mfd->anThis[12];
	int iT32 = mfd->anThis[13];
	int iT42 = mfd->anThis[14];
	int iT03 = mfd->anThis[15];
	int iT13 = mfd->anThis[16];
	int iT23 = mfd->anThis[17];
	int iT33 = mfd->anThis[18];
	int iT43 = mfd->anThis[19];
	int iT04 = mfd->anThis[20];
	int iT14 = mfd->anThis[21];
	int iT24 = mfd->anThis[22];
	int iT34 = mfd->anThis[23];
	int iT44 = mfd->anThis[24];

	bool bUseO = iO00 != 0 || iO10 != 0 || iO20 != 0 || iO30 != 0 || iO40 != 0 ||
					 iO01 != 0 || iO11 != 0 || iO21 != 0 || iO31 != 0 || iO41 != 0 ||
					 iO02 != 0 || iO12 != 0 || iO22 != 0 || iO32 != 0 || iO42 != 0 ||
					 iO03 != 0 || iO13 != 0 || iO23 != 0 || iO33 != 0 || iO43 != 0 ||
					 iO04 != 0 || iO14 != 0 || iO24 != 0 || iO34 != 0 || iO44 != 0;
	bool bUseP = iP00 != 0 || iP10 != 0 || iP20 != 0 || iP30 != 0 || iP40 != 0 ||
					 iP01 != 0 || iP11 != 0 || iP21 != 0 || iP31 != 0 || iP41 != 0 ||
					 iP02 != 0 || iP12 != 0 || iP22 != 0 || iP32 != 0 || iP42 != 0 ||
					 iP03 != 0 || iP13 != 0 || iP23 != 0 || iP33 != 0 || iP43 != 0 ||
					 iP04 != 0 || iP14 != 0 || iP24 != 0 || iP34 != 0 || iP44 != 0;
	bool bUseT = iT00 != 0 || iT10 != 0 || iT20 != 0 || iT30 != 0 || iT40 != 0 ||
					 iT01 != 0 || iT11 != 0 || iT21 != 0 || iT31 != 0 || iT41 != 0 ||
					 iT02 != 0 || iT12 != 0 || iT22 != 0 || iT32 != 0 || iT42 != 0 ||
					 iT03 != 0 || iT13 != 0 || iT23 != 0 || iT33 != 0 || iT43 != 0 ||
					 iT04 != 0 || iT14 != 0 || iT24 != 0 || iT34 != 0 || iT44 != 0;
	bool bUseSize3 = iO11 != 0 || iO21 != 0 || iO31 != 0 || iO12 != 0 || iO32 != 0 || iO13 != 0 || iO23 != 0 || iO33 != 0 ||
						  iP11 != 0 || iP21 != 0 || iP31 != 0 || iP12 != 0 || iP32 != 0 || iP13 != 0 || iP23 != 0 || iP33 != 0 ||
						  iT11 != 0 || iT21 != 0 || iT31 != 0 || iT12 != 0 || iT32 != 0 || iT13 != 0 || iT23 != 0 || iT33 != 0;
	bool bUseSize5 = iO00 != 0 || iO10 != 0 || iO20 != 0 || iO30 != 0 || iO40 != 0 || iO01 != 0 || iO41 != 0 || iO02 != 0 || iO42 != 0 || iO03 != 0 || iO43 != 0 || iO04 != 0 || iO14 != 0 || iO24 != 0 || iO34 != 0 || iO44 != 0 ||
						  iP00 != 0 || iP10 != 0 || iP20 != 0 || iP30 != 0 || iP40 != 0 || iP01 != 0 || iP41 != 0 || iP02 != 0 || iP42 != 0 || iP03 != 0 || iP43 != 0 || iP04 != 0 || iP14 != 0 || iP24 != 0 || iP34 != 0 || iP44 != 0 ||
						  iT00 != 0 || iT10 != 0 || iT20 != 0 || iT30 != 0 || iT40 != 0 || iT01 != 0 || iT41 != 0 || iT02 != 0 || iT42 != 0 || iT03 != 0 || iT43 != 0 || iT04 != 0 || iT14 != 0 || iT24 != 0 || iT34 != 0 || iT44 != 0;

	int iCountO = 0;
	if(fa->pfsi->lCurrentFrame > 1)
		iCountO = iO00 + iO01 + iO02 + iO03 + iO04 +
					 iO10 + iO11 + iO12 + iO13 + iO14 +
					 iO20 + iO21 + iO22 + iO23 + iO24 +
					 iO30 + iO31 + iO32 + iO33 + iO34 +
					 iO40 + iO41 + iO42 + iO43 + iO44;

	int iCountP = 0;
	if(fa->pfsi->lCurrentFrame > 0)
		iCountP = iP00 + iP01 + iP02 + iP03 + iP04 +
					 iP10 + iP11 + iP12 + iP13 + iP14 +
					 iP20 + iP21 + iP22 + iP23 + iP24 +
					 iP30 + iP31 + iP32 + iP33 + iP34 +
					 iP40 + iP41 + iP42 + iP43 + iP44;

	int iCountT = iT00 + iT01 + iT02 + iT03 + iT04 +
					  iT10 + iT11 + iT12 + iT13 + iT14 +
					  iT20 + iT21 + iT22 + iT23 + iT24 +
					  iT30 + iT31 + iT32 + iT33 + iT34 +
					  iT40 + iT41 + iT42 + iT43 + iT44;

	int iCountTot = iCountT + iCountP + iCountO;
	int iCountDiv = 0x100000 / (iCountTot == 0 ? 1 : iCountTot);

	for(y = 0; y < h; y++)
	{
		pbyOR0 = pbyOR1 = pbyOR2 = pbyOR3 = pbyOR4 = mfd->pbyOR + y * w;
		pbyOG0 = pbyOG1 = pbyOG2 = pbyOG3 = pbyOG4 = mfd->pbyOG + y * w;
		pbyOB0 = pbyOB1 = pbyOB2 = pbyOB3 = pbyOB4 = mfd->pbyOB + y * w;
		pbyPR0 = pbyPR1 = pbyPR2 = pbyPR3 = pbyPR4 = mfd->pbyPR + y * w;
		pbyPG0 = pbyPG1 = pbyPG2 = pbyPG3 = pbyPG4 = mfd->pbyPG + y * w;
		pbyPB0 = pbyPB1 = pbyPB2 = pbyPB3 = pbyPB4 = mfd->pbyPB + y * w;
		pbyTR0 = pbyTR1 = pbyTR2 = pbyTR3 = pbyTR4 = mfd->pbyTR + y * w;
		pbyTG0 = pbyTG1 = pbyTG2 = pbyTG3 = pbyTG4 = mfd->pbyTG + y * w;
		pbyTB0 = pbyTB1 = pbyTB2 = pbyTB3 = pbyTB4 = mfd->pbyTB + y * w;

		if(y > 0)
		{
			pbyOR4 -= w;
			pbyOG4 -= w;
			pbyOB4 -= w;
			pbyPR4 -= w;
			pbyPG4 -= w;
			pbyPB4 -= w;
			pbyTR4 -= w;
			pbyTG4 -= w;
			pbyTB4 -= w;

			pbyOR3 -= w;
			pbyOG3 -= w;
			pbyOB3 -= w;
			pbyPR3 -= w;
			pbyPG3 -= w;
			pbyPB3 -= w;
			pbyTR3 -= w;
			pbyTG3 -= w;
			pbyTB3 -= w;

			if(y > 1)
			{
				pbyOR4 -= w;
				pbyOG4 -= w;
				pbyOB4 -= w;
				pbyPR4 -= w;
				pbyPG4 -= w;
				pbyPB4 -= w;
				pbyTR4 -= w;
				pbyTG4 -= w;
				pbyTB4 -= w;
			}
		}

		if(y < h - 1)
		{
			pbyOR1 += w;
			pbyOG1 += w;
			pbyOB1 += w;
			pbyPR1 += w;
			pbyPG1 += w;
			pbyPB1 += w;
			pbyTR1 += w;
			pbyTG1 += w;
			pbyTB1 += w;

			pbyOR0 += w;
			pbyOG0 += w;
			pbyOB0 += w;
			pbyPR0 += w;
			pbyPG0 += w;
			pbyPB0 += w;
			pbyTR0 += w;
			pbyTG0 += w;
			pbyTB0 += w;

			if(y < h - 2)
			{
				pbyOR0 += w;
				pbyOG0 += w;
				pbyOB0 += w;
				pbyPR0 += w;
				pbyPG0 += w;
				pbyPB0 += w;
				pbyTR0 += w;
				pbyTG0 += w;
				pbyTB0 += w;
			}
		}
		
		dst = (Pixel32 *)((char *)fa->dst.data + y * fa->dst.pitch);
		for(x2 = 0; x2 < w; x2++)
		{
			x0 = x2 > 2 ? x2 - 2 : 0;
			x1 = x2 > 1 ? x2 - 1 : 0;
			x3 = x2 < w - 2 ? x2 + 1 : w - 1;
			x4 = x2 < w - 3 ? x2 + 2 : w - 2;

			iR = iG = iB = 0;

			if(bUseT)
			{
				iR += iT22 * pbyTR2[x2];
				iG += iT22 * pbyTG2[x2];
				iB += iT22 * pbyTB2[x2];
				if(bUseSize3)
				{
					iR += iT11 * pbyTR1[x1] + iT21 * pbyTR1[x2] + iT31 * pbyTR1[x3] +
							iT12 * pbyTR2[x1] + iT32 * pbyTR2[x3] +
							iT13 * pbyTR3[x1] + iT23 * pbyTR3[x2] + iT33 * pbyTR3[x3];
					iG += iT11 * pbyTG1[x1] + iT21 * pbyTG1[x2] + iT31 * pbyTG1[x3] +
							iT12 * pbyTG2[x1] + iT32 * pbyTG2[x3] +
							iT13 * pbyTG3[x1] + iT23 * pbyTG3[x2] + iT33 * pbyTG3[x3];
					iB += iT11 * pbyTB1[x1] + iT21 * pbyTB1[x2] + iT31 * pbyTB1[x3] +
							iT12 * pbyTB2[x1] + iT32 * pbyTB2[x3] +
							iT13 * pbyTB3[x1] + iT23 * pbyTB3[x2] + iT33 * pbyTB3[x3];
				}
				if(bUseSize5)
				{
					iR += iT00 * pbyTR0[x0] + iT10 * pbyTR0[x1] + iT20 * pbyTR0[x2] + iT30 * pbyTR0[x3] + iT40 * pbyTR0[x4] +
							iT01 * pbyTR1[x0] + iT41 * pbyTR1[x4] +
							iT02 * pbyTR2[x0] + iT42 * pbyTR2[x4] +
							iT03 * pbyTR3[x0] + iT43 * pbyTR3[x4] +
							iT04 * pbyTR4[x0] + iT14 * pbyTR4[x1] + iT24 * pbyTR4[x2] + iT34 * pbyTR4[x3] + iT44 * pbyTR4[x4];
					iG += iT00 * pbyTG0[x0] + iT10 * pbyTG0[x1] + iT20 * pbyTG0[x2] + iT30 * pbyTG0[x3] + iT40 * pbyTG0[x4] +
							iT01 * pbyTG1[x0] + iT41 * pbyTG1[x4] +
							iT02 * pbyTG2[x0] + iT42 * pbyTG2[x4] +
							iT03 * pbyTG3[x0] + iT43 * pbyTG3[x4] +
							iT04 * pbyTG4[x0] + iT14 * pbyTG4[x1] + iT24 * pbyTG4[x2] + iT34 * pbyTG4[x3] + iT44 * pbyTG4[x4];
					iB += iT00 * pbyTB0[x0] + iT10 * pbyTB0[x1] + iT20 * pbyTB0[x2] + iT30 * pbyTB0[x3] + iT40 * pbyTB0[x4] +
							iT01 * pbyTB1[x0] + iT41 * pbyTB1[x4] +
							iT02 * pbyTB2[x0] + iT42 * pbyTB2[x4] +
							iT03 * pbyTB3[x0] + iT43 * pbyTB3[x4] +
							iT04 * pbyTB4[x0] + iT14 * pbyTB4[x1] + iT24 * pbyTB4[x2] + iT34 * pbyTB4[x3] + iT44 * pbyTB4[x4];
				}
			}

			if(bUseP)
			{
				iR += iP22 * pbyPR2[x2];
				iG += iP22 * pbyPG2[x2];
				iB += iP22 * pbyPB2[x2];
				if(bUseSize3)
				{
					iR += iP11 * pbyPR1[x1] + iP21 * pbyPR1[x2] + iP31 * pbyPR1[x3] +
							iP12 * pbyPR2[x1] + iP32 * pbyPR2[x3] +
							iP13 * pbyPR3[x1] + iP23 * pbyPR3[x2] + iP33 * pbyPR3[x3];
					iG += iP11 * pbyPG1[x1] + iP21 * pbyPG1[x2] + iP31 * pbyPG1[x3] +
							iP12 * pbyPG2[x1] + iP32 * pbyPG2[x3] +
							iP13 * pbyPG3[x1] + iP23 * pbyPG3[x2] + iP33 * pbyPG3[x3];
					iB += iP11 * pbyPB1[x1] + iP21 * pbyPB1[x2] + iP31 * pbyPB1[x3] +
							iP12 * pbyPB2[x1] + iP32 * pbyPB2[x3] +
							iP13 * pbyPB3[x1] + iP23 * pbyPB3[x2] + iP33 * pbyPB3[x3];
				}
				if(bUseSize5)
				{
					iR += iP00 * pbyPR0[x0] + iP10 * pbyPR0[x1] + iP20 * pbyPR0[x2] + iP30 * pbyPR0[x3] + iP40 * pbyPR0[x4] +
							iP01 * pbyPR1[x0] + iP41 * pbyPR1[x4] +
							iP02 * pbyPR2[x0] + iP42 * pbyPR2[x4] +
							iP03 * pbyPR3[x0] + iP43 * pbyPR3[x4] +
							iP04 * pbyPR4[x0] + iP14 * pbyPR4[x1] + iP24 * pbyPR4[x2] + iP34 * pbyPR4[x3] + iP44 * pbyPR4[x4];
					iG += iP00 * pbyPG0[x0] + iP10 * pbyPG0[x1] + iP20 * pbyPG0[x2] + iP30 * pbyPG0[x3] + iP40 * pbyPG0[x4] +
							iP01 * pbyPG1[x0] + iP41 * pbyPG1[x4] +
							iP02 * pbyPG2[x0] + iP42 * pbyPG2[x4] +
							iP03 * pbyPG3[x0] + iP43 * pbyPG3[x4] +
							iP04 * pbyPG4[x0] + iP14 * pbyPG4[x1] + iP24 * pbyPG4[x2] + iP34 * pbyPG4[x3] + iP44 * pbyPG4[x4];
					iB += iP00 * pbyPB0[x0] + iP10 * pbyPB0[x1] + iP20 * pbyPB0[x2] + iP30 * pbyPB0[x3] + iP40 * pbyPB0[x4] +
							iP01 * pbyPB1[x0] + iP41 * pbyPB1[x4] +
							iP02 * pbyPB2[x0] + iP42 * pbyPB2[x4] +
							iP03 * pbyPB3[x0] + iP43 * pbyPB3[x4] +
							iP04 * pbyPB4[x0] + iP14 * pbyPB4[x1] + iP24 * pbyPB4[x2] + iP34 * pbyPB4[x3] + iP44 * pbyPB4[x4];
				}
			}

			if(bUseO)
			{
				iR += iO22 * pbyOR2[x2];
				iG += iO22 * pbyOG2[x2];
				iB += iO22 * pbyOB2[x2];
				if(bUseSize3)
				{
					iR += iO11 * pbyOR1[x1] + iO21 * pbyOR1[x2] + iO31 * pbyOR1[x3] +
							iO12 * pbyOR2[x1] + iO32 * pbyOR2[x3] +
							iO13 * pbyOR3[x1] + iO23 * pbyOR3[x2] + iO33 * pbyOR3[x3];
					iG += iO11 * pbyOG1[x1] + iO21 * pbyOG1[x2] + iO31 * pbyOG1[x3] +
							iO12 * pbyOG2[x1] + iO32 * pbyOG2[x3] +
							iO13 * pbyOG3[x1] + iO23 * pbyOG3[x2] + iO33 * pbyOG3[x3];
					iB += iO11 * pbyOB1[x1] + iO21 * pbyOB1[x2] + iO31 * pbyOB1[x3] +
							iO12 * pbyOB2[x1] + iO32 * pbyOB2[x3] +
							iO13 * pbyOB3[x1] + iO23 * pbyOB3[x2] + iO33 * pbyOB3[x3];
				}
				if(bUseSize5)
				{
					iR += iO00 * pbyOR0[x0] + iO10 * pbyOR0[x1] + iO20 * pbyOR0[x2] + iO30 * pbyOR0[x3] + iO40 * pbyOR0[x4] +
							iO01 * pbyOR1[x0] + iO41 * pbyOR1[x4] +
							iO02 * pbyOR2[x0] + iO42 * pbyOR2[x4] +
							iO03 * pbyOR3[x0] + iO43 * pbyOR3[x4] +
							iO04 * pbyOR4[x0] + iO14 * pbyOR4[x1] + iO24 * pbyOR4[x2] + iO34 * pbyOR4[x3] + iO44 * pbyOR4[x4];
					iG += iO00 * pbyOG0[x0] + iO10 * pbyOG0[x1] + iO20 * pbyOG0[x2] + iO30 * pbyOG0[x3] + iO40 * pbyOG0[x4] +
							iO01 * pbyOG1[x0] + iO41 * pbyOG1[x4] +
							iO02 * pbyOG2[x0] + iO42 * pbyOG2[x4] +
							iO03 * pbyOG3[x0] + iO43 * pbyOG3[x4] +
							iO04 * pbyOG4[x0] + iO14 * pbyOG4[x1] + iO24 * pbyOG4[x2] + iO34 * pbyOG4[x3] + iO44 * pbyOG4[x4];
					iB += iO00 * pbyOB0[x0] + iO10 * pbyOB0[x1] + iO20 * pbyOB0[x2] + iO30 * pbyOB0[x3] + iO40 * pbyOB0[x4] +
							iO01 * pbyOB1[x0] + iO41 * pbyOB1[x4] +
							iO02 * pbyOB2[x0] + iO42 * pbyOB2[x4] +
							iO03 * pbyOB3[x0] + iO43 * pbyOB3[x4] +
							iO04 * pbyOB4[x0] + iO14 * pbyOB4[x1] + iO24 * pbyOB4[x2] + iO34 * pbyOB4[x3] + iO44 * pbyOB4[x4];
				}
			}

			iR = ((iR * iCountDiv) >> 20) + nBias;
			iG = ((iG * iCountDiv) >> 20) + nBias;
			iB = ((iB * iCountDiv) >> 20) + nBias;

			if(iR > 255)
				iR = 255;
			else if(iR < 0)
				iR = 0;
			if(iG > 255)
				iG = 255;
			else if(iG < 0)
				iG = 0;
			if(iB > 255)
				iB = 255;
			else if(iB < 0)
				iB = 0;

			*dst++ = (iR << 16) + (iG << 8) + iB;
		}
	}

	src = mfd->bUseProcPrev ? fa->dst.data : fa->src.data;
	pbyPR0 = mfd->pbyOR;
	pbyPG0 = mfd->pbyOG;
	pbyPB0 = mfd->pbyOB;
	mfd->pbyOR = mfd->pbyPR;
	mfd->pbyOG = mfd->pbyPG;
	mfd->pbyOB = mfd->pbyPB;
	mfd->pbyPR = pbyPR0;
	mfd->pbyPG = pbyPG0;
	mfd->pbyPB = pbyPB0;
	for(y = 0; y < h; y++)
	{
		for(int x = 0; x < w; x++)
		{
			*pbyPR0++ = (BYTE)((*src & 0xff0000) >> 16);
			*pbyPG0++ = (BYTE)((*src & 0xff00) >> 8);
			*pbyPB0++ = (BYTE)(*src++ & 0xff);
		}
		src = (Pixel32 *)((char *)src + (mfd->bUseProcPrev ? fa->dst.modulo : fa->src.modulo));
	}

	return 0;
}

long ParamProc(FilterActivation *fa, const FilterFunctions *ff) {
    return FILTERPARAM_SWAP_BUFFERS;
}


BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
	MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

	UINT auiPrevPrev[] = {ED_O00, ED_O01, ED_O02, ED_O03, ED_O04, ED_O05, ED_O06, ED_O07, ED_O08, ED_O09, ED_O10, ED_O11, ED_O12, ED_O13, ED_O14, ED_O15, ED_O16, ED_O17, ED_O18, ED_O19, ED_O20, ED_O21, ED_O22, ED_O23, ED_O24};
	UINT auiPrev[] = {ED_P00, ED_P01, ED_P02, ED_P03, ED_P04, ED_P05, ED_P06, ED_P07, ED_P08, ED_P09, ED_P10, ED_P11, ED_P12, ED_P13, ED_P14, ED_P15, ED_P16, ED_P17, ED_P18, ED_P19, ED_P20, ED_P21, ED_P22, ED_P23, ED_P24};
	UINT auiThis[] = {ED_T00, ED_T01, ED_T02, ED_T03, ED_T04, ED_T05, ED_T06, ED_T07, ED_T08, ED_T09, ED_T10, ED_T11, ED_T12, ED_T13, ED_T14, ED_T15, ED_T16, ED_T17, ED_T18, ED_T19, ED_T20, ED_T21, ED_T22, ED_T23, ED_T24};
	switch(msg)
	{
		case WM_INITDIALOG:
		{
         SetWindowLong(hdlg, DWL_USER, lParam);
         mfd = (MyFilterData *)lParam;

			char str[10];
			for(int i = 0; i < 25; i++)
			{
				sprintf(str, "%i", mfd->anPrevPrev[i]);
				SetDlgItemText(hdlg, auiPrevPrev[i], str);
				sprintf(str, "%i", mfd->anPrev[i]);
				SetDlgItemText(hdlg, auiPrev[i], str);
				sprintf(str, "%i", mfd->anThis[i]);
				SetDlgItemText(hdlg, auiThis[i], str);
			}
			sprintf(str, "%i", mfd->nBias);
			SetDlgItemText(hdlg, ED_BIAS, str);
			CheckDlgButton(hdlg, CX_USE_PROC_PREV, mfd->bUseProcPrev ? BST_CHECKED:BST_UNCHECKED);
		}

		return TRUE;

		case WM_COMMAND:
			switch(LOWORD(wParam))
			{
				case IDOK:
				{
					char str[10];
					for(int i = 0; i < 25; i++)
					{
						GetDlgItemText(hdlg, auiPrevPrev[i], str, 10);
						mfd->anPrevPrev[i] = atoi(str);
						GetDlgItemText(hdlg, auiPrev[i], str, 10);
						mfd->anPrev[i] = atoi(str);
						GetDlgItemText(hdlg, auiThis[i], str, 10);
						mfd->anThis[i] = atoi(str);
					}
					GetDlgItemText(hdlg, ED_BIAS, str, 10);
					mfd->nBias = atoi(str);
					mfd->bUseProcPrev = !!IsDlgButtonChecked(hdlg, CX_USE_PROC_PREV);
					EndDialog(hdlg, 0);
					return TRUE;
				}
				case IDCANCEL:
					EndDialog(hdlg, 1);
					return FALSE;
            break;
			}
	}

	return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd) {
    return DialogBoxParam(fa->filter->module->hInstModule,
            MAKEINTRESOURCE(IDD_FILTER_GENERAL_CONVOLUTION_3D), hwnd,
            (DLGPROC)ConfigDlgProc, (LPARAM)fa->filter_data);
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str) {
	str[0] = '\0';
}
