/*
    MSharpen Filter for VirtualDub -- performs sharpening
	limited to edge areas of the frame.
	Copyright (C) 1999-2000 Donald A. Graft

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

	The author can be contacted at:
	Donald Graft
	neuron2@home.com.
*/

#include <windows.h>
#include <commctrl.h>
#include <stdio.h>
#include <crtdbg.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

///////////////////////////////////////////////////////////////////////////

int RunProc(const FilterActivation *fa, const FilterFunctions *ff);
int StartProc(FilterActivation *fa, const FilterFunctions *ff);
int EndProc(FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int InitProc(FilterActivation *fa, const FilterFunctions *ff);
int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

typedef struct MyFilterData {
	IFilterPreview		*ifp;
	unsigned char		*blur;
	unsigned char		*work;
	int 				strength;
	int					threshold;
	int					show;
} MyFilterData;

bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	_snprintf(buf, buflen, "Config(%d, %d, %d)",
		mfd->strength,
		mfd->threshold,
		mfd->show);

	return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->strength		= argv[0].asInt();
	mfd->threshold		= argv[1].asInt();
	mfd->show		= !!argv[2].asInt();
}

ScriptFunctionDef func_defs[]={
	{ (ScriptFunctionPtr)ScriptConfig, "Config", "0iii" },
	{ NULL },
};

CScriptObject script_obj={
	NULL, func_defs
};

struct FilterDefinition filterDef_tutorial = {

	NULL, NULL, NULL,		// next, prev, module
	"msharpen (1.0)",	// name
	"sharpen only in edge detail areas",
							// desc
	"Donald Graft", 		// maker
	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size

	InitProc,				// initProc
	NULL,					// deinitProc
	RunProc,				// runProc
	NULL,				// paramProc
	ConfigProc, 			// configProc
	StringProc, 			// stringProc
	StartProc,				// startProc
	EndProc,				// endProc

	&script_obj,			// script_obj
	FssProc,				// fssProc

};

int StartProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->blur = new unsigned char[4*fa->src.w*fa->src.h];
	if (!mfd->blur)
		return 1;
	mfd->work = new unsigned char[4*fa->src.w*fa->src.h];
	if (!mfd->work)
		return 1;

	return 0;
}

int EndProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	delete[] mfd->blur; mfd->blur = NULL;
	delete[] mfd->work; mfd->work = NULL;

	return 0;
}

///////////////////////////////////////////////////////////////////////////

int RunProc(const FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	const long		pitch = fa->src.pitch;
	const PixDim	width = fa->src.w;
	int				bwidth = 4 * width;
	const PixDim	height = fa->src.h;
	unsigned char *src	= (unsigned char *)fa->src.data;
	unsigned char *dst	= (unsigned char *)fa->dst.data;
	unsigned char *srcpp, *srcp, *srcpn, *workp, *blurp, *blurpn, *dstp;
	int r1, r2, r3, r4, g1, g2, g3, g4, b1, b2, b3, b4;
	int x, y, max;
	int strength = mfd->strength, invstrength = 255 - strength;;
	int threshold = mfd->threshold;

	/* Blur the source image prior to detail detection. Separate
	   dimensions for speed. */
	/* Vertical. */
	srcpp = src;
	srcp = srcpp + pitch;
	srcpn = srcp + pitch;
	workp = mfd->work + bwidth;
	for (y = 1; y < height - 1; y++)
	{
		for (x = 0; x < bwidth; x++)
		{
			workp[x] = (srcpp[x] + srcp[x] + srcpn[x]) / 3;
		}
		srcpp += pitch;
		srcp += pitch;
		srcpn += pitch;
		workp += bwidth;
	}

	/* Horizontal. */
	workp  = mfd->work;
	blurp  = mfd->blur;
	for (y = 0; y < height; y++)
	{
		for (x = 4; x < bwidth - 4; x++)
		{
			blurp[x] = (workp[x-4] + workp[x] + workp[x+4]) / 3;
		}
		workp += bwidth;
		blurp += bwidth;
	}

	/* Fix up blur frame borders. */
	srcp = src;
	blurp = mfd->blur;
	memcpy(blurp, srcp, bwidth);
	memcpy(blurp + (height-1)*bwidth, srcp + (height-1)*pitch, bwidth);
	for (y = 0; y < height; y++)
	{
		*((unsigned int *)(&blurp[0])) = *((unsigned int *)(&srcp[0]));
		*((unsigned int *)(&blurp[bwidth-4])) = *((unsigned int *)(&srcp[bwidth-4]));
		srcp += pitch;
		blurp += bwidth;
	}

	/* Diagonal detail detection. */
	blurp = mfd->blur;
	blurpn = blurp + bwidth;
	workp = mfd->work;
	for (y = 0; y < height - 1; y++) 
	{
		b1 = blurp[0];
		g1 = blurp[1];
		r1 = blurp[2];
		b3 = blurpn[0];
		g3 = blurpn[1];
		r3 = blurpn[2];
		for (x = 0; x < bwidth - 4; x+=4)
		{
			b2 = blurp[x+4];
			g2 = blurp[x+5];
			r2 = blurp[x+6];
			b4 = blurpn[x+4];
			g4 = blurpn[x+5];
			r4 = blurpn[x+6];
			if ((abs(b1 - b4) >= threshold) || (abs(g1 - g4) >= threshold) || (abs(r1 - r4) >= threshold) ||
				(abs(b2 - b3) >= threshold) || (abs(g2 - g3) >= threshold) || (abs(g2 - g3) >= threshold))
			{
				*((unsigned int *)(&workp[x])) = 0xffffffff;
			}
			else
			{
				*((unsigned int *)(&workp[x])) = 0x0;
			}
			b1 = b2; b3 = b4;
			g1 = g2; g3 = g4;
			r1 = r2; r3 = r4;
		}
		workp += bwidth;
		blurp += bwidth;
		blurpn += bwidth;
	}

//	if (highq == true)
	if (1)
	{
		/* Vertical detail detection. */
		for (x = 0; x < bwidth; x+=4)
		{
 			blurp = mfd->blur;
			blurpn = blurp + bwidth;
			workp = mfd->work;
			b1 = blurp[x];
			g1 = blurp[x+1];
			r1 = blurp[x+2];
			for (y = 0; y < height - 1; y++)
			{
				b2 = blurpn[x];
				g2 = blurpn[x+1];
				r2 = blurpn[x+2];
				if (abs(b1 - b2) >= threshold || abs(g1 - g2) >= threshold || abs(r1 - r2) >= threshold)
				{
					*((unsigned int *)(&workp[x])) = 0xffffffff;
				}
				b1 = b2;
				g1 = g2;
				r1 = r2;
				workp += bwidth;
				blurp += bwidth;
				blurpn += bwidth;
			}
		}

		/* Horizontal detail detection. */
		blurp = mfd->blur;
		workp = mfd->work;
		for (y = 0; y < height; y++)
		{
			b1 = blurp[0];
			g1 = blurp[1];
			r1 = blurp[2];
			for (x = 0; x < bwidth - 4; x+=4)
			{
				b2 = blurp[x+4];
				g2 = blurp[x+5];
				r2 = blurp[x+6];
				if (abs(b1 - b2) >= threshold || abs(g1 - g2) >= threshold || abs(r1 - r2) >= threshold)
				{
					*((unsigned int *)(&workp[x])) = 0xffffffff;
				}
				b1 = b2;
				g1 = g2;
				r1 = r2;
			}
			workp += bwidth;
			blurp += bwidth;
		}
	}

	/* Fix up detail map borders. */
	memset(mfd->work + (height-1)*bwidth, 0, bwidth);
	workp = mfd->work;
	for (y = 0; y < height; y++)
	{
		*((unsigned int *)(&workp[bwidth-4])) = 0;
		workp += bwidth;
	}

	if (mfd->show)
	{
		workp	= mfd->work;
		dstp	= dst;
		for (y = 0; y < height; y++)
		{
			for (x = 0; x < bwidth; x++)
			{
				dstp[x] = workp[x];
			}
			workp += bwidth;
			dstp = dstp + fa->dst.pitch;
		}
		return 0;
	}

	/* Fix up output frame borders. */
	srcp = src;
	dstp = dst;
	memcpy(dstp, srcp, bwidth);
	memcpy(dstp + (height-1)*pitch, srcp + (height-1)*pitch, bwidth);
	for (y = 0; y < height; y++)
	{
		*((unsigned int *)(&dstp[0])) = *((unsigned int *)(&srcp[0]));
		*((unsigned int *)(&dstp[bwidth-4])) = *((unsigned int *)(&srcp[bwidth-4]));
		srcp += pitch;
		dstp += pitch;
	}

	/* Now sharpen the edge areas and we're done! */
 	srcp = src + pitch;
 	dstp = dst + pitch;
	workp = mfd->work + bwidth;
	blurp = mfd->blur + bwidth;
	for (y = 1; y < height - 1; y++)
	{
		for (x = 4; x < bwidth - 4; x+=4)
		{
			int xplus1 = x + 1, xplus2 = x + 2;

			if (workp[x])
			{
				b4 = (4*(int)srcp[x] - 3*blurp[x]);
				g4 = (4*(int)srcp[x+1] - 3*blurp[x+1]);
				r4 = (4*(int)srcp[x+2] - 3*blurp[x+2]);

				if (b4 < 0) b4 = 0;
				if (g4 < 0) g4 = 0;
				if (r4 < 0) r4 = 0;
				max = b4;
				if (g4 > max) max = g4;
				if (r4 > max) max = r4;
				if (max > 255)
				{
					b4 = (b4 * 255) / max;
					g4 = (g4 * 255) / max;
					r4 = (r4 * 255) / max;
				}
				dstp[x]      = (strength * b4 + invstrength * srcp[x])      >> 8;
				dstp[xplus1] = (strength * g4 + invstrength * srcp[xplus1]) >> 8;
				dstp[xplus2] = (strength * r4 + invstrength * srcp[xplus2]) >> 8;
			}
			else
			{
				dstp[x]   = srcp[x];
				dstp[xplus1] = srcp[xplus1];
				dstp[xplus2] = srcp[xplus2];
			}
		}
		srcp += pitch;
		dstp += pitch;
		workp += bwidth;
		blurp += bwidth;
	}
	return 0;
}

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_tutorial;

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
	if (!(fd_tutorial = ff->addFilter(fm, &filterDef_tutorial, sizeof(FilterDefinition))))
		return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

	return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
	ff->removeFilter(fd_tutorial);
}

int InitProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->strength = 100;
	mfd->threshold = 25;
	mfd->show = FALSE;

	return 0;
}

BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
	MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

	switch(msg) {
		case WM_INITDIALOG:
			SetWindowLong(hdlg, DWL_USER, lParam);
			mfd = (MyFilterData *)lParam;
			HWND hWnd;

			CheckDlgButton(hdlg, IDC_SHOW, mfd->show ? BST_CHECKED : BST_UNCHECKED);

			hWnd = GetDlgItem(hdlg, IDC_SSTRENGTH);
			SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 255));
			SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->strength);
			SetDlgItemInt(hdlg, IDC_STRENGTH, mfd->strength, FALSE);

			hWnd = GetDlgItem(hdlg, IDC_STHRESHOLD);
			SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 255));
			SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->threshold);
			SetDlgItemInt(hdlg, IDC_THRESHOLD, mfd->threshold, FALSE);

			mfd->ifp->InitButton(GetDlgItem(hdlg, IDPREVIEW));

			return TRUE;
		case WM_HSCROLL:
			if ((HWND) lParam == GetDlgItem(hdlg, IDC_SSTRENGTH))
			{
				int strength = SendMessage(GetDlgItem(hdlg, IDC_SSTRENGTH), TBM_GETPOS, 0, 0);
				if (strength != mfd->strength)
				{
					mfd->strength = strength;
					SetDlgItemInt(hdlg, IDC_STRENGTH, mfd->strength, FALSE);
					mfd->ifp->RedoFrame();
				}
			}
			else if ((HWND) lParam == GetDlgItem(hdlg, IDC_STHRESHOLD))
			{
				int threshold = SendMessage(GetDlgItem(hdlg, IDC_STHRESHOLD), TBM_GETPOS, 0, 0);
				if (threshold != mfd->threshold)
				{
					mfd->threshold = threshold;
					SetDlgItemInt(hdlg, IDC_THRESHOLD, mfd->threshold, FALSE);
					mfd->ifp->RedoFrame();
				}
			}
			break;
		case WM_COMMAND:
			switch(LOWORD(wParam))
			{
			case IDPREVIEW:
				mfd->ifp->Toggle(hdlg);
				break;
			case IDOK:
				EndDialog(hdlg, 0);
				return TRUE;
			case IDHELP:
				{
				char prog[256];
				char path[256];
				LPTSTR ptr;

				GetModuleFileName(NULL, prog, 255);
				GetFullPathName(prog, 255, path, &ptr);
				*ptr = 0;
				strcat(path, "plugins\\MSharpen.html");
				ShellExecute(hdlg, "open", path, NULL, NULL, SW_SHOWNORMAL);
				return TRUE;
				}
			case IDCANCEL:
				EndDialog(hdlg, 1);
				return TRUE;
			case IDC_SHOW:
				mfd->show = !!IsDlgButtonChecked(hdlg, IDC_SHOW);
				mfd->ifp->RedoFrame();
				break;
			case IDC_STRENGTHPLUS:
				if (mfd->strength < 255)
				{
					mfd->strength++;
					SetDlgItemInt(hdlg, IDC_STRENGTH, mfd->strength, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_SSTRENGTH), TBM_SETPOS, (WPARAM)TRUE, mfd->strength);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_STRENGTHMINUS:
				if (mfd->strength > 0)
				{
					mfd->strength--;
					SetDlgItemInt(hdlg, IDC_STRENGTH, mfd->strength, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_SSTRENGTH), TBM_SETPOS, (WPARAM)TRUE, mfd->strength);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_THRESHOLDPLUS:
				if (mfd->threshold < 255)
				{
					mfd->threshold++;
					SetDlgItemInt(hdlg, IDC_THRESHOLD, mfd->threshold, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_STHRESHOLD), TBM_SETPOS, (WPARAM)TRUE, mfd->threshold);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_THRESHOLDMINUS:
				if (mfd->threshold > 0)
				{
					mfd->threshold--;
					SetDlgItemInt(hdlg, IDC_THRESHOLD, mfd->threshold, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_STHRESHOLD), TBM_SETPOS, (WPARAM)TRUE, mfd->threshold);
					mfd->ifp->RedoFrame();
				}
				break;
			}
			break;
	}

	return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd)
{
	MyFilterData *mfd = (MyFilterData *) fa->filter_data;
	MyFilterData mfd_old = *mfd;
	int ret;

	mfd->ifp = fa->ifp;
	if (DialogBoxParam(fa->filter->module->hInstModule,
			MAKEINTRESOURCE(IDD_FILTER), hwnd,
			ConfigDlgProc, (LPARAM) mfd))
	{
		*mfd = mfd_old;
		ret = TRUE;
	}
	else
	{
		ret = FALSE;
	}
	return(ret);
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	sprintf(str, " (strength %d, threshold %d, show %s)",
				mfd->strength, mfd->threshold, mfd->show ? "yes" : "no");
}
