/*
    Smart Smoother Filter for VirtualDub -- performs structure
	retaining noise reduction/smoothing.
	Copyright (C) 1999-2000 Donald A. Graft

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

	The author can be contacted at:
	Donald Graft
	neuron2@home.com.
*/

#include <windows.h>
#include <commctrl.h>
#include <stdio.h>
#include <crtdbg.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

///////////////////////////////////////////////////////////////////////////

int RunProc(const FilterActivation *fa, const FilterFunctions *ff);
int StartProc(FilterActivation *fa, const FilterFunctions *ff);
int EndProc(FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int InitProc(FilterActivation *fa, const FilterFunctions *ff);
int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

typedef struct MyFilterData {
	IFilterPreview		*ifp;
	Pixel32				*abuf;
	int 				diameter;
	int					threshold;
	int					interlaced;
} MyFilterData;

bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	_snprintf(buf, buflen, "Config(%d, %d, %d)",
		mfd->diameter,
		mfd->threshold,
		mfd->interlaced);

	return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->diameter		= argv[0].asInt();
	mfd->threshold		= argv[1].asInt();
	mfd->interlaced		= !!argv[2].asInt();
}

ScriptFunctionDef func_defs[]={
	{ (ScriptFunctionPtr)ScriptConfig, "Config", "0iii" },
	{ NULL },
};

CScriptObject script_obj={
	NULL, func_defs
};

struct FilterDefinition filterDef_tutorial = {

	NULL, NULL, NULL,		// next, prev, module
	"smart smoother (1.1)",	// name
	"Smooth/blur while preserving structure.\n\n[MMX optimized by Avery Lee]",
							// desc
	"Donald Graft", 		// maker
	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size

	InitProc,				// initProc
	NULL,					// deinitProc
	RunProc,				// runProc
	ParamProc,				// paramProc
	ConfigProc, 			// configProc
	StringProc, 			// stringProc
	StartProc,				// startProc
	EndProc,				// endProc

	&script_obj,			// script_obj
	FssProc,				// fssProc

};

int StartProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->abuf		= new Pixel32[fa->src.w*fa->src.h];
	if (!mfd->abuf)
		return 1;

	return 0;
}

int EndProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	delete[] mfd->abuf; mfd->abuf = NULL;

	return 0;
}

long ParamProc(FilterActivation *fa, const FilterFunctions *ff)
{
	fa->dst.offset = fa->src.offset;
	return 0;	// run in place
}

///////////////////////////////////////////////////////////////////////////

void __declspec(naked) asm_sum_pixels_MMX(Pixel32 *src, int w2m1, int compareval, Pixel32 nucleus, long *destarray) {
	__asm {
		push	ebp
		push	edi
		push	esi
		push	ebx

		mov			edx,[esp+4+16]		;load start address
		mov			ebp,[esp+8+16]		;load -(2*radius+1)
		pxor		mm7,mm7
		pxor		mm4,mm4
		movd		mm5,[esp+12+16]
		psllq		mm5,32
		movd		mm6,[esp+16+16]
		punpcklbw	mm6,mm7
xloop:
		movd		mm0,[edx]		
		punpcklbw	mm0,mm7
		movq		mm1,mm0
		psubw		mm0,mm6
		pmaddwd		mm0,mm0
		punpckldq	mm2,mm0
		paddd		mm0,mm2
		pcmpgtd		mm0,mm5
		punpckhdq	mm0,mm0
		pandn		mm0,mm1
		paddw		mm4,mm0
		add			edx,4

		inc			ebp
		jne			xloop

		movq		mm0,mm4
		punpcklwd	mm4,mm7
		punpckhwd	mm0,mm7

		mov			ebp,[esp+20+16]		;ebp = destarray
		paddd		mm4,[ebp+0]
		paddd		mm0,[ebp+8]

		movq		[ebp+0],mm4
		movq		[ebp+8],mm0

		pop		ebx
		pop		esi
		pop		edi
		pop		ebp
		ret
	};
}

///////////////////////////////////////////////////////////////////////////

int RunProc(const FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	const long		pitch = fa->src.pitch;
	const PixDim	w = fa->src.w;
	const PixDim	h = fa->src.h;
	Pixel32 *src, *dst;
	int xlo, xhi, ylo, yhi;

	int x, y, a;
	int N = mfd->diameter;
	int Nover2 = N/2;
	int TWOW = w + w;
	int Nsquared = N * N;
	int nucleus, T = (mfd->threshold+2)/3, Tsquared = T * T;

	/* First, decompose the RGB values. */
	src = fa->src.data;
	dst = mfd->abuf;
	for (y = 0; y < h; y++)
	{
		for (x = 0; x < w; x++)
		{
			Pixel c = src[x];

			dst[x] = (c&0xffffff) + 0x01000000;
		}
		src	= (Pixel *)((char *)src + fa->src.pitch);
		dst += w;
	}

	/* Render. */
	dst	= (Pixel *)((char *)fa->src.data);
	src	= mfd->abuf;

	if (mfd->interlaced)
	{
		for (y = 0; y < h; y++)
		{
			for (x = 0; x < w; x++)
			{
				Pixel32 *kernelsrc;
				struct {
					int b_sum, g_sum, r_sum, count;
				} accum;

				nucleus = src[x];
				accum.count = accum.r_sum = accum.g_sum = accum.b_sum = 0;
				xlo = x - Nover2; if (xlo < 0) xlo = 0;
				xhi = x + Nover2; if (xhi >= w) xhi = w - 1;
				ylo = y - N + 1; if (ylo < 0) ylo = y & 1;
				yhi = y + N - 1; if (yhi >= h) yhi = h - 1;
				kernelsrc = mfd->abuf + xlo + ylo*w;
				a = (yhi - ylo) / 2 + 1;
				do
				{
					// If we want |d| < T, add (T-1) to it, and accept if (unsigned)(d + (T-1)) >= 2T-1.
					asm_sum_pixels_MMX(kernelsrc, -(xhi+1-xlo), Tsquared, nucleus, (long *)&accum);
					kernelsrc += TWOW;
				} while (--a);
				accum.r_sum = (accum.r_sum + accum.count/2) / accum.count;
				accum.g_sum = (accum.g_sum + accum.count/2) / accum.count;
				accum.b_sum = (accum.b_sum + accum.count/2) / accum.count;
				dst[x] = (accum.r_sum << 16) + (accum.g_sum << 8) + accum.b_sum;
			}
			dst = (Pixel *)((char *)dst + fa->src.pitch);
			src += w;
		}
	}
	else
	{
		for (y = 0; y < h; y++)
		{
			for (x = 0; x < w; x++)
			{
				Pixel32 *kernelsrc;
				struct {
					int b_sum, g_sum, r_sum, count;
				} accum;

				nucleus = src[x];
				accum.count = accum.r_sum = accum.g_sum = accum.b_sum = 0;
				xlo = x - Nover2; if (xlo < 0) xlo = 0;
				xhi = x + Nover2; if (xhi >= w) xhi = w - 1;
				ylo = y - Nover2; if (ylo < 0) ylo = 0;
				yhi = y + Nover2; if (yhi >= h) yhi = h - 1;
				kernelsrc = mfd->abuf + xlo + ylo*w;
				a = yhi - ylo + 1;
				do
				{
					// If we want |d| < T, add (T-1) to it, and accept if (unsigned)(d + (T-1)) >= 2T-1.
					asm_sum_pixels_MMX(kernelsrc, -(xhi+1-xlo), Tsquared, nucleus, (long *)&accum);
					kernelsrc += w;
				} while (--a);
				accum.r_sum = (accum.r_sum + accum.count/2) / accum.count;
				accum.g_sum = (accum.g_sum + accum.count/2) / accum.count;
				accum.b_sum = (accum.b_sum + accum.count/2) / accum.count;
				dst[x] = (accum.r_sum << 16) + (accum.g_sum << 8) + accum.b_sum;
			}
			dst = (Pixel *)((char *)dst + fa->src.pitch);
			src += w;
		}
	}

	__asm emms

	return 0;
}

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_tutorial;

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
	if (!(fd_tutorial = ff->addFilter(fm, &filterDef_tutorial, sizeof(FilterDefinition))))
		return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

	return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
	ff->removeFilter(fd_tutorial);
}

int InitProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->diameter = 5;
	mfd->threshold = 25;
	mfd->interlaced = FALSE;

	return 0;
}

BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
	MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

	switch(msg) {
		case WM_INITDIALOG:
			SetWindowLong(hdlg, DWL_USER, lParam);
			mfd = (MyFilterData *)lParam;
			HWND hWnd;

			CheckDlgButton(hdlg, IDC_INTERLACED, mfd->interlaced ? BST_CHECKED : BST_UNCHECKED);

			hWnd = GetDlgItem(hdlg, IDC_SDIAMETER);
			SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(1, 5));
			SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->diameter/2);
			SetDlgItemInt(hdlg, IDC_DIAMETER, mfd->diameter, FALSE);

			hWnd = GetDlgItem(hdlg, IDC_STHRESHOLD);
			SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 200));
			SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->threshold);
			SetDlgItemInt(hdlg, IDC_THRESHOLD, mfd->threshold, FALSE);

			mfd->ifp->InitButton(GetDlgItem(hdlg, IDPREVIEW));

			return TRUE;
		case WM_HSCROLL:
			if ((HWND) lParam == GetDlgItem(hdlg, IDC_SDIAMETER))
			{
				int diameter = SendMessage(GetDlgItem(hdlg, IDC_SDIAMETER), TBM_GETPOS, 0, 0) * 2 + 1;
				if (diameter != mfd->diameter)
				{
					mfd->diameter = diameter;
					SetDlgItemInt(hdlg, IDC_DIAMETER, mfd->diameter, FALSE);
					mfd->ifp->RedoFrame();
				}
			}
			else if ((HWND) lParam == GetDlgItem(hdlg, IDC_STHRESHOLD))
			{
				int threshold = SendMessage(GetDlgItem(hdlg, IDC_STHRESHOLD), TBM_GETPOS, 0, 0);
				if (threshold != mfd->threshold)
				{
					mfd->threshold = threshold;
					if (mfd->threshold == 0) mfd->threshold = 1;
					SetDlgItemInt(hdlg, IDC_THRESHOLD, mfd->threshold, FALSE);
					mfd->ifp->RedoFrame();
				}
			}
			break;
		case WM_COMMAND:
			switch(LOWORD(wParam))
			{
			case IDPREVIEW:
				mfd->ifp->Toggle(hdlg);
				break;
			case IDOK:
				EndDialog(hdlg, 0);
				return TRUE;
			case IDHELP:
				{
				char prog[256];
				char path[256];
				LPTSTR ptr;
				GetModuleFileName(NULL, prog, 255);
				GetFullPathName(prog, 255, path, &ptr);
				*ptr = 0;
				strcat(path, "plugins\\Smooth.txt");
				OutputDebugString(path);
				OutputDebugString("\n");
				strcpy(prog, "Notepad ");
				strcat(prog, path);
				WinExec(prog, SW_SHOW);
				return TRUE;
				}
			case IDCANCEL:
				EndDialog(hdlg, 1);
				return TRUE;
			case IDC_INTERLACED:
				mfd->interlaced = !!IsDlgButtonChecked(hdlg, IDC_INTERLACED);
				mfd->ifp->RedoFrame();
				break;
			case IDC_DIAMETERPLUS:
				if (mfd->diameter < 11)
				{
					mfd->diameter += 2;
					SetDlgItemInt(hdlg, IDC_DIAMETER, mfd->diameter, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_SDIAMETER), TBM_SETPOS, (WPARAM)TRUE, mfd->diameter/2);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_DIAMETERMINUS:
				if (mfd->diameter > 3)
				{
					mfd->diameter -= 2;
					SetDlgItemInt(hdlg, IDC_DIAMETER, mfd->diameter, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_SDIAMETER), TBM_SETPOS, (WPARAM)TRUE, mfd->diameter/2);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_THRESHOLDPLUS:
				if (mfd->threshold < 100)
				{
					mfd->threshold += 1;
					SetDlgItemInt(hdlg, IDC_THRESHOLD, mfd->threshold, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_STHRESHOLD), TBM_SETPOS, (WPARAM)TRUE, mfd->threshold);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_THRESHOLDMINUS:
				if (mfd->threshold > 0)
				{
					mfd->threshold -= 1;
					SetDlgItemInt(hdlg, IDC_THRESHOLD, mfd->threshold, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_STHRESHOLD), TBM_SETPOS, (WPARAM)TRUE, mfd->threshold);
					mfd->ifp->RedoFrame();
				}
				break;
			}
			break;
	}

	return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd)
{
	MyFilterData *mfd = (MyFilterData *) fa->filter_data;
	MyFilterData mfd_old = *mfd;
	int ret;

	mfd->ifp = fa->ifp;
	if (DialogBoxParam(fa->filter->module->hInstModule,
			MAKEINTRESOURCE(IDD_FILTER), hwnd,
			ConfigDlgProc, (LPARAM) mfd))
	{
		*mfd = mfd_old;
		ret = TRUE;
	}
	else
	{
		ret = FALSE;
	}
	return(ret);
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	sprintf(str, " (diam %d, str %d, inter %s)",
				mfd->diameter, mfd->threshold, mfd->interlaced ? "yes" : "no");
}
