/*
    2d cleaner filter for VirtualDub -- blends a pixel with pixels 
	surrounding it as long as those pixels are simular to the source
	pixel

    Copyright (C) 2000 Jim Casaburi
		Based on code by Avery Lee
		Useful suggestions and much help from Donald A. Graft

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

	The author can be contacted at:
	Jim Casaburi
	casaburi@earthlink.net

	New versions of the source code and the compiled filter can be 
	found at http://home.earthlink.net/~casaburi/download/
*/


#include "filter.h"
#include "resource.h"
#include <stdio.h>
#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#define throwaway
#define newcalc
#define calcn
#define sep

extern HINSTANCE g_hInst;

typedef struct MyFilterData {
	bool fDebugNoise;
	bool fInterlace;
	int fThreshold;
	int fRadius;
} MyFilterData;

unsigned long ppixels(unsigned long *src, int lowx, int lowy, int highx, int highy);
static int twodclean_run(const FilterActivation *fa, const FilterFunctions *ff);
BOOL CALLBACK twodclean_ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam);
int twodclean_ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void twodclean_StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void twodcleanScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);

#define READSRC(byteoff, lineoff) (unsigned long *)((char *)src + pitch*(lineoff) + (byteoff*4))

ScriptFunctionDef twodclean_func_defs[]={
    { (ScriptFunctionPtr)twodcleanScriptConfig, "Config", "0iii" },
    { NULL },
};

CScriptObject twodclean_obj={
    NULL, twodclean_func_defs
};

static int fThreshold;
static bool fDebugNoise;
static int pitch;
static int multfact;

// takes a pixel and averages it with those around it.  low* and high* defines
// the region to take pixels from (note this code is now integrated into avg_run
unsigned long ppixels(unsigned long *src, int lowx, int lowy, int highx, 
						int highy) {
	register unsigned char red, blue, green;
	register unsigned char fred, fblue, fgreen;
	register int reddif, bluedif, greendif;
	register int a, b;
	register int redcount;
	register int bluecount;
	register int greencount;
	register int totred = 0;
	register int totblue = 0;
	register int totgreen = 0;
	register int pix = 1;
	unsigned long *pixel;

	// read in the source pixel
	pixel = (unsigned long *) ((char *) src + pitch);
	fred = redcount = *((unsigned char *) pixel + 2);
	fgreen = greencount = *((unsigned char *) pixel + 1);
	fblue = bluecount = *((unsigned char *) pixel);
	// get the other pixels (note multfact is 2 if we are dealing with an interlaced
	// source, or 1 otherwise
	for (b = lowy; b < highy; b += multfact) 
		pixel = (unsigned long *) ((char *) src + pitch * b + lowx * 4);
		for (a = lowx; a < highx; a++)
			if (!((a == 0) && (b == 1))) { // as long as it isn't the source pixel
				// extract the r,g,b values
				red = *((unsigned char *) pixel + 2);
				green = *((unsigned char *) pixel + 1);
				blue = *((unsigned char *) pixel);
				// if the abs(red difference) is greater than the threshold
				// then throw out the value otherwise add it to the running 
				// total
				reddif = red - fred;
				if ((reddif > fThreshold) || (reddif < -fThreshold)) 
					totred--;
				else 
					redcount += red;
				// if the abs(blue difference) is greater than the threshold
				// then throw out the value otherwise add it to the running 
				// total
				bluedif = blue - fblue;
				if ((bluedif > fThreshold) || (bluedif < -fThreshold)) 
					totblue--;
				else 
					bluecount += blue;
				// if the abs(green difference) is greater than the threshold
				// then throw out the value otherwise add it to the running 
				// total
				greendif = green - fgreen;
				if ((greendif > fThreshold) || (greendif < -fThreshold)) 
					totgreen--;
				else 
					greencount += green;
				pix++;
				pixel = (unsigned long *) ((char *) pixel + 4);
			}
// now tot* is the number of values that will be averaged
	totred += pix; 
	totblue += pix; 
	totgreen += pix;
// if we are to show edges and less than 1/2 of the rgb values are to be
// averaged together, show a black pixel instead
	if (fDebugNoise && (totblue + totgreen + totred < ((pix * 3)/ 2))) 
		return 0;
	else 
// otherwise average the rgb values and construct a pixel around them
	return (bluecount / totblue + ((greencount / totgreen) << 8) + 
		((redcount / totred) << 16));
}


static int twodclean_run(const FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	unsigned long w,h;
	unsigned long *src = (unsigned long *)fa->src.data, *dst = (unsigned long *)fa->dst.data;
	pitch = fa->src.pitch;
	fThreshold = mfd->fThreshold;
	fDebugNoise = mfd->fDebugNoise;
	int x, y;
	int lowx, lowy;
	int highx, highy;
	int radius = mfd->fRadius;	

	
	unsigned char red, blue, green;
	unsigned char fred, fblue, fgreen;
	int reddif, bluedif, greendif;
	int a, b;
	int redcount;
	int bluecount;
	int greencount;
	int totred = 0;
	int totblue = 0;
	int totgreen = 0;
	int pix = 1;
	unsigned long *pixel;

	src -= pitch>>2;
		
	
	// are we dealing with interlaced source?
	multfact = mfd->fInterlace ? 2 : 1; 
	h = fa->src.h;
	y = 0;
	do { 
		w = fa->src.w;
		x = 0;
		do {
			// set the boundaries of the averaging region
			lowx = -radius;
			lowy = 1 - (radius * multfact);
			highx = radius;
			highy = 1 + (radius * multfact);
			// make sure we aren't exceeding frame boundaries
			if (x + lowx < 0) lowx = 0;
			if (y + lowy < 1) lowy = 1;
			if (x + highx > fa->src.w-1) highx = (fa->src.w - 1)-x;
			if (y + highy > fa->src.h) highy = fa->src.h - y;
//			*dst = ppixels(src, lowx, lowy, highx, highy);			
			

			totred = 0;
			totgreen = 0;
			totblue = 0;
			pix = 1;
			// read in the source pixel
			pixel = (unsigned long *) ((char *) src + pitch);
			fred = redcount = *((unsigned char *) pixel + 2);
			fgreen = greencount = *((unsigned char *) pixel + 1);
			fblue = bluecount = *((unsigned char *) pixel);
			// get the other pixels (note multfact is 2 if we are dealing with an interlaced
			// source, or 1 otherwise
			for (b = lowy; b < highy; b += multfact) {
				pixel = (unsigned long *) ((char *) src + pitch * b + lowx * 4);
				for (a = lowx; a < highx; a++) {
					if (!((a == 0) && (b == 1))) { // as long as it isn't the source pixel
						// extract the r,g,b values
						red = *((unsigned char *) pixel + 2);
						green = *((unsigned char *) pixel + 1);
						blue = *((unsigned char *) pixel);
						// if the abs(red difference) is greater than the threshold
						// then throw out the value otherwise add it to the running 
						// total
						reddif = red - fred;
						if ((reddif > fThreshold) || (reddif < -fThreshold)) 
							totred--;
						else 
							redcount += red;
						// if the abs(blue difference) is greater than the threshold
						// then throw out the value otherwise add it to the running 
						// total
						bluedif = blue - fblue;
						if ((bluedif > fThreshold) || (bluedif < -fThreshold)) 
							totblue--;
							else 
							bluecount += blue;
						// if the abs(green difference) is greater than the threshold
						// then throw out the value otherwise add it to the running 
						// total
						greendif = green - fgreen;
						if ((greendif > fThreshold) || (greendif < -fThreshold)) 
							totgreen--;
							else 
							greencount += green;
						pix++;
						// move to the pixel to the right
						pixel = (unsigned long *) ((char *) pixel + 4);
						}
				}
			}
			// now tot* is the number of values that will be averaged
			totred += pix; 
			totblue += pix; 
			totgreen += pix;
			// if we are to show edges and less than 1/2 of the rgb values are to be
			// averaged together, show a black pixel instead
			if (fDebugNoise && (totblue + totgreen + totred < ((pix * 3)/ 2))) {
				*dst = 0;
			}
			else 
			// otherwise average the rgb values and construct a pixel around them
			*dst = (bluecount / totblue + ((greencount / totgreen) << 8) + 
					((redcount / totred) << 16));
			
			
			++src;
			++dst;
			x++;
		} while (--w);
		src += fa->src.modulo>>2;
		dst += fa->dst.modulo>>2;
		y++;
	} while (--h);
return 0;
}

BOOL CALLBACK twodclean_ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
         MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

         switch(msg) {
             case WM_INITDIALOG:
                 SetWindowLong(hdlg, DWL_USER, lParam);
                 mfd = (MyFilterData *)lParam;

                 CheckDlgButton(hdlg, IDC_DEBUGNOISE, mfd->fDebugNoise?BST_CHECKED:BST_UNCHECKED);
				 CheckDlgButton(hdlg, IDC_INTERLACE, mfd->fInterlace?BST_CHECKED:BST_UNCHECKED);
				 SetDlgItemInt(hdlg, IDC_THRESHOLD, mfd->fThreshold, FALSE);				
				 SetDlgItemInt(hdlg, IDC_RADIUS, mfd->fRadius, FALSE);
                 return TRUE;

             case WM_COMMAND:
                 switch(LOWORD(wParam)) {
                 case IDOK:
					// just added to fix cancel bug
					 long threshold;
					BOOL success;

					threshold = GetDlgItemInt(hdlg, IDC_RADIUS, &success, FALSE);
					if (!success || threshold < 0 || threshold > 10) {
						SetFocus((HWND)lParam);
						MessageBeep(MB_ICONQUESTION);
						return TRUE;
					}

					mfd->fRadius = threshold;

					threshold = GetDlgItemInt(hdlg, IDC_THRESHOLD, &success, FALSE);
					if (!success || threshold < 0 || threshold > 255) {
						SetFocus((HWND)lParam);
						MessageBeep(MB_ICONQUESTION);
						return TRUE;
					}

					mfd->fThreshold = threshold;                     
					
					 mfd->fDebugNoise = !!IsDlgButtonChecked(hdlg, IDC_DEBUGNOISE);
					 mfd->fInterlace = !!IsDlgButtonChecked(hdlg, IDC_INTERLACE);
					 EndDialog(hdlg, 0);
                     return TRUE;
                 case IDCANCEL:
                     EndDialog(hdlg, 1);
                     return FALSE;
				 case IDC_RADIUS:
				if (HIWORD(wParam) == EN_KILLFOCUS) {
				}
				return TRUE;

				 case IDC_THRESHOLD:
				if (HIWORD(wParam) == EN_KILLFOCUS) {
				} 
				return TRUE;

                 }
                 break;
         }

         return FALSE;
     }


int twodclean_ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd) {
         return DialogBoxParam(fa->filter->module->hInstModule,
                 MAKEINTRESOURCE(IDD_2DCLEAN), hwnd,
                 twodclean_ConfigDlgProc, (LPARAM)fa->filter_data);
     }

void twodclean_StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str) {
         const char *modes[2][2]={
			 " (normal processing",
			 " (interlaced processing",
			 " (show edges",
			 " (show edges+interlaced processing"
		 };
         MyFilterData *mfd = (MyFilterData *)fa->filter_data;
		 sprintf(str, "%s, threshold of %d, area of %dx%d)", 
			 modes[mfd->fDebugNoise][mfd->fInterlace], mfd->fThreshold,
			 mfd->fRadius * 2 + 1, mfd->fRadius * 2 + 1);
     }

void twodcleanScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
    FilterActivation *fa = (FilterActivation *)lpVoid;
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

    mfd->fInterlace    = !!argv[0].asInt();
	mfd->fThreshold		= argv[1].asInt();
	mfd->fRadius	   = argv[2].asInt();
}

bool twodcleanFssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

    _snprintf(buf, buflen, "Config(%d, %d, %d)",
        mfd->fInterlace,
		mfd->fThreshold,
		mfd->fRadius);

    return true;
}

static int twodclean_start(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	return 0;
}

int twodclean_init(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->fThreshold = 3;
	mfd->fRadius = 1;
	return 0;
}

FilterDefinition filterDef_2dclean={
	0,0,NULL,
	"2d cleaner",
	"Replaces each pixel with the average of its neighbors, as long as the average falls within a specified threshold",
	"Jim Casaburi",NULL,
	sizeof(MyFilterData),
	twodclean_init,NULL,
	twodclean_run,
	NULL,
	twodclean_ConfigProc,
	twodclean_StringProc,
	NULL,
	NULL,
	&twodclean_obj,          // script_obj
    twodcleanFssProc,        // fssProc

};

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
     extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

     static FilterDefinition *fd_tutorial;

     int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
         if (!(fd_tutorial = ff->addFilter(fm, &filterDef_2dclean, sizeof(FilterDefinition))))
             return 1;

         vdfd_ver    = VIRTUALDUB_FILTERDEF_VERSION;
         vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

         return 0;
     }

     void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
         ff->removeFilter(fd_tutorial);
     }

