/*
    Smart Smoother IQ Filter for VirtualDub -- performs structure
    retaining noise reduction/smoothing, but only on chroma portion.
    Copyright (C) 2002 Tim Park
    Based on Smart Smoother which is
    Copyright (C) 1999-2000 Donald A. Graft
    Optimized for Diameter 11 by Mark DeNies

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    The author can be contacted at:
    Tim Park
    tim@fiction.org
*/

#include <windows.h>
#include <commctrl.h>
#include <stdio.h>
#include <crtdbg.h>
#include <math.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

///////////////////////////////////////////////////////////////////////////

int RunProc(const FilterActivation *fa, const FilterFunctions *ff);
int StartProc(FilterActivation *fa, const FilterFunctions *ff);
int EndProc(FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int InitProc(FilterActivation *fa, const FilterFunctions *ff);
int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

typedef unsigned char YPixel;
typedef signed short IQPixel;

typedef struct MyFilterData {
	YPixel	*ymap;
	IQPixel	*iqmap1;
	IQPixel	*iqmap2;
	IQPixel	*iqmap3;

	IFilterPreview	*ifp;
	int	diameter;
	int	threshold;
	int	interlaced;
} MyFilterData;

bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	_snprintf(buf, buflen, "Config(%d, %d, %d)",
		mfd->diameter,
		mfd->threshold,
		mfd->interlaced);

	return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->diameter		= argv[0].asInt();
	mfd->threshold		= argv[1].asInt();
	mfd->interlaced		= !!argv[2].asInt();
}

ScriptFunctionDef func_defs[]={
	{ (ScriptFunctionPtr)ScriptConfig, "Config", "0iii" },
	{ NULL },
};

CScriptObject script_obj={
	NULL, func_defs
};

struct FilterDefinition filterDef_tutorial = {
	NULL, NULL, NULL,		// next, prev, module
	"smart smoother IQ (0.6)",	// name
	"Smooth/blur chroma information while preserving structure.\n[Based on Donald Graft's Smart Smoother]",  // desc
	"Tim Park", 		// maker
	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size

	InitProc,				// initProc
	NULL,					// deinitProc
	RunProc,				// runProc
	ParamProc,				// paramProc
	ConfigProc, 			// configProc
	StringProc, 			// stringProc
	StartProc,				// startProc
	EndProc,				// endProc

	&script_obj,			// script_obj
	FssProc,				// fssProc
};

int StartProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	if (!(mfd->ymap = new YPixel[fa->dst.w * fa->dst.h]))
		return 1;
	if (!(mfd->iqmap1 = new IQPixel[fa->dst.w * fa->dst.h]))
		return 1;
	if (!(mfd->iqmap2 = new IQPixel[fa->dst.w * fa->dst.h]))
		return 1;
	if (!(mfd->iqmap3 = new IQPixel[fa->dst.w * fa->dst.h]))
		return 1;
	return 0;
}

int EndProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	delete[] mfd->ymap; mfd->ymap = NULL;
	delete[] mfd->iqmap1; mfd->iqmap1 = NULL;
	delete[] mfd->iqmap2; mfd->iqmap2 = NULL;
	delete[] mfd->iqmap3; mfd->iqmap3 = NULL;
	return 0;
}

long ParamProc(FilterActivation *fa, const FilterFunctions *ff)
{
	fa->dst.offset = fa->src.offset;
	return 0;	// run in place
}

///////////////////////////////////////////////////////////////////////////
struct sum {
	int c_sum, count;
};

inline void sum_pixels(IQPixel *src, int w2m1, int compareval, IQPixel nucleus, struct sum *destarray) {
	IQPixel bc;
	int csum = 0, num = 0;
	int low = nucleus - compareval;
	int hi  = nucleus + compareval;

	do
	{
		bc = *src++;
		if (bc <= hi && bc >= low)
		{
			csum += bc;
			num++;
		}
	} while(++w2m1);
	destarray->c_sum += csum;
	destarray->count += num;
}

//****************************************************************
inline void sum_pixels11(IQPixel *src, int compareval, IQPixel nucleus, struct sum *destarray) {
	IQPixel bc;
	int csum = 0, num = 0;
	int low = nucleus - compareval;
	int hi  = nucleus + compareval;

	bc = *src;
	if (bc <= hi && bc >= low) csum += bc, num++;
	bc = *(src+1);
	if (bc <= hi && bc >= low) csum += bc, num++;
	bc = *(src+2);
	if (bc <= hi && bc >= low) csum += bc, num++;
	bc = *(src+3);
	if (bc <= hi && bc >= low) csum += bc, num++;
	bc = *(src+4);
	if (bc <= hi && bc >= low) csum += bc, num++;
	bc = *(src+5);
	if (bc <= hi && bc >= low) csum += bc, num++;
	bc = *(src+6);
	if (bc <= hi && bc >= low) csum += bc, num++;
	bc = *(src+7);
	if (bc <= hi && bc >= low) csum += bc, num++;
	bc = *(src+8);
	if (bc <= hi && bc >= low) csum += bc, num++;
	bc = *(src+9);
	if (bc <= hi && bc >= low) csum += bc, num++;
	bc = *(src+10);
	if (bc <= hi && bc >= low) csum += bc, num++;

	destarray->c_sum += csum;
	destarray->count += num;
}

smoothN(int N, const FilterActivation *fa, MyFilterData *mfd, IQPixel *origsrc, IQPixel *origdst)
{	
	const PixDim	w = fa->src.w;
	const PixDim	h = fa->src.h;
	IQPixel *src, *dst;
	int xlo, ylo, nRow, aStart;
	int xhi, yhi;
	int x, y, a, isum, count;

	int MinusN = -N;
	int Nover2 = N/2;
	int TWOW = w + w;
	int Nsquared = N * N;
	int nucleus, T = (mfd->threshold+2)/3;
	int Tsquared = (T * T)/3;
	int SqrtTsquared = (int) sqrt(Tsquared);

	src = origsrc;
	dst = origdst;
	if (mfd->interlaced)
	{
		for (y = 0; y < h; y++)
		{
			for (x = 0; x < w; x++)
			{
				IQPixel *kernelsrc;
				struct sum accum;

				nucleus = src[x];
				accum.count = accum.c_sum = 0;
				xlo = x - Nover2; if (xlo < 0) xlo = 0;
				xhi = x + Nover2; if (xhi >= w) xhi = w - 1;
				ylo = y - N + 1; if (ylo < 0) ylo = y & 1;
				yhi = y + N - 1; if (yhi >= h) yhi = h - 1;
				kernelsrc = origsrc + xlo + ylo*w;
				a = (yhi - ylo) / 2 + 1;
				do
				{
					sum_pixels(kernelsrc, -(xhi+1-xlo), SqrtTsquared, nucleus, &accum);
					kernelsrc += TWOW;
				} while (--a);
				dst[x] = (accum.c_sum + accum.count/2) / accum.count;
			}
			dst += w;
			src += w;
		}
	}
	else
	{
		int hend = h - Nover2;
		int wend = w - Nover2;
		IQPixel *kernelsrc,*kernelsrc1;
		struct sum accum;

		// Process the first Nover2+1 lines:
		ylo = 0;
		aStart = Nover2 + 1;
		for (y=0; y<Nover2; y++)
		{
			isum = count = xlo = 0;
			nRow = -(Nover2 + 1);
			for (x=0; x<Nover2; x++) {		
				nucleus = src[x];
				kernelsrc = origsrc;
				accum.count = accum.c_sum = 0;
				a = aStart;
				do
				{
					sum_pixels(kernelsrc, nRow, SqrtTsquared, nucleus, &accum);
					kernelsrc += w;
				} while (--a);
  				dst[x] = (accum.c_sum + accum.count/2) / accum.count;
				nRow--;
			}
			for (; x<wend; x++) {
				nucleus = src[x];
				accum.count = accum.c_sum = 0;
				kernelsrc = origsrc + x - Nover2;
				a = aStart;
				do
				{
					sum_pixels(kernelsrc, MinusN, SqrtTsquared, nucleus, &accum);
					kernelsrc += w;
				} while (--a);
  				dst[x] = (accum.c_sum + accum.count/2) / accum.count;
			}
			nRow = MinusN+1;
			for (; x<w; x++) {
				nucleus = src[x];
				accum.count = accum.c_sum = 0;
				kernelsrc = origsrc + x - Nover2;
				a = aStart;
				do
				{
					sum_pixels(kernelsrc, nRow, SqrtTsquared, nucleus, &accum);
					kernelsrc += w;
				} while (--a);
  				dst[x] = (accum.c_sum + accum.count/2) / accum.count;
				nRow++;
			}
			aStart++;
			dst += w;
			src += w;
		}

		// Process the middle lines
		ylo = 0;
		for (; y < hend; y++,ylo++)
		{
			kernelsrc1 = origsrc + ylo*w;
			nRow = -(Nover2 + 1);
			for (x = 0; x<Nover2; x++) {
				nucleus = src[x];
				kernelsrc = kernelsrc1;
				accum.count = accum.c_sum = 0;
				a = N;
				do
				{
					sum_pixels(kernelsrc, nRow, SqrtTsquared, nucleus, &accum);
					kernelsrc += w;
				} while (--a);
  				dst[x] = (accum.c_sum + accum.count/2) / accum.count;
				nRow--;
			}
			xlo = 0;
			for (; x < wend; x++,xlo++)
			{
				IQPixel *kernelsrc;
				struct sum accum;

				nucleus = src[x];
				accum.count = accum.c_sum = 0;
				kernelsrc = origsrc + xlo + ylo*w;
				a = N;
				do
				{
					sum_pixels11(kernelsrc, SqrtTsquared, nucleus, &accum);
					kernelsrc += w;
				} while (--a);
  				dst[x] = (accum.c_sum + accum.count/2) / accum.count;
			}
			nRow = MinusN+1;
			for (; x<w; x++,xlo++) {
				nucleus = src[x];
				accum.count = accum.c_sum = 0;
				kernelsrc = origsrc + xlo + ylo*w;
				a = N;
				do
				{
					sum_pixels(kernelsrc, nRow, SqrtTsquared, nucleus, &accum);
					kernelsrc += w;
				} while (--a);
  				dst[x] = (accum.c_sum + accum.count/2) / accum.count;
				nRow++;
			}
			dst += w;
			src += w;
		}

		//  Process the last Nover2 lines
		aStart = N-1;
		for (; y<h; y++,ylo++) {
			kernelsrc1 = origsrc + ylo*w;
			isum = count = xlo = 0;
			nRow = -(Nover2 + 1);
			for (x=0; x<Nover2; x++) {		
				nucleus = src[x];
				kernelsrc = origsrc + xlo + ylo*w;
				accum.count = accum.c_sum = 0;
				a = aStart;
				do
				{
					sum_pixels(kernelsrc, nRow, SqrtTsquared, nucleus, &accum);
					kernelsrc += w;
				} while (--a);
  				dst[x] = (accum.c_sum + accum.count/2) / accum.count;
				nRow--;
			}
			xlo = 0;
			for (; x<wend; x++,xlo++) {
				nucleus = src[x];
				accum.count = accum.c_sum = 0;
				kernelsrc = origsrc + xlo + ylo*w;
				a = aStart;
				do
				{
					sum_pixels(kernelsrc, MinusN, SqrtTsquared, nucleus, &accum);
					kernelsrc += w;
				} while (--a);
  				dst[x] = (accum.c_sum + accum.count/2) / accum.count;
			}
			nRow = MinusN+1;
			for (; x<w; x++,xlo++) {
				nucleus = src[x];
				accum.count = accum.c_sum = 0;
				kernelsrc = origsrc + xlo + ylo*w;
				a = aStart;
				do
				{
					sum_pixels(kernelsrc, nRow, SqrtTsquared, nucleus, &accum);
					kernelsrc += w;
				} while (--a);
  				dst[x] = (accum.c_sum + accum.count/2) / accum.count;
				nRow++;
			}
			aStart--;
			dst += w;
			src += w;
		}
	}

	return 0;
}

//****************************************************************
int smooth11(const FilterActivation *fa, MyFilterData *mfd, IQPixel *origsrc, IQPixel *origdst)
{
	const PixDim	w = fa->src.w;
	const PixDim	h = fa->src.h;
	IQPixel *src, *dst;
	int xlo, ylo, nRow, aStart;
	int xhi, yhi;
	int x, y, a, isum, count;

	int TWOW = w + w;
	int nucleus, T = (mfd->threshold+2)/3;
	int Tsquared = (T * T)/3;
	int SqrtTsquared = (int) sqrt(Tsquared);

	src = origsrc;
	dst = origdst;
		
	if (mfd->interlaced)
	{
		for (y = 0; y < h; y++)
		{
			for (x = 0; x < w; x++)
			{
				IQPixel *kernelsrc;
				struct sum accum;

				nucleus = src[x];
				accum.count = accum.c_sum = 0;
				xlo = x - 5; if (xlo < 0) xlo = 0;
				xhi = x + 5; if (xhi >= w) xhi = w - 1;
				ylo = y - 11 + 1; if (ylo < 0) ylo = y & 1;
				yhi = y + 11 - 1; if (yhi >= h) yhi = h - 1;
				kernelsrc = origsrc + xlo + ylo*w;
				a = (yhi - ylo) / 2 + 1;
				do
				{
					sum_pixels(kernelsrc, -(xhi+1-xlo), SqrtTsquared, nucleus, &accum);
					kernelsrc += TWOW;
				} while (--a);
				dst[x] = (accum.c_sum + accum.count/2) / accum.count;
			}
			dst += w;
			src += w;
		}
	}
	else
	{
		int hend = h - 5;
		int wend = w - 5;
		IQPixel *kernelsrc,*kernelsrc1;
		struct sum accum;

		// Process the first Nover2+1 lines:
		ylo = 0;
		aStart = 5 + 1;
		for (y=0; y<5; y++)
		{
			isum = count = xlo = 0;
			nRow = -(5 + 1);
			for (x=0; x<5; x++) {		
				nucleus = src[x];
				kernelsrc = origsrc;
				accum.count = accum.c_sum = 0;
				a = aStart;
				do
				{
					sum_pixels(kernelsrc, nRow, SqrtTsquared, nucleus, &accum);
					kernelsrc += w;
				} while (--a);
  				dst[x] = (accum.c_sum + accum.count/2) / accum.count;
				nRow--;
			}
			for (; x<wend; x++) {
				nucleus = src[x];
				accum.count = accum.c_sum = 0;
				kernelsrc = origsrc + x - 5;
				a = aStart;
				do
				{
					sum_pixels(kernelsrc, -11, SqrtTsquared, nucleus, &accum);
					kernelsrc += w;
				} while (--a);
  				dst[x] = (accum.c_sum + accum.count/2) / accum.count;
			}
			nRow = -11+1;
			for (; x<w; x++) {
				nucleus = src[x];
				accum.count = accum.c_sum = 0;
				kernelsrc = origsrc + x - 5;
				a = aStart;
				do
				{
					sum_pixels(kernelsrc, nRow, SqrtTsquared, nucleus, &accum);
					kernelsrc += w;
				} while (--a);
  				dst[x] = (accum.c_sum + accum.count/2) / accum.count;
				nRow++;
			}
			aStart++;
			dst += w;
			src += w;
		}

		// Process the middle lines
		ylo = 0;
		for (; y < hend; y++,ylo++)
			{
			kernelsrc1 = origsrc + ylo*w;
			nRow = -(5 + 1);
			for (x = 0; x<5; x++) {
				nucleus = src[x];
				kernelsrc = kernelsrc1;
				accum.count = accum.c_sum = 0;
				a = 11;
				do
				{
					sum_pixels(kernelsrc, nRow, SqrtTsquared, nucleus, &accum);
					kernelsrc += w;
				} while (--a);
  				dst[x] = (accum.c_sum + accum.count/2) / accum.count;
				nRow--;
			}
			xlo = 0;
			for (; x < wend; x++,xlo++)
			{
				IQPixel *kernelsrc;
				struct sum accum;

				nucleus = src[x];
				accum.count = accum.c_sum = 0;
				kernelsrc = origsrc + xlo + ylo*w;
				a = 11;
				do
				{
					sum_pixels11(kernelsrc, SqrtTsquared, nucleus, &accum);
					kernelsrc += w;
				} while (--a);
  				dst[x] = (accum.c_sum + accum.count/2) / accum.count;
			}
			nRow = -11+1;
			for (; x<w; x++,xlo++) {
				nucleus = src[x];
				accum.count = accum.c_sum = 0;
				kernelsrc = origsrc + xlo + ylo*w;
				a = 11;
				do
				{
					sum_pixels(kernelsrc, nRow, SqrtTsquared, nucleus, &accum);
					kernelsrc += w;
				} while (--a);
  				dst[x] = (accum.c_sum + accum.count/2) / accum.count;
				nRow++;
			}
			dst += w;
			src += w;
		}

		//  Process the last 5 lines
		aStart = 10;
		for (; y<h; y++,ylo++) {
			kernelsrc1 = origsrc + ylo*w;
			isum = count = xlo = 0;
			nRow = -(5 + 1);
			for (x=0; x<5; x++) {		
				nucleus = src[x];
				kernelsrc = origsrc + xlo + ylo*w;
				accum.count = accum.c_sum = 0;
				a = aStart;
				do
				{
					sum_pixels(kernelsrc, nRow, SqrtTsquared, nucleus, &accum);
					kernelsrc += w;
				} while (--a);
  				dst[x] = (accum.c_sum + accum.count/2) / accum.count;
				nRow--;
			}
			xlo = 0;
			for (; x<wend; x++,xlo++) {
				nucleus = src[x];
				accum.count = accum.c_sum = 0;
				kernelsrc = origsrc + xlo + ylo*w;
				a = aStart;
				do
				{
					sum_pixels(kernelsrc, -11, SqrtTsquared, nucleus, &accum);
					kernelsrc += w;
				} while (--a);
  				dst[x] = (accum.c_sum + accum.count/2) / accum.count;
			}
			nRow = -11+1;
			for (; x<w; x++,xlo++) {
				nucleus = src[x];
				accum.count = accum.c_sum = 0;
				kernelsrc = origsrc + xlo + ylo*w;
				a = aStart;
				do
				{
					sum_pixels(kernelsrc, nRow, SqrtTsquared, nucleus, &accum);
					kernelsrc += w;
				} while (--a);
  				dst[x] = (accum.c_sum + accum.count/2) / accum.count;
				nRow++;
			}
			aStart--;
			dst += w;
			src += w;
		}
	}
	return 0;
}

//****************************************************************
int smoothProc(const FilterActivation *fa, IQPixel *origsrc, IQPixel *origdst)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	int N = mfd->diameter;

	if (N == 11) return smooth11(fa,mfd,origsrc,origdst);
	else return smoothN(N,fa,mfd,origsrc,origdst) ;
}

///////////////////////////////////////////////////////////////////////////

int RunProc(const FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	PixDim	w = fa->src.w;
	PixDim	h = fa->src.h;
	Pixel *src, *dst;
	YPixel *ly;
	IQPixel *li, *lq;

	ly = mfd->ymap;
	li = mfd->iqmap1;
	lq = mfd->iqmap2;
	src = (Pixel *)fa->src.data;
	h = fa->dst.h;
	do {
		w = fa->dst.w;
		do {
			Pixel c;
			long r, g, b;
	
			c = *src++;
			r = (c>>16) & 255;
			g = (c>> 8) & 255;
			b = c & 255;

//			*ly++ = (unsigned char)(0.299 * r + 0.587 * g + 0.114 * b);
//			*li++ = (short)(0.596 * r - 0.274 * g - 0.322 * b);
//			*lq++ = (short)(0.212 * r - 0.523 * g + 0.311 * b);
			*ly++ = (unsigned char)((19595 * r + 38470 * g + 7471 * b) >> 16);
			*li++ = (short)((39059 * r - 17957 * g - 21103 * b) >> 16);
			*lq++ = (short)((13894 * r - 34275 * g + 20382 * b) >> 16);
		} while(--w);
		src = (Pixel *)((char *)src + fa->src.modulo);
	} while(--h);

	smoothProc(fa, mfd->iqmap2, mfd->iqmap3);
	smoothProc(fa, mfd->iqmap1, mfd->iqmap2);

	ly = mfd->ymap;
	li = mfd->iqmap2;
	lq = mfd->iqmap3;
	dst = (Pixel *)fa->dst.data;
	h = fa->dst.h;
	do {
		w = fa->dst.w;
		do {
			unsigned char y;
			short i, q, r, g, b;

			y = *ly++;
			i = *li++;
			q = *lq++;

//			r = (short)((double)y + 0.956 * i + 0.621 * q);
//			g = (short)((double)y - 0.272 * i - 0.647 * q);
//			b = (short)((double)y - 1.105 * i + 1.702 * q);
			r = (short)(((y << 16) + 62652 * i + 40698 * q) >> 16);
			g = (short)(((y << 16) - 17826 * i - 42402 * q) >> 16);
			b = (short)(((y << 16) - 72417 * i + 111542 * q) >> 16);

			if (r<0) r=0; else if (r>255) r=255;
			if (g<0) g=0; else if (g>255) g=255;
			if (b<0) b=0; else if (b>255) b=255;
			*dst++ = (r<<16) + (g<<8) + b;
		} while(--w);
		dst = (Pixel *)((char *)dst + fa->dst.modulo);
	} while(--h);

	return 0;
}

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_tutorial;

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
	if (!(fd_tutorial = ff->addFilter(fm, &filterDef_tutorial, sizeof(FilterDefinition))))
		return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

	return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
	ff->removeFilter(fd_tutorial);
}

int InitProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->diameter = 11;
	mfd->threshold = 200;
	mfd->interlaced = FALSE;

	return 0;
}

BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
	MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

	switch(msg) {
		case WM_INITDIALOG:
			SetWindowLong(hdlg, DWL_USER, lParam);
			mfd = (MyFilterData *)lParam;
			HWND hWnd;

			CheckDlgButton(hdlg, IDC_INTERLACED, mfd->interlaced ? BST_CHECKED : BST_UNCHECKED);

			hWnd = GetDlgItem(hdlg, IDC_SDIAMETER);
			SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(1, 5));
			SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->diameter/2);
			SetDlgItemInt(hdlg, IDC_DIAMETER, mfd->diameter, FALSE);

			hWnd = GetDlgItem(hdlg, IDC_STHRESHOLD);
			SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 300));
			SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->threshold);
			SetDlgItemInt(hdlg, IDC_THRESHOLD, mfd->threshold, FALSE);

			mfd->ifp->InitButton(GetDlgItem(hdlg, IDPREVIEW));

			return TRUE;
		case WM_HSCROLL:
			if ((HWND) lParam == GetDlgItem(hdlg, IDC_SDIAMETER))
			{
				int diameter = SendMessage(GetDlgItem(hdlg, IDC_SDIAMETER), TBM_GETPOS, 0, 0) * 2 + 1;
				if (diameter != mfd->diameter)
				{
					mfd->diameter = diameter;
					SetDlgItemInt(hdlg, IDC_DIAMETER, mfd->diameter, FALSE);
					mfd->ifp->RedoFrame();
				}
			}
			else if ((HWND) lParam == GetDlgItem(hdlg, IDC_STHRESHOLD))
			{
				int threshold = SendMessage(GetDlgItem(hdlg, IDC_STHRESHOLD), TBM_GETPOS, 0, 0);
				if (threshold != mfd->threshold)
				{
					mfd->threshold = threshold;
					if (mfd->threshold == 0) mfd->threshold = 1;
					SetDlgItemInt(hdlg, IDC_THRESHOLD, mfd->threshold, FALSE);
					mfd->ifp->RedoFrame();
				}
			}
			break;
		case WM_COMMAND:
			switch(LOWORD(wParam))
			{
			case IDPREVIEW:
				mfd->ifp->Toggle(hdlg);
				break;
			case IDOK:
				EndDialog(hdlg, 0);
				return TRUE;
			case IDHELP:
				{
				char prog[256];
				char path[256];
				LPTSTR ptr;
				GetModuleFileName(NULL, prog, 255);
				GetFullPathName(prog, 255, path, &ptr);
				*ptr = 0;
				strcat(path, "plugins\\SmoothIQ.txt");
				OutputDebugString(path);
				OutputDebugString("\n");
				strcpy(prog, "Notepad ");
				strcat(prog, path);
				WinExec(prog, SW_SHOW);
				return TRUE;
				}
			case IDCANCEL:
				EndDialog(hdlg, 1);
				return TRUE;
			case IDC_INTERLACED:
				mfd->interlaced = !!IsDlgButtonChecked(hdlg, IDC_INTERLACED);
				mfd->ifp->RedoFrame();
				break;
			case IDC_DIAMETERPLUS:
				if (mfd->diameter < 11)
				{
					mfd->diameter += 2;
					SetDlgItemInt(hdlg, IDC_DIAMETER, mfd->diameter, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_SDIAMETER), TBM_SETPOS, (WPARAM)TRUE, mfd->diameter/2);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_DIAMETERMINUS:
				if (mfd->diameter > 3)
				{
					mfd->diameter -= 2;
					SetDlgItemInt(hdlg, IDC_DIAMETER, mfd->diameter, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_SDIAMETER), TBM_SETPOS, (WPARAM)TRUE, mfd->diameter/2);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_THRESHOLDPLUS:
				if (mfd->threshold < 300)
				{
					mfd->threshold += 1;
					SetDlgItemInt(hdlg, IDC_THRESHOLD, mfd->threshold, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_STHRESHOLD), TBM_SETPOS, (WPARAM)TRUE, mfd->threshold);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_THRESHOLDMINUS:
				if (mfd->threshold > 0)
				{
					mfd->threshold -= 1;
					SetDlgItemInt(hdlg, IDC_THRESHOLD, mfd->threshold, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_STHRESHOLD), TBM_SETPOS, (WPARAM)TRUE, mfd->threshold);
					mfd->ifp->RedoFrame();
				}
				break;
			}
			break;
	}

	return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd)
{
	MyFilterData *mfd = (MyFilterData *) fa->filter_data;
	MyFilterData mfd_old = *mfd;
	int ret;

	mfd->ifp = fa->ifp;
	if (DialogBoxParam(fa->filter->module->hInstModule,
			MAKEINTRESOURCE(IDD_FILTER), hwnd,
			ConfigDlgProc, (LPARAM) mfd))
	{
		*mfd = mfd_old;
		ret = TRUE;
	}
	else
	{
		ret = FALSE;
	}
	return(ret);
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	sprintf(str, " (diam %d, str %d, inter %s)",
				mfd->diameter, mfd->threshold, mfd->interlaced ? "yes" : "no");
}
