//	VirtualDub - Video processing and capture application
//	Copyright (C) 1998-2001 Avery Lee
//  Packaged as plug-in and UI enhanced by Donald Graft.
//
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <windows.h>
#include <commctrl.h>
#include <stdio.h>
#include <crtdbg.h>
#include <math.h>
#include <stdlib.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

///////////////////////////////////////////////////////////////////////////

int brightcont_run(const FilterActivation *fa, const FilterFunctions *ff);
long brightcont_param(FilterActivation *fa, const FilterFunctions *ff);
int brightcont_init(FilterActivation *fa, const FilterFunctions *ff);
int brightcont_config(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void brightcont_string(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void brightcont_script_config(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool brightcont_script_line(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

extern "C" void asm_brightcont1_run(
		void *dst,
		unsigned long width,
		unsigned long height,
		unsigned long stride,
		unsigned long multiplier,
		unsigned long adder1,
		unsigned long adder2
		);

extern "C" void asm_brightcont2_run(
		void *dst,
		unsigned long width,
		unsigned long height,
		unsigned long stride,
		unsigned long multiplier,
		unsigned long adder1,
		unsigned long adder2
		);

///////////////////////////////////

typedef struct MyFilterData {
	IFilterPreview		*ifp;
	LONG bright;
	LONG cont;
} MyFilterData;

static bool brightcont_script_line(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	_snprintf(buf, buflen, "Config(%d,%d)", mfd->bright, mfd->cont);

	return true;
}

static void brightcont_script_config(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->bright	= argv[0].asInt();
	mfd->cont		= argv[1].asInt();
}

static ScriptFunctionDef brightcont_func_defs[]={
	{ (ScriptFunctionPtr)brightcont_script_config, "Config", "0ii" },
	{ NULL },
};

static CScriptObject brightcont_obj={
	NULL, brightcont_func_defs
};

FilterDefinition filterDef_brightcont={
	0,0,NULL,
	"brightness/contrast (UI-enhanced)(1.0)",
	"Adjusts brightness and contrast of an image linearly.\n"
	"Main filter code by Avery Lee; UI enhanced by Donald Graft\n\n"
	"[Assembly optimized] [MMX optimized]",
	"Avery Lee/Donald Graft",NULL,
	sizeof(MyFilterData),
	brightcont_init,
	NULL,
	brightcont_run,
	brightcont_param,
	brightcont_config,
	brightcont_string,
	NULL,
	NULL,
	&brightcont_obj,
	brightcont_script_line,
};

int brightcont_run(const FilterActivation *fa, const FilterFunctions *ff) {	
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	if (mfd->bright>=0)
		asm_brightcont2_run(
				fa->src.data,
				fa->src.w,
				fa->src.h,
				fa->src.pitch,
				mfd->cont,
				mfd->bright*0x00100010L,
				mfd->bright*0x00001000L
				);
	else
		asm_brightcont1_run(
				fa->src.data,
				fa->src.w,
				fa->src.h,
				fa->src.pitch,
				mfd->cont,
				(-mfd->bright)*0x00100010L,
				(-mfd->bright)*0x00001000L
				);

	return 0;
}

long brightcont_param(FilterActivation *fa, const FilterFunctions *ff) {
	fa->dst.offset	= fa->src.offset;
	fa->dst.modulo	= fa->src.modulo;
	fa->dst.pitch	= fa->src.pitch;
	return 0;
}

//////////////////

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_brightcont;

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
	if (!(fd_brightcont = ff->addFilter(fm, &filterDef_brightcont, sizeof(FilterDefinition))))
{
OutputDebugString("ouch\n");
		return 1;
}
OutputDebugString("mmmmmm\n");

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

	return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
	ff->removeFilter(fd_brightcont);
}

static int brightcont_init(FilterActivation *fa, const FilterFunctions *ff) {
	((MyFilterData *)fa->filter_data)->bright = 0;
	((MyFilterData *)fa->filter_data)->cont = 16;

	return 0;
}

BOOL CALLBACK brightcontDlgProc( HWND hDlg, UINT message, UINT wParam, LONG lParam)
{
	MyFilterData *mfd = (MyFilterData *)GetWindowLong(hDlg, DWL_USER);

    switch (message)
    {
        case WM_INITDIALOG:
			{
				HWND hWnd;
				char c[2];

				mfd = (MyFilterData *)lParam;
				SetWindowLong(hDlg, DWL_USER, lParam);

				hWnd = GetDlgItem(hDlg, IDC_BRIGHTNESS);
				SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 512));
				SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->bright+256);
				SetDlgItemInt(hDlg, IDC_BRIGHTVAL, (abs(mfd->bright)*25)/64, FALSE);
				c[1] = 0;
				if (mfd->bright < 0) *c = '-';
				else if (mfd->bright == 0) *c = ' ';
				else *c = '+';
				SetDlgItemText(hDlg, IDC_BRIGHTSIGN, c);

				hWnd = GetDlgItem(hDlg, IDC_CONTRAST);
				SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 32));
				SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->cont);
				SetDlgItemInt(hDlg, IDC_CONTVAL, (mfd->cont*25)/4, FALSE);

				mfd->ifp->InitButton(GetDlgItem(hDlg, IDPREVIEW));
				return (TRUE);
			}

        case WM_COMMAND:                      
            if (LOWORD(wParam) == IDOK) {
				EndDialog(hDlg, 0);
				return TRUE;
			} else if (LOWORD(wParam) == IDPREVIEW) {
				mfd->ifp->Toggle(hDlg);
				break;
			} else if (LOWORD(wParam) == IDCANCEL) {
                EndDialog(hDlg, 1);
                return TRUE;
            } else if (LOWORD(wParam) == IDC_NORMAL) {
				SendMessage(GetDlgItem(hDlg, IDC_BRIGHTNESS), TBM_SETPOS, (WPARAM)TRUE, 256);
				SetDlgItemInt(hDlg, IDC_BRIGHTVAL, 0, FALSE);
				SetDlgItemText(hDlg, IDC_BRIGHTSIGN, " ");
				mfd->bright = 0;
				mfd->ifp->RedoFrame();
				return TRUE;
            } else if (LOWORD(wParam) == IDC_100PERCENT) {
				SendMessage(GetDlgItem(hDlg, IDC_CONTRAST), TBM_SETPOS, (WPARAM)TRUE, 16);
				SetDlgItemInt(hDlg, IDC_CONTVAL, 100, FALSE);
				mfd->cont = 16;
				mfd->ifp->RedoFrame();
				return TRUE;
			}
            break;

		case WM_HSCROLL:
			{
				int bright, cont;
				char c[2];

				c[1] = 0;
				bright = SendMessage(GetDlgItem(hDlg, IDC_BRIGHTNESS), TBM_GETPOS, 0, 0) - 256;
				cont = SendMessage(GetDlgItem(hDlg, IDC_CONTRAST), TBM_GETPOS, 0, 0);
				if (bright != mfd->bright)
				{
					mfd->bright = bright;
					SetDlgItemInt(hDlg, IDC_BRIGHTVAL, (abs(bright)*25)/64, FALSE);
					if (bright < 0) *c = '-';
					else if (bright == 0) *c = ' ';
					else *c = '+';
					SetDlgItemText(hDlg, IDC_BRIGHTSIGN, c);
					mfd->bright = bright;
					mfd->ifp->RedoFrame();
				}
				if (cont != mfd->cont)
				{
					SetDlgItemInt(hDlg, IDC_CONTVAL, (cont*25)/4, FALSE);
					mfd->cont = cont;
					mfd->ifp->RedoFrame();
				}
				break;
			}
	}

    return FALSE;
}

int brightcont_config(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	MyFilterData mfd_old = *mfd;
	int ret;

	mfd->ifp = fa->ifp;
OutputDebugString("config\n");

	if (DialogBoxParam(fa->filter->module->hInstModule, MAKEINTRESOURCE(IDD_FILTER_BRIGHTCONT),
		hwnd, brightcontDlgProc, (LPARAM)mfd))
	{
		*mfd = mfd_old;
OutputDebugString("no dialog\n");
		ret = TRUE;
	}
    else
	{
		ret = FALSE;
	}
	return(ret);
}

static void brightcont_string(const FilterActivation *fa, const FilterFunctions *ff, char *buf) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	if (mfd->bright == 0)
		sprintf(buf," (bright %d%%, cont %d%%)", (mfd->bright*25)/64, (mfd->cont*25)/4);
	else
		sprintf(buf," (bright %+d%%, cont %d%%)", (mfd->bright*25)/64, (mfd->cont*25)/4);
}
