// VirtualDub filter: deinterlace - field shift
// by Gunnar Thalin (guth@home.se)

#include <windows.h>
#include <stdio.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

///////////////////////////////////////////////////////////////////////////

int InitProc(FilterActivation *fa, const FilterFunctions *ff);
int RunProc(const FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

typedef struct MyFilterData {
    bool     bShiftEven;
} MyFilterData;


bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	_snprintf(buf, buflen, "Config(%i)",mfd->bShiftEven);

	return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->bShiftEven = !!argv[0].asInt();
}

ScriptFunctionDef func_defs[]={
	{ (ScriptFunctionPtr)ScriptConfig, "Config", "0i" },
	{ NULL },
};

CScriptObject script_obj={
	NULL, func_defs
};


struct FilterDefinition filterDef = {

	NULL, NULL, NULL,	// next, prev, module
	"field shift v1.0",	// name
	"Doubles the height and shifts every other frame down one line to make 50 or 60 fps output flicker less.",	// desc
	"Gunnar Thalin",		// maker
	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size

	InitProc,		// initProc
	NULL,			// deinitProc
	RunProc,		// runProc
	ParamProc,		// paramProc
	ConfigProc,		// configProc
	StringProc,		// stringProc
	NULL,			// startProc
	NULL,			// endProc

	&script_obj,	// script_obj
	FssProc,		// fssProc

};

///////////////////////////////////////////////////////////////////////////

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd;

///////////////////////////////////////////////////////////////////////////

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
    if (!(fd = ff->addFilter(fm, &filterDef, sizeof(FilterDefinition))))
        return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

    return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
    ff->removeFilter(fd);
}

///////////////////////////////////////////////////////////////////////////

int InitProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->bShiftEven = true;
	return 0;
}

int RunProc(const FilterActivation *fa, const FilterFunctions *ff) {
	Pixel32 *src, *dst;
	src = (Pixel32 *)fa->src.data;
	dst = (Pixel32 *)fa->dst.data;
	int iOddEven = ((MyFilterData *)(fa->filter_data))->bShiftEven ? 0 : 1;

	if(fa->pfsi->lCurrentSourceFrame % 2 == iOddEven)
	{
		memset(dst, 0, fa->src.w * sizeof(Pixel32));
		dst = (Pixel32 *)((char *)dst + fa->dst.pitch);
		for(int y = fa->src.h - 2; y >= 0; y--)
		{
			memcpy(dst, src, fa->src.w * sizeof(Pixel32));
			dst = (Pixel32 *)((char *)dst + fa->dst.pitch);
			memcpy(dst, src, fa->src.w * sizeof(Pixel32));
			dst = (Pixel32 *)((char *)dst + fa->dst.pitch);
			src = (Pixel32 *)((char *)src + fa->src.pitch);
		}
		memcpy(dst, src, fa->src.w * sizeof(Pixel32));
	}
	else
	{
		for(int y = fa->src.h - 1; y >= 0; y--)
		{
			memcpy(dst, src, fa->src.w * sizeof(Pixel32));
			dst = (Pixel32 *)((char *)dst + fa->dst.pitch);
			memcpy(dst, src, fa->src.w * sizeof(Pixel32));
			dst = (Pixel32 *)((char *)dst + fa->dst.pitch);
			src = (Pixel32 *)((char *)src + fa->src.pitch);
		}
	}

    return 0;
}


long ParamProc(FilterActivation *fa, const FilterFunctions *ff) {
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

    fa->dst.h *= 2;
    return FILTERPARAM_SWAP_BUFFERS;
}


BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
    MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

    switch(msg) {
        case WM_INITDIALOG:
            SetWindowLong(hdlg, DWL_USER, lParam);
            mfd = (MyFilterData *)lParam;
            CheckDlgButton(hdlg, IDC_SHIFT_EVEN, mfd->bShiftEven?BST_CHECKED:BST_UNCHECKED);
            return TRUE;

        case WM_COMMAND:
            switch(LOWORD(wParam)) {
            case IDOK:
                mfd->bShiftEven = !!IsDlgButtonChecked(hdlg, IDC_SHIFT_EVEN);
                EndDialog(hdlg, 0);
                return TRUE;
            case IDCANCEL:
                EndDialog(hdlg, 1);
                return FALSE;
            }
            break;
    }

    return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd) {
    return DialogBoxParam(fa->filter->module->hInstModule,
            MAKEINTRESOURCE(IDD_FILTER_FIELD_SHIFT), hwnd,
            (DLGPROC)ConfigDlgProc, (LPARAM)fa->filter_data);
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str) {
	if(((MyFilterData *)(fa->filter_data))->bShiftEven)
		strcpy(str, " (even frames)");
	else
		strcpy(str, " (odd frames)");
}
