/*
    Antiflicker Filter for VirtualDub -- removes temporal moire flickering.

	PREPARE PHASE
	*************

		Copyright (C) 2002-2004 Alessandro Malanca, 
		http://web.tiscali.it/minomala/

	original ideas by:

		Copyright (C) 1999-2000 Donald A. Graft

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <windows.h>
#include <commctrl.h>
#include <stdio.h>
#include <direct.h>
#include <math.h>
#include <crtdbg.h>
#include <commdlg.h>
#include <winuser.h>

#include "resource.h"
#include "filter.h"
#include "VdubGfxLib.h"

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

///////////////////////////////////////////////////////////////////////////

int		RunAnalyzing(const FilterActivation *fa, const FilterFunctions *ff);
int		StartProc(FilterActivation *fa, const FilterFunctions *ff);
int		EndProc(FilterActivation *fa, const FilterFunctions *ff);
long	ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int		MyInitProc(FilterActivation *fa, const FilterFunctions *ff);
int		ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void	StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void	ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool	FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

//#define MSGCOLOR	0xFFFF0000
//#define MSGBACKG	0xFFFFFF7F
#define MSGBLACK	0xFF000000
#define MSGWHITE	0xFFFFFFFF
#define MSGBLUE		0xFF0000FF

#define MSG_W 80
#define MSG_H 14

#define DEFAULTFILENAME	"filt.txt"


class LumFile {

protected:

	FILE *rfp;

	// to calculate suggested window
	bool rising;
	long lastLumWritten;
	long lastMaximum;

	long windowMediaElements;
	double windowMedia;
	long currWindowSize;

	#define NumOfHeaderRecord 5
	#define RecordLength (3+2)

public:

	~LumFile() {	doneLastRecord(0); }

	bool init ( char * fileName, bool interlaced ) {

		doneLastRecord(0);

		if ((rfp = fopen(fileName, "w")) == NULL) return false;

		writeNextRec (interlaced?1:0);		// flag to indicate interlaced
		writeNextRec (0);					// suggested window size
		writeNextRec (0);					// file size
		writeNextRec (0);					// file size
		writeNextRec (0);					// file size


		rising=false;
		lastLumWritten=lastMaximum=0;
		windowMedia=0.0;
		currWindowSize=windowMediaElements=0;
		//totalWritten=0;

		return true;

	}

	bool writeNext (long value) {

		writeNextRec(value);

		if (rising && ( value < lastLumWritten ) && (currWindowSize>2) ) {
			// is maximum
			windowMedia = ((windowMediaElements*windowMedia)+currWindowSize)/(windowMediaElements+1);
			++windowMediaElements;
			currWindowSize=1;
		} else {
			currWindowSize++;
		}
		rising = (value >= lastLumWritten);
		lastLumWritten = value;

		return true;
	}

	void doneLastRecord(int lastRecord) {
		if ( rfp!=NULL )  { 
			if (lastRecord!=0)  {
				writeRec(1,(int) windowMedia + 1 );
				writeRec(2,(int) (lastRecord & 0xFF) );
				writeRec(3,(int) ((lastRecord>>8) & 0xFF ) );
				writeRec(4,(int) ((lastRecord>>16) & 0xFF) );
			}
			fclose(rfp); 
			rfp = NULL;
		};
	}

private: 


	void writeNextRec (long value) {
		char tmp[10];
		sprintf(tmp, "%03d\n",  value);
		fprintf(rfp, "%s", tmp);
	}

	int writeRec (long pos, long value) {
		char tmp[10];

		if ( rfp == NULL) return -1;
		if ( fseek(rfp,(long)(RecordLength*pos),SEEK_SET) ) return -1;
	
		sprintf(tmp, "%03d\n",  value);
		fprintf(rfp, "%s", tmp);
		return 0;
	}

	void buildFilePath(char * dest, char * path, char * name ) {
		strcpy(dest, path);
		strcat(dest, name);
	}
#undef NumOfHeaderRecord	
#undef RecordLength
};


typedef struct MyFilterData {

	bool				interlaced;
	char				prepareFileName[MAX_PATH];
    unsigned char		*old_data;
	bool				error;
	long				nextFrameToPrepare;
	LumFile				lumFile;
	int					windowSuggested;

} MyFilterData;

void fileString( char * d, char * s) {
	while( *s != 0 ) {
		if (*s=='\\') *d++ = '\\';
		*d=*s;
		d++;s++;
	}
	*d=0;
}

bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	char ss[MAX_PATH];
	fileString( ss, mfd->prepareFileName);
	_snprintf(buf, buflen, "Config(%d, \"%s\")", mfd->interlaced?1:0, ss);

	return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->interlaced = ((!!argv[0].asInt())==1) ? (true) : (false);
	strcpy(mfd->prepareFileName,*argv[1].asString());
}

ScriptFunctionDef func_defs[]={
	{ (ScriptFunctionPtr)ScriptConfig, "Config", "0is" },
	{ NULL },
};

CScriptObject script_obj={
	NULL, func_defs
};

struct FilterDefinition filterDef_tutorial = {

	NULL, NULL, NULL,					// next, prev, module
	"deflickPrepare(A.2.0)",			// name
	"Remove temporal frame luminance variations (flicker).\nPreprocess video producing a file to be used by Deflick Process filter.",
										// desc
	"Ale Malanca/Donald Graft", 		// maker
	NULL,								// private_data
	sizeof(MyFilterData),				// inst_data_size
	MyInitProc,							// initProc
	NULL,								// deinitProc
	RunAnalyzing,						// RunProc
	ParamProc,							// paramProc
	ConfigProc, 						// configProc
	StringProc, 						// stringProc
	StartProc,							// startProc
	EndProc,							// endProc
	&script_obj,						// script_obj
	FssProc,							// fssProc
};



long ParamProc(FilterActivation *fa, const FilterFunctions *ff) {

//	frame counter output frame
	fa->dst.w = MSG_W;
	fa->dst.pitch = (fa->dst.w*4 + 7) & -8;
	fa->dst.h = MSG_H;

    return FILTERPARAM_SWAP_BUFFERS;
}



int StartProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->error = ! ( mfd->lumFile.init(mfd->prepareFileName,mfd->interlaced));
	mfd->nextFrameToPrepare = 0;

	return  mfd->error ? 1 : 0;
}


int EndProc(FilterActivation *fa, const FilterFunctions *ff) {

	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	mfd->lumFile.doneLastRecord(mfd->nextFrameToPrepare);

	if ( mfd->old_data != NULL ) {
		delete[] mfd->old_data;
		mfd->old_data = NULL;
	}



	return 0;
}

///////////////////////////////////////////////////////////////////////////

void	msgOut(int x, int y, const FilterActivation *fa, char * msg, Pixel32 color)	{
		fa->dst.RectFill(x, y,  MSG_W, MSG_H,  MSGWHITE);
		VGL_DrawString(fa, x+5, y+5, msg, color);
}


void	msgOut(const FilterActivation *fa, char * msg, Pixel32 color)	{
		int x = (fa->dst.w - MSG_W) /2;
		int y = (fa->dst.h - MSG_H) /2;
		msgOut(x,y,fa,msg,color);
		//fa->dst.RectFill(x, y,  MSG_W, MSG_H,  MSGBACKG);
		//VGL_DrawString(fa, x+5, y+5, msg, MSGCOLOR);
}


int RunAnalyzing(const FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	Pixel32 *src;
	Pixel32 inPix;

	const PixDim	w = fa->src.w;
	const PixDim	h = fa->src.h;

	FilterStateInfo *pfsi = fa->pfsi;

/*
	const long rscale = (long)(0.299 * 3.0 * 1024.0);
	const long gscale = (long)(0.587 * 3.0 * 1024.0);
	const long bscale = (long)(0.114 * 3.0 * 1024.0);
*/
	// -----------------------
	//   Prepare video
	// -----------------------
	unsigned long lum_sum[] = {0,0};	  

	int hi,wi;
	src = (Pixel32 *)fa->src.data;
	long row_lum;
	hi = h;
	do {	wi = w;
			row_lum = 0;
			do {	
					inPix = *src++;

					row_lum += 	(long) (
							((inPix >> 16) & 0xff) 	
						+	((inPix >>  8) & 0xff) 
						+	(inPix & 0xff)
					);
/*
					row_lum += 	(long) (
							( (rscale * ((inPix>>16)& 0xff)) >> 10 )
						+	( (gscale * ((inPix>> 8)& 0xff)) >> 10 )
						+	( (bscale * (inPix & 0xff)) >>10)
					);
*/
			} while(--wi);

			lum_sum[hi&1] += row_lum;
			src = (Pixel32 *)((char *)src + fa->src.modulo);

	} while(--hi);

	if ( ! mfd->interlaced ) {
		mfd->lumFile.writeNext( (lum_sum[0]+lum_sum[1]) / (w*h) );
	} else {
		mfd->lumFile.writeNext( (2 * lum_sum[0]) / (w*h) );
		mfd->lumFile.writeNext( (2 * lum_sum[1]) / (w*h) );
	}

	if ( pfsi->lCurrentFrame == mfd->nextFrameToPrepare) 
	{
		char frn [20]; 
		sprintf(frn,"%d",pfsi->lCurrentFrame+1);
		msgOut(fa,frn,MSGBLACK);
		mfd->nextFrameToPrepare++;

	} else {

		msgOut(fa,"error!",MSGBLACK);
		mfd->error = true;
	}

	return 0;
}


extern "C" __declspec(dllexport) int VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" __declspec(dllexport) void VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_tutorial;

int VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
	if ((fd_tutorial = ff->addFilter(fm, &filterDef_tutorial, sizeof(FilterDefinition))) == 0)
		return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

	return 0;
}


void VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
	ff->removeFilter(fd_tutorial);
}


int MyInitProc(FilterActivation *fa, const FilterFunctions *ff) {

	MyFilterData *mfd = (MyFilterData *)fa->filter_data;


	mfd->interlaced = false;
	mfd->error = false;
	mfd->nextFrameToPrepare = 0;
	mfd->windowSuggested = 0;

	char prog[MAX_PATH];
	LPTSTR ptr;
	char	folderPath[MAX_PATH];
 
	GetModuleFileName(NULL, prog, MAX_PATH);
	GetFullPathName(prog, MAX_PATH, folderPath, &ptr);
	*ptr = 0; // ???

	strcat(folderPath, "plugins\\tmp\\");

	if (fopen(folderPath,"r") == NULL) {
		// folder do not exist
		mkdir(folderPath);
	}

	strcpy(mfd->prepareFileName, folderPath);
	strcat(mfd->prepareFileName, DEFAULTFILENAME);

	return 0;
}


OPENFILENAME prepOpFlNm(char * filename, HWND hdlg) {

	char filter[] = "All Files(*,*)\0*.*\0Text Files(*.txt)\0*.txt\0\0";

	OPENFILENAME ofname;
	ZeroMemory(&ofname,sizeof(ofname));

	ofname.hwndOwner = hdlg; 
	ofname.hInstance = NULL; 
	ofname.lStructSize = sizeof(ofname);
	ofname.lpstrFilter =  filter;
	ofname.nFilterIndex = 2; 
	ofname.lpstrDefExt = "txt";
	ofname.Flags= OFN_HIDEREADONLY|OFN_FILEMUSTEXIST|OFN_EXPLORER|OFN_CREATEPROMPT;
	ofname.nMaxFile = sizeof(char)*MAX_PATH;
	ofname.lpstrFile = filename;
	return ofname;
}



BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
	MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

	char filter[] = "All Files(*,*)\0*.*\0Text Files(*.txt)\0*.txt\0\0";


	switch(msg) {

		case WM_INITDIALOG:

			SetWindowLong(hdlg, DWL_USER, lParam);
			mfd = (MyFilterData *)lParam;

			CheckDlgButton(hdlg, IDC_INTERLACED, mfd->interlaced ? BST_CHECKED : BST_UNCHECKED);
			SetDlgItemText(hdlg, IDC_PREP_FILENAME , mfd->prepareFileName);

			return TRUE;

		case WM_COMMAND:
			switch(LOWORD(wParam))
			{
				case IDOK:
					EndDialog(hdlg, 0);
					return TRUE;
					break;

				case IDHELP: {

					char prog[256];
					char path[256];
					LPTSTR ptr;

					GetModuleFileName(NULL, prog, 255);
					GetFullPathName(prog, 255, path, &ptr);
					*ptr = 0;
					strcat(path, "plugins\\DeflickPrep.htm");
					ShellExecute(hdlg, "open", path, NULL, NULL, SW_SHOWNORMAL);
					return TRUE;
				}
				case IDCANCEL:
					EndDialog(hdlg, 1);
					return TRUE;

				case IDC_PREP_FILENAME:
					GetDlgItemText(hdlg, IDC_PREP_FILENAME, mfd->prepareFileName,255);
					break;

				case IDC_PREPFILE_BUTTON: {

					char filename[MAX_PATH];
					strcpy(filename,mfd->prepareFileName);
					OPENFILENAME ofname = prepOpFlNm(filename,hdlg);

					if ( GetSaveFileName(&ofname) != 0 ) {
						// ok !
						SetDlgItemText(hdlg, IDC_PREP_FILENAME ,filename);
						strcpy(mfd->prepareFileName,filename);
					}
					break;
				}	
				case IDC_INTERLACED: 
					mfd->interlaced = ! mfd->interlaced;
					break;
			}

			break;
	}

	return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd)
{
	MyFilterData *mfd = (MyFilterData *) fa->filter_data;
	MyFilterData mfd_old = *mfd;

    extern void Doit(void);

	if (DialogBoxParam(fa->filter->module->hInstModule,
			MAKEINTRESOURCE(IDD_FILTER), hwnd,
			(DLGPROC) ConfigDlgProc, (LPARAM) mfd))
	{
		*mfd = mfd_old;
		return TRUE;
	} 

	return FALSE;
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

		sprintf(str, " ( inter %s, file %s )",
			mfd->interlaced ? "yes" : "no", 
			mfd->prepareFileName
		);

}

