/*
    Colorize Filter for VirtualDub -- Change a video to a monochrome
	video with selectable base color. Copyright (C) 1999-2000 Donald A. Graft

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

	The author can be contacted at:
	Donald Graft
	neuron2@home.com.
*/

#include <windows.h>
#include <commctrl.h>
#include <stdio.h>
#include <crtdbg.h>
#include <math.h>
#include <stdlib.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

///////////////////////////////////////////////////////////////////////////

int RunProc(const FilterActivation *fa, const FilterFunctions *ff);
int StartProc(FilterActivation *fa, const FilterFunctions *ff);
int EndProc(FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int InitProc(FilterActivation *fa, const FilterFunctions *ff);
int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);
void RGBtoHSL(COLORREF rgb, double *H, double *S, double *L);
void HLStoRGB(double H, double L, double S, int *r, int *g, int *b);

///////////////////////////////////////////////////////////////////////////

typedef struct MyFilterData {
	IFilterPreview		*ifp;
	CHOOSECOLOR			cc;
	COLORREF			crCustColors[16];
	double				H, S, L;
	int					h, s, l;
	unsigned long		rx, gx, bx;
	HBRUSH              hbrColor;
} MyFilterData;

bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	_snprintf(buf, buflen, "Config(%d, %d, %d)",
		mfd->h,
		mfd->s,
		mfd->l);

	return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->h		= argv[0].asInt();
	mfd->H		= mfd->h / 240.0;
	mfd->s		= argv[1].asInt();
	mfd->S		= mfd->s / 240.0;
	mfd->l		= argv[2].asInt();
	mfd->L		= mfd->l / 240.0;
}

ScriptFunctionDef func_defs[]={
	{ (ScriptFunctionPtr)ScriptConfig, "Config", "0iii" },
	{ NULL },
};

CScriptObject script_obj={
	NULL, func_defs
};

struct FilterDefinition filterDef_tutorial = {

	NULL, NULL, NULL,		// next, prev, module
	"colorize (1.1 beta 1)",	// name
	"Map a video to selectable monochrome color.",
							// desc
	"Donald Graft", 		// maker
	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size

	InitProc,				// initProc
	NULL,					// deinitProc
	RunProc,				// runProc
	NULL,					// paramProc
	ConfigProc, 			// configProc
	StringProc, 			// stringProc
	StartProc,				// startProc
	EndProc,				// endProc

	&script_obj,			// script_obj
	FssProc,				// fssProc

};

int StartProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->rx = (unsigned long) (0.299 * 65535);
	mfd->gx = (unsigned long) (0.587 * 65535);
	mfd->bx = (unsigned long) (0.114 * 65535);

	return 0;
}

int EndProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	return 0;
}

///////////////////////////////////////////////////////////////////////////

int RunProc(const FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	const PixDim	width = fa->src.w;
	const PixDim	height = fa->src.h;
	Pixel32 *src, *dst;
	int x, y;
	int r, g, b;
	double L;

	src = fa->src.data;
	dst = fa->dst.data;
	for (y = 0; y < height; y++)
	{
		for (x = 0; x < width; x++)
		{
			r = ((src[x] & 0xff0000) >> 16);
			g = ((src[x] & 0xff00) >> 8);
			b = (src[x] & 0xff);
			L = (mfd->rx * r + mfd->gx * g + mfd->bx * b) >> 16;
			L = (L * mfd->L) / 255.0;
			HLStoRGB(mfd->H, L, mfd->S, &r, &g, &b);
			dst[x] = (r << 16) | (g << 8) | (b);			 
		}
		src	= (Pixel *)((char *)src + fa->src.pitch);
		dst	= (Pixel *)((char *)dst + fa->dst.pitch);
	}
	return 0;
}

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_tutorial;

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
	if (!(fd_tutorial = ff->addFilter(fm, &filterDef_tutorial, sizeof(FilterDefinition))))
		return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

	return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
	ff->removeFilter(fd_tutorial);
}

int InitProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	int r, g, b;

	mfd->l = 150;
	mfd->L = 150 / 240.0;
	mfd->s = 150;
	mfd->S = 150 / 240.0;
	HLStoRGB(mfd->H, mfd->L, mfd->S, &r, &g, &b);
	mfd->cc.rgbResult = RGB(r, g, b);
	return 0;
}

void ShowColor(HWND hdlg, MyFilterData *mfd, COLORREF color)
{
		DeleteObject(mfd->hbrColor);
		mfd->hbrColor = CreateSolidBrush(color);
		RedrawWindow(GetDlgItem(hdlg, IDC_COLOR), NULL, NULL, RDW_ERASE|RDW_INVALIDATE|RDW_UPDATENOW);
}

BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
	MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

	switch(msg) {
		HWND hWnd;
		int r, g, b;

		case WM_PAINT:
			HLStoRGB(mfd->H, 150 / 240.0, mfd->S, &r, &g, &b);
			ShowColor(hdlg, mfd, RGB(r, g, b));
			return 0;

		case WM_INITDIALOG:
			SetWindowLong(hdlg, DWL_USER, lParam);
			mfd = (MyFilterData *)lParam;

			hWnd = GetDlgItem(hdlg, IDC_HUE);
			SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 240));
			SendMessage(hWnd, TBM_SETTICFREQ, 10 , 0);
			SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->h);
			SetDlgItemInt(hdlg, IDC_EHUE, mfd->h, FALSE);
			hWnd = GetDlgItem(hdlg, IDC_SAT);
			SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 240));
			SendMessage(hWnd, TBM_SETTICFREQ, 10 , 0);
			SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->s);
			SetDlgItemInt(hdlg, IDC_ESAT, mfd->s, FALSE);
			hWnd = GetDlgItem(hdlg, IDC_LUM);
			SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 239));
			SendMessage(hWnd, TBM_SETTICFREQ, 10 , 0);
			SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->l);
			SetDlgItemInt(hdlg, IDC_ELUM, mfd->l, FALSE);
			HLStoRGB(mfd->H, 150 / 240.0, mfd->S, &r, &g, &b);
			mfd->hbrColor = CreateSolidBrush(RGB(r, g, b));
			ShowColor(hdlg, mfd, RGB(r, g, b));

			mfd->ifp->InitButton(GetDlgItem(hdlg, IDPREVIEW));
			return TRUE;

		case WM_HSCROLL:
			{
				int hue, sat, lum;
				HWND hwnd = GetDlgItem(hdlg, IDC_HUE);
				hue = SendMessage(hwnd, TBM_GETPOS, 0, 0);
				SetDlgItemInt(hdlg, IDC_EHUE, mfd->h, FALSE);
				hwnd = GetDlgItem(hdlg, IDC_SAT);
				sat = SendMessage(hwnd, TBM_GETPOS, 0, 0);
				SetDlgItemInt(hdlg, IDC_ESAT, mfd->s, FALSE);
				hwnd = GetDlgItem(hdlg, IDC_LUM);
				lum = SendMessage(hwnd, TBM_GETPOS, 0, 0);
				SetDlgItemInt(hdlg, IDC_ELUM, mfd->l, FALSE);
				if (hue != mfd->h || sat != mfd->s || lum != mfd->l)
				{
					mfd->h = hue;
					mfd->H = hue / 240.0;
					mfd->s = sat;
					mfd->S = sat / 240.0;
					mfd->l = lum;
					mfd->L = lum / 240.0;
					HLStoRGB(mfd->H, mfd->L, mfd->S, &r, &g, &b);
					mfd->cc.rgbResult = RGB(r, g, b);
					HLStoRGB(mfd->H, 150 / 240.0, mfd->S, &r, &g, &b);
					ShowColor(hdlg, mfd, RGB(r, g, b));
					mfd->ifp->RedoFrame();
				}
				break;
			}

		case WM_COMMAND:
			switch(LOWORD(wParam)) {
			case IDPREVIEW:
				mfd->ifp->Toggle(hdlg);
				break;
			case IDOK:
				EndDialog(hdlg, 0);
				return TRUE;
 			case IDHELP:
				{
				char prog[256];
				char path[256];
				LPTSTR ptr;

				GetModuleFileName(NULL, prog, 255);
				GetFullPathName(prog, 255, path, &ptr);
				*ptr = 0;
				strcat(path, "plugins\\Colorize.html");
				ShellExecute(hdlg, "open", path, NULL, NULL, SW_SHOWNORMAL);
				return TRUE;
				}
			case IDCANCEL:
				EndDialog(hdlg, 1);
				return TRUE;
			case IDC_HUEPLUS:
				if (mfd->h < 239)
				{
					mfd->h += 1;
					mfd->H = mfd->h / 240.0;
					SetDlgItemInt(hdlg, IDC_EHUE, mfd->h, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_HUE), TBM_SETPOS, (WPARAM)TRUE, mfd->h);
					HLStoRGB(mfd->H, mfd->L, mfd->S, &r, &g, &b);
					mfd->cc.rgbResult = RGB(r, g, b);
					HLStoRGB(mfd->H, 150 / 240.0, mfd->S, &r, &g, &b);
					ShowColor(hdlg, mfd, RGB(r, g, b));
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_HUEMINUS:
				if (mfd->h > 0)
				{
					mfd->h -= 1;
					mfd->H = mfd->h / 240.0;
					SetDlgItemInt(hdlg, IDC_EHUE, mfd->h, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_HUE), TBM_SETPOS, (WPARAM)TRUE, mfd->h);
					HLStoRGB(mfd->H, mfd->L, mfd->S, &r, &g, &b);
					mfd->cc.rgbResult = RGB(r, g, b);
					HLStoRGB(mfd->H, 150 / 240.0, mfd->S, &r, &g, &b);
					ShowColor(hdlg, mfd, RGB(r, g, b));
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_SATPLUS:
				if (mfd->s < 239)
				{
					mfd->s += 1;
					mfd->S = mfd->s / 240.0;
					SetDlgItemInt(hdlg, IDC_ESAT, mfd->s, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_SAT), TBM_SETPOS, (WPARAM)TRUE, mfd->s);
					HLStoRGB(mfd->H, mfd->L, mfd->S, &r, &g, &b);
					mfd->cc.rgbResult = RGB(r, g, b);
					HLStoRGB(mfd->H, 150 / 240.0, mfd->S, &r, &g, &b);
					ShowColor(hdlg, mfd, RGB(r, g, b));
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_SATMINUS:
				if (mfd->s > 0)
				{
					mfd->s -= 1;
					mfd->S = mfd->s / 240.0;
					SetDlgItemInt(hdlg, IDC_ESAT, mfd->s, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_SAT), TBM_SETPOS, (WPARAM)TRUE, mfd->s);
					HLStoRGB(mfd->H, mfd->L, mfd->S, &r, &g, &b);
					mfd->cc.rgbResult = RGB(r, g, b);
					HLStoRGB(mfd->H, 150 / 240.0, mfd->S, &r, &g, &b);
					ShowColor(hdlg, mfd, RGB(r, g, b));
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_LUMPLUS:
				if (mfd->l < 239)
				{
					mfd->l += 1;
					mfd->L = mfd->l / 240.0;
					SetDlgItemInt(hdlg, IDC_ELUM, mfd->l, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_LUM), TBM_SETPOS, (WPARAM)TRUE, mfd->l);
					HLStoRGB(mfd->H, mfd->L, mfd->S, &r, &g, &b);
					mfd->cc.rgbResult = RGB(r, g, b);
					HLStoRGB(mfd->H, 150 / 240.0, mfd->S, &r, &g, &b);
					ShowColor(hdlg, mfd, RGB(r, g, b));
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_LUMMINUS:
				if (mfd->l > 0)
				{
					mfd->l -= 1;
					mfd->L = mfd->l / 240.0;
					SetDlgItemInt(hdlg, IDC_ELUM, mfd->l, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_LUM), TBM_SETPOS, (WPARAM)TRUE, mfd->l);
					HLStoRGB(mfd->H, mfd->L, mfd->S, &r, &g, &b);
					mfd->cc.rgbResult = RGB(r, g, b);
					HLStoRGB(mfd->H, 150 / 240.0, mfd->S, &r, &g, &b);
					ShowColor(hdlg, mfd, RGB(r, g, b));
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_PICKCOLOR:
				{
				mfd->cc.lStructSize = sizeof(CHOOSECOLOR);
				mfd->cc.hwndOwner = NULL;
				mfd->cc.hInstance = NULL;
				mfd->cc.lpCustColors = mfd->crCustColors;
				mfd->cc.Flags = CC_RGBINIT | CC_FULLOPEN;
				mfd->cc.lCustData = 0L;
				mfd->cc.lpfnHook = NULL;
				mfd->cc.lpTemplateName = NULL;
				ChooseColor(&mfd->cc);
				RGBtoHSL(mfd->cc.rgbResult, &mfd->H, &mfd->S, &mfd->L);
				mfd->h = (int) (mfd->H * 240);
				hWnd = GetDlgItem(hdlg, IDC_HUE);
				SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->h);
				SetDlgItemInt(hdlg, IDC_EHUE, mfd->h, FALSE);
				mfd->s = (int) (mfd->S * 240);
				hWnd = GetDlgItem(hdlg, IDC_SAT);
				SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->s);
				SetDlgItemInt(hdlg, IDC_ESAT, mfd->s, FALSE);
				mfd->l = (int) (mfd->L * 240);
				hWnd = GetDlgItem(hdlg, IDC_LUM);
				SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->l);
				SetDlgItemInt(hdlg, IDC_ELUM, mfd->l, FALSE);
				HLStoRGB(mfd->H, 150 / 240.0, mfd->S, &r, &g, &b);
				ShowColor(hdlg, mfd, RGB(r, g, b));
				mfd->ifp->RedoFrame();
				return TRUE;
				}
			}
			break;
		case WM_CTLCOLORSTATIC:
			if (GetWindowLong((HWND)lParam, GWL_ID) == IDC_COLOR)
				return (BOOL)mfd->hbrColor;
			break;
	}

	return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	MyFilterData mfd_old = *mfd;
	int ret;

	mfd->ifp = fa->ifp;
	mfd->hbrColor = NULL;

	ret = DialogBoxParam(fa->filter->module->hInstModule, MAKEINTRESOURCE(IDD_FILTER),
						 hwnd, ConfigDlgProc, (LPARAM) mfd);
	if (mfd->hbrColor)
	{
		DeleteObject(mfd->hbrColor);
		mfd->hbrColor = NULL;
	}
	if (ret)
	{
		*mfd = mfd_old;
		ret = TRUE;
	}
    else
	{
		ret = FALSE;
	}
	return(ret);
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	sprintf(str, " (hue %d, sat %d, lum %d)", mfd->h, mfd->s, mfd->l);
}

void RGBtoHSL(COLORREF rgb, double *H, double *S, double *L)
{
   double delta;
   double r = (double)GetRValue(rgb)/255;
   double g = (double)GetGValue(rgb)/255;
   double b = (double)GetBValue(rgb)/255;
   double cmax = max(r,max(g,b));
   double cmin = min(r,min(g,b));
   *L=(cmax+cmin)/2.0;
   if(cmax==cmin) {
      *S = 0;
      *H = 0; // it's really undefined
   } else {
      if(*L < 0.5) 
         *S = (cmax-cmin)/(cmax+cmin);
      else
         *S = (cmax-cmin)/(2.0-cmax-cmin);
      delta = cmax - cmin;
      if(r==cmax)
          *H = (g-b)/delta;
      else if(g==cmax)
          *H = 2.0 +(b-r)/delta;
      else
          *H=4.0+(r-g)/delta;
      *H /= 6.0;
      if(*H < 0.0)
          *H += 1;
   }
}

int HuetoRGB(double m1, double m2, double h)
{
   if( h < 0 ) h += 1.0;
   if( h > 1 ) h -= 1.0;
   if( 6.0*h < 1 )
      return ((int) (255 * (m1+(m2-m1)*h*6.0)));
   if( 2.0*h < 1 )
      return ((int) (255 * m2));
   if( 3.0*h < 2.0 )
      return ((int) (255 * (m1+(m2-m1)*((2.0/3.0)-h)*6.0)));
   return ((int) (255 * m1));
}

void HLStoRGB(double H, double L, double S, int *r, int *g, int *b)
{
  double m1, m2;

  if(S==0) {
     *r = *g = *b = (int) (255 * L);
  } else {
     if(L <=0.5)
        m2 = L*(1.0+S);
     else
        m2 = L+S-L*S;
     m1 = 2.0*L-m2;
     *r = HuetoRGB(m1,m2,H+1.0/3.0);
     *g = HuetoRGB(m1,m2,H);
     *b = HuetoRGB(m1,m2,H-1.0/3.0);
  }
}

