/*
    Pseudocolor Filter for VirtualDub -- Copyright (C) 2002
	Donald A. Graft

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

	The author can be contacted at:
	Donald Graft
	neuron2@attbi.com.
*/

#include <windows.h>
#include <commctrl.h>
#include <stdio.h>
#include <crtdbg.h>
#include <math.h>
#include <stdlib.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

///////////////////////////////////////////////////////////////////////////

int RunProc(const FilterActivation *fa, const FilterFunctions *ff);
int StartProc(FilterActivation *fa, const FilterFunctions *ff);
int EndProc(FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int InitProc(FilterActivation *fa, const FilterFunctions *ff);
int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

#define MAX_COLORS 1000

typedef struct MyFilterData
{
	IFilterPreview		*ifp;
	unsigned int		palette[MAX_COLORS];
	int					palette_size;
	char				filename[1024];
} MyFilterData;

void Load(MyFilterData *mfd)
{
	FILE *fp;
	char path[1024];
	char line[1024];
	int r, g, b;
	int i;

	strcpy(path, mfd->filename);
 	if ((fp = fopen(path, "r")) == NULL)
	{
		return;
	}
	i = 0;
	while (i < MAX_COLORS && fgets(line, 255, fp) != NULL)
	{
		sscanf(line, "%d %d %d", &r, &g, &b);
		mfd->palette[i++] = (r << 16) + (g << 8) + b;
	}
	mfd->palette_size = i;
	fclose(fp);
	return;
}

bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
    char save[255];
	int i, j;

	i = j = 0;
	while(mfd->filename[i])
	{
		if (mfd->filename[i] == '\\')
		{
			save[j++] = '\\';
			save[j++] = '\\';
			i++;
		}
		else
		{
			save[j++] = mfd->filename[i++];
		}
	}
	save[j] = 0;
	_snprintf(buf, buflen, "Config(\"%s\")", save);

	return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc)
{
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	strcpy(mfd->filename, *argv[0].asString());
}

ScriptFunctionDef func_defs[]=
{
	{ (ScriptFunctionPtr)ScriptConfig, "Config", "0s" },
	{ NULL },
};

CScriptObject script_obj=
{
	NULL, func_defs
};

struct FilterDefinition filterDef_tutorial =
{

	NULL, NULL, NULL,		// next, prev, module
	"pseudocolor (1.0)",	// name
	"Map a video to a pseudocolor palette.",
							// desc
	"Donald Graft", 		// maker
	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size

	InitProc,				// initProc
	NULL,					// deinitProc
	RunProc,				// runProc
	NULL,					// paramProc
	ConfigProc, 			// configProc
	StringProc, 			// stringProc
	StartProc,				// startProc
	EndProc,				// endProc

	&script_obj,			// script_obj
	FssProc,				// fssProc
};

int StartProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	Load(mfd);
	return 0;
}

int EndProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	return 0;
}

///////////////////////////////////////////////////////////////////////////

int RunProc(const FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	const PixDim	width = fa->src.w;
	const PixDim	height = fa->src.h;
	Pixel32 *src, *dst;
	int x, y;
	int r, g, b;

	src = fa->src.data;
	dst = fa->dst.data;
	for (y = 0; y < height; y++)
	{
		for (x = 0; x < width; x++)
		{
			r = ((src[x] & 0xff0000) >> 16);
			g = ((src[x] & 0xff00) >> 8);
			b = (src[x] & 0xff);
			dst[x] = mfd->palette[(((3*r+5*g+2*b)/10)*mfd->palette_size)/256];
		}
		src	= (Pixel *)((char *)src + fa->src.pitch);
		dst	= (Pixel *)((char *)dst + fa->dst.pitch);
	}
	return 0;
}

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm,
											const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm,
																		   const FilterFunctions *ff);

static FilterDefinition *fd_tutorial;

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff,
															  int& vdfd_ver, int& vdfd_compat)
{
	if (!(fd_tutorial = ff->addFilter(fm, &filterDef_tutorial, sizeof(FilterDefinition))))
		return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

	return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff)
{
	ff->removeFilter(fd_tutorial);
}

int InitProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	return 0;
}

BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
	MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

	switch(msg)
	{
		case WM_INITDIALOG:
			SetWindowLong(hdlg, DWL_USER, lParam);
			mfd = (MyFilterData *)lParam;

			SetWindowText(GetDlgItem(hdlg, IDC_FILENAME), mfd->filename);
			mfd->ifp->InitButton(GetDlgItem(hdlg, IDPREVIEW));
			return TRUE;

		case WM_COMMAND:
			switch(LOWORD(wParam))
			{
			case IDPREVIEW:
				/* Temporarily disabled by making the button invisible.
				   There are problems with the Browse window coming up
				   behind the filter configuration window when preview is
				   enabled. Not yet solved, so preview is disabled. */
				mfd->ifp->Toggle(hdlg);
				break;
			case IDOK:
				EndDialog(hdlg, 0);
				return TRUE;
 			case IDHELP:
				{
				char prog[256];
				char path[256];
				LPTSTR ptr;

				GetModuleFileName(NULL, prog, 255);
				GetFullPathName(prog, 255, path, &ptr);
				*ptr = 0;
				strcat(path, "plugins\\Pseudocolor.html");
				ShellExecute(hdlg, "open", path, NULL, NULL, SW_SHOWNORMAL);
				return TRUE;
				}
			case IDCANCEL:
				EndDialog(hdlg, 1);
				return TRUE;
			case IDC_BROWSE:
			{
				OPENFILENAME ofn;
				mfd->filename[0] = NULL;
				ofn.lStructSize = sizeof(OPENFILENAME);
				ofn.hwndOwner = hdlg;
				ofn.hInstance = NULL;
				ofn.lpTemplateName = NULL;
				ofn.lpstrFilter = "All Files\0*.*\0\0";
				ofn.lpstrCustomFilter = NULL;
				ofn.nMaxCustFilter = 0;
				ofn.nFilterIndex = 1;
				ofn.lpstrFile = mfd->filename;
				ofn.nMaxFile = 1024;
				ofn.lpstrFileTitle = NULL;
				ofn.nMaxFileTitle = 0;
				ofn.lpstrInitialDir = NULL;
				ofn.lpstrTitle = "Choose Logo Bitmap File";
				ofn.Flags = OFN_CREATEPROMPT;
				ofn.nFileOffset = 0;
				ofn.nFileExtension = 0;
				ofn.lpstrDefExt = NULL;
				ofn.lCustData = 0;
				ofn.lpfnHook = NULL;
				GetOpenFileName(&ofn);
				if (mfd->filename[0] != 0)
				{
					SetWindowText(GetDlgItem(hdlg, IDC_FILENAME), mfd->filename);
					Load(mfd);
					mfd->ifp->UndoSystem();
					mfd->ifp->RedoSystem();
				}
				break;
			}
		}
	}

	return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	MyFilterData mfd_old = *mfd;
	int ret;

	mfd->ifp = fa->ifp;

	ret = DialogBoxParam(fa->filter->module->hInstModule, MAKEINTRESOURCE(IDD_FILTER),
						 hwnd, ConfigDlgProc, (LPARAM) mfd);
	if (ret)
	{
		*mfd = mfd_old;
		ret = TRUE;
	}
    else
	{
		ret = FALSE;
	}
	return(ret);
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

}
