/*//////////////////////////////////////////////////////////////////////////

	flaXen's Temporal Stabilizer Filter for VirtualDub
	(c) Copyright 2000 Dan Flower (flaXen)

	Reduce or eliminate chromatic morie	paterns, abrupt chromatic
	deviations, and noise by use of a temporal accumulation buffer.
	(Works on any video source, but was built for VHS)

*//////////////////////////////////////////////////////////////////////////

#include <windows.h>
#include <stdio.h>
#include <math.h>

#include "resource.h"
#include "filter.h"
#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#define	FORMAT_YUV	0
#define	FORMAT_YIQ	1
#define PI			3.14159265358979

typedef struct MyFilterData {
	Pixel32	*TmpBuff;
	long	*ytab, *itab, *qtab;
	long	*rtab, *gtab, *btab;
	long	*cytf, *Div3Tab, *sdi;
	int		*stBuffY, *stBuffI, *stBuffQ, *stCount, *stError;
	unsigned char	*stlBuffY, *stlBuffI, *stlBuffQ, *stPrevY;
	unsigned char	*stpBuffY, *stpBuffI, *stpBuffQ;
	long	mmx;
	int		stChromaThresh, stLumaThresh, stTempError;
	int		stTempBias, stPixelLock, stYIQin, stYIQout;
	int		stShowMotion, stShowTemp, stShowLock;
	int		stYIQ, stSceneDet, stAffectLC, stAffectCO;
	int		dnEnable, dnRadius, dnAmt, stHMSD, stHMSDI;
	int		acRange, acAmp, stScales;
} MyFilterData;

///////////////////////////////////////////////////////////////////////////

int		fxTempRunProc(const FilterActivation *fa, const FilterFunctions *ff);
void	fxTemp_Temporal(Pixel32 *, Pixel32 *, long, long, long, MyFilterData *);
void	fxTemp_TemporalTest(Pixel32 *, Pixel32 *, long, long, long, MyFilterData *);
void	fxTemp_RGB2YIQ(Pixel32 *, Pixel32 *, long, long, long, MyFilterData *);
void	fxTemp_YIQ2RGB(Pixel32 *, Pixel32 *, long, long, long, MyFilterData *);
void	fxTemp_DeNoise(Pixel32 *, Pixel32 *, long, long, long, int, int);
long	fxTemp_DeNoiseSingle(Pixel32 *, long, int, int);
void	fxColorSpace_GenTabs(long, MyFilterData *);
int		fxTempStartProc(FilterActivation *fa, const FilterFunctions *ff);
int		fxTempEndProc(FilterActivation *fa, const FilterFunctions *ff); 
int		fxTempInitProc(FilterActivation *fa, const FilterFunctions *ff);
int		fxTempConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void	fxTempStringProc(const FilterActivation *fa, const FilterFunctions *ff, char *buf);
void	fxTempScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool	fxTempFssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

ScriptFunctionDef fxTemp_func_defs[]={
    { (ScriptFunctionPtr)fxTempScriptConfig, "Config", "0iiiiiiiiiiiiiii" },
    { NULL },
};

CScriptObject fxTempobj={
    NULL, fxTemp_func_defs
};

struct FilterDefinition filterDef_fxTemp = {
	NULL, NULL, NULL,		// next, prev, module

	"Stabilizer v2.1",			// name
	"flaXen's Advanced Temporal Stabilizer v2.1b",	// desc
	"flaXen",				// maker

	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size
	fxTempInitProc,			// initProc
	NULL,					// deinitProc
	fxTempRunProc,			// runProc
	NULL,					// paramProc
	fxTempConfigProc,		// configProc
	fxTempStringProc,		// stringProc
	fxTempStartProc,			// startProc
	fxTempEndProc,			// endProc
	&fxTempobj,				// script_obj
	fxTempFssProc,			// fssProc
};

///////////////////////////////////////////////////////////////////////////

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_fxTemp;

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
	if (!(fd_fxTemp = ff->addFilter(fm, &filterDef_fxTemp, sizeof(FilterDefinition))))
		return 1;

	vdfd_ver    = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;
	return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
	ff->removeFilter(fd_fxTemp);
}

///////////////////////////////////////////////////////////////////////////

int	fxTempRunProc(const FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData	*mfd = (MyFilterData *)fa->filter_data;
	Pixel32		*isrc, *idst, *itmp;
	const long	swid = fa->src.w, shei = fa->src.h;		// So I'm lazy...
	const long	spit = fa->src.pitch >> 2;

	isrc = (Pixel32 *)fa->src.data;
	idst = (Pixel32 *)fa->dst.data;
	itmp = (Pixel32 *)mfd->TmpBuff;

	if (mfd->stYIQ == BST_CHECKED)
		fxColorSpace_GenTabs(FORMAT_YIQ, mfd);
	else
		fxColorSpace_GenTabs(FORMAT_YUV, mfd);

	if (mfd->stShowMotion == BST_CHECKED || mfd->stShowTemp == BST_CHECKED || mfd->stShowLock == BST_CHECKED) {
		if (mfd->stYIQin == BST_UNCHECKED) {
			fxTemp_RGB2YIQ(isrc, itmp, swid, shei, spit, mfd);
			fxTemp_TemporalTest(itmp, idst, swid, shei, spit, mfd);
		} else {
			fxTemp_TemporalTest(isrc, idst, swid, shei, spit, mfd);
		}
	} else {
		if (mfd->stYIQin == BST_CHECKED) {
			// Ycc input
			if (mfd->stYIQout == BST_CHECKED) {
				// Ycc input and Ycc output
				fxTemp_Temporal(isrc, idst, swid, shei, spit, mfd);
			} else {
				// Ycc input and RGB output
				fxTemp_Temporal(isrc, itmp, swid, shei, spit, mfd);
				fxTemp_YIQ2RGB(itmp, idst, swid, shei, spit, mfd);
			}
		} else {
			// RGB input
			if (mfd->stYIQout == BST_CHECKED) {
				// RGB input and Ycc output
				fxTemp_RGB2YIQ(isrc, itmp, swid, shei, spit, mfd);
				fxTemp_Temporal(itmp, idst, swid, shei, spit, mfd);
			} else {
				// RGB input and RGB output
				fxTemp_RGB2YIQ(isrc, itmp, swid, shei, spit, mfd);
				fxTemp_Temporal(itmp, itmp, swid, shei, spit, mfd);
				fxTemp_YIQ2RGB(itmp, idst, swid, shei, spit, mfd);
			}
		}
	}

	return 0;
}

///////////////////////////////////////////////////////////////////////////

void	fxTemp_Temporal(Pixel32 *isrc, Pixel32 *idst, long swid, long shei, long spit, MyFilterData *mfd) {
	Pixel32	*src, *dst;
	long	x, y, c, t, pl = mfd->stPixelLock;
	long	cr, cg, cb, tcy, cy, ci, cq, oy, Rad, flag;
	long	sci[256], sc, sd, d;
	int		*byi, *bii, *bqi, *bci, *bei;
	int		*by = (int *)mfd->stBuffY;
	int		*bu = (int *)mfd->stBuffI;
	int		*bv = (int *)mfd->stBuffQ;
	int		*bc = (int *)mfd->stCount;
	int		*be = (int *)mfd->stError;
	unsigned char	*lbyi, *lbii, *lbqi;
	unsigned char	*pbyi, *pbii, *pbqi, *pyi;
	unsigned char	*lby = (unsigned char *)mfd->stlBuffY;
	unsigned char	*lbu = (unsigned char *)mfd->stlBuffI;
	unsigned char	*lbv = (unsigned char *)mfd->stlBuffQ;
	unsigned char	*pby = (unsigned char *)mfd->stpBuffY;
	unsigned char	*pbu = (unsigned char *)mfd->stpBuffI;
	unsigned char	*pbv = (unsigned char *)mfd->stpBuffQ;
	unsigned char	*py = (unsigned char *)mfd->stPrevY;
	const long	lThresh = mfd->stLumaThresh;
	const long	cThresh = mfd->stChromaThresh;
	const long	ssize = spit * shei * 4;

	if (pl == 0) pl = 2147483647;
	Rad = mfd->dnRadius; sd = 0;
	for (d = 0; d < 256; d++)
		sci[d] = d;

	if (mfd->stSceneDet > 0) {
		t = 0; sc = 0;
		for (y = 0; y < shei; y++) {
			pyi = py + y * swid;;
			src = isrc + y * spit;
			for (x = 0; x < swid; x++) {
				cy = (*src++ >> 16) & 255;
				sc += cy;
				cr = abs(cy - *pyi);
				*pyi++ = (unsigned char)cy;
				if (cr > 4) t++;
			}
		}

		sc /= (swid * shei);
		for (d = 0; d < mfd->acRange; d++) {
			sci[d] = d + (long)(sin(PI / (double)mfd->acRange / 2.0 * (double)d) * (double)(mfd->acRange - d) * 1.0 / 255.0 * (double)(255 - sc) * (0.01 * (double)mfd->acAmp) + 0.5);
			if (sci[d] < 0)
				sci[d] = 0;
			else if (sci[d] > 255)
				sci[d] = 255;
		}

		sd = (long)(255.0 / (float)(swid * shei) * (float)t);
		if ((long)(100.0 / (float)(swid * shei) * (float)t) > mfd->stSceneDet) {
			for (y = 0; y < shei; y++) {
				x = y * swid;
				byi = by + x; bii = bu + x;	bqi = bv + x;
				bci = bc + x; bei = be + x;
				lbyi = lby + x; lbii = lbu + x; lbqi = lbv + x;
				pbyi = pby + x; pbii = pbu + x; pbqi = pbv + x;
				src = isrc + y * spit;
				for (x = 0; x < swid; x++) {
					c = *src++;
					cy = (c >> 16) & 255; ci = (c >> 8) & 255; cq = c & 255;
					*byi++ = cy; *bii++ = ci; *bqi++ = cq; *bci++ = 1; *bei++ = 0;
					*lbyi++ = (unsigned char)cy; *lbii++ = (unsigned char)ci; *lbqi++ = (unsigned char)cq;
					*pbyi++ = (unsigned char)cy; *pbii++ = (unsigned char)ci; *pbqi++ = (unsigned char)cq;
				}
			}
			if (mfd->dnEnable == BST_CHECKED) {
				fxTemp_DeNoise(isrc, idst, swid, shei, spit, mfd->dnAmt, mfd->dnRadius);
			} else
				memcpy(idst, isrc, ssize);
			return;
		}
	}

	// Vertical loop
	for (y = Rad; y < shei - Rad; y++) {
		// Prepare all sorts of pointers...
		// b?i pointers reference the temporal accumulation buffer
		// bci references the counter portion of the accumulation buffer
		x = y * swid + Rad;
		byi = by + x; bii = bu + x;	bqi = bv + x;
		// bci and bei reference temporal counters
		bci = bc + x; bei = be + x;
		// lb?i pointers reference the LAST frame (no temporal functions)
		lbyi = lby + x; lbii = lbu + x; lbqi = lbv + x;
		// pb?i pointers reference the LAST accumulated frame
		pbyi = pby + x; pbii = pbu + x; pbqi = pbv + x;
		// src and dst reference the actual source and destination frames
		x = y * spit + Rad;
		src = isrc + x; dst = idst + x;

		// Horizontal loop
		for (x = Rad; x < swid - Rad; x++) {
			// Read de-noised luminance
			tcy = fxTemp_DeNoiseSingle(src, spit, mfd->dnAmt, Rad) >> 16;
			// Read YIQ values
			_asm {
				mov		ebx, [src]
				add		[src], 4
				mov		eax, [ebx]
				mov		ebx, eax
				and		ebx, 0x00FFL
				mov		[cq], ebx
				shr		eax, 8
				mov		ebx, eax
				and		ebx, 0x00FFL
				mov		[ci], ebx
				shr		eax, 8
				and		eax, 0x00FFL
				mov		[cy], eax
			}

			oy = cy;	// Original Luminance
			flag = 0;

			// Find difference between current and previously accumulated
			// luma's for use in the range check and storing temporal error
			cr = mfd->sdi[sci[abs((long)*pbyi - cy)]];
			// Find difference between current Luma and Luma from previous frame
			// and test against the Luma Threshold
			//if ((abs((long)*lbyi - tcy) >> sms) <= lThresh && (cr >> sms) <= lThresh) {
			if (mfd->sdi[sci[abs((long)*lbyi - tcy)]] <= lThresh && cr <= lThresh) {
				cg = mfd->sdi[sci[abs((long)*pbii - ci)]];
				cb = mfd->sdi[sci[abs((long)*pbqi - cq)]];
				// If within range, check Chroma Thresholds in the same way as
				// the Luma Threshold check
				if (mfd->sdi[sci[abs((long)*lbii - ci)]] <= cThresh && mfd->sdi[sci[abs((long)*lbqi - cq)]] <= cThresh && cg <= cThresh && cb <= cThresh) {
					// If BOTH Chroma's fall within the Chroma Threshold, add
					// all signal components to the temporal accumulation buffer
					// and increment the accumulation counter
					t = *bci;
					if (t <= pl) {
						*byi += (unsigned char)cy; *bii += (unsigned char)ci; *bqi += (unsigned char)cq;
						*bci = ++t;
					}
					// Store error count
					*bei = *bei + mfd->Div3Tab[cr + cg + cb];
					if (*bei >= mfd->stTempError) {
						*byi = (unsigned char)cy; *bii = (unsigned char)ci; *bqi = (unsigned char)cq;
						*bci = 1; *bei = 0; flag = 1;
					}
				} else {
					// If one (or both) of the Chroma's are outside of the
					// Threshold, replace the data in the temporal accumulation
					// buffer with new values and reset the accumulation counters
					*byi = (unsigned char)cy; *bii = (unsigned char)ci; *bqi = (unsigned char)cq;
					*bci = 1; *bei = 0; flag = 1;
				}
			} else {
				// Same as above. Replace old components with new and reset counters
				*byi = (unsigned char)cy; *bii = (unsigned char)ci; *bqi = (unsigned char)cq;
				*bci = 1; *bei = 0; flag = 1;
			}

			// Store current video components in the previous frame buffer
			*lbyi++ = (unsigned char)tcy; *lbii++ = (unsigned char)ci; *lbqi++ = (unsigned char)cq;

			// Calculate new components by dividing the temporal accumulation
			// buffer's data by the accumulation count (averages all collected
			// samples together)
			if (flag && mfd->dnEnable == BST_CHECKED) {
				cy = tcy;
			} else {
				cy = *byi / *bci;
				ci = *bii / *bci;
				cq = *bqi / *bci;
			}
			// Store results in previous frame buffer
			*pbyi++ = (unsigned char)cy; *pbii++ = (unsigned char)ci; *pbqi++ = (unsigned char)cq;

			// Store new Ycc values in temporary framebuffer
			if (mfd->stAffectLC == BST_CHECKED) cy = oy;
			__asm {
				mov		eax, [cy]
				shl		eax, 8
				or		eax, [ci]
				shl		eax, 8
				or		eax, [cq]
				mov		ebx, [dst]
				add		[dst], 4
				mov		[ebx], eax
			}

			// Increment temporal accumulation buffer pointers
			byi++; bii++; bqi++; bci++; bei++;
		}
	}

	if (mfd->stScales == BST_CHECKED) {
		for (c = 0; c < 255; c++) {
			dst = idst + c + mfd->sdi[c] * spit;
			*dst = 0x00FF00FFL;
			if (c == sd) {
				for (y = mfd->sdi[c]; y <= mfd->sdi[c] + 4; y++) {
					dst = idst + c + y * spit;
					*dst = 0x00FF00FFL;
				}
			}
			dst = idst + c + sci[c] * spit;
			*dst = 0x00FFFF00L;
			if (c == sc) {
				for (y = sci[c]; y <= sci[c] + 4; y++) {
					dst = idst + c + y * spit;
					*dst = 0x00FFFF00L;
				}
			}
		}
	}
}

///////////////////////////////////////////////////////////////////////////

void	fxTemp_TemporalTest(Pixel32 *isrc, Pixel32 *idst, long swid, long shei, long spit, MyFilterData *mfd) {
	Pixel32	*src, *dst;
	long	x, y, c, cl, t, pl = mfd->stPixelLock;
	long	cr, cg, cb, tcy, cy, ci, cq, Rad, flag;
	long	sci[256], sc, d;
	int		*byi, *bii, *bqi, *bci, *bei;
	int		*by = (int *)mfd->stBuffY;
	int		*bu = (int *)mfd->stBuffI;
	int		*bv = (int *)mfd->stBuffQ;
	int		*bc = (int *)mfd->stCount;
	int		*be = (int *)mfd->stError;
	unsigned char	*lbyi, *lbii, *lbqi;
	unsigned char	*pbyi, *pbii, *pbqi, *pyi;
	unsigned char	*lby = (unsigned char *)mfd->stlBuffY;
	unsigned char	*lbu = (unsigned char *)mfd->stlBuffI;
	unsigned char	*lbv = (unsigned char *)mfd->stlBuffQ;
	unsigned char	*pby = (unsigned char *)mfd->stpBuffY;
	unsigned char	*pbu = (unsigned char *)mfd->stpBuffI;
	unsigned char	*pbv = (unsigned char *)mfd->stpBuffQ;
	unsigned char	*py = (unsigned char *)mfd->stPrevY;
	const long	lThresh = mfd->stLumaThresh;
	const long	cThresh = mfd->stChromaThresh;
	const long	ssize = spit * shei * 4;

	if (pl == 0) pl = 2147483647;
	Rad = mfd->dnRadius;
	for (d = 0; d < 256; d++)
		sci[d] = d;

	if (mfd->stSceneDet > 0) {
		t = 0; sc = 0;
		for (y = 0; y < shei; y++) {
			pyi = py + y * swid;;
			src = isrc + y * spit;
			for (x = 0; x < swid; x++) {
				cy = (*src++ >> 16) & 255;
				sc += cy;
				cr = abs(cy - *pyi);
				*pyi++ = (char)cy;
				if (cr > 4) t++;
			}
		}

		sc /= (swid * shei);
		for (d = 0; d < 256; d++) {
			sci[d] = d + (long)(sin(PI / (double)mfd->acRange / 4.0 * (double)d) * (double)(mfd->acRange - d) * 1.0 / 255.0 * (double)(255 - sc) * (0.01 * (double)mfd->acAmp) + 0.5);
			if (sci[d] < 0)
				sci[d] = 0;
			else if (sci[d] > 255)
				sci[d] = 255;
		}

		if ((long)(100.0 / (double)(swid * shei) * (double)t) > mfd->stSceneDet) {
			for (y = 0; y < shei; y++) {
				x = y * swid;
				byi = by + x; bii = bu + x;	bqi = bv + x;
				bci = bc + x; bei = be + x;
				lbyi = lby + x; lbii = lbu + x; lbqi = lbv + x;
				pbyi = pby + x; pbii = pbu + x; pbqi = pbv + x;
				src = isrc + y * spit;
				dst = idst + y * spit;
				for (x = 0; x < swid; x++) {
					c = *src++;
					cy = (c >> 16) & 255; ci = (c >> 8) & 255; cq = c & 255;
					*byi++ = cy; *bii++ = ci; *bqi++ = cq; *bci++ = 1; *bei++ = 0;
					*lbyi++ = (char)cy; *lbii++ = (char)ci; *lbqi++ = (char)cq;
					*pbyi++ = (char)cy; *pbii++ = (char)ci; *pbqi++ = (char)cq;
					if (mfd->stShowMotion == BST_CHECKED) *dst++ = 0xFF0000L;
				}
			}
			return;
		}
	}

	// Vertical loop
	for (y = Rad; y < shei - Rad; y++) {
		// Prepare all sorts of pointers...
		// b?i pointers reference the temporal accumulation buffer
		// bci references the counter portion of the accumulation buffer
		x = y * swid + Rad;
		byi = by + x; bii = bu + x;	bqi = bv + x;
		// bci and bei reference temporal counters
		bci = bc + x; bei = be + x;
		// lb?i pointers reference the LAST frame (no temporal functions)
		lbyi = lby + x; lbii = lbu + x; lbqi = lbv + x;
		// pb?i pointers reference the LAST accumulated frame
		pbyi = pby + x; pbii = pbu + x; pbqi = pbv + x;
		// src and dst reference the actual source and destination frames
		x = y * spit + Rad;
		src = isrc + x; dst = idst + x;

		// Horizontal loop
		for (x = Rad; x < swid - Rad; x++) {
			// Read denoised luminance value
			tcy = fxTemp_DeNoiseSingle(src, spit, mfd->dnAmt, Rad) >> 16;
			// Read YIQ values
			c = *src; cl = 0;
			cy = (c >> 16) & 255; ci = (c >> 8) & 255; cq = c & 255;
			flag = 0;
			// Find difference between current and previously accumulated
			// luma's for use in the range check and storing temporal error
			cr = sci[mfd->sdi[abs((long)*pbyi - cy)]];
			// Find difference between current Luma and Luma from previous frame
			// and test against the Luma Threshold
			//if ((abs((long)*lbyi - tcy) >> sms) <= lThresh && cr <= lThresh) {
			if (sci[mfd->sdi[abs((long)*lbyi - tcy)]] <= lThresh && cr <= lThresh) {
				cg = sci[mfd->sdi[abs((long)*pbii - ci)]];
				cb = sci[mfd->sdi[abs((long)*pbqi - cq)]];
				// If within range, check Chroma Thresholds in the same way as
				// the Luma Threshold check
				if (sci[mfd->sdi[abs((long)*lbii - ci)]] <= cThresh && sci[mfd->sdi[abs((long)*lbqi - cq)]] <= cThresh && cg <= cThresh && cb <= cThresh) {
					// If BOTH Chroma's fall within the Chroma Threshold, add
					// all signal components to the temporal accumulation buffer
					// and increment the accumulation counter
					t = *bci;
					if (t <= pl) {					
						*byi += cy; *bii += ci; *bqi += cq;
						*bci = *bci + 1;
					} else {
						if (mfd->stShowLock == BST_CHECKED && pl <= 255) cl |= 0x0000FFL;
					}
					// Store error count
					*bei = *bei + mfd->Div3Tab[cr + cg + cb];
					if (*bei >= mfd->stTempError) {
						if (mfd->stShowTemp == BST_CHECKED) cl |= 0x00FF00L;
						*byi = cy; *bii = ci; *bqi = cq;
						*bci = 1; *bei = 0; flag = 1;
					}
				} else {
					// If one (or both) of the Chroma's are outside of the
					// Threshold, replace the data in the temporal accumulation
					// buffer with new values and reset the accumulation counters
					if (mfd->stShowMotion == BST_CHECKED) cl |= 0xFF0000L;
					*byi = cy; *bii = ci; *bqi = cq;
					*bci = 1; *bei = 0; flag = 1;
				}
			} else {
				// Same as above. Replace old components with new and reset counters
				if (mfd->stShowMotion == BST_CHECKED) cl |= 0xFF0000L;
				*byi = cy; *bii = ci; *bqi = cq;
				*bci = 1; *bei = 0; flag = 1;
			}

			// Store current video components in the previous frame buffer
			*lbyi++ = (unsigned char)tcy; *lbii++ = (unsigned char)ci; *lbqi++ = (unsigned char)cq;

			// Calculate new components by dividing the temporal accumulation
			// buffer's data by the accumulation count (averages all collected
			// samples together)
			if (flag && mfd->dnEnable == BST_CHECKED) {
				cy = tcy;
			} else {
				cy = *byi / *bci;
				ci = *bii / *bci;
				cq = *bqi / *bci;
			}
			// Store results in previous frame buffer
			*pbyi++ = (unsigned char)cy; *pbii++ = (unsigned char)ci; *pbqi++ = (unsigned char)cq;

			// Store new Ycc values in temporary framebuffer
			c >>= 17;
			cr = c + (cl >> 16);
			if (cr > 255) cr = 255;
			cg = c + ((cl >> 8) & 255);
			if (cg > 255) cg = 255;
			cb = c + (cl & 255);
			if (cb > 255) cb = 255;
			src++;
			*dst++ = (cr << 16) | (cg << 8) | cb;

			// Increment temporal accumulation buffer pointers
			byi++; bii++; bqi++; bci++; bei++;
		}
	}

	if (mfd->stScales == BST_CHECKED) {
		for (c = 0; c < 255; c++) {
			dst = idst + c + mfd->sdi[c] * spit;
			*dst = 0x00FF00FFL;
			dst = idst + c + sci[c] * spit;
			*dst = 0x00FFFF00L;
		}
	}
}

///////////////////////////////////////////////////////////////////////////

void fxTemp_DeNoise(Pixel32 *isrc, Pixel32 *idst, long sw, long sh, long spit, int Thresh, int Rad) {
	Pixel32	*src, *dst, *ofs;
	PixDim	w, h, x, y;
	long	cr, cg, cb, tr, tg, tb;
	long	ar, ag, ab, c, count;

	if (sw - (Rad * 2 + 1) <= 0 || sh - (Rad * 2 + 1) <= 0)
		return;

	for (h = Rad; h < sh - Rad; h++) {
		src = isrc + h * spit + Rad;
		dst = idst + h * spit + Rad;
		for (w = Rad; w < sw - Rad; w++) {
			c = *src++;
			cr = (c>>16)&255; cg = (c>>8)&255; cb = c&255;
			ar = cr; ag = cg; ab = cb; count = 1;
			for (y = h - Rad; y < h + Rad; y++) {
				ofs = isrc + y * spit + w - Rad;
				for (x = w - Rad; x < w + Rad; x++) {
					c = *ofs++;
					tr = (c>>16)&255; tg = (c>>8)&255; tb = c&255;
					if (abs(cr - tr) < Thresh && abs(cg - tg) < Thresh && abs(cb - tb) < Thresh) {
						ar += tr; ag += tg; ab += tb; count++;
					}
				}
			}
			cr = ar / count; cg = ag / count; cb = ab / count;
			*dst++ = (cr<<16) | (cg<<8) | cb;
		}
	}
}

///////////////////////////////////////////////////////////////////////////

long fxTemp_DeNoiseSingle(Pixel32 *src, long spit, int Thresh, int Rad) {
	//Pixel32	*src, *dst, *ofs;
	//PixDim	w, h, x, y;
	Pixel32	*ofs;
	long	x, y, c, count;
	long	cy, ciq, ty, ay;

	Thresh <<= 16;
	c = *src;
	cy = c & 0x00FF0000L; ciq = c & 0x0000FFFFL;
	ay = cy; count = 1;
	for (y = -Rad; y <= Rad; y++) {
		ofs = src + y * spit - Rad;
		for (x = -Rad; x <= Rad; x++) {
			c = *ofs++;
			ty = c & 0x00FF0000L;
			if (abs(cy - ty) < Thresh) {
				ay += ty; count++;
			}
		}
	}
	cy = ay / count;
	return (cy & 0x00FF0000L) | ciq;
}

///////////////////////////////////////////////////////////////////////////

void	fxTemp_RGB2YIQ(Pixel32 *isrc, Pixel32 *idst, long swid, long shei, long spit, MyFilterData *mfd) {
	Pixel32	*src, *dst;
	long	y, c;
	long	*ytab = (long *)mfd->ytab;
	long	*itab = (long *)mfd->itab;
	long	*qtab = (long *)mfd->qtab;

	if (mfd->stYIQ == BST_CHECKED) {
		// Convert RGB -> YIQ
		for (y = 0; y < shei; y++) {
			src = isrc + y * spit;
			dst = idst + y * spit;
			__asm {
				push	esi
				push	edi
				mov		ecx, [swid]

cty10:			mov		esi, [src]
				add		[src], 4
				mov		edx, [esi]
				mov		ebx, edx
				shr		ebx, 6
				and		ebx, 0x0003FCL
				mov		eax, edx
				shr		eax, 14
				and		eax, 0x0003FCL
				shl		edx, 2
				and		edx, 0x0003FCL
				add		ebx, 1024
				add		edx, 2048

				mov		esi, [ytab]
				mov		edi, [esi + eax]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cty20
				test	edi, 0xFF000000L
				jz		cty21
				mov		edi, 0x00FF0000L
				jmp		cty21
cty20:			xor		edi, edi
cty21:			mov		[c], edi

				mov		esi, [itab]
				mov		edi, [esi + eax]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cty30
				test	edi, 0xFF000000L
				jz		cty31
				mov		edi, 0x00FF0000L
				jmp		cty31
cty30:			xor		edi, edi
cty31:			shr		edi, 8
				or		[c], edi

				mov		esi, [qtab]
				mov		edi, [esi + eax]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cty40
				test	edi, 0xFF000000L
				jz		cty41
				mov		edi, 0x00FF0000L
cty40:			xor		edi, edi
cty41:			shr		edi, 16
				or		[c], edi

				mov		edi, [dst]
				add		[dst], 4
				mov		eax, [c]
				mov		[edi], eax

				dec		ecx
				jnz		cty10
				pop		edi
				pop		esi
			}
		}
	} else {
		// Convert RGB -> YUV
		for (y = 0; y < shei; y++) {
			src = isrc + y * spit;
			dst = idst + y * spit;
			__asm {
				push	esi
				push	edi
				mov		ecx, [swid]

ctu10:			mov		esi, [src]
				add		[src], 4
				mov		edx, [esi]
				mov		ebx, edx
				shr		ebx, 6
				and		ebx, 0x0003FCL
				mov		eax, edx
				shr		eax, 14
				and		eax, 0x0003FCL
				shl		edx, 2
				and		edx, 0x0003FCL
				add		ebx, 1024
				add		edx, 2048

				mov		esi, [ytab]
				mov		edi, [esi + eax]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		ctu20
				test	edi, 0xFF000000L
				jz		ctu21
				mov		edi, 0x00FF0000L
				jmp		ctu21
ctu20:			xor		edi, edi
ctu21:			mov		[c], edi

				mov		esi, [itab]
				mov		edi, [esi + eax]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		ctu30
				test	edi, 0xFF000000L
				jz		ctu31
				mov		edi, 0x00FF0000L
				jmp		ctu31
ctu30:			xor		edi, edi
ctu31:			shr		edi, 8
				or		[c], edi

				mov		esi, [qtab]
				mov		edi, [esi + eax]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		ctu40
				test	edi, 0xFF000000L
				jz		ctu41
				mov		edi, 0x00FF0000L
				jmp		ctu41
ctu40:			xor		edi, edi
ctu41:			shr		edi, 16
				or		[c], edi

				mov		edi, [dst]
				add		[dst], 4
				mov		eax, [c]
				mov		[edi], eax

				dec		ecx
				jnz		ctu10
				pop		edi
				pop		esi
			}
		}
	}
}

///////////////////////////////////////////////////////////////////////////

void	fxTemp_YIQ2RGB(Pixel32 *isrc, Pixel32 *idst, long swid, long shei, long spit, MyFilterData *mfd) {
	Pixel32	*src, *dst;
	long	y, c;
	long	*rtab = (long *)mfd->rtab;
	long	*gtab = (long *)mfd->gtab;
	long	*btab = (long *)mfd->btab;
	long	*cytf = (long *)mfd->cytf;

	if (mfd->stYIQ == BST_CHECKED) {
		// Convert YIQ -> RGB
		for (y = 0; y < shei; y++) {
			src = isrc + y * spit;
			dst = idst + y * spit;
			__asm {
				push	esi
				push	edi
				mov		ecx, [swid]

cfy10:			mov		esi, [src]
				add		[src], 4
				mov		edx, [esi]
				mov		ebx, edx
				shr		ebx, 6
				and		ebx, 0x0003FCL
				mov		eax, edx
				and		eax, 0x00FF0000L
				shl		edx, 2
				and		edx, 0x0003FCL
				add		edx, 1024

				mov		edi, eax
				mov		esi, [rtab]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cfy20
				test	edi, 0xFF000000L
				jz		cfy21
				mov		edi, 0x00FF0000L
				jmp		cfy21
cfy20:			xor		edi, edi
cfy21:			mov		[c], edi

				mov		edi, eax
				mov		esi, [gtab]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cfy30
				test	edi, 0xFF000000L
				jz		cfy31
				mov		edi, 0x00FF0000L
				jmp		cfy31
cfy30:			xor		edi, edi
cfy31:			shr		edi, 8
				or		[c], edi

				mov		edi, eax
				mov		esi, [btab]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cfy40
				test	edi, 0xFF000000L
				jz		cfy41
				mov		edi, 0x00FF0000L
				jmp		cfy41
cfy40:			xor		edi, edi
cfy41:			shr		edi, 16
				or		[c], edi

				mov		edi, [dst]
				add		[dst], 4
				mov		eax, [c]
				mov		[edi], eax

				dec		ecx
				jnz		cfy10
				pop		edi
				pop		esi
			}
		}
	} else {
		// Convert YUV -> RGB
		for (y = 0; y < shei; y++) {
			src = isrc + y * spit;
			dst = idst + y * spit;
			__asm {
				push	esi
				push	edi
				mov		ecx, [swid]

cfu10:			mov		esi, [src]
				add		[src], 4
				mov		edx, [esi]
				mov		ebx, edx
				shr		ebx, 6
				and		ebx, 0x0003FCL
				mov		eax, edx
				shr		eax, 14
				and		eax, 0x0003FCL
				shl		edx, 2
				and		edx, 0x0003FCL
				add		edx, 1024

				mov		esi, [cytf]
				mov		eax, [esi + eax]

				mov		edi, eax
				mov		esi, [rtab]
				add		edi, [esi + ebx]
				and		edi, 0xFFFF0000L
				js		cfu20
				test	edi, 0xFF000000L
				jz		cfu21
				mov		edi, 0x00FF0000L
				jmp		cfu21
cfu20:			xor		edi, edi
cfu21:			mov		[c], edi

				mov		edi, eax
				mov		esi, [gtab]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cfu30
				test	edi, 0xFF000000L
				jz		cfu31
				mov		edi, 0x00FF0000L
				jmp		cfu31
cfu30:			xor		edi, edi
cfu31:			shr		edi, 8
				or		[c], edi

				mov		edi, eax
				mov		esi, [btab]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cfu40
				test	edi, 0xFF000000L
				jz		cfu41
				mov		edi, 0x00FF0000L
				jmp		cfu41
cfu40:			xor		edi, edi
cfu41:			shr		edi, 16
				or		[c], edi

				mov		edi, [dst]
				add		[dst], 4
				mov		eax, [c]
				mov		[edi], eax

				dec		ecx
				jnz		cfu10
				pop		edi
				pop		esi
			}
		}
	}
}

///////////////////////////////////////////////////////////////////////////

void	fxColorSpace_GenTabs(long format, MyFilterData *mfd) {
	long	d;

	// Calculate data for RGB -> Ycc conversion tables
	for (d = 0; d < 256; d++) {
		if (mfd->stYIQ == BST_CHECKED) {
			// RGB -> YIQ transformation matrix
			mfd->ytab[d      ] = (long)(+0.299 * d * 65536 + 0.5);
			mfd->ytab[d + 256] = (long)(+0.587 * d * 65536 + 0.5);
			mfd->ytab[d + 512] = (long)(+0.114 * d * 65536 + 0.5);
			mfd->itab[d      ] = (long)(+0.596 * d * 65536 + 0.5) + 8388608;
			mfd->itab[d + 256] = (long)(-0.275 * d * 65536 + 0.5);
			mfd->itab[d + 512] = (long)(-0.321 * d * 65536 + 0.5);
			mfd->qtab[d      ] = (long)(+0.212 * d * 65536 + 0.5) + 8388608;
			mfd->qtab[d + 256] = (long)(-0.523 * d * 65536 + 0.5);
			mfd->qtab[d + 512] = (long)(+0.311 * d * 65536 + 0.5);
			// YIQ -> RGB transformation matrix
			mfd->rtab[d      ] = (long)(+0.956 * (d-128) * 65536 + 0.5);
			mfd->rtab[d + 256] = (long)(+0.621 * (d-128) * 65536 + 0.5);
			mfd->gtab[d      ] = (long)(-0.272 * (d-128) * 65536 + 0.5);
			mfd->gtab[d + 256] = (long)(-0.647 * (d-128) * 65536 + 0.5);
			mfd->btab[d      ] = (long)(-1.105 * (d-128) * 65536 + 0.5);
			mfd->btab[d + 256] = (long)(+1.702 * (d-128) * 65536 + 0.5);
		} else {
			// RGB -> YUV transformation matrix
			mfd->ytab[d      ] = (long)(+0.257 * d * 65536 + 0.5) + 1048576;
			mfd->ytab[d + 256] = (long)(+0.504 * d * 65536 + 0.5);
			mfd->ytab[d + 512] = (long)(+0.098 * d * 65536 + 0.5);
			mfd->itab[d      ] = (long)(+0.439 * d * 65536 + 0.5) + 8388608;
			mfd->itab[d + 256] = (long)(-0.368 * d * 65536 + 0.5);
			mfd->itab[d + 512] = (long)(-0.071 * d * 65536 + 0.5);
			mfd->qtab[d      ] = (long)(-0.148 * d * 65536 + 0.5) + 8388608;
			mfd->qtab[d + 256] = (long)(-0.291 * d * 65536 + 0.5);
			mfd->qtab[d + 512] = (long)(+0.439 * d * 65536 + 0.5);
			// YUV -> RGB transformation matrix
			mfd->rtab[d      ] = (long)(+1.596 * (d-128) * 65536 + 0.5);
			mfd->rtab[d + 256] = 0;
			mfd->gtab[d      ] = (long)(-0.813 * (d-128) * 65536 + 0.5);
			mfd->gtab[d + 256] = (long)(-0.391 * (d-128) * 65536 + 0.5);
			mfd->btab[d      ] = 0;
			mfd->btab[d + 256] = (long)(+2.018 * (d-128) * 65536 + 0.5);
			mfd->cytf[d] = (long)(((d - 16) * 1.164) * 65536.0 + 0.5);	// Y transform
		}
	}
}

///////////////////////////////////////////////////////////////////////////

int fxTempStartProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	long	d;
	double	e, i;

	// Detect if MMX is available and should be used
	mfd->mmx = ff->isMMXEnabled();
	if (fa->src.w & 1) mfd->mmx = 0;

	// Create temporary frame buffer
	d = fa->src.pitch / 4 * fa->src.h;
	if (!(mfd->TmpBuff = new Pixel32[d])) return 1;

	// Chroma Buffers for the chroma stabilizer
	d = fa->src.w * fa->src.h;
	if (!(mfd->stCount = new int[d])) return 1;
	if (!(mfd->stError = new int[d])) return 1;
	if (!(mfd->stBuffY = new int[d])) return 1;
	if (!(mfd->stBuffI = new int[d])) return 1;
	if (!(mfd->stBuffQ = new int[d])) return 1;

	if (!(mfd->stPrevY = new unsigned char[d])) return 1;
	if (!(mfd->stlBuffY = new unsigned char[d])) return 1;
	if (!(mfd->stlBuffI = new unsigned char[d])) return 1;
	if (!(mfd->stlBuffQ = new unsigned char[d])) return 1;
	if (!(mfd->stpBuffY = new unsigned char[d])) return 1;
	if (!(mfd->stpBuffI = new unsigned char[d])) return 1;
	if (!(mfd->stpBuffQ = new unsigned char[d])) return 1;

	memset(mfd->stPrevY, 0, d);
	memset(mfd->stlBuffY, 0, d);
	memset(mfd->stlBuffI, 0, d);
	memset(mfd->stlBuffQ, 0, d);
	memset(mfd->stpBuffY, 80, d);
	memset(mfd->stpBuffI, 0, d);
	memset(mfd->stpBuffQ, 0, d);

	d <<= 1;
	memset(mfd->stCount, 0, d);
	memset(mfd->stError, 0, d);
	memset(mfd->stBuffY, 0, d);
	memset(mfd->stBuffI, 0, d);
	memset(mfd->stBuffQ, 0, d);
	
	// Y transform for YUV -> RGB conversion
	if (!(mfd->cytf = new long[256])) return 1;

	// RGB -> Ycc conversion tables
	if (!(mfd->ytab = new long[768])) return 1;
	if (!(mfd->itab = new long[768])) return 1;
	if (!(mfd->qtab = new long[768])) return 1;

	// Ycc -> RGB conversion tables
	if (!(mfd->rtab = new long[512])) return 1;
	if (!(mfd->gtab = new long[512])) return 1;
	if (!(mfd->btab = new long[512])) return 1;

	// Division by 3 table
	if (!(mfd->Div3Tab = new long[768])) return 1;
	for (d = 0; d < 768; d++) {
		//mfd->Div3Tab[d] = d / 3 - mfd->stTempBias;
		mfd->Div3Tab[d] = d - mfd->stTempBias;
		if (mfd->Div3Tab[d] < 0) mfd->Div3Tab[d] = 0;
	}

	// Scene Detection <-> Motion Intensity
	if (!(mfd->sdi = new long[255])) return 1;
	for (d = 0; d < 256; d++)
		mfd->sdi[d] = d;

	i = (2.56 * (double)mfd->stHMSD);
	for (d = 0; d < (long)i; d++) {
		e = (double)d - sin(PI / (double)i / 2.0 * (double)-d) * (double)mfd->stHMSDI - (double)mfd->stHMSDI;
		if (e < 0) e = 0;
		mfd->sdi[d] = (long)e;
	}

	return 0;
}

///////////////////////////////////////////////////////////////////////////

int fxTempEndProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	// Delete temporary frame buffer
	delete[] mfd->TmpBuff; mfd->TmpBuff = NULL;

	// Delete the Y transformation table
	delete[] mfd->cytf; mfd->cytf = NULL;

	// Delete RGB -> YIQ/YUV conversion tables
	delete[] mfd->ytab; mfd->ytab = NULL;
	delete[] mfd->itab; mfd->itab = NULL;
	delete[] mfd->qtab; mfd->qtab = NULL;
	
	delete[] mfd->rtab; mfd->rtab = NULL;
	delete[] mfd->gtab; mfd->gtab = NULL;
	delete[] mfd->btab; mfd->btab = NULL;

	// Delete frame buffers
	delete[] mfd->stBuffY; mfd->stBuffY = NULL;
	delete[] mfd->stBuffI; mfd->stBuffI = NULL;
	delete[] mfd->stBuffQ; mfd->stBuffQ = NULL;
	delete[] mfd->stlBuffY; mfd->stlBuffY = NULL;
	delete[] mfd->stlBuffI; mfd->stlBuffI = NULL;
	delete[] mfd->stlBuffQ; mfd->stlBuffQ = NULL;
	delete[] mfd->stpBuffY; mfd->stpBuffY = NULL;
	delete[] mfd->stpBuffI; mfd->stpBuffI = NULL;
	delete[] mfd->stpBuffQ; mfd->stpBuffQ = NULL;
	delete[] mfd->stCount; mfd->stCount = NULL;
	delete[] mfd->stError; mfd->stError = NULL;

	delete[] mfd->stPrevY; mfd->stPrevY = NULL;

	delete[] mfd->Div3Tab; mfd->Div3Tab = NULL;
	delete[] mfd->sdi; mfd->sdi = NULL;

	return 0;
}

///////////////////////////////////////////////////////////////////////////

int fxTempInitProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->stLumaThresh = 4;
	mfd->stChromaThresh = 10;
	mfd->stTempError = 1;
	mfd->stTempBias = 6;
	mfd->stPixelLock = 4;
	mfd->stSceneDet = 50;
	mfd->stShowMotion = BST_UNCHECKED;
	mfd->stShowTemp = BST_UNCHECKED;
	mfd->stShowLock = BST_UNCHECKED;
	mfd->stYIQin = BST_UNCHECKED;
	mfd->stYIQout = BST_UNCHECKED;
	mfd->stYIQ = BST_CHECKED;
	mfd->stAffectLC = BST_UNCHECKED;
	mfd->stAffectCO = BST_CHECKED;
	mfd->dnEnable = BST_CHECKED;
	mfd->dnRadius = 2;
	mfd->dnAmt = 8;
	mfd->stHMSD = 10;
	mfd->stHMSDI = 20;
	mfd->acRange = 64;
	mfd->acAmp = 100;

	return 0;
}

///////////////////////////////////////////////////////////////////////////

BOOL CALLBACK fxTempConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
	MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

	switch(msg) {
	case WM_INITDIALOG:
		SetWindowLong(hdlg, DWL_USER, lParam);
		mfd = (MyFilterData *)lParam;
		SetDlgItemInt(hdlg, IDC_ST_LUMA, mfd->stLumaThresh, FALSE);
		SetDlgItemInt(hdlg, IDC_ST_CHROMA, mfd->stChromaThresh, FALSE);
		SetDlgItemInt(hdlg, IDC_ST_ERROR, mfd->stTempError, FALSE);
		SetDlgItemInt(hdlg, IDC_ST_BIAS, mfd->stTempBias, FALSE);
		SetDlgItemInt(hdlg, IDC_ST_PIXLOCK, mfd->stPixelLock, FALSE);
		SetDlgItemInt(hdlg, IDC_ST_SCNDET, mfd->stSceneDet, FALSE);
		SetDlgItemInt(hdlg, IDC_SD_HMSD, mfd->stHMSD, FALSE);
		SetDlgItemInt(hdlg, IDC_SD_HMSDI, mfd->stHMSDI, FALSE);
		SetDlgItemInt(hdlg, IDC_DN_RADIUS, mfd->dnRadius, FALSE);
		SetDlgItemInt(hdlg, IDC_DN_AMT, mfd->dnAmt, FALSE);
		SetDlgItemInt(hdlg, IDC_AC_RANGE, mfd->acRange, FALSE);
		SetDlgItemInt(hdlg, IDC_AC_AMP, mfd->acAmp, FALSE);
		CheckDlgButton(hdlg, IDC_ST_YIQIN, mfd->stYIQin ? BST_CHECKED : BST_UNCHECKED);
		CheckDlgButton(hdlg, IDC_ST_YIQOUT, mfd->stYIQout ? BST_CHECKED : BST_UNCHECKED);
		CheckDlgButton(hdlg, IDC_ST_SHMOTION, mfd->stShowMotion ? BST_CHECKED : BST_UNCHECKED);
		CheckDlgButton(hdlg, IDC_ST_SHTEMP, mfd->stShowTemp ? BST_CHECKED : BST_UNCHECKED);
		CheckDlgButton(hdlg, IDC_ST_SHLOCK, mfd->stShowLock ? BST_CHECKED : BST_UNCHECKED);
		CheckDlgButton(hdlg, IDC_ST_YIQ, mfd->stYIQ ? BST_CHECKED : BST_UNCHECKED);
		CheckDlgButton(hdlg, IDC_ST_AFLC, mfd->stAffectLC ? BST_CHECKED : BST_UNCHECKED);
		CheckDlgButton(hdlg, IDC_ST_AFCO, mfd->stAffectCO ? BST_CHECKED : BST_UNCHECKED);
		CheckDlgButton(hdlg, IDC_ST_SCALES, mfd->stScales ? BST_CHECKED : BST_UNCHECKED);
		CheckDlgButton(hdlg, IDC_DN_ENABLE, mfd->dnEnable ? BST_CHECKED : BST_UNCHECKED);
		return TRUE;

	case WM_COMMAND:
		switch(LOWORD(wParam)) {
		case IDOK:
			mfd->stLumaThresh = GetDlgItemInt(hdlg, IDC_ST_LUMA, &mfd->stLumaThresh, FALSE);
			mfd->stChromaThresh = GetDlgItemInt(hdlg, IDC_ST_CHROMA, &mfd->stChromaThresh, FALSE);
			mfd->stTempError = GetDlgItemInt(hdlg, IDC_ST_ERROR, &mfd->stTempError, FALSE);
			mfd->stTempBias = GetDlgItemInt(hdlg, IDC_ST_BIAS, &mfd->stTempBias, FALSE);
			mfd->stPixelLock = GetDlgItemInt(hdlg, IDC_ST_PIXLOCK, &mfd->stPixelLock, FALSE);
			mfd->stSceneDet = GetDlgItemInt(hdlg, IDC_ST_SCNDET, &mfd->stSceneDet, FALSE);
			mfd->stHMSD = GetDlgItemInt(hdlg, IDC_SD_HMSD, &mfd->stHMSD, FALSE);
			mfd->stHMSDI = GetDlgItemInt(hdlg, IDC_SD_HMSDI, &mfd->stHMSDI, FALSE);
			mfd->dnRadius = GetDlgItemInt(hdlg, IDC_DN_RADIUS, &mfd->dnRadius, FALSE);
			mfd->dnAmt = GetDlgItemInt(hdlg, IDC_DN_AMT, &mfd->dnAmt, FALSE);
			mfd->acRange = GetDlgItemInt(hdlg, IDC_AC_RANGE, &mfd->acRange, FALSE);
			mfd->acAmp = GetDlgItemInt(hdlg, IDC_AC_AMP, &mfd->acAmp, FALSE);
			mfd->stYIQin = !!IsDlgButtonChecked(hdlg, IDC_ST_YIQIN);
			mfd->stYIQout = !!IsDlgButtonChecked(hdlg, IDC_ST_YIQOUT);
			mfd->stShowMotion = !!IsDlgButtonChecked(hdlg, IDC_ST_SHMOTION);
			mfd->stShowTemp = !!IsDlgButtonChecked(hdlg, IDC_ST_SHTEMP);
			mfd->stShowLock = !!IsDlgButtonChecked(hdlg, IDC_ST_SHLOCK);
			mfd->stScales = !!IsDlgButtonChecked(hdlg, IDC_ST_SCALES);
			mfd->stYIQ = !!IsDlgButtonChecked(hdlg, IDC_ST_YIQ);
			mfd->stAffectLC = !!IsDlgButtonChecked(hdlg, IDC_ST_AFLC);
			mfd->stAffectCO = !!IsDlgButtonChecked(hdlg, IDC_ST_AFCO);
			mfd->dnEnable = !!IsDlgButtonChecked(hdlg, IDC_DN_ENABLE);
			EndDialog(hdlg, 0);

			if (mfd->stLumaThresh > 255) mfd->stLumaThresh = 255;
			if (mfd->stChromaThresh > 255) mfd->stChromaThresh = 255;
			if (mfd->stTempError == 0) mfd->stTempError = 1;
			if (mfd->stTempBias > 255) mfd->stTempBias = 255;
			if (mfd->stPixelLock > 255) mfd->stPixelLock = 255;
			if (mfd->stSceneDet > 100) mfd->stSceneDet = 100;
			if (mfd->stAffectLC == BST_UNCHECKED && mfd->stAffectCO == BST_UNCHECKED) mfd->stAffectLC = BST_CHECKED;
			if (mfd->dnRadius > 10) mfd->dnRadius = 10;
			if (mfd->dnAmt > 255) mfd->dnAmt = 255;
			if (mfd->stHMSD > 100) mfd->stHMSD = 100;
			if (mfd->acRange > 255) mfd->acRange = 255;

			return TRUE;
		case IDCANCEL:
			EndDialog(hdlg, 1);
			return FALSE;
		}
		break;
	}

	return FALSE;
}

///////////////////////////////////////////////////////////////////////////

int fxTempConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd) {
	return DialogBoxParam(fa->filter->module->hInstModule,
	       MAKEINTRESOURCE(IDD_FXTEMP_CONFIG), hwnd,
		   fxTempConfigDlgProc, (LPARAM)fa->filter_data);
}

///////////////////////////////////////////////////////////////////////////

void fxTempStringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	char	fstr[10];

	sprintf(fstr, mfd->stYIQ == BST_CHECKED ? "YIQ" : "YUV");

	sprintf(str, ":  %d, %d, %d, %d, %d, %d%% (%s -> %s)", mfd->stLumaThresh, mfd->stChromaThresh,
		mfd->stTempError, mfd->stTempBias, mfd->stPixelLock, mfd->stSceneDet,
		mfd->stYIQin == BST_CHECKED ? fstr : "RGB", mfd->stYIQout == BST_CHECKED ? fstr : "RGB");
}

///////////////////////////////////////////////////////////////////////////

void fxTempScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
    FilterActivation *fa = (FilterActivation *)lpVoid;
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->stLumaThresh = argv[0].asInt();
	mfd->stChromaThresh = argv[1].asInt();
	mfd->stTempError = argv[2].asInt();
	mfd->stTempBias = argv[3].asInt();
	mfd->stPixelLock = argv[4].asInt();
	mfd->stSceneDet = argv[5].asInt();
	mfd->stYIQin = argv[6].asInt() == 1 ? BST_CHECKED : BST_UNCHECKED;
	mfd->stYIQout = argv[7].asInt() == 1 ? BST_CHECKED : BST_UNCHECKED;
	mfd->stYIQ = argv[8].asInt() == 1 ? BST_CHECKED : BST_UNCHECKED;
	mfd->stAffectLC = argv[9].asInt() == 1 ? BST_CHECKED : BST_UNCHECKED;
	if (mfd->stAffectLC	== BST_CHECKED)
		mfd->stAffectCO = BST_UNCHECKED;
	mfd->dnEnable = argv[10].asInt() == 1 ? BST_CHECKED : BST_UNCHECKED;
	mfd->dnRadius = argv[11].asInt();
	mfd->dnAmt = argv[12].asInt();
	mfd->stHMSD = argv[13].asInt() & 0x00FFL;
	mfd->stHMSDI = argv[13].asInt() >> 8;
	mfd->acRange = argv[14].asInt() & 0x00FFL;
	mfd->acAmp = argv[14].asInt() >> 8;
}

///////////////////////////////////////////////////////////////////////////

bool fxTempFssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	_snprintf(buf, buflen, "Config(%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d)", 
		mfd->stLumaThresh, mfd->stChromaThresh, mfd->stTempError, mfd->stTempBias, mfd->stPixelLock, 
		mfd->stSceneDet, mfd->stYIQin == BST_CHECKED ? 1 : 0, mfd->stYIQout == BST_CHECKED ? 1 : 0,
		mfd->stYIQ == BST_CHECKED ? 1 : 0, mfd->stAffectLC == BST_CHECKED ? 1 : 0, 
		mfd->dnEnable == BST_CHECKED ? 1 : 0, mfd->dnRadius, mfd->dnAmt, mfd->stHMSD | (mfd->stHMSDI << 8), mfd->acRange | (mfd->acAmp << 8));

    return true;
}