// VirtualDub filter: deinterlace - area based
// by Gunnar Thalin (guth@home.se)

#include <windows.h>
#include <stdio.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

///////////////////////////////////////////////////////////////////////////

int InitProc(FilterActivation *fa, const FilterFunctions *ff);
int StartProc(FilterActivation *fa, const FilterFunctions *ff);
int EndProc(FilterActivation *fa, const FilterFunctions *ff);
int RunProc(const FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
void ScriptConfig2(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

typedef struct MyFilterData {
	bool    bShowDeinterlacedAreaOnly;
	bool    bBlend;
	int     iThreshold;
	int     iEdgeDetect;
	bool    bLog;
	float   fLogInterlacePercent;
	HANDLE  hLog;
} MyFilterData;


bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	_snprintf(buf, buflen, "Config(%i, %i, %i, %i, %i, \"%g\")",
		mfd->bShowDeinterlacedAreaOnly,
		mfd->bBlend,
		mfd->iThreshold,
		mfd->iEdgeDetect,
		mfd->bLog,
		mfd->fLogInterlacePercent);

	return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->bShowDeinterlacedAreaOnly = !!argv[0].asInt();
	mfd->bBlend = !!argv[1].asInt();
	mfd->iThreshold = argv[2].asInt();
	mfd->iEdgeDetect = argv[3].asInt();
}

void ScriptConfig2(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->bShowDeinterlacedAreaOnly = !!argv[0].asInt();
	mfd->bBlend = !!argv[1].asInt();
	mfd->iThreshold = argv[2].asInt();
	mfd->iEdgeDetect = argv[3].asInt();
	mfd->bLog = !!argv[4].asInt();
	mfd->fLogInterlacePercent = (float)atof(*(argv[5].asString()));
}

ScriptFunctionDef func_defs[]={
	{ (ScriptFunctionPtr)ScriptConfig, "Config", "0iiii" },
	{ (ScriptFunctionPtr)ScriptConfig2, "Config", "0iiiiis" },
	{ NULL },
};

CScriptObject script_obj={
	NULL, func_defs
};


struct FilterDefinition filterDef = {

	NULL, NULL, NULL,	// next, prev, module
	"deinterlace - area based v1.4",	// name
	"Removes interlace lines only where they are visible. Each frame is deinterlaced individually.",	// desc
	"Gunnar Thalin",		// maker
	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size

	InitProc,		// initProc
	NULL,			// deinitProc
	RunProc,		// runProc
	ParamProc,		// paramProc
	ConfigProc,		// configProc
	StringProc,		// stringProc
	StartProc,			// startProc
	EndProc,			// endProc

	&script_obj,	// script_obj
	FssProc,		// fssProc

};


///////////////////////////////////////////////////////////////////////////

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd;

///////////////////////////////////////////////////////////////////////////

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
    if (!(fd = ff->addFilter(fm, &filterDef, sizeof(FilterDefinition))))
        return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

    return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
    ff->removeFilter(fd);
}

///////////////////////////////////////////////////////////////////////////

int InitProc(FilterActivation *fa, const FilterFunctions *ff) {
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->bShowDeinterlacedAreaOnly = false;
	mfd->bBlend = true;
	mfd->iThreshold = 27;
	mfd->iEdgeDetect = 25;
	mfd->bLog = false;;
	mfd->fLogInterlacePercent = 0.0f;

    return 0;
}

int StartProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	if(mfd->bLog)
	{
		mfd->hLog = CreateFile("DeinterlaceAreaBased.log", GENERIC_WRITE, FILE_SHARE_READ, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
		if(mfd->hLog != NULL)
		{
			DWORD dw;
			WriteFile(mfd->hLog, "Frame\tInterlaced area (%)\r\n", 27, &dw, NULL);
		}
	}

	return 0;
}

int EndProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	if(mfd->bLog && mfd->hLog != NULL)
		CloseHandle(mfd->hLog);
	return 0;
}

int RunProc(const FilterActivation *fa, const FilterFunctions *ff) {
	Pixel32 *psrc = (Pixel32 *)fa->src.data;
	Pixel32 *pdst = (Pixel32 *)fa->dst.data;
   MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	int iThreshold = mfd->iThreshold;
	iThreshold = iThreshold * iThreshold * 4;
	int iEdgeDetect = mfd->iEdgeDetect;
	if(iEdgeDetect > 180)
		iEdgeDetect = 180;	// We don't want an integer overflow in the interlace calculation.
	iEdgeDetect = iEdgeDetect * iEdgeDetect;
	bool bShowDeinterlacedAreaOnly = mfd->bShowDeinterlacedAreaOnly;
	bool bBlend = mfd->bBlend;

	int iR0, iG0, iB0, iR1, iG1, iB1, iR2, iG2, iB2, iR3, iG3, iB3;
	Pixel32 *psrc1, *psrc2, *psrc3, *pdst1;
	int iInterlaceValue0, iInterlaceValue1, iInterlaceValue2;
	int iDeinterlacedPixels = 0;
	iR1 = iG1 = iB1 = 0;	// Avoid compiler warning. The value is not used.
	for(int x = 0; x < fa->src.w; x++)
	{
		psrc3 = psrc + x;
		iR3 = (*psrc3 >> 16) & 0xff;
		iG3 = (*psrc3 >> 8) & 0xff;
		iB3 = *psrc3 & 0xff;
		psrc2 = (Pixel32 *)((char *)psrc3 + fa->src.pitch);
		iR2 = (*psrc2 >> 16) & 0xff;
		iG2 = (*psrc2 >> 8) & 0xff;
		iB2 = *psrc2 & 0xff;
		pdst1 = pdst + x;
		iInterlaceValue1 = iInterlaceValue2 = 0;
		for(int y = 0; y <= fa->src.h; y++)
		{
			psrc1 = psrc2;
			psrc2 = psrc3;
			psrc3 = (Pixel32 *)((char *)psrc3 + fa->src.pitch);
			iR0 = iR1;
			iG0 = iG1;
			iB0 = iB1;
			iR1 = iR2;
			iG1 = iG2;
			iB1 = iB2;
			iR2 = iR3;
			iG2 = iG3;
			iB2 = iB3;
			if(y < fa->src.h - 1)
			{
				iR3 = (*psrc3 >> 16) & 0xff;
				iG3 = (*psrc3 >> 8) & 0xff;
				iB3 = *psrc3 & 0xff;
			}
			else
			{
				iR3 = iR1;
				iG3 = iG1;
				iB3 = iB1;
			}

			iInterlaceValue0 = iInterlaceValue1;
			iInterlaceValue1 = iInterlaceValue2;
			if(y < fa->src.h)
				// Calculate the interlace value by checking if the pixel color on previous row differs much
				// from this row and next row differs much (with same sign) from this row.
				// Detect edges by checking so that pixel color on previous row doesn't differ too much from next row.
				// If it does, it's probably just an ordinary edge.
				// 3, 6 and 1 are approximate values for converting R, G, B to intensity.
				iInterlaceValue2 = (3 * ((iR1 - iR2) * (iR3 - iR2) - ((iEdgeDetect * (iR1 - iR3) * (iR1 - iR3)) >> 12)) +
									6 * ((iG1 - iG2) * (iG3 - iG2) - ((iEdgeDetect * (iG1 - iG3) * (iG1 - iG3)) >> 12)) +
									(iB1 - iB2) * (iB3 - iB2) - ((iEdgeDetect * (iB1 - iB3) * (iB1 - iB3)) >> 12));
			else
				iInterlaceValue2 = 0;

			if(y > 0)
			{	// New in version 1.1: Get mean interlace value of 3 rows (i.e. 5 pixels are examined).
				// Middle row has twice the weight.
				if(iInterlaceValue0 + 2 * iInterlaceValue1 + iInterlaceValue2 > iThreshold)
				{
					// Blend: Get mean value of previous and next row (weight 0.25) and this row (weight 0.5).
					// Interpolate: Odd lines: Copy from source. Even lines: Get mean value of previous and next row
					*pdst1 = bBlend ? (((iR0 + 2 * iR1 + iR2) >> 2) << 16) + (((iG0 + 2 * iG1 + iG2) >> 2) << 8) + ((iB0 + 2 * iB1 + iB2) >> 2)
									  :
									  (y % 2 == 1 ? *psrc1 : (((iR0 + iR2) >> 1) << 16) + (((iG0 + iG2) >> 1) << 8) + ((iB0 + iB2) >> 1));

					iDeinterlacedPixels++;
				}
				else
				{
					*pdst1 = bShowDeinterlacedAreaOnly ? (0x00787878 + (iR1 >> 4 << 16) + (iG1 >> 4 << 8) + (iB1 >> 4)) : *psrc1;
				}
				pdst1 = (Pixel32 *)((char *)pdst1 + fa->dst.pitch);
			}
		}
	}

	if(mfd->bLog)
	{
		float fDeinterlacedPercent = 100.0f * iDeinterlacedPixels / fa->src.w / fa->src.h;
		if(fDeinterlacedPercent > mfd->fLogInterlacePercent && mfd->hLog != NULL)
		{
			char szLog[25];
			sprintf(szLog, "%li\t%5.1f\r\n", fa->pfsi->lCurrentSourceFrame, fDeinterlacedPercent);
			DWORD dw;
			WriteFile(mfd->hLog, szLog, strlen(szLog), &dw, NULL);
		}
	}


	return 0;
}

long ParamProc(FilterActivation *fa, const FilterFunctions *ff) {
    return FILTERPARAM_SWAP_BUFFERS;
}


BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
    MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

    switch(msg) {
        case WM_INITDIALOG:
            SetWindowLong(hdlg, DWL_USER, lParam);
            mfd = (MyFilterData *)lParam;

            CheckDlgButton(hdlg, IDC_SHOW_DEINTERLACED_AREAS_ONLY, mfd->bShowDeinterlacedAreaOnly ? BST_CHECKED:BST_UNCHECKED);
            CheckDlgButton(hdlg, IDC_BLEND, mfd->bBlend ? BST_CHECKED:BST_UNCHECKED);
            char str[10];
            sprintf(str, "%i", mfd->iThreshold);
            SetDlgItemText(hdlg, IDC_THRESHOLD, str);
            sprintf(str, "%i", mfd->iEdgeDetect);
            SetDlgItemText(hdlg, IDC_EDGE_DETECT, str);
            CheckDlgButton(hdlg, IDC_LOG, mfd->bLog ? BST_CHECKED:BST_UNCHECKED);
            sprintf(str, "%g", mfd->fLogInterlacePercent);
            SetDlgItemText(hdlg, IDC_INTERLACE_PERCENT, str);

            return TRUE;

        case WM_COMMAND:
            switch(LOWORD(wParam)) {
            case IDOK:
                mfd->bShowDeinterlacedAreaOnly = !!IsDlgButtonChecked(hdlg, IDC_SHOW_DEINTERLACED_AREAS_ONLY);
                mfd->bBlend = !!IsDlgButtonChecked(hdlg, IDC_BLEND);
                char str[20];
                GetDlgItemText(hdlg, IDC_THRESHOLD, str, 20);
                mfd->iThreshold = atoi(str);
                GetDlgItemText(hdlg, IDC_EDGE_DETECT, str, 20);
                mfd->iEdgeDetect = atoi(str);
                mfd->bLog = !!IsDlgButtonChecked(hdlg, IDC_LOG);
                GetDlgItemText(hdlg, IDC_INTERLACE_PERCENT, str, 20);
                mfd->fLogInterlacePercent = (float)atof(str);
                EndDialog(hdlg, 0);
                return TRUE;
            case IDCANCEL:
                EndDialog(hdlg, 1);
                return FALSE;
            }
            break;
    }

    return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd) {
    return DialogBoxParam(fa->filter->module->hInstModule,
            MAKEINTRESOURCE(IDD_FILTER_DEINTERLACE_AREA_BASED), hwnd,
            (DLGPROC)ConfigDlgProc, (LPARAM)fa->filter_data);
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str) {
	sprintf(str, " (thresh %i, edge %i, %s%s)",
		((MyFilterData *)(fa->filter_data))->iThreshold,
		((MyFilterData *)(fa->filter_data))->iEdgeDetect,
		((MyFilterData *)(fa->filter_data))->bBlend ? "blend" : "interp",
		((MyFilterData *)(fa->filter_data))->bShowDeinterlacedAreaOnly ? ", show areas" : "");
}
