/*
    Antiflicker Filter for VirtualDub -- removes temporal moire flickering.
	Copyright (C) 1999-2000 Donald A. Graft

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

	The author can be contacted at:
	Donald Graft
	neuron2@home.com.
*/

#include <windows.h>
#include <commctrl.h>
#include <stdio.h>
#include <math.h>
#include <crtdbg.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

///////////////////////////////////////////////////////////////////////////

long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int RunProc(const FilterActivation *fa, const FilterFunctions *ff);
int StartProc(FilterActivation *fa, const FilterFunctions *ff);
int EndProc(FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int MyInitProc(FilterActivation *fa, const FilterFunctions *ff);
int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

#define ANALYZING 0
#define PROCESSING 1

typedef struct MyFilterData {
	IFilterPreview		*ifp;
	FILE				*fp;
	int 				window;
	int					softening;
	int					mode;
	int					interlaced;
	char				filtpath[256];
	char				logpath[256];
    unsigned char		*old_data;
    long				size;
} MyFilterData;

bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	_snprintf(buf, buflen, "Config(%d, %d, %d, %d)",
		mfd->window,
		mfd->softening,
		mfd->mode,
		mfd->interlaced);

	return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->window		= argv[0].asInt();
	mfd->softening	= argv[1].asInt();
	mfd->mode		= !!argv[2].asInt();
	mfd->interlaced = !!argv[3].asInt();
}

ScriptFunctionDef func_defs[]={
	{ (ScriptFunctionPtr)ScriptConfig, "Config", "0iiii" },
	{ NULL },
};

CScriptObject script_obj={
	NULL, func_defs
};

struct FilterDefinition filterDef_tutorial = {

	NULL, NULL, NULL,		// next, prev, module
	"deflicker (1.1)",	// name
	"Remove temporal frame luminance variations (flicker).",
							// desc
	"Donald Graft", 		// maker
	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size

	MyInitProc,				// initProc
	NULL,					// deinitProc
	RunProc,				// runProc
	NULL,				// paramProc
	ConfigProc, 			// configProc
	StringProc, 			// stringProc
	StartProc,				// startProc
	EndProc,				// endProc

	&script_obj,			// script_obj
	FssProc,				// fssProc

};

int StartProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	char prog[256];
	char path[256];
	LPTSTR ptr;

	GetModuleFileName(NULL, prog, 255);
	GetFullPathName(prog, 255, path, &ptr);
	*ptr = 0;
	strcat(path, "log");
	strcpy(mfd->logpath, path);
	GetModuleFileName(NULL, prog, 255);
	GetFullPathName(prog, 255, path, &ptr);
	*ptr = 0;
	strcat(path, "filt");
	strcpy(mfd->filtpath, path);

	if (mfd->mode == ANALYZING)
	{
		/* Open file to store frame luminance values. */
		if ((mfd->fp = fopen(mfd->logpath, "w")) == NULL)
		{
			MessageBox(NULL, "Cannot open temp file", "Error", MB_OK);
			return 1;
		}
	}
	else
	{
	    /* Make buffer and clear it. */
		mfd->size = fa->src.Size();
		if ((mfd->old_data = new unsigned char [mfd->size]) == 0)
			return 1;
		memset (mfd->old_data, 0, mfd->size);
		/* Open filtered frame luminance file. */
		if ((mfd->fp = fopen(mfd->filtpath, "r")) == NULL)
		{
			MessageBox(NULL, "Cannot open filtered data file", "Error", MB_OK);
			return 1;
		}
	}
	return 0;
}

#define MAX_KERNEL 100
#define MAX_SOFTENING 31

int EndProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	unsigned long data[MAX_KERNEL], d;
	double kernel[MAX_KERNEL], filt;
	char tmp[80], *p;
	int i, frame;
	FILE *rfp, *wfp;
	bool first = TRUE;
	double FIR[] =
	{
		/* Requires window size of 25. */
		0.000520,
		-0.007933,
		0.007639,
		0.000683,
		0.042691,
		-0.119935,
		0.090397,
		0.052361,
		-0.097280,
		-0.015221,
		0.030142,
		0.175937,
		0.680000,
		0.175937,
		0.030142,
		-0.015221,
		-0.097280,
		0.052361,
		0.090397,
		-0.119935,
		0.042691,
		0.000683,
		0.007639,
		-0.007933,
		0.000520
	};

	if (mfd->mode == ANALYZING)
	{
		fclose(mfd->fp);
		rfp = fopen(mfd->logpath, "r");
		wfp = fopen(mfd->filtpath, "w");

		for (i = 0; i < mfd->window; i++)
		{
			/* This kernel makes a moving average filter. */
			kernel[i] = 1.0 / mfd->window;
			/* This one allows for an arbitrary FIR filter. */
//			kernel[i] = FIR[i];
		}
		while (fgets(tmp, 80, rfp) != NULL)
		{
			p = tmp;
			frame = atoi(p);
			while (*p++ != ',');
			for (i = 0; i < mfd->window - 1; i++) data[i] = data[i+1];
			data[mfd->window - 1] = atoi(p);
			if (first == TRUE)
			{
				first = FALSE;
				for (i = 0; i < mfd->window - 1; i++) data[i] = data[mfd->window - 1];
			}
			for (i = 0, filt = 0; i < mfd->window; i++) filt += kernel[i] * data[i];
			d = data[mfd->window - 1];
			fprintf(wfp, "%d,%f\n", frame, d == 0 ? 0.0 : filt/d);
		}
		fclose(rfp);
		fclose(wfp);
	}
	else
	{
		delete[] mfd->old_data;
		mfd->old_data = NULL;
 		fclose(mfd->fp);
	}

	return 0;
}

///////////////////////////////////////////////////////////////////////////

int RunProc(const FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	const PixDim	w = fa->src.w;
	const PixDim	h = fa->src.h;
	Pixel32 *src, *dst;
	int field, x, y;
	unsigned long lum_sum = 0;
	char tmp[80];
	double scale, s;
	int r, g, b, max;
	FilterStateInfo *pfsi = fa->pfsi;
    int current;

	char *p;
	int FIELDS = mfd->interlaced + 1;

	if (mfd->mode == ANALYZING)
	{
	  for (field = 0; field <= FIELDS - 1; field++)
	  {
		src = (Pixel32 *)((char *)fa->src.data + field * fa->src.pitch);
		for (y = field; y < h; y += FIELDS)
		{
			for (x = 0; x < w; x++)
			{
				lum_sum += (src[x] >> 16) & 0xff;
				lum_sum += (src[x] >> 8) & 0xff;
				lum_sum += src[x] & 0xff;
			}
			src	= (Pixel32 *)((char *)src + FIELDS * fa->src.pitch);
		}
		lum_sum /= (w * h * 3);
		sprintf(tmp, "%d,%d\n", FIELDS * pfsi->lCurrentFrame + field, lum_sum);
		fprintf(mfd->fp, "%s", tmp);
	  }
	}
	else
	{
		/* Luminance adjustment phase. */
		for (field = 0; field <= FIELDS - 1; field++)
		{
			if (fgets(tmp, 80, mfd->fp) == NULL)
			{
//				MessageBox(NULL, "Not enough frames in filtered data file", "Error", MB_OK);
				return 0;
			}
            /* Make sure current frame/field is correct. */
			p = tmp;
            current = atoi(p);
          	if (current != FIELDS * pfsi->lCurrentFrame + field)
            {
				if (current > FIELDS * pfsi->lCurrentFrame + field)
				{
					fclose(mfd->fp);
					mfd->fp = fopen(mfd->filtpath, "r");
				}
				while (current != FIELDS * pfsi->lCurrentFrame + field)
				{
					if (fgets(tmp, 80, mfd->fp) == NULL)
					{
//						MessageBox(NULL, "Not enough frames/fields in filtered data file", "Error", MB_OK);
						return 0;
					}
					p = tmp;
					current = atoi(p);
				}
			}

			/* Skip field number field. */
			while (*p++ != ',');
			scale = atof(p);
			src = (Pixel32 *)((char *)fa->src.data + field * fa->src.pitch);
			dst = (Pixel32 *)((char *)fa->dst.data + field * fa->dst.pitch);
			for (y = field; y < h; y += FIELDS)
			{
				for (x = 0; x < w; x++)
				{
					r = (src[x] >> 16) & 0xff;
					g = (src[x] >> 8) & 0xff;
					b = src[x] & 0xff;
					max = r;
					if (g > max) max = g;
					if (b > max) max = b;
					if (scale * max > 255) s = 255.0 / max;
					else s = scale;
					r = (int) (s * r);
					g = (int) (s * g);
					b = (int) (s * b);
					dst[x] = (r << 16) | (g << 8) | b;
				}
				src	= (Pixel32 *)((char *)src + FIELDS * fa->src.pitch);
				dst	= (Pixel32 *)((char *)dst + FIELDS * fa->dst.pitch);
			}
		}

		/* Temporal softening phase. Adapted from code by Steven Don. */
		unsigned char *src1, *src2;
		long diff, ofs, sum;

		if (mfd->softening == 0) return 0;

		ofs = mfd->size;
		src1 = (unsigned char *) fa->dst.data;
		src2 = (unsigned char *) mfd->old_data;
		do
		{
                        diff = abs(*src1 - *src2);
			if (diff < mfd->softening)
			{
				if (diff > (mfd->softening >> 1))
				{
					sum = *src1 + *src1 + *src2;
					*src2 = sum / 3;
				}
			}
			else
			{
				*src2 = *src1;
			}
			*src1 = *src2;
			src1++; src2++;
		} while (--ofs);
	}
	return 0;
}

extern "C" __declspec(dllexport) int VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" __declspec(dllexport) void VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_tutorial;

int VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
	if ((fd_tutorial = ff->addFilter(fm, &filterDef_tutorial, sizeof(FilterDefinition))) == 0)
		return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

	return 0;
}

void VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
	ff->removeFilter(fd_tutorial);
}

int MyInitProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->window = 25;
	mfd->softening = 18;
	mfd->mode = ANALYZING;
	mfd->interlaced = 0;

	return 0;
}

BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
	MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

	switch(msg) {
		case WM_INITDIALOG:
			SetWindowLong(hdlg, DWL_USER, lParam);
			mfd = (MyFilterData *)lParam;

			CheckDlgButton(hdlg, IDC_ANALYZING, mfd->mode == ANALYZING ? BST_CHECKED : BST_UNCHECKED);
			CheckDlgButton(hdlg, IDC_PROCESSING, mfd->mode == PROCESSING ? BST_CHECKED : BST_UNCHECKED);
			SetDlgItemInt(hdlg, IDC_WINDOW, mfd->window, FALSE);
			SetDlgItemInt(hdlg, IDC_SOFTENING, mfd->softening, FALSE);
			CheckDlgButton(hdlg, IDC_FIELD, mfd->interlaced == 1 ? BST_CHECKED : BST_UNCHECKED);

			return TRUE;

		case WM_COMMAND:
			switch(LOWORD(wParam))
			{
			case IDOK:
				EndDialog(hdlg, 0);
				return TRUE;
			case IDHELP:
				{
				char prog[256];
				char path[256];
				LPTSTR ptr;
				GetModuleFileName(NULL, prog, 255);
				GetFullPathName(prog, 255, path, &ptr);
				*ptr = 0;
				strcat(path, "plugins\\Deflick.txt");
				strcpy(prog, "Notepad ");
				strcat(prog, path);
				WinExec(prog, SW_SHOW);
				return TRUE;
				}
			case IDCANCEL:
				EndDialog(hdlg, 1);
				return TRUE;
			case IDC_WINDOW:
				mfd->window = GetDlgItemInt(hdlg, IDC_WINDOW, &mfd->window, FALSE);
				if (mfd->window > MAX_KERNEL)
				{
					mfd->window = MAX_KERNEL;
					SetDlgItemInt(hdlg, IDC_WINDOW, mfd->window, FALSE);
				}
				break;
			case IDC_SOFTENING:
				mfd->softening = GetDlgItemInt(hdlg, IDC_SOFTENING, &mfd->softening, FALSE);
				if (mfd->softening > MAX_SOFTENING)
				{
					mfd->softening = MAX_SOFTENING;
					SetDlgItemInt(hdlg, IDC_SOFTENING, mfd->softening, FALSE);
				}
				break;
			case IDC_ANALYZING:
				mfd->mode = ANALYZING;
				CheckDlgButton(hdlg, IDC_ANALYZING, BST_CHECKED);
				break;
			case IDC_PROCESSING:
				mfd->mode = PROCESSING;
				CheckDlgButton(hdlg, IDC_PROCESSING, BST_CHECKED);
				break;
			case IDC_FIELD:
				mfd->interlaced = !mfd->interlaced;
				break;
			}
			break;
	}

	return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd)
{
	MyFilterData *mfd = (MyFilterData *) fa->filter_data;
	MyFilterData mfd_old = *mfd;
	int ret;
    extern void Doit(void);

	if (DialogBoxParam(fa->filter->module->hInstModule,
			MAKEINTRESOURCE(IDD_FILTER), hwnd,
			(DLGPROC) ConfigDlgProc, (LPARAM) mfd))
	{
		*mfd = mfd_old;
		ret = TRUE;
	}
	else
	{
		ret = FALSE;
	}
	return(ret);
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	sprintf(str, " (wind %d, soft %d, mode %s, inter %s)",
				mfd->window, mfd->softening, mfd->mode == ANALYZING ? "prepare" : "process",
				mfd->interlaced ? "yes" : "no");
}

