/*
    Antiflicker Filter for VirtualDub -- removes temporal moire flickering.
	Copyright (C) 1999-2006 Donald A. Graft

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

	The author can be contacted at:
	Donald Graft
	neuron2@comcast.net
*/

#include <windows.h>
#include <commctrl.h>
#include <stdio.h>
#include <math.h>
#include <crtdbg.h>
#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"
#include "resource.h"
#include "filter.h"

///////////////////////////////////////////////////////////////////////////

long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int RunProc(const FilterActivation *fa, const FilterFunctions *ff);
int StartProc(FilterActivation *fa, const FilterFunctions *ff);
int EndProc(FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int InitProc(FilterActivation *fa, const FilterFunctions *ff);
int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

#define MAX_WINDOW 100
#define MAX_SOFTENING 31

typedef struct FilterData
{
	IFilterPreview		*ifp;
	int 				window;
	int					softening;
	int					interlaced;
	int					scene_change_threshold;
	int					show_scene_changes;
    unsigned char		*old_data;
    long				size;
	unsigned long		lumnData[MAX_WINDOW];
} FilterData;

bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen)
{
	FilterData *mfd = (FilterData *)fa->filter_data;

	_snprintf(buf, buflen, "Config(%d, %d, %d, %d, %d)",
		mfd->window,
		mfd->softening,
		mfd->interlaced,
		mfd->scene_change_threshold,
		mfd->show_scene_changes);

	return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc)
{
	FilterActivation *fa = (FilterActivation *)lpVoid;
	FilterData *mfd = (FilterData *)fa->filter_data;

	mfd->window		= argv[0].asInt();
	mfd->softening	= argv[1].asInt();
	mfd->interlaced = !!argv[2].asInt();
	mfd->scene_change_threshold = argv[3].asInt();
	mfd->show_scene_changes = !!argv[4].asInt();
}

ScriptFunctionDef func_defs[] =
{
	{ (ScriptFunctionPtr)ScriptConfig, "Config", "0iiiii" },
	{ NULL },
};

CScriptObject script_obj =
{
	NULL, func_defs
};

struct FilterDefinition filterDef_tutorial =
{

	NULL, NULL, NULL,		// next, prev, module
	"deflicker (1.3b1)",		// name
	"Remove temporal frame luminance variations (flicker).",
							// desc
	"Donald Graft", 		// maker
	NULL,					// private_data
	sizeof(FilterData),		// inst_data_size
	InitProc,				// initProc
	NULL,					// deinitProc
	RunProc,				// runProc
	NULL,					// paramProc
	ConfigProc, 			// configProc
	StringProc, 			// stringProc
	StartProc,				// startProc
	EndProc,				// endProc
	&script_obj,			// script_obj
	FssProc,				// fssProc

};

int StartProc(FilterActivation *fa, const FilterFunctions *ff)
{
	FilterData *mfd = (FilterData *)fa->filter_data;

	// first frame infication.
	mfd->lumnData[0] = 256;
	/* Make buffer and clear it. */
	mfd->size = fa->src.Size();
	if ((mfd->old_data = new unsigned char [mfd->size]) == 0)
		return 1;
	memset (mfd->old_data, 0, mfd->size);
	return 0;
}

int EndProc(FilterActivation *fa, const FilterFunctions *ff)
{
	FilterData *mfd = (FilterData *)fa->filter_data;

	delete[] mfd->old_data;
	mfd->old_data = NULL;
	return 0;
}

int RunProc(const FilterActivation *fa, const FilterFunctions *ff)
{
	FilterData *mfd = (FilterData *)fa->filter_data;
	const PixDim w = fa->src.w;
	const PixDim h = fa->src.h;
	Pixel32 *src, *dst;
	int i, field, x, y, FIELDS;
	unsigned long lum_sum = 0;
	double scale, s;
	int r, g, b, max, scene_change;

	FIELDS = mfd->interlaced + 1;
	for (field = 0; field <= FIELDS - 1; field++)
	{
		// Calculate the luminance of the current frame.
		src = (Pixel32 *)((char *)fa->src.data + field * fa->src.pitch);
		for (y = field; y < h; y += FIELDS)
		{
			for (x = 0; x < w; x++)
			{
				lum_sum += (src[x] >> 16) & 0xff;
				lum_sum += (src[x] >> 8) & 0xff;
				lum_sum += src[x] & 0xff;
			}
			src	= (Pixel32 *)((char *)src + FIELDS * fa->src.pitch);
		}
		lum_sum /= (w * h * 3);

		// Do scene change processing.
		scene_change = 0;
		if (mfd->scene_change_threshold < 256 &&
			mfd->lumnData[0] != 256 &&
			abs((int)lum_sum - (int)mfd->lumnData[mfd->window - 1]) >= mfd->scene_change_threshold)
		{
			mfd->lumnData[0] = 256;
			scene_change = 1;
		}

		// Calculate the adjustment factor for the current frame.
		// The adjustment factor is determined from a moving average
		// of the luminances of the past frames in the window.
		scale = 1.0;
		if (mfd->lumnData[0] > 255 )
		{
			for (i = 0; i < mfd->window; i++) 
				mfd->lumnData[i] = lum_sum;
		}
		else
		{
			for (i = 0; i < mfd->window - 1; i++) 
				mfd->lumnData[i] = mfd->lumnData[i+1];
			mfd->lumnData[mfd->window - 1] = lum_sum;
			if (lum_sum > 0)
			{
				scale = 1.0f/lum_sum;
				double filt = 0;
				for (int i = 0; i < mfd->window; i++) 
					filt += (float) mfd->lumnData[i] / mfd->window;
				scale *= filt;
			}
		}

		// Adjust the current frame.
		src = (Pixel32 *)((char *)fa->src.data + field * fa->src.pitch);
		dst = (Pixel32 *)((char *)fa->dst.data + field * fa->dst.pitch);
		for (y = field; y < h; y += FIELDS)
		{
			for (x = 0; x < w; x++)
			{
				if (mfd->show_scene_changes && scene_change)
				{
					dst[x] = 0xff0000;
				}
				else
				{
					r = (src[x] >> 16) & 0xff;
					g = (src[x] >> 8) & 0xff;
					b = src[x] & 0xff;
					max = r;
					if (g > max) max = g;
					if (b > max) max = b;
					if (scale * max > 255) s = 255.0 / max;
					else s = scale;
					r = (int) (s * r);
					g = (int) (s * g);
					b = (int) (s * b);
					dst[x] = (r << 16) | (g << 8) | b;
				}
			}
			src	= (Pixel32 *)((char *)src + FIELDS * fa->src.pitch);
			dst	= (Pixel32 *)((char *)dst + FIELDS * fa->dst.pitch);
		}
	}

	/* Temporal softening phase. Adapted from code by Steven Don. */
	unsigned char *src1, *src2;
	long diff, ofs, sum;

	if (scene_change || mfd->softening == 0) return 0;

	ofs = mfd->size;
	src1 = (unsigned char *) fa->dst.data;
	src2 = (unsigned char *) mfd->old_data;
	do
	{
                    diff = abs(*src1 - *src2);
		if (diff < mfd->softening)
		{
			if (diff > (mfd->softening >> 1))
			{
				sum = *src1 + *src1 + *src2;
				*src2 = sum / 3;
			}
		}
		else
		{
			*src2 = *src1;
		}
		*src1 = *src2;
		src1++; src2++;
	} while (--ofs);
	return 0;
}

extern "C" __declspec(dllexport) int VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" __declspec(dllexport) void VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_tutorial;

int VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat)
{
	if ((fd_tutorial = ff->addFilter(fm, &filterDef_tutorial, sizeof(FilterDefinition))) == 0)
		return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

	return 0;
}

void VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff)
{
	ff->removeFilter(fd_tutorial);
}

int InitProc(FilterActivation *fa, const FilterFunctions *ff) {
	FilterData *mfd = (FilterData *)fa->filter_data;

	mfd->window = 8;
	mfd->softening = 10;
	mfd->interlaced = 0;
	mfd->scene_change_threshold = 256;
	mfd->show_scene_changes = 0;

	return 0;
}

BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
	FilterData *mfd = (FilterData *)GetWindowLong(hdlg, DWL_USER);

	switch(msg) {
		case WM_INITDIALOG:
			SetWindowLong(hdlg, DWL_USER, lParam);
			mfd = (FilterData *)lParam;

			SetDlgItemInt(hdlg, IDC_WINDOW, mfd->window, FALSE);
			SetDlgItemInt(hdlg, IDC_SOFTENING, mfd->softening, FALSE);
			CheckDlgButton(hdlg, IDC_FIELD, mfd->interlaced == 1 ? BST_CHECKED : BST_UNCHECKED);
			SetDlgItemInt(hdlg, IDC_SCT, mfd->scene_change_threshold, FALSE);
			CheckDlgButton(hdlg, IDC_SHOW, mfd->show_scene_changes == 1 ? BST_CHECKED : BST_UNCHECKED);

			return TRUE;

		case WM_COMMAND:
			switch(LOWORD(wParam))
			{
			case IDOK:
				EndDialog(hdlg, 0);
				return TRUE;
			case IDHELP:
				{
				char prog[256];
				char path[256];
				LPTSTR ptr;

				GetModuleFileName(NULL, prog, 255);
				GetFullPathName(prog, 255, path, &ptr);
				*ptr = 0;
				strcat(path, "plugins\\Deflick.html");
				ShellExecute(hdlg, "open", path, NULL, NULL, SW_SHOWNORMAL);
				return TRUE;
				}
			case IDCANCEL:
				EndDialog(hdlg, 1);
				return TRUE;
			case IDC_WINDOW:
				mfd->window = GetDlgItemInt(hdlg, IDC_WINDOW, &mfd->window, FALSE);
				if (mfd->window > MAX_WINDOW)
				{
					mfd->window = MAX_WINDOW;
					SetDlgItemInt(hdlg, IDC_WINDOW, mfd->window, FALSE);
				}
				break;
			case IDC_SOFTENING:
				mfd->softening = GetDlgItemInt(hdlg, IDC_SOFTENING, &mfd->softening, FALSE);
				if (mfd->softening > MAX_SOFTENING)
				{
					mfd->softening = MAX_SOFTENING;
					SetDlgItemInt(hdlg, IDC_SOFTENING, mfd->softening, FALSE);
				}
				break;
			case IDC_FIELD:
				mfd->interlaced = !mfd->interlaced;
				break;
			case IDC_SCT:
				mfd->scene_change_threshold = GetDlgItemInt(hdlg, IDC_SCT, &mfd->scene_change_threshold, FALSE);
				if (mfd->scene_change_threshold > 256)
				{
					mfd->scene_change_threshold = 256;
					SetDlgItemInt(hdlg, IDC_SCT, mfd->scene_change_threshold, FALSE);
				}
				break;
			case IDC_SHOW:
				mfd->show_scene_changes = !mfd->show_scene_changes;
				break;
			}
			break;
	}

	return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd)
{
	FilterData *mfd = (FilterData *) fa->filter_data;
	FilterData mfd_old = *mfd;
	int ret;
    extern void Doit(void);

	if (DialogBoxParam(fa->filter->module->hInstModule,
			MAKEINTRESOURCE(IDD_FILTER), hwnd,
			(DLGPROC) ConfigDlgProc, (LPARAM) mfd))
	{
		*mfd = mfd_old;
		ret = TRUE;
	}
	else
	{
		ret = FALSE;
	}
	return(ret);
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str)
{
	FilterData *mfd = (FilterData *)fa->filter_data;

	sprintf(str, " (wind %d, soft %d, inter %s, sct %d, show %d)",
			mfd->window, mfd->softening, mfd->interlaced ? "yes" : "no", mfd->scene_change_threshold, mfd->show_scene_changes);
}

