#include <windows.h>
#include <stdio.h>
#include <math.h>
#include <commctrl.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"
/*
*******************************************************************************

	Virtualdub filter to correct:

			colours brightness contrast and gamma

	-----------------------------------------------------------


		Copyright (C) 2002-2004 Alessandro Malanca, 

		http://web.tiscali.it/minomala/
 
	-----------------------------------------------------------

	This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

*******************************************************************************
*/

int  coloursRunProc ( const FilterActivation *fa, const FilterFunctions *ff);
int  coloursStartProc ( FilterActivation *fa, const FilterFunctions *ff);
int  coloursEndProc ( FilterActivation *fa, const FilterFunctions *ff);
long coloursParamProc ( FilterActivation *fa, const FilterFunctions *ff);
int  coloursConfigProc ( FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void coloursStringProc ( const FilterActivation *fa, const FilterFunctions *ff, char *str);
void coloursScriptConfig ( IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool coloursFssProc ( FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);
int  coloursInitProc ( FilterActivation *fa, const FilterFunctions *ff);


///////////////////////////////////////////////////////////////////////////



#define MAX_CONTRAST	200	
#define MAX_BRIGHTNESS	200
#define MAX_GAMMA		200
#define MAX_BALANCE		255

typedef enum Channel { Global, Red, Green, Blue } Channel;
typedef enum ChanParam { Brightness, Contrast, Gamma, Balance } ChanParam;
#define Channel_Lenght (Blue+1)
#define ChanParam_Lenght (Balance+1)

const int MaxControl[ChanParam_Lenght] = {MAX_BRIGHTNESS, MAX_CONTRAST, MAX_GAMMA, MAX_BALANCE};
const int MinControl[ChanParam_Lenght] = {-MAX_BRIGHTNESS, -MAX_CONTRAST, -MAX_GAMMA, 0};



const char * ChannelName[4] = {"Global", "Red", "Green", "Blue"};

typedef struct ChannelControl_s {
	int		value [ChanParam_Lenght];
} ChannelControl [Channel_Lenght];

typedef struct FilterParameters {

	ChannelControl  chn;

} FilterParameters;

#define BALANCE_DEFAULT 0

typedef Channel ShowingSelection;

class FrameHistogram {

#define NUM_SAMPLES (256/2)

public:

void setShowingSelection ( ShowingSelection s ) {

	showingSelection = s;
}

Channel getShowingSelection ( ) {
	return showingSelection;
}

Channel nextShowingSelection() {
	if (showingSelection == Blue) {
		showingSelection = Global;
	} else {
		showingSelection = (ShowingSelection) ((int)(showingSelection)+1);
	}
	return showingSelection;
}


int getMaxHistoValue() {
	return maxHistoValue;
}


void  paintHisto (HWND hdlg) {

	RECT	rHistoCtrl;

	HDC hdc;
	PAINTSTRUCT ps;

	HPEN whitePen = (HPEN)GetStockObject(WHITE_PEN);
	HPEN blackPen = (HPEN)GetStockObject(BLACK_PEN);

	hdc = BeginPaint(GetDlgItem(hdlg,IDC_HISTO), &ps);

	GetClientRect(GetDlgItem(hdlg, IDC_HISTO), &rHistoCtrl);
	paintBackGr(hdc,rHistoCtrl);

	RECT rHisto;
	int lineSize = calcPaintArea (rHisto, rHistoCtrl );
	FillRect(hdc, &rHisto, (HBRUSH)GetStockObject(LTGRAY_BRUSH));

	HPEN pen;

	switch(showingSelection) {
		case Red:	
			pen = CreatePen( PS_SOLID, 1, RGB(200,0,0));
			break;
		case Blue:	
			pen = CreatePen( PS_SOLID, 1, RGB(0,0,200));
			break;
		case Green:	
			pen = CreatePen( PS_SOLID, 1, RGB(0,240,0));
			break;
		default:
			pen = whitePen;
	}
	
	int xPos = rHisto.left;

	SelectObject( hdc, pen);

	for ( int j=0; j<NUM_SAMPLES; j++ ) {

		if ( j == histoMediaBalance ) {
			SelectObject( hdc, blackPen);
		} else {
			SelectObject( hdc, pen);
		}

		int h = scaleValue(histogram[j],rHisto.bottom-rHisto.top);
		for ( int l=0; l<lineSize; l++) {
			MoveToEx(hdc, xPos+l, rHisto.bottom, NULL);		
			LineTo(hdc,	xPos+l, rHisto.bottom-h);
		}
		xPos += lineSize;
	}

	DeleteObject(pen); 
	EndPaint(hdlg, &ps);
	return;
}


void invalidate(HWND hdlg) {
	InvalidateRgn(hdlg,NULL,FALSE);
}

void calculateHisto( const FilterActivation *fa, int mediaBalance ) {

	int w = fa->dst.w;
	int h = fa->dst.h;

	Pixel32 *dst =  (Pixel32 *)fa->dst.data;

	long summs[256]; 

	for (int j=0; j<256; j++) { summs[j] = 0; }

    int y = h;
	do {int x = w;
		do {
			Pixel32 pixel = *dst++;

			int v;

			switch (showingSelection) {
				case Global:
					v = ((pixel & 0xFF0000)>>16) +
						((pixel & 0x00FF00)>>8) +
						(pixel & 0x0000FF);
					v /= 3;
					break;
				case Red:
					v = (pixel & 0xFF0000)>>16;
					break;
				case Green:
					v = (pixel & 0x00FF00)>>8;
					break;
				case Blue:
					v = (pixel & 0x0000FF);
					break;
			}
			summs[v]++;	
		} while(--x);
		dst = (Pixel32 *)((char *)dst + fa->dst.modulo);
    } while(--y);

	double max = 0; int next = 0;
//	int step = 256/NUM_SAMPLES;

/*
	const int step[] =  { 
		4, 3, 3, 3, 3, 3, 3, 3, 3, 3, 4, 3, 3, 3, 3, 3, 3, 3, 3, 4,
		4, 3, 3, 3, 3, 3, 3, 3, 3, 3, 4, 3, 3, 3, 3, 3, 3, 3, 3, 4,
		4, 3, 3, 3, 3, 3, 3, 3, 3, 3, 4, 3, 3, 3, 3, 3, 3, 3, 3, 4,
		4, 3, 3, 3, 3, 3, 3, 3, 3, 3, 4, 3, 3, 3, 3, 3, 3, 3, 3, 4,
	};
*/
	int step=2; /* NUM_SAPLES = 256/2 */

	for( j=0; j<NUM_SAMPLES; j++) {
		histogram[j] = 0;
		for ( int k=0; k < step /*step[j]*/ ; k++ ) {
			if ( next == mediaBalance ) {
				histoMediaBalance = j;
			}
			histogram[j] += summs[next++];
			if (histogram[j] > max) max = histogram[j];
		}
	}

	for( j=0; j<NUM_SAMPLES; j++) {
		histogram[j] = histogram[j] / max; 
	}

	maxHistoValue = (long)max;
	// in 0,0 .. 1,0
}

private:

	double		histogram[NUM_SAMPLES];
	ShowingSelection	showingSelection;
	long		maxHistoValue;
	long		histoMediaBalance;

void paintBackGr( HDC hdc, RECT r ) {

	FillRect(hdc, &r, (HBRUSH)GetStockObject(LTGRAY_BRUSH));

	HPEN blackPen = (HPEN)GetStockObject(BLACK_PEN);
	SelectObject( hdc, blackPen);

	MoveToEx(hdc, r.left, r.top, NULL);
	LineTo(hdc,	r.right-1,r.top);
	LineTo(hdc,	r.right-1,r.bottom-1);
	LineTo(hdc,	r.left,r.bottom-1);
	LineTo(hdc,	r.left,r.top);

}

int calcPaintArea ( RECT & res, RECT rCtrl ) {

	int step = (rCtrl.right-rCtrl.left) / NUM_SAMPLES;
	int left = (rCtrl.right - rCtrl.left - (step * NUM_SAMPLES)) / 2;
	res.left = rCtrl.left+left;
	res.right = rCtrl.left+left+step*NUM_SAMPLES;
	res.bottom = rCtrl.bottom-4;
	res.top = rCtrl.top+4;
	return step;

}

int scaleValue(double histo, int max) {

	// histo in 0..1

	if ( histo <= 0 ) return 0;
	if ( histo >= 1 ) return max;
	return (int) ( histo * (double) max );

}

}; // FrameHistogram


typedef struct PixelAdjust {

	Pixel32 red[256];
	Pixel32 green[256];
	Pixel32 blue[256];

} PixelAdjust;


typedef struct MyFilterData {
	
	bool				disableFilter;
	FilterParameters	par,save;
	IFilterPreview		*ifp;
	FrameHistogram      frameHistogram;
	bool				displayingPanel;
	PixelAdjust			*pixelAdjust; 

} MyFilterData;


ScriptFunctionDef colours_func_defs[]={
    { (ScriptFunctionPtr)coloursScriptConfig, "Config", "0iiiiiiiiiiiiiiii" }, 
    { NULL },
};

CScriptObject colours_obj = {
    NULL, colours_func_defs
};

struct FilterDefinition filterDef_colours = {

    NULL, NULL, NULL,			// next, prev, module
    "brightness-contrast-gamma [1.0]",	// name
    "rgb colours brightness contrast and gamma correction",	// desc
    "Alessandro Malanca",		// maker
    NULL,						// private_data
    sizeof(MyFilterData),		// inst_data_size
    coloursInitProc,			// initProc
    NULL,						// deinitProc
    coloursRunProc,				// runProc
    NULL,						// ParamProc
    coloursConfigProc,			// ConfigProc
    coloursStringProc,			// stringProc
    coloursStartProc,			// startProc
    coloursEndProc,				// endProc
    &colours_obj,				// script_obj
    coloursFssProc,				// fssProc

};

///////////////////////////////////////////////////////////////////////////

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_colours;

///////////////////////////////////////////////////////////////////////////

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
    if (!(fd_colours = ff->addFilter(fm, &filterDef_colours, sizeof(FilterDefinition))))
        return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

    return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
    ff->removeFilter(fd_colours);
}

///////////////////////////////////////////////////////////////////////////


int coloursInitProc(FilterActivation *fa, const FilterFunctions *ff) {

	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	mfd->displayingPanel = false;
	for (Channel s = Global; s <= Blue; s = (Channel)(s+1) ) {
		mfd->par.chn[s].value[Balance] = BALANCE_DEFAULT;
	}
//	mfd->par.contrastBalance = 120;
	mfd->frameHistogram.setShowingSelection(Global);
	return 0;

}




double gammaC(double r, double gamma_corr) {
	//	x' =  ((x/255) ^ kk) * 255; 
	// per kk == 1 invariato
	return (pow ((r/255.0), gamma_corr ) * 255.0 );
}


double gammaPar ( long gm ) {

	if ( gm >= 0)  {
		return 1.0 - ( ((double)gm) / 200.0 ); // 0..2
	} else {
		return 1.0 - ( ((double)gm) / 100.0 ); // 0..2
	}

}

PixelAdjust * prepareToAdjust(FilterParameters par) {

	long r,g,b;
	
	PixelAdjust * res;

	res = new PixelAdjust();

	for (int i=0; i<256; i++) {

		r = g = b = i;

		// ------------------------------
		// -----  gamma correction   ----
		// ------------------------------

		double gamma_corr = gammaPar ( par.chn[Global].value[Gamma]);

		r = (long) gammaC (r, gamma_corr);
		g = (long) gammaC (g, gamma_corr);
		b = (long) gammaC (b, gamma_corr);

		r = (long) gammaC (r, gammaPar(par.chn[Red].value[Gamma]));
		g = (long) gammaC (g, gammaPar(par.chn[Green].value[Gamma]));
		b = (long) gammaC (b, gammaPar(par.chn[Blue].value[Gamma]));


		// -------------------------------
		// --- adjust colours contrast ---
		// -------------------------------
/*		
		was integer (same!)
		===========

		long glb_contr = 256 + par.chn[Global].value[Contrast] ;

		r = ( ( (r - balance) * glb_contr ) / 256 ) + balance;
		g = ( ( (g - balance) * glb_contr ) / 256 ) + balance;
		b = ( ( (b - balance) * glb_contr ) / 256 ) + balance;

		long red_contr = 256 + par.chn[Red].value[Contrast];
		long gre_contr = 256 + par.chn[Green].value[Contrast];
		long blu_contr = 256 + par.chn[Blue].value[Contrast];

		r = ( ( (r - par.chn[Red].value[Balance])   * red_contr ) / 256) + par.chn[Red].value[Balance];
		g = ( ( (g - par.chn[Green].value[Balance]) * gre_contr ) / 256) + par.chn[Green].value[Balance];
		b = ( ( (b - par.chn[Blue].value[Balance])  * blu_contr ) / 256) + par.chn[Blue].value[Balance];

*/
		double globalControl = 1.0 + par.chn[Global].value[Contrast] / 255.0;

		long balance = par.chn[Global].value[Balance];


		r = (long)(( (double)(r - balance) * globalControl ) + balance);
		g = (long)(( (double)(g - balance) * globalControl ) + balance);
		b = (long)(( (double)(b - balance) * globalControl ) + balance);

		double redContr = 1.0 + par.chn[Red].value[Contrast]   / 255.0;
		double greContr = 1.0 + par.chn[Green].value[Contrast] / 255.0;
		double bluContr = 1.0 + par.chn[Blue].value[Contrast]  / 255.0;
 
		r = (long)(( (double)(r - par.chn[Red].value[Balance])   * redContr ) + par.chn[Red].value[Balance]);
		g = (long)(( (double)(g - par.chn[Green].value[Balance]) * greContr ) + par.chn[Green].value[Balance]);
		b = (long)(( (double)(b - par.chn[Blue].value[Balance])  * bluContr ) + par.chn[Blue].value[Balance]);

		// -----------------------------------
		// ---  adjust colours brightness  ---
		// -----------------------------------

		r += par.chn[Global].value[Brightness] + par.chn[Red].value[Brightness];
		g += par.chn[Global].value[Brightness] + par.chn[Green].value[Brightness];
		b += par.chn[Global].value[Brightness] + par.chn[Blue].value[Brightness];

		if (r < 0) r=0; else if (r > 255) r=255;
		if (g < 0) g=0; else if (g > 255) g=255;
		if (b < 0) b=0; else if (b > 255) b=255;

		res->red[i]	  = r<<16;
		res->green[i] = g<<8;
		res->blue[i]  = b;

	}

	return res;

}


int coloursStartProc(FilterActivation *fa, const FilterFunctions *ff) {

    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->pixelAdjust = prepareToAdjust(mfd->par);

    return 0;

}


int coloursRunProc(const FilterActivation *fa, const FilterFunctions *ff) {

    MyFilterData *mfd = (MyFilterData *)fa->filter_data;
    PixDim w, h;
    Pixel32 *src, *dst;

	if (mfd->disableFilter) {
		fa->dst.data = fa->src.data;
		int s = mfd->frameHistogram.getShowingSelection();
		mfd->frameHistogram.calculateHisto(fa, mfd->par.chn[s].value[Balance]);
		return 0;
	}

    src = (Pixel32 *)fa->src.data;
    dst = (Pixel32 *)fa->dst.data;

	Pixel32 * red_pix = mfd->pixelAdjust->red;
	Pixel32 * gre_pix = mfd->pixelAdjust->green;
	Pixel32 * blu_pix = mfd->pixelAdjust->blue;

    h = fa->src.h;
    do { 
		w = fa->src.w;
		do {

			// adjust colors brightness and contrast 

			*dst++ =  red_pix [ (*src & 0xFF0000) >> 16 ] 
					| gre_pix [ (*src & 0x00FF00) >> 8 ] 
					| blu_pix [ (*src & 0x0000FF) ];

			src++;

		} while(--w);

	    src = (Pixel32 *)((char *)src + fa->src.modulo);
		dst = (Pixel32 *)((char *)dst + fa->dst.modulo);

    } while(--h);

	if (mfd->displayingPanel) {
		int s = mfd->frameHistogram.getShowingSelection();
		mfd->frameHistogram.calculateHisto(fa, mfd->par.chn[s].value[Balance]);
	}

    return 0;
}


int coloursEndProc(FilterActivation *fa, const FilterFunctions *ff) {
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	if ( mfd->pixelAdjust != NULL ) {
		delete mfd->pixelAdjust;
		mfd->pixelAdjust=NULL;
	}

    return 0;
}


const int IDC [Channel_Lenght][ChanParam_Lenght]   = { 
	{IDC_BRIGHTNESS_VALUE, IDC_CONTRAST_VALUE, IDC_GAMMA_VALUE, IDC_MEDIA_VALUE},
	{IDC_BRIGHT_RED_VALUE,IDC_CONTR_RED_VALUE ,IDC_GAMMA_RED_VALUE , IDC_MEDIA_RED_VALUE},
	{IDC_BRIGHT_GREEN_VALUE,IDC_CONTR_GREEN_VALUE ,IDC_GAMMA_GREEN_VALUE ,IDC_MEDIA_GREEN_VALUE },
	{IDC_BRIGHT_BLUE_VALUE, IDC_CONTR_BLUE_VALUE,IDC_GAMMA_BLUE_VALUE ,IDC_MEDIA_BLUE_VALUE },
};



Channel lookUpChannel ( int find) {
	for ( int i=0; i<Channel_Lenght;i++) 
		for (int v=0; v < ChanParam_Lenght ; v++) {
			if (find == IDC[i][v] ) return (Channel)i;
	}
	return Global;
};

ChanParam lookUpParam ( int find ) {
	for ( int i=0; i<Channel_Lenght;i++) 
		for (int v=0; v < ChanParam_Lenght ; v++) {
			if (find == IDC[i][v] ) return (ChanParam)v;
	}
	return Brightness;
}


void setSliderRange(HWND hdlg, MyFilterData *mfd) {

	HWND hWnd;

	int curr = mfd->frameHistogram.getShowingSelection();

	hWnd = GetDlgItem(hdlg,IDC_CONTRAST_SLIDER);
	SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(-MAX_CONTRAST, MAX_CONTRAST));

	hWnd = GetDlgItem(hdlg,IDC_BRIGHTNESS_SLIDER);
	SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(-MAX_BRIGHTNESS, MAX_BRIGHTNESS));

	hWnd = GetDlgItem(hdlg,IDC_GAMMA_SLIDER);
	SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(-MAX_GAMMA, MAX_GAMMA));

}

void setSliderPositions(HWND hdlg, MyFilterData *mfd) {

	HWND hWnd;

	int curr = mfd->frameHistogram.getShowingSelection();

	hWnd = GetDlgItem(hdlg,IDC_CONTRAST_SLIDER);
	SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->par.chn[curr].value[Contrast] );

	hWnd = GetDlgItem(hdlg,IDC_BRIGHTNESS_SLIDER);
	SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->par.chn[curr].value[Brightness]);

	hWnd = GetDlgItem(hdlg,IDC_GAMMA_SLIDER);
	SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->par.chn[curr].value[Gamma]);


}

void setDialogFeedbackValues (HWND hdlg, MyFilterData *mfd) {

	int s = mfd->frameHistogram.getShowingSelection();

	SetDlgItemText(hdlg, IDC_SHOW_SELECTION, ChannelName[s]);

	SetDlgItemInt(hdlg, IDC_CONTRAST_BALANCE, mfd->par.chn[s].value[Balance], TRUE);

	for ( int i=0; i<Channel_Lenght;i++) 
		for (int v=0; v < ChanParam_Lenght ; v++) {
			SetDlgItemInt(hdlg, IDC[i][v], mfd->par.chn[i].value[v], TRUE);
	}
	setSliderPositions( hdlg,  mfd);

}


void redoSystem( MyFilterData * mfd, HWND hdlg) {		

	mfd->ifp->UndoSystem();
	mfd->ifp->RedoSystem();
//	setDialogFeedbackValues(hdlg,mfd);
	mfd->frameHistogram.invalidate(hdlg);
}

void getHorSliderValues(MyFilterData *mfd,HWND hdlg) {

	int s = mfd->frameHistogram.getShowingSelection();

	int brightness = SendMessage(GetDlgItem(hdlg, IDC_BRIGHTNESS_SLIDER), TBM_GETPOS, 0, 0);
	if ( brightness != mfd->par.chn[s].value[Brightness] ) {
		mfd->par.chn[s].value[Brightness] = brightness;
		setDialogFeedbackValues(hdlg,mfd);
		redoSystem( mfd, hdlg );
		return ;
	}

	int contrast = SendMessage(GetDlgItem(hdlg, IDC_CONTRAST_SLIDER), TBM_GETPOS, 0, 0);
	if ( contrast != mfd->par.chn[s].value[Contrast] ) {
		mfd->par.chn[s].value[Contrast] = contrast;
		setDialogFeedbackValues(hdlg,mfd);
		redoSystem( mfd, hdlg );
		return ;
	}
	
	int gamma = SendMessage(GetDlgItem(hdlg, IDC_GAMMA_SLIDER), TBM_GETPOS, 0, 0);
	if ( gamma != mfd->par.chn[s].value[Gamma] ) {
		mfd->par.chn[s].value[Gamma] = gamma;
		setDialogFeedbackValues(hdlg,mfd);
		redoSystem( mfd, hdlg );
		return ;
	}

}

static void levelsButtonCallback(bool fNewState, void *pvData)
{
	HWND hdlg = (HWND)pvData;
	EnableWindow(GetDlgItem(hdlg, IDSAMPLE), fNewState);
	EnableWindow(GetDlgItem(hdlg, IDC_DISABLE_CHECK), fNewState);
//	EnableWindow(GetDlgItem(hdlg, IDC_HISTO_SELECTION), fNewState);

	HWND wHist = GetDlgItem(hdlg, IDC_HISTO);
	if (fNewState == TRUE) {
		ShowWindow(wHist, SW_SHOWNORMAL);
	} else {
		ShowWindow(wHist, SW_HIDE);
	}

}



BOOL CALLBACK coloursConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {

    MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

    switch(msg) {

        case WM_INITDIALOG:
            SetWindowLong(hdlg, DWL_USER, lParam);
            mfd = (MyFilterData *)lParam;

			mfd->displayingPanel = true;
			EnableWindow(GetDlgItem(hdlg, IDSAMPLE), false);

			levelsButtonCallback(false, (void *)hdlg);
			mfd->ifp->SetButtonCallback(levelsButtonCallback, (void *)hdlg);
//			SetDlgItemInt(hdlg, IDC_CONTRAST_BALANCE, mfd->par.contrastBalance, TRUE);

			// save for cancel ...
			mfd->save = mfd->par;
			HWND hWnd;

			hWnd = GetDlgItem(hdlg,IDPREVIEW);
			mfd->ifp->InitButton(hWnd);

			setSliderRange(hdlg,mfd);
			setSliderPositions(hdlg,mfd);

			// update vales in boxes
			setDialogFeedbackValues(hdlg,mfd);

			return TRUE;


        case WM_COMMAND:

            switch(LOWORD(wParam)) {

            case IDOK:
				mfd->disableFilter = false;
				mfd->displayingPanel = false;
                EndDialog(hdlg, 0);
                return TRUE;

			case IDHELP:	{
				char prog[256];
				char path[256];
				LPTSTR ptr;

				GetModuleFileName(NULL, prog, 255);
				GetFullPathName(prog, 255, path, &ptr);
				*ptr = 0;
				strcat(path, "plugins\\ColourBriContr.htm");
				ShellExecute(hdlg, "open", path, NULL, NULL, SW_SHOWNORMAL);
				return TRUE;
				}

            case IDCANCEL: 
				mfd->par = mfd->save;
				mfd->disableFilter = false;
				mfd->displayingPanel = false;
                EndDialog(hdlg, 1);
                return FALSE;

			case IDPREVIEW: 
				mfd->ifp->RedoSystem();
				mfd->ifp->Toggle(GetDlgItem(hdlg,IDPREVIEW));
				break;
			
			case IDSAMPLE:
				mfd->frameHistogram.invalidate(hdlg);
				break;

			case IDC_BRIGHTNESS_DEFAULT : {
				int s = mfd->frameHistogram.getShowingSelection();
				mfd->par.chn[s].value[Brightness] =0;
				setDialogFeedbackValues ( hdlg, mfd);
				SendMessage(GetDlgItem(hdlg,IDC_BRIGHTNESS_SLIDER), 
					TBM_SETPOS, (WPARAM)TRUE, 0);
				redoSystem( mfd, hdlg );
				} break;

			case IDC_CONTRAST_DEFAULT : {
				int s = mfd->frameHistogram.getShowingSelection();
				mfd->par.chn[s].value[Contrast] = 0;
				mfd->par.chn[s].value[Balance] = BALANCE_DEFAULT;
				setDialogFeedbackValues ( hdlg, mfd);
				SendMessage(GetDlgItem(hdlg,IDC_CONTRAST_SLIDER), 
					TBM_SETPOS, (WPARAM)TRUE, 0);
				redoSystem( mfd, hdlg );
				} break;

			case IDC_GAMMA_DEFAULT : {
				int s = mfd->frameHistogram.getShowingSelection();
				mfd->par.chn[s].value[Gamma] = 0;
				setDialogFeedbackValues ( hdlg, mfd);
				SendMessage(GetDlgItem(hdlg,IDC_GAMMA_SLIDER), 
					TBM_SETPOS, (WPARAM)TRUE, 0);
				redoSystem( mfd, hdlg );
				} break;

			case IDC_DISABLE_CHECK:
				mfd->disableFilter = ! mfd->disableFilter;
				mfd->ifp->RedoFrame();
				mfd->frameHistogram.invalidate(hdlg);
				break;

			case IDC_CHAN_SELECTION:
				mfd->frameHistogram.nextShowingSelection();
				setDialogFeedbackValues(hdlg,mfd);
				setSliderPositions(hdlg,mfd);
				mfd->ifp->RedoFrame();
				mfd->frameHistogram.invalidate(hdlg);
				break;

			case IDC_CONTRAST_BALANCE: {
				int balance = GetDlgItemInt(hdlg, IDC_CONTRAST_BALANCE, NULL, FALSE);
				int s = mfd->frameHistogram.getShowingSelection();
				if (balance !=	mfd->par.chn[s].value[Balance] ) {
					if ( balance > MaxControl[Balance] )  {
						balance = MaxControl[Balance] ;
						SetDlgItemInt(hdlg, IDC_CONTRAST_BALANCE, balance, FALSE);
					}
					SetDlgItemInt(hdlg, IDC[s][Balance] , balance, FALSE);
					mfd->par.chn[s].value[Balance] = balance; 
					redoSystem( mfd, hdlg );
				}
				} break;


			case IDC_CONTRAST_VALUE:
			case IDC_CONTR_RED_VALUE:
			case IDC_CONTR_GREEN_VALUE:
			case IDC_CONTR_BLUE_VALUE:	
	
			case IDC_GAMMA_VALUE:
			case IDC_GAMMA_RED_VALUE:
			case IDC_GAMMA_GREEN_VALUE:
			case IDC_GAMMA_BLUE_VALUE:

			case IDC_MEDIA_VALUE:
			case IDC_MEDIA_RED_VALUE:
			case IDC_MEDIA_GREEN_VALUE:
			case IDC_MEDIA_BLUE_VALUE:

			case IDC_BRIGHT_GREEN_VALUE:
			case IDC_BRIGHT_BLUE_VALUE: 
			case IDC_BRIGHTNESS_VALUE:
			case IDC_BRIGHT_RED_VALUE: {

				int cc = lookUpChannel(LOWORD(wParam));
				int vv = lookUpParam(LOWORD(wParam));
				int k =  GetDlgItemInt(hdlg,  LOWORD(wParam), NULL, TRUE);
				if ( k != mfd->par.chn[cc].value[vv] ) {
					int max = MaxControl[vv];
					int min = MinControl[vv];
					if (k>max) {
						mfd->par.chn[cc].value[vv] = max;
						setDialogFeedbackValues(hdlg,mfd);
					} else if (k<min) {
						mfd->par.chn[cc].value[vv] = min;
						setDialogFeedbackValues(hdlg,mfd);
					} else {
						mfd->par.chn[cc].value[vv] = k;
					}
					setSliderPositions(hdlg,mfd);
					redoSystem( mfd, hdlg );
				}
		
				} break;


/*
			case IDC_BRIGHT_GREEN_VALUE :
			case IDC_BRIGHT_BLUE_VALUE  : 
			case IDC_BRIGHTNESS_VALUE :
			case IDC_BRIGHT_RED_VALUE : {

				int cc = lookUpChannel(LOWORD(wParam));
				int vv = lookUpParam(LOWORD(wParam));
				int k =  GetDlgItemInt(hdlg,  LOWORD(wParam), NULL, TRUE);
				if ( k != mfd->par.chn[cc].value[vv] ) {
					int max = MaxControl[cc];
					if (k>max) {
						mfd->par.chn[cc].value[vv] = max;
						setDialogFeedbackValues(hdlg,mfd);
					} else if (k<-max) {
						mfd->par.chn[cc].value[vv] = -max;
						setDialogFeedbackValues(hdlg,mfd);
					} else {
						mfd->par.chn[cc].value[vv] = k;
					}
					setSliderPositions(hdlg,mfd);
					redoSystem( mfd, hdlg );
				}
		
				} break;
*/

			case IDC_CLEAR_BUTTON: {
				for ( int s=0; s<Channel_Lenght; s++ ) 
					for (int p=0;p<ChanParam_Lenght;p++) {
					mfd->par.chn[s].value[p] = 0;
				}
				setDialogFeedbackValues(hdlg,mfd);
				redoSystem( mfd, hdlg );
				} break;

		}	// case WM_COMMAND:

		case WM_HSCROLL: 
			if (LOWORD(wParam) == TB_ENDTRACK)  
				getHorSliderValues(mfd,hdlg);
			break;

		case WM_PAINT:
			SetDlgItemInt(hdlg, IDC_MAX_VALUE, mfd->frameHistogram.getMaxHistoValue(), TRUE);
			mfd->frameHistogram.paintHisto(hdlg);			
			break;
    }

    return FALSE;
}

int coloursConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd) {

	MyFilterData * mfd = (MyFilterData *)fa->filter_data;
	mfd->ifp = fa->ifp;
	mfd->displayingPanel = false;
    return DialogBoxParam(fa->filter->module->hInstModule,
            MAKEINTRESOURCE(IDD_FILTER_COLOURS), hwnd,
            coloursConfigDlgProc, (LPARAM)fa->filter_data
	);
}


void coloursStringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str) {

    MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	sprintf(str,"(global b=%d c=%d g=%d m=%d, red b=%d c=%d g=%d m=%d, green b=%d c=%d g=%d m=%d, blue b=%d c=%d g=%d m=%d)",
		mfd->par.chn[Global].value[Brightness], mfd->par.chn[Global].value[Contrast], mfd->par.chn[Global].value[Gamma], mfd->par.chn[Global].value[Balance],
		mfd->par.chn[Red].value[Brightness], mfd->par.chn[Red].value[Contrast], mfd->par.chn[Red].value[Gamma], mfd->par.chn[Red].value[Balance],
		mfd->par.chn[Green].value[Brightness], mfd->par.chn[Green].value[Contrast], mfd->par.chn[Green].value[Gamma], mfd->par.chn[Green].value[Balance],
		mfd->par.chn[Blue].value[Brightness], mfd->par.chn[Blue].value[Contrast], mfd->par.chn[Blue].value[Gamma], mfd->par.chn[Blue].value[Balance]
	);

}

void coloursScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {

    FilterActivation *fa = (FilterActivation *)lpVoid;
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	int ctr=0;
	mfd->par.chn[Global].value[Brightness]	= argv[ctr++].asInt();
	mfd->par.chn[Global].value[Contrast]	= argv[ctr++].asInt();
	mfd->par.chn[Global].value[Gamma]		= argv[ctr++].asInt();
	mfd->par.chn[Global].value[Balance]	= argv[ctr++].asInt();

	mfd->par.chn[Red].value[Brightness]	= argv[ctr++].asInt();
	mfd->par.chn[Red].value[Contrast]		= argv[ctr++].asInt();
	mfd->par.chn[Red].value[Gamma]			= argv[ctr++].asInt();
	mfd->par.chn[Red].value[Balance]		= argv[ctr++].asInt();

	mfd->par.chn[Green].value[Brightness]	= argv[ctr++].asInt();
	mfd->par.chn[Green].value[Contrast]	= argv[ctr++].asInt();
	mfd->par.chn[Green].value[Gamma]		= argv[ctr++].asInt();
	mfd->par.chn[Green].value[Balance]		= argv[ctr++].asInt();

	mfd->par.chn[Blue].value[Brightness]	= argv[ctr++].asInt();
	mfd->par.chn[Blue].value[Contrast]		= argv[ctr++].asInt();
	mfd->par.chn[Blue].value[Gamma]		= argv[ctr++].asInt();
	mfd->par.chn[Blue].value[Balance]		= argv[ctr++].asInt();

}


bool coloursFssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

    _snprintf(buf, buflen, "Config(%d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d)",
		mfd->par.chn[Global].value[Brightness],mfd->par.chn[Global].value[Contrast], 	mfd->par.chn[Global].value[Gamma], mfd->par.chn[Global].value[Balance],
		mfd->par.chn[Red].value[Brightness],	mfd->par.chn[Red].value[Contrast], 	mfd->par.chn[Red].value[Gamma],	mfd->par.chn[Red].value[Balance],
		mfd->par.chn[Green].value[Brightness],	mfd->par.chn[Green].value[Contrast], 	mfd->par.chn[Green].value[Gamma],	mfd->par.chn[Green].value[Balance],
		mfd->par.chn[Blue].value[Brightness],	mfd->par.chn[Blue].value[Contrast], 	mfd->par.chn[Blue].value[Gamma],	mfd->par.chn[Blue].value[Balance]

	);

    return true;
}
