/*
    Hue/Saturation/Intensity Filter for VirtualDub -- allows adjustment
	of hue, saturation, and intensity. Copyright (C) 1999-2000 Donald A. Graft

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

	The author can be contacted at:
	Donald Graft
	neuron2@attbi.com.
*/

#include <windows.h>
#include <commctrl.h>
#include <stdio.h>
#include <crtdbg.h>
#include <math.h>
#include <stdlib.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

///////////////////////////////////////////////////////////////////////////

int RunProc(const FilterActivation *fa, const FilterFunctions *ff);
int StartProc(FilterActivation *fa, const FilterFunctions *ff);
int EndProc(FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int InitProc(FilterActivation *fa, const FilterFunctions *ff);
int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);
void identmat(float mat[4][4]);
void cscalemat(float mat[4][4], float rscale, float gscale, float bscale);
void saturatemat(float mat[4][4], float sat);
void simplehuerotatemat(float mat[4][4], float rot);
void huerotatemat(float mat[4][4], float rot);
void mat2lmat(float mat[4][4], long lmat[4][4]);
void lxformRGB(long mat[4][4], int *r, int *g, int *b);
void xformRGB(float mat[4][4], int *r, int *g, int *b);

///////////////////////////////////////////////////////////////////////////

typedef struct MyFilterData {
	IFilterPreview		*ifp;
	int					hue;
	int					preserve;
	int					saturation;
	int					intensity;
	int					applyto;
    float				mat[4][4];
	long				lmat[4][4];
} MyFilterData;

bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	_snprintf(buf, buflen, "Config(%d, %d, %d, %d, %d)",
		mfd->hue,
		mfd->preserve,
		mfd->saturation,
		mfd->intensity,
		mfd->applyto);

	return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->hue			= argv[0].asInt();
	mfd->preserve		= argv[1].asInt();
	mfd->saturation		= argv[2].asInt();
	mfd->intensity		= argv[3].asInt();
	mfd->applyto		= argv[4].asInt();
}

ScriptFunctionDef func_defs[]={
	{ (ScriptFunctionPtr)ScriptConfig, "Config", "0iiiii" },
	{ NULL },
};

CScriptObject script_obj={
	NULL, func_defs
};

struct FilterDefinition filterDef_tutorial = {

	NULL, NULL, NULL,		// next, prev, module
	"hue/saturation/intensity (1.2)",	// name
	"Adjust hue, saturation, and intensity.",
							// desc
	"Donald Graft", 		// maker
	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size

	InitProc,				// initProc
	NULL,					// deinitProc
	RunProc,				// runProc
	NULL,					// paramProc
	ConfigProc, 			// configProc
	StringProc, 			// stringProc
	StartProc,				// startProc
	EndProc,				// endProc

	&script_obj,			// script_obj
	FssProc,				// fssProc

};

void InitMatrix(MyFilterData *mfd)
{
	float i = (float) (1.0 + mfd->intensity * 0.01);
	float s = (float) (1.0 + mfd->saturation * 0.01);
	float h = (float) mfd->hue;

	/* Create the transformation matrix. */
	identmat(mfd->mat);
	cscalemat(mfd->mat, i, i, i);
	saturatemat(mfd->mat, s);
	if (mfd->preserve)
		huerotatemat(mfd->mat, h);
	else
		simplehuerotatemat(mfd->mat, h);
	mat2lmat(mfd->mat, mfd->lmat);
}

int StartProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	InitMatrix(mfd);	
	return 0;
}

int EndProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	return 0;
}

///////////////////////////////////////////////////////////////////////////

#define C_SCALE 32
#define RED			0x01
#define GREEN		0x02
#define BLUE		0x04
#define YELLOW		0x08
#define CYAN		0x10
#define MAGENTA		0x20
#define ALL			(RED | GREEN | BLUE | YELLOW | CYAN | MAGENTA)

int RunProc(const FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	const PixDim	width = fa->src.w;
	const PixDim	height = fa->src.h;
	Pixel32 *src, *dst;
	int x, y;
	int r, g, b, min, doit;
//	double H, S, L;
	int red = mfd->applyto & RED;
	int green = mfd->applyto & GREEN;
	int blue = mfd->applyto & BLUE;
	int yellow = mfd->applyto & YELLOW;
	int cyan = mfd->applyto & CYAN;
	int magenta = mfd->applyto & MAGENTA;

	src = fa->src.data;
	dst = fa->dst.data;
	for (y = 0; y < height; y++)
	{
		for (x = 0; x < width; x++)
		{
			r = ((src[x] & 0xff0000) >> 16);
			g = ((src[x] & 0xff00) >> 8);
			b = (src[x] & 0xff);
			if (mfd->applyto == ALL)
			{
				lxformRGB(mfd->lmat, &r, &g, &b);
			}
			else
			{
				min = r;
				doit = FALSE;
				if (g < min) min = g;
				if (b < min) min = b;
				if (min == b)
				{
					if (g < 2*r/3)
					{
						if (red) doit = TRUE;
					}
					else if (r < 2*g/3)
					{
						if (green) doit = TRUE;
					}
					else if (yellow) doit = TRUE;
				}
				else if (min == r)
				{
					if (g < 2*b/3)
					{
						if (blue) doit = TRUE;
					}
					else if (b < 2*g/3)
					{
						if (green) doit = TRUE;
					}
					else if (cyan) doit = TRUE;
				}
				else
				{
					if (r < 2*b/3)
					{
						if (blue) doit = TRUE;
					}
					else if (b < 2*r/3)
					{
						if (red) doit = TRUE;
					}
					else if (magenta) doit = TRUE;
				}
				if (doit == TRUE)
					lxformRGB(mfd->lmat, &r, &g, &b);
			}
			dst[x] = (r << 16) | (g << 8) | (b);			 
		}
		src	= (Pixel *)((char *)src + fa->src.pitch);
		dst	= (Pixel *)((char *)dst + fa->dst.pitch);
	}

	return 0;
}

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_tutorial;

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
	if (!(fd_tutorial = ff->addFilter(fm, &filterDef_tutorial, sizeof(FilterDefinition))))
		return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

	return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
	ff->removeFilter(fd_tutorial);
}

int InitProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->hue = 0;
	mfd->preserve = 0;
	mfd->saturation = 0;
	mfd->intensity = 0;
	mfd->applyto = ALL;

	return 0;
}

BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
	MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

	switch(msg) {
        char buf[10];
		case WM_INITDIALOG:
			{
			HWND hWnd;

			SetWindowLong(hdlg, DWL_USER, lParam);
			mfd = (MyFilterData *)lParam;

			hWnd = GetDlgItem(hdlg, IDC_HUE);
			SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 360));
			SendMessage(hWnd, TBM_SETTICFREQ, 10 , 0);
			SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->hue+180);
			sprintf(buf, "%d", mfd->hue);
			SetWindowText(GetDlgItem(hdlg, IDC_HUETEXT), buf);
			hWnd = GetDlgItem(hdlg, IDC_SATURATION);
			SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 512));
			SendMessage(hWnd, TBM_SETTICFREQ, 16, 0);
			SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->saturation+100);
			sprintf(buf, "%2.2f", 1.0 + mfd->saturation * 0.01);
			SetWindowText(GetDlgItem(hdlg, IDC_SATTEXT), buf);
			hWnd = GetDlgItem(hdlg, IDC_INTENSITY);
			SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 256));
			SendMessage(hWnd, TBM_SETTICFREQ, 16, 0);
			SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->intensity+100);
			sprintf(buf, "%2.2f", 1.0 + mfd->intensity * 0.01);
			SetWindowText(GetDlgItem(hdlg, IDC_INTTEXT), buf);
			if (mfd->preserve) CheckDlgButton(hdlg, IDC_PRESERVE, BST_CHECKED);
			if (mfd->applyto & RED) CheckDlgButton(hdlg, IDC_RED, BST_CHECKED);
			if (mfd->applyto & GREEN) CheckDlgButton(hdlg, IDC_GREEN, BST_CHECKED);
			if (mfd->applyto & BLUE) CheckDlgButton(hdlg, IDC_BLUE, BST_CHECKED);
			if (mfd->applyto & YELLOW) CheckDlgButton(hdlg, IDC_YELLOW, BST_CHECKED);
			if (mfd->applyto & CYAN) CheckDlgButton(hdlg, IDC_CYAN, BST_CHECKED);
			if (mfd->applyto & MAGENTA) CheckDlgButton(hdlg, IDC_MAGENTA, BST_CHECKED);
			mfd->ifp->InitButton(GetDlgItem(hdlg, IDPREVIEW));

			return TRUE;
			}
		case WM_COMMAND:
			switch(LOWORD(wParam)) {
			case IDPREVIEW:
				mfd->ifp->Toggle(hdlg);
				break;
			case IDOK:
				EndDialog(hdlg, 0);
				return TRUE;
			case IDHELP:
				{
				char prog[256];
				char path[256];
				LPTSTR ptr;
				GetModuleFileName(NULL, prog, 255);
				GetFullPathName(prog, 255, path, &ptr);
				*ptr = 0;
				strcat(path, "plugins\\Hue.txt");
				OutputDebugString(path);
				OutputDebugString("\n");
				strcpy(prog, "Notepad ");
				strcat(prog, path);
				WinExec(prog, SW_SHOW);
				return TRUE;
				}
			case IDCANCEL:
				EndDialog(hdlg, 1);
				return TRUE;
			case IDC_ZEROH:
				SendMessage(GetDlgItem(hdlg, IDC_HUE), TBM_SETPOS, (WPARAM)TRUE, 180);
				mfd->hue = 0;
				sprintf(buf, "%d", mfd->hue);
				SetWindowText(GetDlgItem(hdlg, IDC_HUETEXT), buf);
				InitMatrix(mfd);
				mfd->ifp->RedoFrame();
				return TRUE;
			case IDC_ZEROS:
				SendMessage(GetDlgItem(hdlg, IDC_SATURATION), TBM_SETPOS, (WPARAM)TRUE, 100);
				mfd->saturation = 0;
				sprintf(buf, "%2.2f", 1.0 + mfd->saturation * 0.01);
				SetWindowText(GetDlgItem(hdlg, IDC_SATTEXT), buf);
				InitMatrix(mfd);
				mfd->ifp->RedoFrame();
				return TRUE;
			case IDC_ZEROI:
				SendMessage(GetDlgItem(hdlg, IDC_INTENSITY), TBM_SETPOS, (WPARAM)TRUE, 100);
				mfd->intensity = 0;
				sprintf(buf, "%2.2f", 1.0 + mfd->intensity * 0.01);
				SetWindowText(GetDlgItem(hdlg, IDC_INTTEXT), buf);
				InitMatrix(mfd);
				mfd->ifp->RedoFrame();
				return TRUE;
			case IDC_PRESERVE:
				if (IsDlgButtonChecked(hdlg, IDC_PRESERVE))
					mfd->preserve = 1;
				else
					mfd->preserve = 0;
				InitMatrix(mfd);
				mfd->ifp->RedoFrame();
				break;
			case IDC_RED:
				if (IsDlgButtonChecked(hdlg, IDC_RED))
					mfd->applyto |= RED;
				else
					mfd->applyto &= ~RED;
				mfd->ifp->RedoFrame();
				break;
			case IDC_GREEN:
				if (IsDlgButtonChecked(hdlg, IDC_GREEN))
					mfd->applyto |= GREEN;
				else
					mfd->applyto &= ~GREEN;
				mfd->ifp->RedoFrame();
				break;
			case IDC_BLUE:
				if (IsDlgButtonChecked(hdlg, IDC_BLUE))
					mfd->applyto |= BLUE;
				else
					mfd->applyto &= ~BLUE;
				mfd->ifp->RedoFrame();
				break;
			case IDC_YELLOW:
				if (IsDlgButtonChecked(hdlg, IDC_YELLOW))
					mfd->applyto |= YELLOW;
				else
					mfd->applyto &= ~YELLOW;
				mfd->ifp->RedoFrame();
				break;
			case IDC_CYAN:
				if (IsDlgButtonChecked(hdlg, IDC_CYAN))
					mfd->applyto |= CYAN;
				else
					mfd->applyto &= ~CYAN;
				mfd->ifp->RedoFrame();
				break;
			case IDC_MAGENTA:
				if (IsDlgButtonChecked(hdlg, IDC_MAGENTA))
					mfd->applyto |= MAGENTA;
				else
					mfd->applyto &= ~MAGENTA;
				mfd->ifp->RedoFrame();
				break;
			}
			break;
		case WM_HSCROLL:
			{
				int hue, sat, inten;
				HWND hwnd = GetDlgItem(hdlg, IDC_HUE);
				hue = SendMessage(hwnd, TBM_GETPOS, 0, 0)-180;
				sprintf(buf, "%d", hue);
				SetWindowText(GetDlgItem(hdlg, IDC_HUETEXT), buf);
				sat = SendMessage(GetDlgItem(hdlg, IDC_SATURATION), TBM_GETPOS, 0, 0)-100;
				sprintf(buf, "%2.2f", 1.0 + sat * 0.01);
				SetWindowText(GetDlgItem(hdlg, IDC_SATTEXT), buf);
				inten = SendMessage(GetDlgItem(hdlg, IDC_INTENSITY), TBM_GETPOS, 0, 0)-100;
				sprintf(buf, "%2.2f", 1.0 +inten * 0.01);
				SetWindowText(GetDlgItem(hdlg, IDC_INTTEXT), buf);
				if (hue != mfd->hue || sat != mfd->saturation || inten != mfd->intensity)
				{
					mfd->hue = hue;
					mfd->saturation = sat;
					mfd->intensity = inten;
					InitMatrix(mfd);
					mfd->ifp->RedoFrame();
				}
				break;
			}
	}

	return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	MyFilterData mfd_old = *mfd;
	int ret;

	mfd->ifp = fa->ifp;

	if (DialogBoxParam(fa->filter->module->hInstModule, MAKEINTRESOURCE(IDD_FILTER),
		hwnd, ConfigDlgProc, (LPARAM)mfd))
	{
		*mfd = mfd_old;
		ret = TRUE;
	}
    else
	{
		ret = FALSE;
	}
	return(ret);
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	char temp[10] = "";

	if (mfd->applyto & RED) strcat(temp, "r");
	if (mfd->applyto & GREEN) strcat(temp, "g");
	if (mfd->applyto & BLUE) strcat(temp, "b");
	if (mfd->applyto & YELLOW) strcat(temp, "y");
	if (mfd->applyto & CYAN) strcat(temp, "c");
	if (mfd->applyto & MAGENTA) strcat(temp, "m");
	sprintf(str, " (hue %d, sat %.2f, inten %.2f, %s)",
				mfd->hue, 1.0 + mfd->saturation * 0.01, 1.0 + mfd->intensity * 0.01, temp);
}

#define M_PI (3.1415926)

#define RLUM    ((float) (0.3086))
#define GLUM    ((float) (0.6094))
#define BLUM    ((float) (0.0820))

/* 
 *	matrixmult -	
 *		multiply two matricies
 */
void matrixmult(float a[4][4], float b[4][4], float c[4][4])
{
    int x, y;
    float temp[4][4];

    for(y=0; y<4 ; y++)
        for(x=0 ; x<4 ; x++) {
            temp[y][x] = b[y][0] * a[0][x]
                       + b[y][1] * a[1][x]
                       + b[y][2] * a[2][x]
                       + b[y][3] * a[3][x];
        }
    for(y=0; y<4; y++)
        for(x=0; x<4; x++)
            c[y][x] = temp[y][x];
}

/* 
 *	identmat -	
 *		make an identity matrix
 */
void identmat(float mat[4][4])
{
	float *matrix = (float *) mat;

    *matrix++ = 1.0;    /* row 1        */
    *matrix++ = 0.0;
    *matrix++ = 0.0;
    *matrix++ = 0.0;
    *matrix++ = 0.0;    /* row 2        */
    *matrix++ = 1.0;
    *matrix++ = 0.0;
    *matrix++ = 0.0;
    *matrix++ = 0.0;    /* row 3        */
    *matrix++ = 0.0;
    *matrix++ = 1.0;
    *matrix++ = 0.0;
    *matrix++ = 0.0;    /* row 4        */
    *matrix++ = 0.0;
    *matrix++ = 0.0;
    *matrix++ = 1.0;
}

/* 
 *	xformpnt -	
 *		transform a 3D point using a matrix
 */
void xformpnt(float matrix[4][4], float x, float y, float z, float *tx, float *ty, float *tz)
{
	x = y;
	*tx = x;
    *tx = x*matrix[0][0] + y*matrix[1][0] + z*matrix[2][0] + matrix[3][0];
    *ty = x*matrix[0][1] + y*matrix[1][1] + z*matrix[2][1] + matrix[3][1];
    *tz = x*matrix[0][2] + y*matrix[1][2] + z*matrix[2][2] + matrix[3][2];
}

/* 
 *	cscalemat -	
 *		make a color scale marix
 */
void cscalemat(float mat[4][4], float rscale, float gscale, float bscale)
{
    float mmat[4][4];

    mmat[0][0] = rscale;
    mmat[0][1] = 0.0;
    mmat[0][2] = 0.0;
    mmat[0][3] = 0.0;

    mmat[1][0] = 0.0;
    mmat[1][1] = gscale;
    mmat[1][2] = 0.0;
    mmat[1][3] = 0.0;


    mmat[2][0] = 0.0;
    mmat[2][1] = 0.0;
    mmat[2][2] = bscale;
    mmat[2][3] = 0.0;

    mmat[3][0] = 0.0;
    mmat[3][1] = 0.0;
    mmat[3][2] = 0.0;
    mmat[3][3] = 1.0;
    matrixmult(mmat,mat,mat);
}

/* 
 *	saturatemat -	
 *		make a saturation marix
 */
void saturatemat(float mat[4][4], float sat)
{
    float mmat[4][4];
    float a, b, c, d, e, f, g, h, i;
    float rwgt, gwgt, bwgt;

    rwgt = RLUM;
    gwgt = GLUM;
    bwgt = BLUM;

    a = (float) ((1.0-sat)*rwgt + sat);
    b = (float) ((1.0-sat)*rwgt);
    c = (float) ((1.0-sat)*rwgt);
    d = (float) ((1.0-sat)*gwgt);
    e = (float) ((1.0-sat)*gwgt + sat);
    f = (float) ((1.0-sat)*gwgt);
    g = (float) ((1.0-sat)*bwgt);
    h = (float) ((1.0-sat)*bwgt);
    i = (float) ((1.0-sat)*bwgt + sat);
    mmat[0][0] = a;
    mmat[0][1] = b;
    mmat[0][2] = c;
    mmat[0][3] = 0.0;

    mmat[1][0] = d;
    mmat[1][1] = e;
    mmat[1][2] = f;
    mmat[1][3] = 0.0;

    mmat[2][0] = g;
    mmat[2][1] = h;
    mmat[2][2] = i;
    mmat[2][3] = 0.0;

    mmat[3][0] = 0.0;
    mmat[3][1] = 0.0;
    mmat[3][2] = 0.0;
    mmat[3][3] = 1.0;
    matrixmult(mmat,mat,mat);
}

/* 
 *	xrotate -	
 *		rotate about the x (red) axis
 */
void xrotatemat(float mat[4][4], float rs, float rc)
{
    float mmat[4][4];

    mmat[0][0] = 1.0;
    mmat[0][1] = 0.0;
    mmat[0][2] = 0.0;
    mmat[0][3] = 0.0;

    mmat[1][0] = 0.0;
    mmat[1][1] = rc;
    mmat[1][2] = rs;
    mmat[1][3] = 0.0;

    mmat[2][0] = 0.0;
    mmat[2][1] = -rs;
    mmat[2][2] = rc;
    mmat[2][3] = 0.0;

    mmat[3][0] = 0.0;
    mmat[3][1] = 0.0;
    mmat[3][2] = 0.0;
    mmat[3][3] = 1.0;
    matrixmult(mmat,mat,mat);
}

/* 
 *	yrotate -	
 *		rotate about the y (green) axis
 */
void yrotatemat(float mat[4][4], float rs, float rc)
{
    float mmat[4][4];

    mmat[0][0] = rc;
    mmat[0][1] = 0.0;
    mmat[0][2] = -rs;
    mmat[0][3] = 0.0;

    mmat[1][0] = 0.0;
    mmat[1][1] = 1.0;
    mmat[1][2] = 0.0;
    mmat[1][3] = 0.0;

    mmat[2][0] = rs;
    mmat[2][1] = 0.0;
    mmat[2][2] = rc;
    mmat[2][3] = 0.0;

    mmat[3][0] = 0.0;
    mmat[3][1] = 0.0;
    mmat[3][2] = 0.0;
    mmat[3][3] = 1.0;
    matrixmult(mmat,mat,mat);
}

/* 
 *	zrotate -	
 *		rotate about the z (blue) axis
 */
void zrotatemat(float mat[4][4], float rs, float rc)
{
    float mmat[4][4];

    mmat[0][0] = rc;
    mmat[0][1] = rs;
    mmat[0][2] = 0.0;
    mmat[0][3] = 0.0;

    mmat[1][0] = -rs;
    mmat[1][1] = rc;
    mmat[1][2] = 0.0;
    mmat[1][3] = 0.0;

    mmat[2][0] = 0.0;
    mmat[2][1] = 0.0;
    mmat[2][2] = 1.0;
    mmat[2][3] = 0.0;

    mmat[3][0] = 0.0;
    mmat[3][1] = 0.0;
    mmat[3][2] = 0.0;
    mmat[3][3] = 1.0;
    matrixmult(mmat,mat,mat);
}

/* 
 *	zshear -	
 *		shear z using x and y.
 */
void zshearmat(float mat[4][4], float dx, float dy)
{
    float mmat[4][4];

    mmat[0][0] = 1.0;
    mmat[0][1] = 0.0;
    mmat[0][2] = dx;
    mmat[0][3] = 0.0;

    mmat[1][0] = 0.0;
    mmat[1][1] = 1.0;
    mmat[1][2] = dy;
    mmat[1][3] = 0.0;

    mmat[2][0] = 0.0;
    mmat[2][1] = 0.0;
    mmat[2][2] = 1.0;
    mmat[2][3] = 0.0;

    mmat[3][0] = 0.0;
    mmat[3][1] = 0.0;
    mmat[3][2] = 0.0;
    mmat[3][3] = 1.0;
    matrixmult(mmat,mat,mat);
}

/* 
 *	simplehuerotatemat -	
 *		simple hue rotation. This changes luminance 
 */
void simplehuerotatemat(float mat[4][4], float rot)
{
    float mag;
    float xrs, xrc;
    float yrs, yrc;
    float zrs, zrc;

/* rotate the grey vector into positive Z */
    mag = (float) sqrt(2.0);
    xrs = (float) 1.0/mag;
    xrc = (float) 1.0/mag;
    xrotatemat(mat,xrs,xrc);

    mag = (float) sqrt(3.0);
    yrs = (float) -1.0/mag;
    yrc = (float) sqrt(2.0)/mag;
    yrotatemat(mat,yrs,yrc);

/* rotate the hue */
    zrs = (float) sin(rot*M_PI/180.0);
    zrc = (float) cos(rot*M_PI/180.0);
    zrotatemat(mat,zrs,zrc);

/* rotate the grey vector back into place */
    yrotatemat(mat,-yrs,yrc);
    xrotatemat(mat,-xrs,xrc);
}

/* 
 *	huerotatemat -	
 *		rotate the hue, while maintaining luminance.
 */
void huerotatemat(float mat[4][4], float rot)
{
    float mag;
    float lx, ly, lz;
    float xrs, xrc;
    float yrs, yrc;
    float zrs, zrc;
    float zsx, zsy;

/* rotate the grey vector into positive Z */
    mag = (float) sqrt(2.0);
    xrs = (float) 1.0/mag;
    xrc = (float) 1.0/mag;
    xrotatemat(mat,xrs,xrc);
    mag = (float) sqrt(3.0);
    yrs = (float) -1.0/mag;
    yrc = (float) sqrt(2.0)/mag;
    yrotatemat(mat,yrs,yrc);

/* shear the space to make the luminance plane horizontal */
    xformpnt(mat,RLUM,GLUM,BLUM,&lx,&ly,&lz);
    zsx = lx/lz;
    zsy = ly/lz;
    zshearmat(mat,zsx,zsy);

/* rotate the hue */
    zrs = (float) sin(rot*M_PI/180.0);
    zrc = (float) cos(rot*M_PI/180.0);
    zrotatemat(mat,zrs,zrc);

/* unshear the space to put the luminance plane back */
    zshearmat(mat,-zsx,-zsy);

/* rotate the grey vector back into place */
    yrotatemat(mat,-yrs,yrc);
    xrotatemat(mat,-xrs,xrc);
}

#define DENOM 0x10000

inline void lxformRGB(long mat[4][4], int *r, int *g, int *b)
{
    int ir, ig, ib;

	ir = *r;
	ig = *g;
	ib = *b;
	*r = (int) ((ir*mat[0][0] + ig*mat[1][0] + ib*mat[2][0] /*+ mat[3][0]*/)/DENOM);
	*g = (int) ((ir*mat[0][1] + ig*mat[1][1] + ib*mat[2][1] /*+ mat[3][1]*/)/DENOM);
	*b = (int) ((ir*mat[0][2] + ig*mat[1][2] + ib*mat[2][2] /*+ mat[3][2]*/)/DENOM);
	if(*r<0) *r = 0;
	else if(*r>255) *r = 255;
	if(*g<0) *g = 0;
	else if(*g>255) *g = 255;
	if(*b<0) *b = 0;
	else if(*b>255) *b = 255;
}

void mat2lmat(float mat[4][4], long lmat[4][4])
{
	int x, y;

	for (y = 0; y < 4; y++)
		for (x = 0; x < 4; x++) 
			lmat[y][x] = (long) (mat[y][x] * DENOM);
}
