/*
    Temporal cleaner filter for VirtualDub -- Blends a frame with the
	previous frame in cases where the differences between the pixels
	are less than the threshold.

    Copyright (C) 2000 Jim Casaburi
		Based on code by Avery Lee and Ben Rudiak-Gould
		Useful suggestions and much help from Donald A. Graft, Steven Don, 
		and "X-Bios"
		

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

	The author can be contacted at:
	Jim Casaburi
	casaburi@earthlink.net

	New versions of the source code and the compiled filter can be 
	found at http://home.earthlink.net/~casaburi/download/
*/

#include <stdio.h>

#include <windows.h>
#include <commctrl.h>
#include <math.h>
#include "resource.h"
#include "filter.h"
#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"
#include <string.h>

extern HINSTANCE g_hInst;


///////////////////////////////////

typedef struct MyFilterData {
	bool is_first_frame;
	bool fDebugMotion;
	int fPartial;
	int fThreshold;
	int fThreshold2;
	int fPixellock;
	int fPixellock2;
	int fScene;
	int fYUV;
	unsigned char *lastframeYUV;
	Pixel32 *lastframe;
	Pixel32 *origframe;
	unsigned char lookup[256][256];
	unsigned char *lockhistory;
} MyFilterData;

///////////////////////////////////
// The following two functions, and other RGB->YUV code are borrowed from 
// Ben Rudiak-Gould's excellent HuffYUV capture codec
///////////////////////////////////
static unsigned char clip[896];

static void InitClip() {
  memset(clip, 0, 320);
  for (int i=0; i<256; ++i) clip[i+320] = i;
  memset(clip+320+256, 255, 320);
}

static inline unsigned char Clip(int x)
  { return clip[320 + ((x+0x8000) >> 16)]; }



// The main function
static int tclean_run(const FilterActivation *fa, const FilterFunctions *ff) {	
	const int cyb = int(0.114*219/255*65536+0.5);
    const int cyg = int(0.587*219/255*65536+0.5);
    const int cyr = int(0.299*219/255*65536+0.5);
    const int crv = int(1.596*65536+0.5);
    const int cgv = int(0.813*65536+0.5);
    const int cgu = int(0.391*65536+0.5);
    const int cbu = int(2.018*65536+0.5);
	int b_y;
	int r_y;
	int scaled_y;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	unsigned char *lockhistory = mfd->lockhistory;
	Pixel *dst = (Pixel *)fa->dst.data;
	Pixel *source = (Pixel *)fa->src.data;
	unsigned char *lastYUV = mfd->lastframeYUV;
	Pixel32 *origframe;
	origframe = mfd->origframe;
	int totlocks = 0;	
	int r1, r2, g1, g2, b1, b2;
	int Y1, Y2, U1, U2, V1, V2;
	int threshY, threshU, threshV;
	long w, h;
	// if we are dealing with the first frame, just make a copy
	if (mfd->is_first_frame) { 
		Pixel *src = (Pixel *)fa->dst.data;
		unsigned long *last = mfd->lastframe;
		h = fa->dst.h;
		do {
			w = fa->dst.w;
			do {
				*last++ = *src;
				if (mfd->fYUV) {
					r2 = *((unsigned char *) src + 2);
					g2 = *((unsigned char *) src + 1);
					b2 = *((unsigned char *) src);
				    Y2 = (cyb*b2 + cyg*g2 + cyr*r2 + 0x108000) >> 16;
				    scaled_y = (2 * Y2 - 32) * int(255.0/219.0*32768+0.5);
					b_y = (r2 << 16) - scaled_y;
					r_y = (b2 << 16) - scaled_y;
					V2 = Clip((b_y >> 10) * int(1/2.018*1024+0.5) + 0x800000);
					U2 = Clip((r_y >> 10) * int(1/1.596*1024+0.5) + 0x800000);
					lastYUV[0] = Y2;
					lastYUV[1] = U2;
					lastYUV[2] = V2;
					lastYUV += 3;
				}
				*src++;

			} while (--w);
		r1 = fa->dst.modulo;
		src = (Pixel *)((char *)src + r1);
		last = (Pixel *) ((char *)last + r1);
		lastYUV = (unsigned char *) (lastYUV + r1 * 3);
		} while (--h);
		mfd->is_first_frame = FALSE	;
		return 0;
	}
	Pixel *src = (Pixel *)fa->dst.data;
	// make sure to preserve the existing frame in case this is a scene change
	source = origframe;
	h = fa->dst.h;
		do {
			w = fa->dst.w;
			do {
				*source++ = *src++;
			} while (--w);
		r1 = fa->dst.modulo;
		src = (Pixel *)((char *)src + r1);
		source = (Pixel *) ((char *)source + r1);		
		} while (--h);
	src = (Pixel *)mfd->lastframe;
	source = origframe;
	h = fa->dst.h;
	lastYUV = mfd->lastframeYUV;
	do {
		w = fa->dst.w;
		do {
			// get the components of the current pixel
			r1 = *((unsigned char *) dst + 2);
			r2 = *((unsigned char *) src + 2);
			g1 = *((unsigned char *) dst + 1);
			g2 = *((unsigned char *) src + 1);
			b1 = *((unsigned char *) dst);
			b2 = *((unsigned char *) src);
			if (mfd->fYUV) {
				// convert the pixel to its YUV equiv
			    Y1 = (cyb*b1 + cyg*g1 + cyr*r1 + 0x108000) >> 16;
			    scaled_y = (2 * Y1 - 32) * int(255.0/219.0*32768+0.5);
				b_y = (r1 << 16) - scaled_y;
				r_y = (b1 << 16) - scaled_y;
				V1 = Clip((b_y >> 10) * int(1/2.018*1024+0.5) + 0x800000);
				U1 = Clip((r_y >> 10) * int(1/1.596*1024+0.5) + 0x800000);
				Y2 = lastYUV[0];
				U2 = lastYUV[1];
				V2 = lastYUV[2];
				// lets lookup the biased differences between the pixels
				threshY = mfd->lookup[Y1][Y2];
				threshU = mfd->lookup[U1][U2];
				threshV = mfd->lookup[V1][V2];
			}
			else {
				// lets lookup the biased differences between the pixels
				threshY = mfd->lookup[r1][r2];
				threshU = mfd->lookup[g1][g2];
				threshV = mfd->lookup[b1][b2];
			}
			// figure out what kind of processing we want to do
			if (mfd->fPartial && mfd->fYUV) {
				if ((threshY < mfd->fPixellock) && (threshU < mfd->fPixellock2) &&
					(threshV < mfd->fPixellock2)) { 
					// we're doing a full pixel lock since we're under all
					// thresholds
					if (*lockhistory > 30) {
					// if we've locked more than 30 times at this point, let's refresh the pixel
					*lockhistory = 0;
		  			r1 = (r1 + r2) / 2;
					g1 = (g1 + g2) / 2;
					b1 = (b1 + b2) / 2;
					*((unsigned char *) src + 2) = r1;
					*((unsigned char *) src + 1) = g1;
					*((unsigned char *) src) = b1;
					if (mfd->fDebugMotion) *dst = 0xff;
					else *dst = *src;
					Y1 = (Y1 + Y2) / 2;
					U1 = (U1 + U2) / 2;
					V1 = (V1 + V2) / 2;
					lastYUV[0] = Y1;
					lastYUV[1] = U1;
					lastYUV[2] = V1;
					}
					else {
				if (mfd->fDebugMotion) *dst = 0xff;
				else
				*dst = *src;
				*lockhistory = *lockhistory + 1;
				}
				}
				else
				{
				if ((threshY < mfd->fPixellock) && (threshU < mfd->fThreshold2) &&
					(threshV < mfd->fThreshold2)) {
					// If the luma is within pixellock, and the chroma is within
					// blend, lets blend the chroma and lock the luma (this is 
					// the ONLY case the calculated YUV values are used in the
					// output
					*lockhistory = 0;
					Y1 = Y2;
					U1 = (U1 + U2) >> 1;
					V1 = (V1 + V2) >> 1;
					scaled_y = (Y1 - 16) * int((255.0/219.0)*65536+0.5);
					b1 = Clip(scaled_y + (U1-128) * cbu); // blue
					g1 = Clip(scaled_y - (U1-128) * cgu - (V1-128) * cgv); // green
					r1 = Clip(scaled_y + (V1-128) * crv); // red
					*((unsigned char *) src + 2) = r1;
					*((unsigned char *) src + 1) = g1;
					*((unsigned char *) src) = b1;
					lastYUV[0] = Y1;
					lastYUV[1] = U1;
					lastYUV[2] = V1;
					if (mfd->fDebugMotion) *dst = 0x777777;
					else *src = *dst;
					}
				else {
					if ((threshY < mfd->fThreshold) && (threshU < mfd->fThreshold2) && (threshV < mfd->fThreshold2)) {
						// We are above pixellock in luma and chroma, but below the
						// blend thresholds in both, so let's blend
						*lockhistory = 0;
		  				r1 = (r1 + r2) / 2;
						g1 = (g1 + g2) / 2;
						b1 = (b1 + b2) / 2;
						*((unsigned char *) src + 2) = r1;
						*((unsigned char *) src + 1) = g1;
						*((unsigned char *) src) = b1;
						*dst = *src;
						Y1 = (Y1 + Y2) / 2;
						U1 = (U1 + U2) / 2;
						V1 = (V1 + V2) / 2;
						lastYUV[0] = Y1;
						lastYUV[1] = U1;
						lastYUV[2] = V1;
						}
					else {
					// if we are above all thresholds, just leave the output
					// untouched
							*lockhistory = 0;
							*src = *dst;
							if (mfd->fDebugMotion) *dst = 0;
							totlocks++;
							lastYUV[0] = Y1;
							lastYUV[1] = U1;
							lastYUV[2] = V1;
						}
					}
				}
			}
			else {
			if ((threshY < mfd->fPixellock) && (threshU < mfd->fPixellock2) 
				&& (threshV < mfd->fPixellock2)) { 
				// beneath pixellock so lets keep the existing pixel (most
				// likely)
					if (*lockhistory > 30) {
					// if we've locked more than 30 times at this point, 
				    // let's refresh the pixel
					*lockhistory = 0;
		  			r1 = (r1 + r2) / 2;
					g1 = (g1 + g2) / 2;
					b1 = (b1 + b2) / 2;
					*((unsigned char *) src + 2) = r1;
					*((unsigned char *) src + 1) = g1;
					*((unsigned char *) src) = b1;
					if (mfd->fDebugMotion) *dst = 0xff;
					else *dst = *src;
					Y1 = (Y1 + Y2) / 2;
					U1 = (U1 + U2) / 2;
					V1 = (V1 + V2) / 2;
					lastYUV[0] = Y1;
					lastYUV[1] = U1;
					lastYUV[2] = V1;
					}
					else {
				if (mfd->fDebugMotion) *dst = 0xff;
				else
				*dst = *src;
				*lockhistory = *lockhistory + 1;
			}}
			else {
				if ((threshY < mfd->fThreshold) && (threshU < mfd->fThreshold2) 
					&& (threshV < mfd->fThreshold2)) { 
					// we are above pixellock, but below the blend threshold
					// so we want to blend
					*lockhistory = 0;
		  			r1 = (r1 + r2) / 2;
					g1 = (g1 + g2) / 2;
					b1 = (b1 + b2) / 2;
					*((unsigned char *) src + 2) = r1;
					*((unsigned char *) src + 1) = g1;
					*((unsigned char *) src) = b1;
					*dst = *src;
					if (mfd->fYUV) {
						Y1 = (Y1 + Y2) / 2;
						U1 = (U1 + U2) / 2;
						V1 = (V1 + V2) / 2;
						lastYUV[0] = Y1;
						lastYUV[1] = U1;
						lastYUV[2] = V1;
					}
				}
				else { // it's beyond the thresholds, just leave it alone
					*lockhistory = 0;
					*src = *dst;
					if (mfd->fDebugMotion) *dst = 0;
					totlocks++;
					if (mfd->fYUV) {
						lastYUV[0] = Y1;
						lastYUV[1] = U1;
						lastYUV[2] = V1;
					}
				}
			}
			}  
			++dst;
			++src;
			++lockhistory;
			lastYUV += 3;
		} while(--w);
		r1 = fa->dst.modulo;
		dst = (Pixel *)((char *)dst + r1);
		src = (Pixel *)((char *)src + r1);
		lastYUV = (unsigned char *) (lastYUV + r1 * 3);
		lockhistory += r1;
	} while(--h);
	dst = (Pixel *)fa->dst.data;
	src = mfd->lastframe;
	long totpixels = fa->dst.h * fa->dst.w;
	totpixels *= mfd->fScene;
	totpixels /= 100;
	// If more than the specified percent of pixels have exceeded all thresholds
	// then we restore the saved frame.  (this doesn't happen very often 
	// hopefully)  We also set the pixellock history to 0 for all frames
	if (totlocks > totpixels) {
		lockhistory = mfd->lockhistory;
		h = fa->dst.h;
		do {
			w = fa->dst.w;
			do {
				if (mfd->fDebugMotion)  {
					*dst++ = 0xff0000;
					*src++ = *source++;
					*lockhistory++ = 0;
				}
				else {
				*src++ = *dst++ = *source++;
				*lockhistory++ = 0;
				}
			} while (--w);
		r1 = fa->dst.modulo;
		src = (Pixel *)((char *)src + r1);
		dst = (Pixel *) ((char *)dst + r1);
		source = (Pixel *) ((char *) source + r1);
		lockhistory += r1;
		} while (--h);
	}
	return 0;
}

static long tclean_param(FilterActivation *fa, const FilterFunctions *ff) {
	fa->dst.offset	= fa->src.offset;
	fa->dst.modulo	= fa->src.modulo;
	fa->dst.pitch	= fa->src.pitch;
	return 0;
}

static int tclean_start(FilterActivation *fa, const FilterFunctions *ff) {
	if (!fa->filter_data)
		if (!(fa->filter_data = (void *)new MyFilterData)) return 1;
	InitClip();
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	mfd->is_first_frame = TRUE;
	mfd->lastframe = new unsigned long[fa->src.h * fa->src.pitch];
	mfd->origframe = new unsigned long[fa->src.h * fa->src.pitch];
	mfd->lastframeYUV = new unsigned char[fa->src.h * fa->src.pitch * 3];
	mfd->lockhistory = new unsigned char[fa->src.h * fa->src.pitch];
	for (int c = 0; c < fa->src.h; c++)
		for (int d = 0; d < fa->src.pitch; d++) 
			mfd->lockhistory[d + c * fa->src.pitch] = 0;
	double low1, low2;
	double high1, high2;
	int dif1, dif2;
	// setup a biased thresholding difference matrix
	// this is an expensive operation we only want to to once
	for (int a = 0; a < 256; a++) {
		for (int b = 0; b < 256; b++) {
			// instead of scaling linearly
			// we scale according to the following formulas
			// val1 = 256 * (x / 256) ^ .9
			// and 
			// val2 = 256 * (x / 256) ^ (1/.9)
			// and we choose the maximum distance between two points
			// based on these two scales
			low1 = a;
			low2 = b;
			low1 = low1 / 256;
			low1 = 256 * pow(low1, .9);
			low2 = low2 / 256;
			low2 = 256 * pow(low2, .9);
			// the low scale should make all values larger
			// and the high scale should make all values smaller
			high1 = a;
			high2 = b;
			high1 = high1 / 256;
			high2 = high2 / 256;
			high1 = 256 * pow(high1, 1.0/.9);
			high2 = 256 * pow(high2, 1.0/.9);
			dif1 = (int) (low1 - low2);
			if (dif1 < 0) dif1 *= -1;
			dif2 = (int) (high1 - high2);
			if (dif2 < 0) dif2 *= -1;
			dif1 = (dif1 > dif2) ? dif1 : dif2;
			mfd->lookup[a][b] = dif1;
		}
	}
	return 0;
}

static int tclean_stop(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	delete mfd->lastframe;
	delete mfd->origframe;
	delete mfd->lastframeYUV;
	delete mfd->lockhistory;
	return 0;
}


BOOL CALLBACK tcleanConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
         MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

         switch(msg) {
             case WM_INITDIALOG:
                 SetWindowLong(hdlg, DWL_USER, lParam);
                 mfd = (MyFilterData *)lParam;

                 CheckDlgButton(hdlg, IDC_DEBUGMOTION, mfd->fDebugMotion?BST_CHECKED:BST_UNCHECKED);
				 SetDlgItemInt(hdlg, IDC_THRESHOLD, mfd->fThreshold, FALSE);
				 SetDlgItemInt(hdlg, IDC_PIXELLOCK, mfd->fPixellock, FALSE);
				 SetDlgItemInt(hdlg, IDC_SCENE, mfd->fScene, FALSE);
				 SetDlgItemInt(hdlg, IDC_THRESHOLD2, mfd->fThreshold2, FALSE);
				 SetDlgItemInt(hdlg, IDC_PIXELLOCK2, mfd->fPixellock2, FALSE);
				 CheckDlgButton(hdlg, IDC_PARTIAL, mfd->fPartial?BST_CHECKED:BST_UNCHECKED);
				 CheckDlgButton(hdlg, IDC_YUV, mfd->fYUV?BST_CHECKED:BST_UNCHECKED);
                 return TRUE;

             case WM_COMMAND:
                 switch(LOWORD(wParam)) {
                 case IDOK:
                     mfd->fDebugMotion = !!IsDlgButtonChecked(hdlg, IDC_DEBUGMOTION);
					 mfd->fPartial = !!IsDlgButtonChecked(hdlg, IDC_PARTIAL);
					 mfd->fYUV = !!IsDlgButtonChecked(hdlg, IDC_YUV);
					 long threshold;
					 BOOL success;
  					 threshold = GetDlgItemInt(hdlg, IDC_PIXELLOCK, &success, FALSE);
					 if (!success || threshold < 0 || threshold > 255) {
						SetFocus((HWND)lParam);
						MessageBeep(MB_ICONQUESTION);
						return TRUE;
					 }
 					 mfd->fPixellock = threshold;
  					 threshold = GetDlgItemInt(hdlg, IDC_PIXELLOCK2, &success, FALSE);
					 if (!success || threshold < 0 || threshold > 255) {
						SetFocus((HWND)lParam);
						MessageBeep(MB_ICONQUESTION);
						return TRUE;
					 }
 					 mfd->fPixellock2 = threshold;
					threshold = GetDlgItemInt(hdlg, IDC_SCENE, &success, FALSE);
					if (!success || threshold < 0 || threshold > 255) {
						SetFocus((HWND)lParam);
						MessageBeep(MB_ICONQUESTION);
						return TRUE;
					}
					 mfd->fScene = threshold;
					threshold = GetDlgItemInt(hdlg, IDC_THRESHOLD, &success, FALSE);
					if (!success || threshold < 0 || threshold > 255) {
						SetFocus((HWND)lParam);
						MessageBeep(MB_ICONQUESTION);
						return TRUE;
					}
					mfd->fThreshold = threshold;
					threshold = GetDlgItemInt(hdlg, IDC_THRESHOLD2, &success, FALSE);
					if (!success || threshold < 0 || threshold > 255) {
						SetFocus((HWND)lParam);
						MessageBeep(MB_ICONQUESTION);
						return TRUE;
					}
					mfd->fThreshold2 = threshold;
					if (!mfd->fYUV) {
						mfd->fThreshold2 = mfd->fThreshold;
						mfd->fPixellock2 = mfd->fPixellock;
					}

                     
					 EndDialog(hdlg, 0);
                     return TRUE;
                 case IDCANCEL:
                     EndDialog(hdlg, 1);
                     return FALSE;
				 case IDC_PIXELLOCK2:
					 if (HIWORD(wParam) == EN_KILLFOCUS) {
					 }
					 return TRUE;
				 case IDC_THRESHOLD2:
					 if (HIWORD(wParam) == EN_KILLFOCUS) {
					 }
					 return TRUE;
				 case IDC_PARTIAL:
					 if (HIWORD(wParam) == EN_KILLFOCUS) {
					 }
					 return TRUE;
				 case IDC_YUV:
					 if (HIWORD(wParam) == EN_KILLFOCUS) {
					 }
					 return TRUE;
				 case IDC_PIXELLOCK:
				if (HIWORD(wParam) == EN_KILLFOCUS) {
				}
				return TRUE;
				 case IDC_SCENE:
				if (HIWORD(wParam) == EN_KILLFOCUS) {
				}
				return TRUE;


				 case IDC_THRESHOLD:
				if (HIWORD(wParam) == EN_KILLFOCUS) {
				}
				return TRUE;

               }
                 break;
         }

         return FALSE;
     }

int tcleanConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd) {
         return DialogBoxParam(fa->filter->module->hInstModule,
                 MAKEINTRESOURCE(IDD_TSOFTEN_TUTORIAL), hwnd,
                 tcleanConfigDlgProc, (LPARAM)fa->filter_data);
     }

void tcleanStringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str) {
         const char *modes[2]={
             "(",
             " (SHOW MOTION, ",
         };
		 const char *othermodes[2]={
			 "",
			 ", LUM LOCK",
		 };
		 const char *colorspace[2]={
			 "RGB",
			 "YUV",
		 };
         MyFilterData *mfd = (MyFilterData *)fa->filter_data;
		 sprintf(str, "%sYT: %d, YL: %d, UVT: %d, UVL: %d, SCENE: %d, %s%s)", 
			 modes[mfd->fDebugMotion], mfd->fThreshold, mfd->fPixellock, mfd->fThreshold2, mfd->fPixellock2, mfd->fScene, colorspace[mfd->fYUV], othermodes[mfd->fPartial]);
     }


void tcleanScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
    FilterActivation *fa = (FilterActivation *)lpVoid;
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->fThreshold		= argv[0].asInt();
	mfd->fPixellock		= argv[1].asInt();
	mfd->fThreshold2	= argv[2].asInt();
	mfd->fPixellock2	= argv[3].asInt();
	mfd->fScene			= argv[4].asInt();
	mfd->fPartial		= !!argv[5].asInt();	
	mfd->fYUV			= !!argv[6].asInt();
}

bool tcleanFssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

    _snprintf(buf, buflen, "Config(%d, %d, %d, %d, %d, %d, %d)",
		mfd->fThreshold, mfd->fPixellock, mfd->fThreshold2, mfd->fPixellock2, mfd->fScene, mfd->fPartial, mfd->fYUV);

    return true;
}

ScriptFunctionDef tclean_func_defs[]={
    { (ScriptFunctionPtr)tcleanScriptConfig, "Config", "0iiiiiii" },
    { NULL },
};

CScriptObject tclean_obj={
    NULL, tclean_func_defs
};

int tclean_init(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	mfd->fThreshold = 10;
	mfd->fPixellock = 4;
	mfd->fThreshold2 = 16;
	mfd->fPixellock2 = 8;
	mfd->fScene = 30;
	mfd->fPartial = 0;
	mfd->fYUV = 1;
	return 0;
}

FilterDefinition filterDef_tclean={
	0,0,NULL,
	"temporal cleaner",
	"reduces changes across frames without blending motion.\n\n",
	"Jim Casaburi",NULL,
	sizeof(MyFilterData),
	tclean_init,NULL,
	tclean_run,
	tclean_param,
    tcleanConfigProc,     // configProc
    tcleanStringProc,     // stringProc
	tclean_start,
	tclean_stop,
	&tclean_obj,          // script_obj
    tcleanFssProc,        // fssProc

};

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
     extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

     static FilterDefinition *fd_tutorial;

     int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
         if (!(fd_tutorial = ff->addFilter(fm, &filterDef_tclean, sizeof(FilterDefinition))))
             return 1;

         vdfd_ver    = VIRTUALDUB_FILTERDEF_VERSION;
         vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

         return 0;
     }

     void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
         ff->removeFilter(fd_tutorial);
     }

