// VirtualDub filter: deinterlace - PAL movie
// by Gunnar Thalin (guth@home.se)

#include <windows.h>
#include <stdio.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

///////////////////////////////////////////////////////////////////////////

int InitProc(FilterActivation *fa, const FilterFunctions *ff);
int RunProc(const FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

typedef struct MyFilterData {
    bool     bSwap;
} MyFilterData;


bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	_snprintf(buf, buflen, "Config(%i)", mfd->bSwap);

	return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->bSwap = !!argv[0].asInt();
}

ScriptFunctionDef func_defs[]={
	{ (ScriptFunctionPtr)ScriptConfig, "Config", "0i" },
	{ NULL },
};

CScriptObject script_obj={
	NULL, func_defs
};


struct FilterDefinition filterDef = {

	NULL, NULL, NULL,			// next, prev, module
	"deinterlace - PAL movie v1.1",	// name
	"Fixes video that should be non-interlaced (such as most PAL movies) but still show interlace lines.",	// desc
	"Gunnar Thalin",		// maker
	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size

	InitProc,		// initProc
	NULL,			// deinitProc
	RunProc,		// runProc
	ParamProc,		// paramProc
	ConfigProc,		// configProc
	StringProc,		// stringProc
	NULL,			// startProc
	NULL,			// endProc

	&script_obj,	// script_obj
	FssProc,		// fssProc

};

///////////////////////////////////////////////////////////////////////////

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd;

///////////////////////////////////////////////////////////////////////////

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
    if (!(fd = ff->addFilter(fm, &filterDef, sizeof(FilterDefinition))))
        return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

    return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
    ff->removeFilter(fd);
}

///////////////////////////////////////////////////////////////////////////

int InitProc(FilterActivation *fa, const FilterFunctions *ff) {
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->bSwap = false;

	return 0;
}

int RunProc(const FilterActivation *fa, const FilterFunctions *ff) {
	Pixel32 *src, *dst, *last;

	src = (Pixel32 *)fa->src.data;
	dst = (Pixel32 *)fa->dst.data;
	last = (Pixel32 *)fa->last->data;
	int iOddEven = ((MyFilterData *)(fa->filter_data))->bSwap ? 0 : 1;

	for(int y = 0; y < fa->src.h; y++)
	{
		if((y % 2) == iOddEven)
		{	 // Take line from the previous frame.
			if(fa->pfsi->lCurrentFrame > 0)
				memcpy(dst, last, fa->src.w * sizeof(Pixel32));
			else
			{	// There is no previous frame. Interpolate from line above and below instead.
				Pixel32 *pprevrow = (y - 1 >= 0 ? (Pixel32 *)((char *)src - fa->src.pitch) : src);
				Pixel32 *pnextrow = (Pixel32 *)(y + 1 < fa->src.h ? (Pixel32 *)((char *)src + fa->src.pitch) : pprevrow);
				if(pprevrow == src)
					pprevrow = pnextrow;
				Pixel32 prevrow, nextrow;
				for(int x = 0; x < fa->src.w; x++)
				{
					prevrow = *pprevrow++;
					nextrow = *pnextrow++;
					*(dst + x) = ((prevrow & 0xff) + (nextrow & 0xff)) / 2 +
					(((((prevrow >> 8) & 0xff) + ((nextrow >> 8) & 0xff)) / 2) << 8) + 
					(((((prevrow >> 16) & 0xff) + ((nextrow >> 16) & 0xff)) / 2) << 16);
				}
			}
		}
		else
			// Take line from the this frame.
			memcpy(dst, src, fa->src.w * sizeof(Pixel32));

		src = (Pixel32 *)((char *)src + fa->src.pitch);
		dst = (Pixel32 *)((char *)dst + fa->dst.pitch);
		if(last != NULL)
			last = (Pixel32 *)((char *)last + fa->last->pitch);
	}

	return 0;
}

long ParamProc(FilterActivation *fa, const FilterFunctions *ff) {
    return FILTERPARAM_NEEDS_LAST | FILTERPARAM_SWAP_BUFFERS;
}


BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
    MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

    switch(msg) {
        case WM_INITDIALOG:
            SetWindowLong(hdlg, DWL_USER, lParam);
            mfd = (MyFilterData *)lParam;

            CheckDlgButton(hdlg, IDC_SWAP, mfd->bSwap?BST_CHECKED:BST_UNCHECKED);

            return TRUE;

        case WM_COMMAND:
            switch(LOWORD(wParam)) {
            case IDOK:
                mfd->bSwap = !!IsDlgButtonChecked(hdlg, IDC_SWAP);
                EndDialog(hdlg, 0);
                return TRUE;
            case IDCANCEL:
                EndDialog(hdlg, 1);
                return FALSE;
            }
            break;
    }

    return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd) {
    return DialogBoxParam(fa->filter->module->hInstModule,
            MAKEINTRESOURCE(IDD_FILTER_DEINTERLACE_PALMOVIE), hwnd,
            (DLGPROC)ConfigDlgProc, (LPARAM)fa->filter_data);
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str) {
	if(((MyFilterData *)(fa->filter_data))->bSwap)
		strcpy(str, " (swap fields)");
	else
		strcpy(str, " (normal)");
}
