/*
    Coring filter for virtualdub - Copyright 2001 Paul Currie

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    The author can be contacted at:
    Paul Currie
    bogo@gimle.nu
*/

#include <windows.h>
#include <stdio.h>
#include <commctrl.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"



#define CORE_MAX 64
#define CORE_MIN 0
#define BOOST_MAX 128
#define BOOST_MIN 0




///////////////////////////////////////////////////////////////////////////

int coringRunProc(const FilterActivation *fa, const FilterFunctions *ff);
int coringStartProc(FilterActivation *fa, const FilterFunctions *ff);
int coringEndProc(FilterActivation *fa, const FilterFunctions *ff);
long coringParamProc(FilterActivation *fa, const FilterFunctions *ff);
int coringConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void coringStringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void coreScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool coreFssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

#define uint unsigned int

struct CoringData {
	int threshold;
	bool corevalue;

	// Preview stuff
	bool test;
	bool green;
	int boost;

	Pixel32 *red;
	Pixel32 *grn;
	Pixel32 *blu;

	IFilterPreview *ifp;

	// Backup of settings
	int oldthreshold;
	bool oldcorevalue;
};

ScriptFunctionDef core_func_defs[]={
    { (ScriptFunctionPtr)coreScriptConfig, "Config", "0ii" },
    { NULL },
};

CScriptObject core_obj={
    NULL, core_func_defs
};


struct FilterDefinition filterDef_coring = {

    NULL, NULL, NULL,       // next, prev, module
    "coring",			    // name
    "Any pixel below a certain threshold value is set to black.\n",
                            // desc
    "Paul Currie",          // maker
    NULL,                   // private_data
    sizeof(CoringData),	    // inst_data_size

	NULL,			        // initProc
	NULL,                   // deinitProc
    coringRunProc,          // runProc
    coringParamProc,	    // paramProc
    coringConfigProc,	    // configProc
    coringStringProc,		// stringProc
    coringStartProc,        // startProc
    coringEndProc,          // endProc
    
	&core_obj,				// script_obj
    coreFssProc,			// fssProc

};

///////////////////////////////////////////////////////////////////////////

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_coring;


int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
    if (!(fd_coring = ff->addFilter(fm, &filterDef_coring, sizeof(FilterDefinition))))
        return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

    return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
    ff->removeFilter(fd_coring);
}

///////////////////////////////////////////////////////////////////////////


// Precalculate boost values
void makeBoostArray(CoringData *mfd) {
	int boost, temp, i;

	if ( mfd->red == NULL ) {
	   if (!(mfd->red = new Pixel32[256 * 3]))
	       return;
		mfd->grn = mfd->red + 256;
		mfd->blu = mfd->red + 512;
	}

	boost = mfd->boost;
	boost = __max(boost, BOOST_MIN);
	boost = __min(boost, BOOST_MAX);
	
	// 0 is special case
	mfd->red[0] = boost << 16;
	mfd->grn[0] = boost << 8;
	mfd->blu[0] = boost;

	for ( i = 0 ; i < 255 ; i++ ) {
		// pixel = pixel + boost - boost*pixel/256
		temp = (( (i << 8) + (boost << 8) - (i * boost) ) >> 8) & 0xff;

		mfd->red[i] = temp << 16;
		mfd->grn[i] = temp << 8;
		mfd->blu[i] = temp;
	}

	// 255 is special case
	mfd->red[255] = 255 << 16;
	mfd->grn[255] = 255 << 8;
	mfd->blu[255] = 255;
}





int coringStartProc(FilterActivation *fa, const FilterFunctions *ff) {
    CoringData *mfd = (CoringData *)fa->filter_data;

	makeBoostArray(mfd);
    
	return 0;
}

int coringEndProc(FilterActivation *fa, const FilterFunctions *ff) {
    CoringData *mfd = (CoringData *)fa->filter_data;

	if ( mfd->red != NULL ) {
		delete[] mfd->red;
		mfd->red = NULL;
	}

	return 0;
}






int coringRunProc(const FilterActivation *fa, const FilterFunctions *ff) {
    PixDim w, h;
    Pixel32 *src, old_pixel, *red, *grn, *blu;
	uint threshold;
	int coreval;

    src = (Pixel32 *)fa->src.data;

	threshold = ((CoringData *) fa->filter_data)->threshold;
	threshold = __max(threshold, CORE_MIN);
	threshold = __min(threshold, CORE_MAX);

	if ( ((CoringData *) fa->filter_data)->test == false ) {
		//
		// Main algorithm (non-test)
		// 
		if ( ((CoringData *) fa->filter_data)->corevalue == true ) {
			coreval = ((threshold & 0xff) << 16) | ((threshold & 0xff) << 8) | (threshold & 0xff);
		} else {
			coreval = 0; // Core to black
		}

		h = fa->src.h;
		do {
			w = fa->src.w;

			do {
				old_pixel = *src;

				if (   ((old_pixel>>16) & 0xff) < threshold
					&& ((old_pixel>>8) & 0xff) < threshold
					&& (old_pixel & 0xff) < threshold)
					*src = coreval;
				*src++;
			} while(--w);

			src = (Pixel32 *)((char *)src + fa->src.modulo);
		} while(--h);
	} else {
		//
		// Test algotrithm
		//
		red = ((CoringData *) fa->filter_data)->red;
		grn = ((CoringData *) fa->filter_data)->grn;
		blu = ((CoringData *) fa->filter_data)->blu;
		
		if ( ((CoringData *) fa->filter_data)->green ) 
			coreval = 255<<8; // Core to green
		else {
			if ( ((CoringData *) fa->filter_data)->corevalue == true ) {
				coreval = red[threshold] | grn[threshold] | blu[threshold]; // Core to "threshold"
			} else {
				coreval = red[0] | grn[0] | blu[0]; // Core to "black"
			}
		}

		h = fa->src.h;
		do {
			w = fa->src.w;

			do {
				old_pixel = *src;

				if (   ((old_pixel>>16) & 0xff) < threshold
					&& ((old_pixel>>8) & 0xff) < threshold
					&& (old_pixel & 0xff) < threshold) {
					*src++ = coreval;
				} else {
					*src++ = red[ (old_pixel >> 16) & 0xff]
						|    grn[ (old_pixel >> 8)  & 0xff]
						|    blu[  old_pixel        & 0xff];
				}
			} while(--w);

			src = (Pixel32 *)((char *)src + fa->src.modulo);
		} while(--h);

	}

    return 0;

}



long coringParamProc(FilterActivation *fa, const FilterFunctions *ff) {
	fa->dst.offset = fa->src.offset;

	return 0;
}



// Creat config gui
BOOL CALLBACK coringConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
    CoringData *mfd = (CoringData *)GetWindowLong(hdlg, DWL_USER);
	int temp, temp2;

    switch(msg) {
    case WM_INITDIALOG:
        SetWindowLong(hdlg, DWL_USER, lParam);
        mfd = (CoringData *)lParam;

		// Store current config
		mfd->oldthreshold = mfd->threshold;
		mfd->oldcorevalue = mfd->corevalue;
	
		// checkbox
        CheckDlgButton(hdlg, IDC_GREEN, mfd->green?BST_CHECKED:BST_UNCHECKED);
        CheckDlgButton(hdlg, IDC_CORE, mfd->corevalue?BST_CHECKED:BST_UNCHECKED);

		// Threshold value
		SetDlgItemInt(hdlg, IDC_BOTTOM, mfd->threshold, true);
		SendMessage(GetDlgItem(hdlg, IDC_BOTTOM_SLIDER), TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(CORE_MIN, CORE_MAX));
		SendMessage(GetDlgItem(hdlg, IDC_BOTTOM_SLIDER), TBM_SETPOS, (WPARAM)TRUE, mfd->threshold);

		// Boost slider
		SetDlgItemInt(hdlg, IDC_BOOST, mfd->boost, true);
		SendMessage(GetDlgItem(hdlg, IDC_BOOST_SLIDER), TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(BOOST_MIN, BOOST_MAX));
		SendMessage(GetDlgItem(hdlg, IDC_BOOST_SLIDER), TBM_SETPOS, (WPARAM)TRUE, mfd->boost);

		// Preview button
		mfd->ifp->InitButton(GetDlgItem(hdlg, IDC_PREVIEW));

        return TRUE;
	case WM_HSCROLL:
		switch(LOWORD(wParam)) {
		case SB_THUMBTRACK:
		case SB_THUMBPOSITION:
			// Scroll bar released
			if ( lParam == (long) GetDlgItem(hdlg, IDC_BOTTOM_SLIDER) ) {
				SetDlgItemInt(hdlg, IDC_BOTTOM, HIWORD(wParam), true);
				mfd->threshold = HIWORD(wParam);
			}
			if ( lParam == (long) GetDlgItem(hdlg, IDC_BOOST_SLIDER) ) {
				SetDlgItemInt(hdlg, IDC_BOOST, HIWORD(wParam), true);
				mfd->boost = HIWORD(wParam);
				makeBoostArray(mfd);
			}
			mfd->ifp->RedoFrame();
			return 0;
		}
		break;
    case WM_COMMAND:
        switch(LOWORD(wParam)) {
        case IDOK:
			// Store new checkboxes
            mfd->green = !!IsDlgButtonChecked(hdlg, IDC_GREEN);
            mfd->corevalue = !!IsDlgButtonChecked(hdlg, IDC_CORE);

			// Threshold
			mfd->threshold = (int) GetDlgItemInt(hdlg, IDC_BOTTOM, &temp, true);
			if (temp == false)  // error
				mfd->threshold = 0;
			mfd->threshold = __max(mfd->threshold, CORE_MIN);
			mfd->threshold = __min(mfd->threshold, CORE_MAX);

			// Boost
			mfd->boost = (int) GetDlgItemInt(hdlg, IDC_BOOST, &temp, true);
			if (temp == false) 
				mfd->boost = 0;
			mfd->boost = __max(mfd->boost, BOOST_MIN);
			mfd->boost = __min(mfd->boost, BOOST_MAX);

			makeBoostArray(mfd);

			// Test is never active unless in config
			mfd->test = false;

            EndDialog(hdlg, 0);
            return TRUE;
        case IDCANCEL:
			// Restore old backed up values
			mfd->threshold = mfd->oldthreshold;
			mfd->corevalue = mfd->oldcorevalue;

			makeBoostArray(mfd);
            EndDialog(hdlg, 1);
            return FALSE;
		case IDC_PREVIEW:
			mfd->ifp->Toggle(GetDlgItem(hdlg, IDC_PREVIEW));
			break;
		case IDC_BOTTOM:
			switch(HIWORD(wParam)) {
			case EN_KILLFOCUS:
				temp2 = (int) GetDlgItemInt(hdlg, IDC_BOTTOM, &temp, true);
				if (temp == false) 
					temp2 = 0;
				SendMessage(GetDlgItem(hdlg, IDC_BOTTOM_SLIDER), TBM_SETPOS, (WPARAM)TRUE, temp2);
				mfd->threshold = temp2;
				mfd->ifp->RedoFrame();
				break;
			}
			break;
		case IDC_BOOST:
			switch(HIWORD(wParam)) {
			case EN_KILLFOCUS:
				temp2 = (int) GetDlgItemInt(hdlg, IDC_BOOST, &temp, true);
				if (temp == false) 
					temp2 = 0;
				SendMessage(GetDlgItem(hdlg, IDC_BOOST_SLIDER), TBM_SETPOS, (WPARAM)TRUE, temp2);
				mfd->boost = temp;
//				makeBoostArray(mfd);
				mfd->ifp->RedoFrame();
				break;
			}
		case IDC_GREEN:
			mfd->green = !!IsDlgButtonChecked(hdlg, IDC_GREEN);
			mfd->ifp->RedoFrame();
			break;
		case IDC_CORE:
            mfd->corevalue = !!IsDlgButtonChecked(hdlg, IDC_CORE);
			mfd->ifp->RedoFrame();
		break;
		}
	break;
	}

    return FALSE;
}

int coringConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd) {
    CoringData *mfd = (CoringData *) fa->filter_data;

	mfd->ifp = fa->ifp;
	mfd->test = true;

	return DialogBoxParam(fa->filter->module->hInstModule,
            MAKEINTRESOURCE(IDD_FILTER_CORING), hwnd,
            coringConfigDlgProc, (LPARAM)fa->filter_data);
}

void coringStringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str) {
	if ( ((CoringData*) fa->filter_data)->corevalue == true ) 
		sprintf(str, " Threshold: %d  Core to threshold", ((CoringData*) fa->filter_data)->threshold);
	else
		sprintf(str, " Threshold: %d  Core to black", ((CoringData*) fa->filter_data)->threshold);
}


// Batch job/script support

void coreScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
    FilterActivation *fa = (FilterActivation *)lpVoid;
    CoringData *mfd = (CoringData *)fa->filter_data;

    mfd->threshold  = argv[0].asInt();
    mfd->corevalue  = !!argv[1].asInt();
}

bool coreFssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
    CoringData *mfd = (CoringData *)fa->filter_data;

    _snprintf(buf, buflen, "Config(%d, %d)",
        mfd->threshold,
        (int) mfd->corevalue);

    return true;
}

