/*
NRS - "Noise reduction suite" filter for VirtualDub
Copyright (C) 2003 Antonio Foranna

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation.
	
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
		
You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
			
The author can be contacted at:
kreel@tiscali.it
*/

#include <windows.h>
#include <commctrl.h>
#include <stdio.h>

#include "resource.h"
#include "filter.h"
#include "Cregistry.h"
#include "Define.h"
#include "Struct.h"
#include "Crop.h"
#include "Gamma.h"
#include "DNR.h"
#include "SmartBlur.h"
#include "Run.h"

//************************************************************************************************

extern void LoadReg(MyFilterData *mfd);
extern void SaveReg(MyFilterData *mfd);

extern HBITMAP hBmRefresh;
extern HBITMAP hBmPicker;

//************************************************************************************************

void SetThresholdLabels(HWND hdlg, int L1, int L2, BYTE Min, BYTE Max)
{
	if(!hdlg || !L2 || !L2)
		return;
	
WORD	i;
float	step=(float)(Max-Min)/(THRESHOLDTABLELEN-1);

WORD	tLog[]={2,4,8,16,32,64,128,256};
char	tmp[11+1], // "<xxx: yyy, "
		buf[THRESHOLDTABLELEN*11+1];
		
	 if(Max<=Min)
		return;

	*buf=0;
	for(i=0; i<THRESHOLDTABLELEN/2-1; i++)
	{
		sprintf(tmp,"<%d: %d, ",tLog[i],(BYTE)(Min+step*i+0.5));
		strcat(buf,tmp);
	}
	sprintf(tmp,"<%d: %d",tLog[i],(BYTE)(Min+step*i+0.5));
	strcat(buf,tmp);
	SetDlgItemText(hdlg, L1, buf); 

	*buf=0;
	for(i=THRESHOLDTABLELEN/2; i<THRESHOLDTABLELEN-1; i++)
	{
		sprintf(tmp,"<%d: %d, ",tLog[i],(BYTE)(Min+step*i+0.5));
		strcat(buf,tmp);
	} 
	sprintf(tmp,"<%d: %d",tLog[i],(BYTE)(Min+step*i+0.5));
	strcat(buf,tmp);
	SetDlgItemText(hdlg, L2, buf); 
}
//------------------------------------------------------------------------------------------------

void InitSliderEdit(HWND hdlg, DWORD S, DWORD E, DWORD Freq, int min, int Max, int Val)
{
char buf[10];
	if(!hdlg)
		return;
	EnableWindow(GetDlgItem(hdlg, S), min || Max);
	SendDlgItemMessage(hdlg, S, TBM_SETTICFREQ, Freq, 0);
	SendDlgItemMessage(hdlg, S, TBM_SETRANGE, true, MAKELONG (min, Max));
	SendDlgItemMessage(hdlg, S, TBM_SETPOS, true, Val);
	sprintf(buf, "%i", Val);
	SetDlgItemText(hdlg, E, buf);
}
//------------------------------------------------------------------------------------------------

void InitSlider(HWND hdlg, DWORD S, DWORD Freq, int min, int Max, int Val)
{
	if(!hdlg)
		return;
	EnableWindow(GetDlgItem(hdlg, S), min || Max);
	SendDlgItemMessage(hdlg, S, TBM_SETTICFREQ, Freq, 0);
	SendDlgItemMessage(hdlg, S, TBM_SETRANGE, true, MAKELONG (min, Max));
    SendDlgItemMessage(hdlg, S, TBM_SETPOS, true, Val);
}
//------------------------------------------------------------------------------------------------

void UpSliderEdit(HWND hdlg, DWORD S, DWORD E, int Val)
{
char buf[10];
	if(!hdlg)
		return;
	SendDlgItemMessage(hdlg, S, TBM_SETPOS, true, Val);
	sprintf(buf, "%i", Val);
	SetDlgItemText(hdlg, E, buf);
}
//------------------------------------------------------------------------------------------------

void UpSlider(HWND hdlg, DWORD S, int Val)
{
	if(!hdlg)
		return;
    SendDlgItemMessage(hdlg, S, TBM_SETPOS, true, Val);
}
//------------------------------------------------------------------------------------------------

void UpEdit(HWND hdlg, DWORD E, int Val)
{
char buf[10];
	if(!hdlg)
		return;
	sprintf(buf, "%i", Val);
	SetDlgItemText(hdlg, E, buf);
}
//------------------------------------------------------------------------------------------------

void InitControls(HWND hdlg, MyFilterData *mfd)
{
	InitSliderEdit(hdlg,IDC_SLIDER_MINT_DNR,IDC_E_MINT_DNR,FREQTHRESHOLD,0,MAXTHRESHOLD_DNR,mfd->MinThresholdDNR);
	InitSliderEdit(hdlg,IDC_SLIDER_MAXT_DNR,IDC_E_MAXT_DNR,FREQTHRESHOLD,1,MAXTHRESHOLD_DNR,mfd->MaxThresholdDNR);
	InitSliderEdit(hdlg,IDC_SLIDER_SCD,IDC_E_SCD,FREQTHRESHOLD,0,MAXTHRESHOLD_SCD,mfd->SCD_threshold);
	InitSliderEdit(hdlg,IDC_SLIDER_EDGES_DNR,IDC_E_EDGES_DNR,FREQTHRESHOLD,2,MAXTHRESHOLD_EDGES,mfd->EdgesThreshold);
	InitSliderEdit(hdlg,IDC_SLIDER_SHARPEN,IDC_E_SHARPEN,FREQTHRESHOLD,-MAXSHARPEN_SB,MAXSHARPEN_SB,mfd->SharpenStrength);
	InitSliderEdit(hdlg,IDC_SLIDER_RADIUS,IDC_E_RADIUS,FREQTHRESHOLD,1,MAXRADIUS_SB,mfd->radius);
	InitSliderEdit(hdlg,IDC_SLIDER_MINT_SB,IDC_E_MINT_SB,FREQTHRESHOLD,0,MAXTHRESHOLD_SB,mfd->MinThresholdSB);
	InitSliderEdit(hdlg,IDC_SLIDER_MAXT_SB,IDC_E_MAXT_SB,FREQTHRESHOLD,1,MAXTHRESHOLD_SB,mfd->MaxThresholdSB);
	InitSliderEdit(hdlg,IDC_SLIDER_TOP,IDC_E_TOP,FREQTHRESHOLD,0,mfd->h,mfd->top);
	InitSliderEdit(hdlg,IDC_SLIDER_BOTTOM,IDC_E_BOTTOM,FREQTHRESHOLD,0,mfd->h,mfd->bottom);
	InitSliderEdit(hdlg,IDC_SLIDER_LEFT,IDC_E_LEFT,FREQTHRESHOLD,0,mfd->w,mfd->left);
	InitSliderEdit(hdlg,IDC_SLIDER_RIGHT,IDC_E_RIGHT,FREQTHRESHOLD,0,mfd->w,mfd->right);
	InitSliderEdit(hdlg,IDC_SLIDER_X,IDC_E_X,FREQTHRESHOLD,-mfd->w,mfd->w,mfd->xShift);
	InitSliderEdit(hdlg,IDC_SLIDER_Y,IDC_E_Y,FREQTHRESHOLD,-mfd->h,mfd->h,mfd->yShift);
	InitSliderEdit(hdlg,IDC_SLIDER_R,IDC_E_R,FREQTHRESHOLD,0,MAXCOLORS,mfd->R);
	InitSliderEdit(hdlg,IDC_SLIDER_G,IDC_E_G,FREQTHRESHOLD,0,MAXCOLORS,mfd->G);
	InitSliderEdit(hdlg,IDC_SLIDER_B,IDC_E_B,FREQTHRESHOLD,0,MAXCOLORS,mfd->B);
	InitSliderEdit(hdlg,IDC_SLIDER_GAMMA,IDC_E_GAMMA,FREQTHRESHOLD,-MAX_GAMMA,MAX_GAMMA,mfd->GammaVal);
	InitSliderEdit(hdlg,IDC_SLIDER_DARK,IDC_E_DARK,FREQTHRESHOLD,0,MAX_DARK,mfd->darkLimit);
	InitSliderEdit(hdlg,IDC_SLIDER_BRIGHT,IDC_E_BRIGHT,FREQTHRESHOLD,1,MAX_BRIGHT,mfd->brightLimit);
	CheckDlgButton(hdlg,IDC_CHK_DNR,mfd->Use_DNR);
	CheckDlgButton(hdlg,IDC_CHK_DNRMP,mfd->UseDnrMultiPass);
	CheckDlgButton(hdlg,IDC_CHK_SHOWBLENDED,mfd->show_blended);
	CheckDlgButton(hdlg,IDC_CHK_SB,mfd->Use_SB);
	CheckDlgButton(hdlg,IDC_CHK_SHOWEDGES,mfd->show_edges);
	CheckDlgButton(hdlg,IDC_CHK_SHARPENEDGES,mfd->SharpenEdges);
	CheckDlgButton(hdlg,IDC_CHK_CROP,mfd->Use_crop);
	CheckDlgButton(hdlg,IDC_CHK_GAMMA,mfd->Use_gamma);
	CheckDlgButton(hdlg,IDC_CHK_INTERLACED,mfd->Interlaced);
	CheckDlgButton(hdlg,IDC_CHK_DISABLEMMX,mfd->DisableMMX);
}
//------------------------------------------------------------------------------------------------

void EnableSliders(HWND hdlg, MyFilterData *mfd)
{
	InitSlider(hdlg,IDC_SLIDER_TOP,FREQTHRESHOLD,0,mfd->h,mfd->top);
	InitSlider(hdlg,IDC_SLIDER_BOTTOM,FREQTHRESHOLD,0,mfd->h,mfd->bottom);
	InitSlider(hdlg,IDC_SLIDER_LEFT,FREQTHRESHOLD,0,mfd->w,mfd->left);
	InitSlider(hdlg,IDC_SLIDER_RIGHT,FREQTHRESHOLD,0,mfd->w,mfd->right);
	InitSlider(hdlg,IDC_SLIDER_X,FREQTHRESHOLD,-mfd->w,mfd->w,mfd->xShift);
	InitSlider(hdlg,IDC_SLIDER_Y,FREQTHRESHOLD,-mfd->h,mfd->h,mfd->yShift);
}
//------------------------------------------------------------------------------------------------

inline void RedoPreview(HWND hdlg, FilterActivation *fa, BYTE ResetData, BYTE ChangeProc, BYTE InitProc)
{
MyFilterData *mfd=(MyFilterData *)fa->filter_data;
int (*RunProcSav)(DWORD *dst, DWORD *src, MyFilterData *mfd)=mfd->pRunProc;

	mfd->ResetOldData=ResetData;
	if(ResetData && InitProc!=INIT_DNR)
		if(InitDNR(mfd,fa->src.data))
			PostMessage(hdlg, WM_COMMAND, IDCANCEL, 0);
	switch(InitProc)
	{
	case INIT_CROP:
		InitCrop(mfd,fa->dst.data);
		if(mfd->pRunProc==RunShiftY && mfd->dst_width!=mfd->w)
			RunProcSav=mfd->pRunProc=RunCrop;
		else
			if(mfd->pRunProc==RunCrop && mfd->dst_width==mfd->wnc)
				RunProcSav=mfd->pRunProc=RunShiftY;
		if(ChangeProc)
		{
			mfd->pRunProc=RunCrop;
			InitCrop(mfd,fa->dst.data);
			InitGamma(mfd);
			if(InitDNR(mfd,fa->src.data))
				PostMessage(hdlg, WM_COMMAND, IDCANCEL, 0);
			if(InitSB(mfd))
				PostMessage(hdlg, WM_COMMAND, IDCANCEL, 0);
		}
		else
			if(mfd->Use_SB)
				if(InitSB(mfd))
					PostMessage(hdlg, WM_COMMAND, IDCANCEL, 0);
		break;
	case INIT_GAMMA:
		InitGamma(mfd);
		if(ChangeProc)
			mfd->pRunProc=RunGamma;
		break;
	case INIT_DNR:
		if(InitDNR(mfd,fa->src.data))
			PostMessage(hdlg, WM_COMMAND, IDCANCEL, 0);
		break;
	case INIT_SB:
		if(InitSB(mfd))
			PostMessage(hdlg, WM_COMMAND, IDCANCEL, 0);
		break;
	case INIT_ALL:
		InitCrop(mfd,fa->dst.data);
		InitGamma(mfd);
		if(InitDNR(mfd,fa->src.data))
			PostMessage(hdlg, WM_COMMAND, IDCANCEL, 0);
		if(InitSB(mfd))
			PostMessage(hdlg, WM_COMMAND, IDCANCEL, 0);
		break;
	}
	if(mfd->PreviewMode)
		fa->ifp->RedoFrame();
	mfd->pRunProc=RunProcSav;
}
//------------------------------------------------------------------------------------------------

BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
FilterActivation *fa=(FilterActivation *)GetWindowLong(hdlg, DWL_USER);
MyFilterData *mfd;
	if(fa)
		mfd=(MyFilterData *)fa->filter_data;

    switch(msg)
	{
        case WM_INITDIALOG:
            {
                SetWindowLong(hdlg, DWL_USER, lParam);
			FilterActivation *fa=(FilterActivation *)GetWindowLong(hdlg, DWL_USER);
			MyFilterData *mfd;
				if(fa)
					mfd=(MyFilterData *)fa->filter_data;

				SendMessage(GetDlgItem(hdlg, IDC_BTN_RESETOLD), BM_SETIMAGE, IMAGE_BITMAP, (LPARAM) hBmRefresh);
				SendMessage(GetDlgItem(hdlg, IDC_BTN_GETLIMITS), BM_SETIMAGE, IMAGE_BITMAP, (LPARAM) hBmPicker);

				fa->ifp->InitButton(GetDlgItem(hdlg,IDC_BTN_PREVIEW));
				mfd->PreviewMode=0;
				mfd->NewImage=1;

				SetThresholdLabels(hdlg,IDC_L_THRESHOLD1_DNR,IDC_L_THRESHOLD2_DNR,mfd->MinThresholdDNR,mfd->MaxThresholdDNR);
				SetThresholdLabels(hdlg,IDC_L_THRESHOLD1_SB,IDC_L_THRESHOLD2_SB,mfd->MinThresholdSB,mfd->MaxThresholdSB);

				InitControls(hdlg,mfd);
				EnableWindow(GetDlgItem(hdlg,IDC_SLIDER_EDGES_DNR),mfd->UseDnrMultiPass);
				EnableWindow(GetDlgItem(hdlg,IDC_E_EDGES_DNR),mfd->UseDnrMultiPass);
				PostMessage(hdlg, WM_HSCROLL, SB_ENDSCROLL, IDC_SLIDER_EDGES_DNR);

				EnableWindow(GetDlgItem(hdlg,IDC_SLIDER_SHARPEN),mfd->SharpenEdges);
				EnableWindow(GetDlgItem(hdlg,IDC_E_SHARPEN),mfd->SharpenEdges);
				PostMessage(hdlg, WM_HSCROLL, SB_ENDSCROLL, IDC_SLIDER_SHARPEN);

				PostMessage(hdlg, WM_HSCROLL, SB_ENDSCROLL, IDC_SLIDER_TOP);
				PostMessage(hdlg, WM_HSCROLL, SB_ENDSCROLL, IDC_SLIDER_BOTTOM);
				PostMessage(hdlg, WM_HSCROLL, SB_ENDSCROLL, IDC_SLIDER_LEFT);
				PostMessage(hdlg, WM_HSCROLL, SB_ENDSCROLL, IDC_SLIDER_RIGHT);
			}
            break;
			// END WM_INITDIALOG

        case WM_HSCROLL:
			{
				switch(GetDlgCtrlID((HWND)lParam))
				{
				case IDC_SLIDER_EDGES_DNR:
					mfd->EdgesThreshold=(BYTE)SendDlgItemMessage(hdlg, IDC_SLIDER_EDGES_DNR, TBM_GETPOS, 0, 0);
					if(mfd->EdgesThreshold<2)
					{
						mfd->EdgesThreshold=2;
						UpSlider(hdlg,IDC_SLIDER_EDGES_DNR,mfd->EdgesThreshold);
					}
					UpEdit(hdlg,IDC_E_EDGES_DNR,mfd->EdgesThreshold);
					if(mfd->UseDnrMultiPass)
						if(g_MMXenabled)
							SetLogThresholdSquares(mfd,mfd->pLogThresholdEdgesDNR,mfd->EdgesThreshold-2,mfd->EdgesThreshold+2);
//							SetLogThresholdABS(mfd,mfd->pLogThresholdEdgesDNR,mfd->EdgesThreshold-2,mfd->EdgesThreshold+2);
						else
							SetLogThresholdSquares(mfd,mfd->pLogThresholdEdgesDNR,mfd->EdgesThreshold-2,mfd->EdgesThreshold+2);
					if(LOWORD(wParam)==SB_ENDSCROLL)
					{
						if(mfd->Use_SB)
						{
							UpSlider(hdlg,IDC_SLIDER_MAXT_SB,mfd->EdgesThreshold+2);
							PostMessage(hdlg, WM_HSCROLL, SB_ENDSCROLL, (long)GetDlgItem(hdlg,IDC_SLIDER_MAXT_SB));
						}

						RedoPreview(hdlg,fa,1,0,INIT_DNR);
					}
					break;

				case IDC_SLIDER_SCD:
					mfd->SCD_threshold=(BYTE)SendDlgItemMessage(hdlg, IDC_SLIDER_SCD, TBM_GETPOS, 0, 0);
					mfd->SCD_NumOfPixels=((mfd->sizeDst>>2)*mfd->SCD_threshold)/100;
					UpEdit(hdlg,IDC_E_SCD,mfd->SCD_threshold);
					if(LOWORD(wParam)==SB_ENDSCROLL)
						RedoPreview(hdlg,fa,0,0,INIT_DNR);
					break;

				case IDC_SLIDER_MINT_DNR:
					mfd->MinThresholdDNR=(BYTE)SendDlgItemMessage(hdlg, IDC_SLIDER_MINT_DNR, TBM_GETPOS, 0, 0);
					if(mfd->MinThresholdDNR>=mfd->MaxThresholdDNR)
					{
						mfd->MinThresholdDNR=mfd->MaxThresholdDNR-1;
						UpSlider(hdlg,IDC_SLIDER_MINT_DNR,mfd->MinThresholdDNR);
					}
					UpEdit(hdlg,IDC_E_MINT_DNR,mfd->MinThresholdDNR);

					SetThresholdLabels(hdlg,IDC_L_THRESHOLD1_DNR,IDC_L_THRESHOLD2_DNR,mfd->MinThresholdDNR,mfd->MaxThresholdDNR);
					SetLogThresholdSpace(mfd,mfd->pLogThresholdDNR,mfd->MinThresholdDNR,mfd->MaxThresholdDNR);
					if(LOWORD(wParam)==SB_ENDSCROLL)
					{
//						UpSlider(hdlg,IDC_SLIDER_MAXT_DNR,(mfd->MinThresholdDNR*2)<=MAXTHRESHOLD_DNR ? mfd->MinThresholdDNR*2 : MAXTHRESHOLD_DNR);
//						PostMessage(hdlg, WM_HSCROLL, 0, (long)GetDlgItem(hdlg,IDC_SLIDER_MAXT_DNR));

						RedoPreview(hdlg,fa,0,0,INIT_DNR);
					}
					break;
				case IDC_SLIDER_MAXT_DNR:
					mfd->MaxThresholdDNR=(BYTE)SendDlgItemMessage(hdlg, IDC_SLIDER_MAXT_DNR, TBM_GETPOS, 0, 0);
/*					if(mfd->MaxThresholdDNR<=mfd->MinThresholdDNR)
					{
						mfd->MaxThresholdDNR=mfd->MinThresholdDNR+1;
						UpSlider(hdlg,IDC_SLIDER_MAXT_DNR,mfd->MaxThresholdDNR);
					}*/
					UpEdit(hdlg,IDC_E_MAXT_DNR,mfd->MaxThresholdDNR);

					SetThresholdLabels(hdlg,IDC_L_THRESHOLD1_DNR,IDC_L_THRESHOLD2_DNR,mfd->MinThresholdDNR,mfd->MaxThresholdDNR);
					SetLogThresholdSpace(mfd,mfd->pLogThresholdDNR,mfd->MinThresholdDNR,mfd->MaxThresholdDNR);
					if(LOWORD(wParam)==SB_ENDSCROLL)
					{
						UpSlider(hdlg,IDC_SLIDER_MINT_DNR,mfd->MaxThresholdDNR-4);
						PostMessage(hdlg, WM_HSCROLL, 0, (long)GetDlgItem(hdlg,IDC_SLIDER_MINT_DNR));

						RedoPreview(hdlg,fa,0,0,INIT_DNR);
					}
					break;

				case IDC_SLIDER_SHARPEN:
					mfd->SharpenStrength=(BYTE)SendDlgItemMessage(hdlg, IDC_SLIDER_SHARPEN, TBM_GETPOS, 0, 0);
					UpEdit(hdlg,IDC_E_SHARPEN,mfd->SharpenStrength);
					if(LOWORD(wParam)==SB_ENDSCROLL)
						RedoPreview(hdlg,fa,1,0,INIT_SB);
					break;

				case IDC_SLIDER_RADIUS:
					mfd->radius=(BYTE)SendDlgItemMessage(hdlg, IDC_SLIDER_RADIUS, TBM_GETPOS, 0, 0);
					UpEdit(hdlg,IDC_E_RADIUS,mfd->radius);
					if(LOWORD(wParam)==SB_ENDSCROLL)
						RedoPreview(hdlg,fa,1,0,INIT_SB);
					break;

				case IDC_SLIDER_MINT_SB:
					mfd->MinThresholdSB=(BYTE)SendDlgItemMessage(hdlg, IDC_SLIDER_MINT_SB, TBM_GETPOS, 0, 0);
					if(mfd->MinThresholdSB>=mfd->MaxThresholdSB)
					{
						mfd->MinThresholdSB=mfd->MaxThresholdSB-1;
						UpSlider(hdlg,IDC_SLIDER_MINT_SB,mfd->MinThresholdSB);
					}
					UpEdit(hdlg,IDC_E_MINT_SB,mfd->MinThresholdSB);

					SetThresholdLabels(hdlg,IDC_L_THRESHOLD1_SB,IDC_L_THRESHOLD2_SB,mfd->MinThresholdSB,mfd->MaxThresholdSB);
					if(g_MMXenabled)
						SetLogThresholdABS(mfd,mfd->pLogThresholdSB,mfd->MinThresholdSB,mfd->MaxThresholdSB);
					if(LOWORD(wParam)==SB_ENDSCROLL)
					{
//						UpSlider(hdlg,IDC_SLIDER_MAXT_SB,(mfd->MinThresholdSB*2)<=MAXTHRESHOLD_SB ? mfd->MinThresholdSB*2 : MAXTHRESHOLD_SB);
//						PostMessage(hdlg, WM_HSCROLL, 0, (long)GetDlgItem(hdlg,IDC_SLIDER_MAXT_SB));

						RedoPreview(hdlg,fa,1,0,INIT_SB);
					}
					break;
				case IDC_SLIDER_MAXT_SB:
					mfd->MaxThresholdSB=(BYTE)SendDlgItemMessage(hdlg, IDC_SLIDER_MAXT_SB, TBM_GETPOS, 0, 0);
/*					if(mfd->MaxThresholdSB<=mfd->MinThresholdSB)
					{
						mfd->MaxThresholdSB=mfd->MinThresholdSB+1;
						UpSlider(hdlg,IDC_SLIDER_MAXT_SB,mfd->MaxThresholdSB);
					}*/
					UpEdit(hdlg,IDC_E_MAXT_SB,mfd->MaxThresholdSB);

					SetThresholdLabels(hdlg,IDC_L_THRESHOLD1_SB,IDC_L_THRESHOLD2_SB,mfd->MinThresholdSB,mfd->MaxThresholdSB);
					if(g_MMXenabled)
						SetLogThresholdABS(mfd,mfd->pLogThresholdSB,mfd->MinThresholdSB,mfd->MaxThresholdSB);
					if(LOWORD(wParam)==SB_ENDSCROLL)
					{
						UpSlider(hdlg,IDC_SLIDER_MINT_SB,mfd->MaxThresholdSB-4);
						PostMessage(hdlg, WM_HSCROLL, 0, (long)GetDlgItem(hdlg,IDC_SLIDER_MINT_SB));

						if(mfd->Use_DNR && mfd->UseDnrMultiPass)
						{
							UpSlider(hdlg,IDC_SLIDER_EDGES_DNR,mfd->MaxThresholdSB-2);
							PostMessage(hdlg, WM_HSCROLL, 0, (long)GetDlgItem(hdlg,IDC_SLIDER_EDGES_DNR));
						}

						RedoPreview(hdlg,fa,1,0,INIT_SB);
					}
					break;

				case IDC_SLIDER_TOP:
					mfd->top=(WORD)SendDlgItemMessage(hdlg, IDC_SLIDER_TOP, TBM_GETPOS, 0, 0);
					if(mfd->top+mfd->bottom>mfd->h)
					{
						mfd->top=mfd->h-mfd->bottom;
						UpSlider(hdlg,IDC_SLIDER_TOP,mfd->top);
					}
					UpEdit(hdlg,IDC_E_TOP,mfd->top);
					RedoPreview(hdlg,fa,1,LOWORD(wParam)!=SB_ENDSCROLL && mfd->PreviewMode,INIT_CROP);
					break;
				case IDC_SLIDER_BOTTOM:
					mfd->bottom=(WORD)SendDlgItemMessage(hdlg, IDC_SLIDER_BOTTOM, TBM_GETPOS, 0, 0);
					if(mfd->top+mfd->bottom>mfd->h)
					{
						mfd->bottom=mfd->h-mfd->top;
						UpSlider(hdlg,IDC_SLIDER_BOTTOM,mfd->bottom);
					}
					UpEdit(hdlg,IDC_E_BOTTOM,mfd->bottom);
					RedoPreview(hdlg,fa,1,LOWORD(wParam)!=SB_ENDSCROLL && mfd->PreviewMode,INIT_CROP);
					break;
				case IDC_SLIDER_LEFT:
					mfd->left=(WORD)SendDlgItemMessage(hdlg, IDC_SLIDER_LEFT, TBM_GETPOS, 0, 0);
					if(mfd->left+mfd->right>mfd->w)
					{
						mfd->left=mfd->w-mfd->right;
						UpSlider(hdlg,IDC_SLIDER_LEFT,mfd->left);
					}
					UpEdit(hdlg,IDC_E_LEFT,mfd->left);
					RedoPreview(hdlg,fa,1,LOWORD(wParam)!=SB_ENDSCROLL && mfd->PreviewMode,INIT_CROP);
					break;
				case IDC_SLIDER_RIGHT:
					mfd->right=(WORD)SendDlgItemMessage(hdlg, IDC_SLIDER_RIGHT, TBM_GETPOS, 0, 0);
					if(mfd->left+mfd->right>mfd->w)
					{
						mfd->right=mfd->w-mfd->left;
						UpSlider(hdlg,IDC_SLIDER_RIGHT,mfd->right);
					}
					UpEdit(hdlg,IDC_E_RIGHT,mfd->right);
					RedoPreview(hdlg,fa,1,LOWORD(wParam)!=SB_ENDSCROLL && mfd->PreviewMode,INIT_CROP);
					break;
				case IDC_SLIDER_X:
					mfd->xShift=(WORD)SendDlgItemMessage(hdlg, IDC_SLIDER_X, TBM_GETPOS, 0, 0);
					if(mfd->xShift+((mfd->left+mfd->right)>>1)>mfd->w)
					{
						mfd->xShift=mfd->w-((mfd->left+mfd->right)>>1);
						UpSlider(hdlg,IDC_SLIDER_X,mfd->xShift);
					}
					else
						if(mfd->xShift+((mfd->left+mfd->right)>>1)+(mfd->w-mfd->right-mfd->left)<0)
						{
							mfd->xShift=-(((mfd->left+mfd->right)>>1)+(mfd->w-mfd->right-mfd->left));
							UpSlider(hdlg,IDC_SLIDER_X,mfd->xShift);
						}
					UpEdit(hdlg,IDC_E_X,mfd->xShift);
					RedoPreview(hdlg,fa,1,LOWORD(wParam)!=SB_ENDSCROLL && mfd->PreviewMode,INIT_CROP);
					break;
				case IDC_SLIDER_Y:
					mfd->yShift=(WORD)SendDlgItemMessage(hdlg, IDC_SLIDER_Y, TBM_GETPOS, 0, 0);
					if(mfd->yShift+((mfd->top+mfd->bottom)>>1)>mfd->h)
					{
						mfd->yShift=mfd->h-((mfd->top+mfd->bottom)>>1);
						UpSlider(hdlg,IDC_SLIDER_Y,mfd->yShift);
					}
					else
						if(mfd->yShift+((mfd->top+mfd->bottom)>>1)+(mfd->h-mfd->bottom-mfd->top)<0)
						{
							mfd->yShift=-(((mfd->top+mfd->bottom)>>1)+(mfd->h-mfd->bottom-mfd->top));
							UpSlider(hdlg,IDC_SLIDER_Y,mfd->yShift);
						}
					UpEdit(hdlg,IDC_E_Y,mfd->yShift);
					RedoPreview(hdlg,fa,1,LOWORD(wParam)!=SB_ENDSCROLL && mfd->PreviewMode,INIT_CROP);
					break;

				case IDC_SLIDER_R:
					mfd->R=(BYTE)SendDlgItemMessage(hdlg, IDC_SLIDER_R, TBM_GETPOS, 0, 0);
					UpEdit(hdlg,IDC_E_R,mfd->R);
					RedoPreview(hdlg,fa,0,LOWORD(wParam)!=SB_ENDSCROLL && mfd->PreviewMode,INIT_CROP);
					break;
				case IDC_SLIDER_G:
					mfd->G=(BYTE)SendDlgItemMessage(hdlg, IDC_SLIDER_G, TBM_GETPOS, 0, 0);
					UpEdit(hdlg,IDC_E_G,mfd->G);
					RedoPreview(hdlg,fa,0,LOWORD(wParam)!=SB_ENDSCROLL && mfd->PreviewMode,INIT_CROP);
					break;
				case IDC_SLIDER_B:
					mfd->B=(BYTE)SendDlgItemMessage(hdlg, IDC_SLIDER_B, TBM_GETPOS, 0, 0);
					UpEdit(hdlg,IDC_E_B,mfd->B);
					RedoPreview(hdlg,fa,0,LOWORD(wParam)!=SB_ENDSCROLL && mfd->PreviewMode,INIT_CROP);
					break;

				case IDC_SLIDER_GAMMA:
					mfd->GammaVal=(BYTE)SendDlgItemMessage(hdlg, IDC_SLIDER_GAMMA, TBM_GETPOS, 0, 0);
					if(mfd->GammaVal>=-10 && mfd->GammaVal<0)
					{
						mfd->GammaVal=10;
						UpSlider(hdlg,IDC_SLIDER_GAMMA,mfd->GammaVal);
					}
					else
						if(mfd->GammaVal>=0 && mfd->GammaVal<10)
						{
							mfd->GammaVal=-11;
							UpSlider(hdlg,IDC_SLIDER_GAMMA,mfd->GammaVal);
						}
					UpEdit(hdlg,IDC_E_GAMMA,mfd->GammaVal);
					RedoPreview(hdlg,fa,1,LOWORD(wParam)!=SB_ENDSCROLL && mfd->PreviewMode,INIT_GAMMA);
					break;
				case IDC_SLIDER_DARK:
					mfd->darkLimit=(BYTE)SendDlgItemMessage(hdlg, IDC_SLIDER_DARK, TBM_GETPOS, 0, 0);
					if(mfd->darkLimit>=mfd->brightLimit)
					{
						mfd->darkLimit=mfd->brightLimit-1;
						UpSlider(hdlg,IDC_SLIDER_DARK,mfd->darkLimit);
					}
					UpEdit(hdlg,IDC_E_DARK,mfd->darkLimit);
					RedoPreview(hdlg,fa,1,LOWORD(wParam)!=SB_ENDSCROLL && mfd->PreviewMode,INIT_GAMMA);
					break;
				case IDC_SLIDER_BRIGHT:
					mfd->brightLimit=(BYTE)SendDlgItemMessage(hdlg, IDC_SLIDER_BRIGHT, TBM_GETPOS, 0, 0);
					if(mfd->brightLimit<=mfd->darkLimit)
					{
						mfd->brightLimit=mfd->darkLimit+1;
						UpSlider(hdlg,IDC_SLIDER_BRIGHT,mfd->brightLimit);
					}
					UpEdit(hdlg,IDC_E_BRIGHT,mfd->brightLimit);
					RedoPreview(hdlg,fa,1,LOWORD(wParam)!=SB_ENDSCROLL && mfd->PreviewMode,INIT_GAMMA);
					break;
				}
			}
			break;
			// END WM_HSCROLL

		case WM_COMMAND:
		{
			switch(HIWORD(wParam))
			{
			case EN_KILLFOCUS:
				switch(LOWORD(wParam))
				{
				case IDC_E_EDGES_DNR:
					mfd->EdgesThreshold=GetDlgItemInt (hdlg, IDC_E_EDGES_DNR,0,true);
					UpSlider(hdlg,IDC_SLIDER_EDGES_DNR,mfd->EdgesThreshold);
					PostMessage(hdlg, WM_HSCROLL, SB_ENDSCROLL, (long)GetDlgItem(hdlg,IDC_SLIDER_EDGES_DNR));
					RedoPreview(hdlg,fa,1,0,INIT_DNR);
					break;

				case IDC_E_SCD:
					mfd->SCD_threshold=GetDlgItemInt (hdlg, IDC_E_SCD,0,true);
					UpSlider(hdlg,IDC_SLIDER_SCD,mfd->SCD_threshold);
					RedoPreview(hdlg,fa,0,0,INIT_DNR);
					break;

				case IDC_E_MINT_DNR:
					mfd->MinThresholdDNR=GetDlgItemInt (hdlg, IDC_E_MINT_DNR,0,true);
					UpSlider(hdlg,IDC_SLIDER_MINT_DNR,mfd->MinThresholdDNR);
					PostMessage(hdlg, WM_HSCROLL, SB_ENDSCROLL, (long)GetDlgItem(hdlg,IDC_SLIDER_MINT_DNR));
					RedoPreview(hdlg,fa,0,0,INIT_DNR);
					break;
				case IDC_E_MAXT_DNR:
					mfd->MaxThresholdDNR=GetDlgItemInt (hdlg, IDC_E_MAXT_DNR,0,true);
					UpSlider(hdlg,IDC_SLIDER_MAXT_DNR,mfd->MaxThresholdDNR);
					PostMessage(hdlg, WM_HSCROLL, SB_ENDSCROLL, (long)GetDlgItem(hdlg,IDC_SLIDER_MAXT_DNR));
					RedoPreview(hdlg,fa,0,0,INIT_DNR);
					break;

				case IDC_E_SHARPEN:
					mfd->SharpenStrength=GetDlgItemInt (hdlg, IDC_E_SHARPEN,0,true);
					UpSliderEdit(hdlg,IDC_SLIDER_SHARPEN,IDC_E_SHARPEN,mfd->SharpenStrength);
					RedoPreview(hdlg,fa,1,0,INIT_SB);
					break;

				case IDC_E_RADIUS:
					mfd->radius=GetDlgItemInt (hdlg, IDC_E_RADIUS,0,true);
					UpSliderEdit(hdlg,IDC_SLIDER_RADIUS,IDC_E_RADIUS,mfd->radius);
					RedoPreview(hdlg,fa,1,0,INIT_SB);
					break;

				case IDC_E_MINT_SB:
					mfd->MinThresholdSB=GetDlgItemInt (hdlg, IDC_E_MINT_SB,0,true);
					UpSliderEdit(hdlg,IDC_SLIDER_MINT_SB,IDC_E_MINT_SB,mfd->MinThresholdSB);
					PostMessage(hdlg, WM_HSCROLL, SB_ENDSCROLL, (long)GetDlgItem(hdlg,IDC_SLIDER_MINT_SB));
					RedoPreview(hdlg,fa,1,0,INIT_SB);
					break;
				case IDC_E_MAXT_SB:
					mfd->MaxThresholdSB=GetDlgItemInt (hdlg, IDC_E_MAXT_SB,0,true);
					UpSliderEdit(hdlg,IDC_SLIDER_MAXT_SB,IDC_E_MAXT_SB,mfd->MaxThresholdSB);
					PostMessage(hdlg, WM_HSCROLL, SB_ENDSCROLL, (long)GetDlgItem(hdlg,IDC_SLIDER_MAXT_SB));
					RedoPreview(hdlg,fa,1,0,INIT_SB);
					break;

				case IDC_E_TOP:
					mfd->top=GetDlgItemInt (hdlg, IDC_E_TOP,0,true);
					UpSliderEdit(hdlg,IDC_SLIDER_TOP,IDC_E_TOP,mfd->top);
					RedoPreview(hdlg,fa,1,0,INIT_CROP);
					break;
				case IDC_E_BOTTOM:
					mfd->bottom=GetDlgItemInt (hdlg, IDC_E_BOTTOM,0,true);
					UpSliderEdit(hdlg,IDC_SLIDER_BOTTOM,IDC_E_BOTTOM,mfd->bottom);
					RedoPreview(hdlg,fa,1,0,INIT_CROP);
					break;
				case IDC_E_LEFT:
					mfd->left=GetDlgItemInt (hdlg, IDC_E_LEFT,0,true);
					UpSliderEdit(hdlg,IDC_SLIDER_LEFT,IDC_E_LEFT,mfd->left);
					RedoPreview(hdlg,fa,1,0,INIT_CROP);
					break;
				case IDC_E_RIGHT:
					mfd->right=GetDlgItemInt (hdlg, IDC_E_RIGHT,0,true);
					UpSliderEdit(hdlg,IDC_SLIDER_RIGHT,IDC_E_RIGHT,mfd->right);
					RedoPreview(hdlg,fa,1,0,INIT_CROP);
					break;
				case IDC_E_X:
					mfd->xShift=GetDlgItemInt (hdlg, IDC_E_X,0,true); 
					UpSlider(hdlg,IDC_SLIDER_X,mfd->xShift);
					RedoPreview(hdlg,fa,1,0,INIT_CROP);
					break;
				case IDC_E_Y:
					mfd->yShift=(int)GetDlgItemInt (hdlg, IDC_E_Y,0,true);
					UpSlider(hdlg,IDC_SLIDER_Y,mfd->yShift);
					RedoPreview(hdlg,fa,1,0,INIT_CROP);
					break;

				case IDC_E_R:
					mfd->R=(int)GetDlgItemInt (hdlg, IDC_E_R,0,true);
					UpSliderEdit(hdlg,IDC_SLIDER_R,IDC_E_R,mfd->R);
					RedoPreview(hdlg,fa,0,0,INIT_CROP);
					break;
				case IDC_E_G:
					mfd->G=(int)GetDlgItemInt (hdlg, IDC_E_G,0,true);
					UpSliderEdit(hdlg,IDC_SLIDER_G,IDC_E_G,mfd->G);
					RedoPreview(hdlg,fa,0,0,INIT_CROP);
					break;
				case IDC_E_B:
					mfd->B=(int)GetDlgItemInt (hdlg, IDC_E_B,0,true);
					UpSliderEdit(hdlg,IDC_SLIDER_B,IDC_E_B,mfd->B);
					RedoPreview(hdlg,fa,0,0,INIT_CROP);
					break;

				case IDC_E_GAMMA:
					mfd->GammaVal=(int)GetDlgItemInt (hdlg, IDC_E_GAMMA,0,true);
					UpSliderEdit(hdlg,IDC_SLIDER_GAMMA,IDC_E_GAMMA,mfd->GammaVal);
					RedoPreview(hdlg,fa,1,0,INIT_GAMMA);
					break;
				case IDC_E_DARK:
					mfd->darkLimit=(int)GetDlgItemInt (hdlg, IDC_E_DARK,0,true);
					UpSliderEdit(hdlg,IDC_SLIDER_DARK,IDC_E_DARK,mfd->darkLimit);
					RedoPreview(hdlg,fa,1,0,INIT_GAMMA);
					break;
				case IDC_E_BRIGHT:
					mfd->brightLimit=(int)GetDlgItemInt (hdlg, IDC_E_BRIGHT,0,true);
					UpSliderEdit(hdlg,IDC_SLIDER_BRIGHT,IDC_E_BRIGHT,mfd->brightLimit);
					RedoPreview(hdlg,fa,1,0,INIT_GAMMA);
					break;
				// END EN_KILLFOCUS
             }
			break;

			default:
				switch(LOWORD(wParam))
				{
					case IDOK:
						SaveReg(mfd);

						if(fa->ifp->isPreviewEnabled())
							fa->ifp->Close();
						mfd->PreviewMode=0;

						EndDialog(hdlg, 0);
					break;

		            case IDCANCEL:
						if(fa->ifp->isPreviewEnabled())
							fa->ifp->Close();
						mfd->PreviewMode=0;

						EndDialog(hdlg, 1);

//						LoadReg(mfd);
		                return FALSE;

					case IDC_BTN_RELOAD:
						{
							LoadReg(mfd);
							SetThresholdLabels(hdlg,IDC_L_THRESHOLD1_DNR,IDC_L_THRESHOLD2_DNR,mfd->MinThresholdDNR,mfd->MaxThresholdDNR);
							InitControls(hdlg,mfd);
							RedoPreview(hdlg,fa,1,0,INIT_ALL);
						}
						break;

					case IDC_BTN_RESET:
						{
							mfd->MinThresholdDNR = DEF_MINTHRESHOLD_DNR;
							mfd->MaxThresholdDNR = DEF_MAXTHRESHOLD_DNR;
							mfd->SCD_threshold = DEF_SCD;
							mfd->UseDnrMultiPass= DEF_USE_DNRMP;
							SetThresholdLabels(hdlg,IDC_L_THRESHOLD1_DNR,IDC_L_THRESHOLD2_DNR,mfd->MinThresholdDNR,mfd->MaxThresholdDNR);
	
							mfd->MinThresholdSB = DEF_MINTHRESHOLD_SB;
							mfd->MaxThresholdSB = DEF_MAXTHRESHOLD_SB;
							mfd->radius = DEF_RADIUS_SB;
							mfd->Interlaced = DEF_INTERLACED;
							SetThresholdLabels(hdlg,IDC_L_THRESHOLD1_SB,IDC_L_THRESHOLD2_SB,mfd->MinThresholdSB,mfd->MaxThresholdSB);

							mfd->left = DEF_CROP;
							mfd->right = DEF_CROP;
							mfd->top = DEF_CROP;
							mfd->bottom = DEF_CROP;
							mfd->xShift = DEF_SHIFT;
							mfd->yShift = DEF_SHIFT;
							mfd->R = DEF_COLORS;
							mfd->G = DEF_COLORS;
							mfd->B = DEF_COLORS;

							mfd->GammaVal=DEF_GAMMA;
							mfd->darkLimit=DEF_DARK_LIMIT;
							mfd->brightLimit=DEF_BRIGHT_LIMIT;
						
							InitControls(hdlg,mfd);
							RedoPreview(hdlg,fa,1,0,INIT_ALL);
						}
						break;

					case IDC_BTN_PREVIEW:
						mfd->PreviewMode=!mfd->PreviewMode;
						InitFuncs(mfd);
						fa->ifp->Toggle(hdlg);
						InitFuncs(mfd); // this to activate MMX
						EnableWindow(GetDlgItem(hdlg, IDC_BTN_RESETOLD), mfd->PreviewMode);
						EnableWindow(GetDlgItem(hdlg, IDC_BTN_GETLIMITS), mfd->PreviewMode);
						EnableWindow(GetDlgItem(hdlg, IDC_CHK_DISABLEALL), mfd->PreviewMode);
						if(mfd->PreviewMode)
							EnableSliders(hdlg,mfd);
						break;

					case IDC_BTN_ABOUT:
						{
						char buf[256];
						sprintf(buf,	APP_NAME " plugin " APP_VER " by Antonio Foranna\n\n"
										"Compiled on %s",
										__DATE__
										);
							MessageBox(hdlg, buf, "About", MB_OK);
						}
						break;

					case IDC_BTN_RESETOLD:
						if(mfd->Use_DNR)
							RedoPreview(hdlg,fa,1,0,INIT_ALL);
						break;

					case IDC_BTN_GETLIMITS:
						{
						short	xOffset=0,yOffset=0;
						BYTE	*bsrc=(BYTE *)((DWORD *)fa->src.data+mfd->src_shift);
							mfd->darkLimit=255;
							mfd->brightLimit=0;
							while(yOffset<mfd->dst_heigth)
							{
								mfd->darkLimit=(BYTE)Min(mfd->darkLimit,(*(bsrc+2)*299L+*(bsrc+1)*587L+*bsrc*114L)/1000);
								mfd->brightLimit=(BYTE)Max(mfd->brightLimit,(*(bsrc+2)*299L+*(bsrc+1)*587L+*bsrc*114L)/1000);
								bsrc+=4;
								if(++xOffset>=mfd->dst_width)
								{
									xOffset=0;
									yOffset++;
									bsrc+=mfd->xGap<<2;
								}
						    }
							UpSliderEdit(hdlg,IDC_SLIDER_DARK,IDC_E_DARK,mfd->darkLimit);
							UpSliderEdit(hdlg,IDC_SLIDER_BRIGHT,IDC_E_BRIGHT,mfd->brightLimit);
							if(mfd->Use_gamma)
								RedoPreview(hdlg,fa,1,0,INIT_GAMMA);
						}
						break;
				
					case IDC_CHK_CROP:
						if(!IsDlgButtonChecked(hdlg, IDC_CHK_DISABLEALL))
						{
							mfd->Use_crop=IsDlgButtonChecked(hdlg, IDC_CHK_CROP);
							InitFuncs(mfd);
							RedoPreview(hdlg,fa,1,0,INIT_CROP);
						}
						break;

					case IDC_CHK_DNR:
					case IDC_CHK_SHOWBLENDED:
						if(IsDlgButtonChecked(hdlg, IDC_CHK_DISABLEALL))
							break;
						mfd->Use_DNR=IsDlgButtonChecked(hdlg, IDC_CHK_DNR);
						mfd->show_blended=IsDlgButtonChecked(hdlg, IDC_CHK_SHOWBLENDED);
						if(mfd->show_blended && mfd->Use_DNR)
						{
							if(mfd->show_edges)
							{
								CheckDlgButton(hdlg,IDC_CHK_SHOWEDGES,false);
								mfd->show_edges=0;
							}
//							mfd->Use_SB=0;
//							mfd->Use_gamma=0;
						}
						else
						{
							if(mfd->Use_SB!=IsDlgButtonChecked(hdlg, IDC_CHK_SB))
								PostMessage(hdlg, WM_COMMAND, IDC_CHK_SB, 0);
							if(mfd->Use_gamma!=IsDlgButtonChecked(hdlg, IDC_CHK_GAMMA))
								PostMessage(hdlg, WM_COMMAND, IDC_CHK_GAMMA, 0);
							mfd->Use_SB=IsDlgButtonChecked(hdlg, IDC_CHK_SB);
							mfd->Use_gamma=IsDlgButtonChecked(hdlg, IDC_CHK_GAMMA);
						}
						InitGamma(mfd);
						InitFuncs(mfd);
						RedoPreview(hdlg,fa,1,0,INIT_DNR);
						break;

					case IDC_CHK_DNREDGES:
						mfd->show_DNR_edges=IsDlgButtonChecked(hdlg, IDC_CHK_DNREDGES);
						if(mfd->show_DNR_edges && !IsDlgButtonChecked(hdlg, IDC_CHK_SHOWBLENDED))
						{
							CheckDlgButton(hdlg,IDC_CHK_SHOWBLENDED,true);
							PostMessage(hdlg, WM_COMMAND, IDC_CHK_SHOWBLENDED, 0);
						}
						else
							RedoPreview(hdlg,fa,1,0,0);
						break;

					case IDC_CHK_DNRMP:
						mfd->UseDnrMultiPass=IsDlgButtonChecked(hdlg, IDC_CHK_DNRMP);
						EnableWindow(GetDlgItem(hdlg,IDC_SLIDER_EDGES_DNR),mfd->UseDnrMultiPass);
						EnableWindow(GetDlgItem(hdlg,IDC_E_EDGES_DNR),mfd->UseDnrMultiPass);
						InitFuncs(mfd);
						RedoPreview(hdlg,fa,1,0,INIT_ALL);
						break;

					case IDC_CHK_SHARPENEDGES:
						mfd->SharpenEdges=IsDlgButtonChecked(hdlg, IDC_CHK_SHARPENEDGES);
						EnableWindow(GetDlgItem(hdlg,IDC_SLIDER_SHARPEN),mfd->SharpenEdges);
						EnableWindow(GetDlgItem(hdlg,IDC_E_SHARPEN),mfd->SharpenEdges);
						InitFuncs(mfd);
						if(mfd->SharpenEdges)
							CheckDlgButton(hdlg,IDC_CHK_SB,true);
						else
						{
							RedoPreview(hdlg,fa,1,0,INIT_SB);
							break;
						}

					case IDC_CHK_SB:
					case IDC_CHK_SHOWEDGES:
						if(IsDlgButtonChecked(hdlg, IDC_CHK_DISABLEALL))
							break;
						mfd->Use_SB=IsDlgButtonChecked(hdlg, IDC_CHK_SB);
						mfd->show_edges=IsDlgButtonChecked(hdlg, IDC_CHK_SHOWEDGES);
						if(mfd->show_edges && mfd->Use_SB && mfd->show_blended)
						{
							CheckDlgButton(hdlg,IDC_CHK_SHOWBLENDED,false);
							mfd->show_blended=0;
						}
/*						if(mfd->show_edges && mfd->Use_SB)
						{
							if(mfd->show_blended)
							{
								CheckDlgButton(hdlg,IDC_CHK_SHOWBLENDED,false);
								mfd->show_blended=0;
							}
							mfd->Use_DNR=0;
							mfd->Use_gamma=0;
						}
						else
						{
							if(mfd->Use_DNR!=IsDlgButtonChecked(hdlg, IDC_CHK_DNR))
								PostMessage(hdlg, WM_COMMAND, 0, IDC_CHK_DNR);
							if(mfd->Use_gamma!=IsDlgButtonChecked(hdlg, IDC_CHK_GAMMA))
								PostMessage(hdlg, WM_COMMAND, 0, IDC_CHK_GAMMA);
							mfd->Use_DNR=IsDlgButtonChecked(hdlg, IDC_CHK_DNR);
							mfd->Use_gamma=IsDlgButtonChecked(hdlg, IDC_CHK_GAMMA);
						}*/
						InitGamma(mfd);
						InitFuncs(mfd);
						RedoPreview(hdlg,fa,1,0,INIT_SB);
						break;

					case IDC_CHK_GAMMA:
						if((!mfd->show_edges || !mfd->Use_SB) && !IsDlgButtonChecked(hdlg, IDC_CHK_DISABLEALL))
						{
							mfd->Use_gamma=IsDlgButtonChecked(hdlg, IDC_CHK_GAMMA);
							InitGamma(mfd);
							InitFuncs(mfd);
							RedoPreview(hdlg,fa,1,0,INIT_GAMMA);
						}
						break;

					case IDC_CHK_INTERLACED:
						if(IsDlgButtonChecked(hdlg, IDC_CHK_INTERLACED))
							mfd->Interlaced=1;
						else
							mfd->Interlaced=0;
						RedoPreview(hdlg,fa,1,0,INIT_SB);
						break;

					case IDC_CHK_DISABLEMMX:
						mfd->DisableMMX=IsDlgButtonChecked(hdlg, IDC_CHK_DISABLEMMX);
					    g_MMXenabled=mfd->ff->isMMXEnabled() && !mfd->DisableMMX;
						InitFuncs(mfd);
						RedoPreview(hdlg,fa,1,0,INIT_ALL);
						break;

					case IDC_CHK_DISABLEALL:
						if(IsDlgButtonChecked(hdlg, IDC_CHK_DISABLEALL))
						{
							mfd->Use_crop=0;
							mfd->Use_DNR=0;
							mfd->Use_SB=0;
							mfd->show_edges=0;
							mfd->Use_gamma=0;
						}
						else
						{
							mfd->Use_crop=IsDlgButtonChecked(hdlg, IDC_CHK_CROP);
							mfd->Use_DNR=IsDlgButtonChecked(hdlg, IDC_CHK_DNR);
							mfd->Use_SB=IsDlgButtonChecked(hdlg, IDC_CHK_SB);
							mfd->show_edges=IsDlgButtonChecked(hdlg, IDC_CHK_SHOWEDGES);
							mfd->Use_gamma=IsDlgButtonChecked(hdlg, IDC_CHK_GAMMA);
							if(mfd->show_edges)
							{
								mfd->Use_DNR=0;
								mfd->Use_gamma=0;
							}
						}
						InitFuncs(mfd);
						RedoPreview(hdlg,fa,1,0,INIT_ALL);
						break;
				}

			}
			// END switch(HIWORD(wParam))
		break;
		}
		// END WM_COMMAND
		
		default: return FALSE;
	}

	return TRUE;
}
