/*
NRS - "Noise reduction suite" filter for VirtualDub
Copyright (C) 2003 Antonio Foranna

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation.
	
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
		
You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
			
The author can be contacted at:
kreel@tiscali.it
*/

#include <windows.h>
#include <stdio.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"
#include "CGamma.h"
#include "Cregistry.h"
#include "Define.h"
#include "Struct.h"
#include "Dialog.h"
#include "Crop.h"
#include "Gamma.h"
#include "DNR.h"
#include "SmartBlur.h"
#include "Run.h"

//************************************************************************************************

int nrsInitProc(FilterActivation *fa, const FilterFunctions *ff);
int nrsStartProc(FilterActivation *fa, const FilterFunctions *ff);
int nrsRunProc(const FilterActivation *fa, const FilterFunctions *ff);
int nrsEndProc(FilterActivation *fa, const FilterFunctions *ff);
void nrsFilterDeinitProc(FilterActivation *fa, const FilterFunctions *ff);
int nrsConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void nrsStringProc(const FilterActivation *fa, const FilterFunctions *ff, char *buf); 

//long nrsParamProc(FilterActivation *fa, const FilterFunctions *ff);
void nrsScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool nrsFssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

//************************************************************************************************

bool g_MMXenabled;

#pragma data_seg("SHARED")

HBITMAP hBmRefresh=NULL;
HBITMAP hBmPicker=NULL;

#pragma data_seg()

//************************************************************************************************

ScriptFunctionDef func_defs[]={
	{ (ScriptFunctionPtr)nrsScriptConfig, "Config", "0iiiiiiii" },
	{ NULL },
};
//------------------------------------------------------------------------------------------------

CScriptObject script_obj={
	NULL, func_defs
};
//------------------------------------------------------------------------------------------------

struct FilterDefinition filterDEF_USE_DNR = {

    NULL, NULL, NULL,						// next, prev, module
    APP_NAME,								// name
    "Reduces changes across frames,\n"
	"blurs preserving structure,\n"
	"crops and centers images,\n"
	"adjusts gamma.",						// description

	"Antonio Foranna",						// maker
    NULL,									// private_data
    sizeof(MyFilterData),					// inst_data_size

    nrsInitProc,							// initProc
    nrsFilterDeinitProc,					// deinitProc
    nrsRunProc,								// runProc
    NULL,	//nrsParamProc,					// paramProc
    nrsConfigProc,							// configProc
    nrsStringProc,							// stringProc
    nrsStartProc,							// startProc
    nrsEndProc,								// endProc

    &script_obj,							// script_obj
    nrsFssProc,								// fssProc

};

//************************************************************************************************

BOOL WINAPI DllMain (HANDLE hModule, DWORD fdwReason, LPVOID lpReserved)
{
	switch (fdwReason)
	{
	case DLL_PROCESS_ATTACH:
	/*	Code from LibMain inserted here.  Return TRUE to keep the
		DLL loaded or return FALSE to fail loading the DLL.
		
		You may have to modify the code in your original LibMain to
		account for the fact that it may be called more than once.
		You will get one DLL_PROCESS_ATTACH for each process that
		loads the DLL. This is different from LibMain which gets
		called only once when the DLL is loaded. The only time this
		is critical is when you are using shared data sections.
		If you are using shared data sections for statically
		allocated data, you will need to be careful to initialize it
		only once. Check your code carefully.
		  
		Certain one-time initializations may now need to be done for
		each process that attaches. You may also not need code from
		your original LibMain because the operating system may now
		be doing it for you.
	*/
		if(!hBmRefresh)	
			hBmRefresh=LoadBitmap((HINSTANCE__ *)hModule, MAKEINTRESOURCE(IDB_REFRESH));
		if(!hBmPicker)
			hBmPicker=LoadBitmap((HINSTANCE__ *)hModule, MAKEINTRESOURCE(IDB_PICKER));
		break;
		
	case DLL_THREAD_ATTACH:
	/*	Called each time a thread is created in a process that has
		already loaded (attached to) this DLL. Does not get called
		for each thread that exists in the process before it loaded
		the DLL.
		Do thread-specific initialization here.
	*/
		break;
		
	case DLL_THREAD_DETACH:
	/*	Same as above, but called when a thread in the process exits.
		Do thread-specific cleanup here.
	*/
		break;
		
	case DLL_PROCESS_DETACH:
	/*	Code from _WEP inserted here.  This code may (like the
		LibMain) not be necessary.
		Check to make certain that the operating system is not
		doing it for you.
	*/
		if(hBmRefresh)
			DeleteObject(hBmRefresh);
		hBmRefresh=NULL;
		if(hBmPicker)
			DeleteObject(hBmPicker);
		hBmPicker=NULL;
		break;
	}
	
	/*	The return value is only used for DLL_PROCESS_ATTACH;
		all other conditions are ignored.  
	*/
	return TRUE;   // successful DLL_PROCESS_ATTACH
}
//************************************************************************************************

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);
//------------------------------------------------------------------------------------------------

static FilterDefinition *fd_;
//------------------------------------------------------------------------------------------------

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
    if (!(fd_ = ff->addFilter(fm, &filterDEF_USE_DNR, sizeof(FilterDefinition))))
        return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

    return 0;
}
//------------------------------------------------------------------------------------------------

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
    ff->removeFilter(fd_);
}

//************************************************************************************************

int nrsRunProc(const FilterActivation *fa, const FilterFunctions *ff)
{
MyFilterData *mfd=(MyFilterData *)fa->filter_data;

	mfd->pRunProc(fa->dst.data,fa->src.data,mfd);

    return 0;
}
//************************************************************************************************

int nrsStartProc(FilterActivation *fa, const FilterFunctions *ff)
{
MyFilterData *mfd = (MyFilterData *)fa->filter_data;

    mfd->sizeSrc = fa->src.Size();
    mfd->sizeDst = fa->dst.Size();
	if(!mfd->sizeSrc || !mfd->sizeDst)
		return 1;

    g_MMXenabled=ff->isMMXEnabled() && !mfd->DisableMMX;
	mfd->NewImage=1;
	mfd->w=(WORD)fa->src.w;
	mfd->wnc=(WORD)fa->src.pitch>>2;
	mfd->h=(WORD)fa->src.h;

	InitCrop(mfd,fa->dst.data);
	mfd->xGap=(short)mfd->wnc-mfd->dst_width;
	InitGamma(mfd);
	if(InitDNR(mfd,fa->src.data))
		return 1;
	if(InitSB(mfd))
		return 1;
	if(!mfd->PreviewMode)
		InitFuncs(mfd);

    return 0;
}
//************************************************************************************************

int nrsEndProc(FilterActivation *fa, const FilterFunctions *ff)
{
//MyFilterData *mfd = (MyFilterData *)fa->filter_data;
    return 0;
}
//************************************************************************************************

void LoadReg(MyFilterData *mfd)
{
CRegistry Registry;

	if(Registry.openCreateReg(HKEY_LOCAL_MACHINE,REGISTRY_APP_NAME))
	{
		mfd->DisableMMX=Registry.getSetRegByte(REG_DISABLEMMX, DEF_DISABLEMMX);
		mfd->Interlaced=Registry.getSetRegByte(REG_INTERLACED, DEF_INTERLACED);

		mfd->Use_DNR=Registry.getSetRegByte(REG_USE_DNR, DEF_USE_DNR);
		mfd->MinThresholdDNR=Registry.getSetRegByte(REG_MINT_DNR, DEF_MINTHRESHOLD_DNR);
		mfd->MaxThresholdDNR=Registry.getSetRegByte(REG_MAXT_DNR, DEF_MAXTHRESHOLD_DNR);
		mfd->SCD_threshold=Registry.getSetRegByte(REG_SCD, DEF_SCD);
		mfd->show_blended=Registry.getSetRegByte(REG_SHOWBLENDED_DNR, DEF_SHOWBLENDED_DNR);
		mfd->UseDnrMultiPass=Registry.getSetRegByte(REG_USE_DNRMP, DEF_USE_DNRMP);
		mfd->EdgesThreshold=Registry.getSetRegByte(REG_EDGES_DNR, DEF_EDGES_DNR);

		mfd->Use_SB=Registry.getSetRegByte(REG_USE_SB, DEF_USE_SB);
		mfd->MinThresholdSB=Registry.getSetRegByte(REG_MINT_SB, DEF_MINTHRESHOLD_SB);
		mfd->MaxThresholdSB=Registry.getSetRegByte(REG_MAXT_SB, DEF_MAXTHRESHOLD_SB);
		mfd->radius=Registry.getSetRegByte(REG_RADIUS_SB, DEF_RADIUS_SB);
		mfd->SharpenEdges=Registry.getSetRegByte(REG_SHARPENEDGES, DEF_SHARPENEDGES);
		mfd->SharpenStrength=Registry.getSetRegByte(REG_SHARPENSTRENGTH, DEF_SHARPENSTRENGTH);

		mfd->top=Registry.getSetRegWord(REG_TOP,DEF_CROP);
		mfd->bottom=Registry.getSetRegWord(REG_BOTTOM,DEF_CROP);
		mfd->left=Registry.getSetRegWord(REG_LEFT,DEF_CROP);
		mfd->right=Registry.getSetRegWord(REG_RIGHT,DEF_CROP);
		mfd->xShift=Registry.getSetRegWord(REG_XSHIFT,DEF_SHIFT);
		mfd->yShift=Registry.getSetRegWord(REG_YSHIFT,DEF_SHIFT);
		mfd->R=Registry.getSetRegByte(REG_R,DEF_COLORS);
		mfd->G=Registry.getSetRegByte(REG_G,DEF_COLORS);
		mfd->B=Registry.getSetRegByte(REG_B,DEF_COLORS);
		mfd->Use_crop=Registry.getSetRegByte(REG_USE_CROP, DEF_USE_CROP);

		mfd->GammaVal=Registry.getSetRegByte(REG_GAMMA,DEF_GAMMA);
		mfd->darkLimit=Registry.getSetRegByte(REG_DARK_LIMIT,DEF_DARK_LIMIT);
		mfd->brightLimit=Registry.getSetRegByte(REG_BRIGHT_LIMIT,DEF_BRIGHT_LIMIT);
		mfd->Use_gamma=Registry.getSetRegByte(REG_USE_GAMMA,DEF_USE_GAMMA);
	}
}
//------------------------------------------------------------------------------------------------

void SaveReg(MyFilterData *mfd)
{
CRegistry Registry;
	if(Registry.openCreateReg(HKEY_LOCAL_MACHINE,REGISTRY_APP_NAME))
	{
		Registry.setRegByte(REG_DISABLEMMX, mfd->DisableMMX);
		Registry.setRegByte(REG_INTERLACED, mfd->Interlaced);

		Registry.setRegByte(REG_USE_DNR, mfd->Use_DNR);
		Registry.setRegByte(REG_MINT_DNR, mfd->MinThresholdDNR);
		Registry.setRegByte(REG_MAXT_DNR, mfd->MaxThresholdDNR);
		Registry.setRegByte(REG_SCD, mfd->SCD_threshold);
		Registry.setRegByte(REG_SHOWBLENDED_DNR, mfd->show_blended);
		Registry.setRegByte(REG_USE_DNRMP, mfd->UseDnrMultiPass);
		Registry.setRegByte(REG_EDGES_DNR, mfd->EdgesThreshold);

		Registry.setRegByte(REG_USE_SB, mfd->Use_SB);
		Registry.setRegByte(REG_MINT_SB, mfd->MinThresholdSB);
		Registry.setRegByte(REG_MAXT_SB, mfd->MaxThresholdSB);
		Registry.setRegByte(REG_RADIUS_SB, mfd->radius);
		Registry.setRegByte(REG_SHARPENEDGES, mfd->SharpenEdges);
		Registry.setRegByte(REG_SHARPENSTRENGTH, mfd->SharpenStrength);

		Registry.setRegWord(REG_TOP, mfd->top);
		Registry.setRegWord(REG_BOTTOM, mfd->bottom);
		Registry.setRegWord(REG_LEFT, mfd->left);
		Registry.setRegWord(REG_RIGHT, mfd->right);
		Registry.setRegWord(REG_XSHIFT, mfd->xShift);
		Registry.setRegWord(REG_YSHIFT, mfd->yShift);
		Registry.setRegByte(REG_R, mfd->R);
		Registry.setRegByte(REG_G, mfd->G);
		Registry.setRegByte(REG_B, mfd->B);
		Registry.setRegByte(REG_USE_CROP, mfd->Use_crop);

		Registry.setRegByte(REG_GAMMA,mfd->GammaVal);
		Registry.setRegByte(REG_DARK_LIMIT,mfd->darkLimit);
		Registry.setRegByte(REG_BRIGHT_LIMIT,mfd->brightLimit);
		Registry.setRegByte(REG_USE_GAMMA,mfd->Use_gamma);
	}
}
//------------------------------------------------------------------------------------------------

int nrsInitProc(FilterActivation *fa, const FilterFunctions *ff)
{
MyFilterData *mfd = (MyFilterData *)fa->filter_data;

//    g_MMXenabled=ff->isMMXEnabled() && !mfd->DisableMMX;

	memset(mfd,0,sizeof(*mfd));

	if(!(mfd->pGamma=new CGamma()))
		return 1;

	mfd->ff=(FilterFunctions *)ff;
	LoadReg(mfd);
	InitFuncs(mfd);

    return 0;
}
//************************************************************************************************

void nrsFilterDeinitProc(FilterActivation *fa, const FilterFunctions *ff)
{
MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	DEL(mfd->pGamma);
	DELETE_ARRAY(mfd->pLogThresholdDNR);
	DELETE_ARRAY(mfd->pLogThresholdSB);
	DELETE_ARRAY(mfd->sav_data);
	DELETE_ARRAY(mfd->old_data);
	DELETE_ARRAY(mfd->DiffAcc);
	DELETE_ARRAY(mfd->Yold);
	DELETE_ARRAY(mfd->pWeight);
	DELETE_ARRAY(mfd->pScaleWeight);
	DELETE_ARRAY(mfd->LUTminx);
	DELETE_ARRAY(mfd->LUTmaxx);
	DELETE_ARRAY(mfd->LUTminy);
	DELETE_ARRAY(mfd->LUTmaxy);
}
//************************************************************************************************

void nrsStringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str)
{
MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	*str=0;

	if(mfd->Use_DNR)
		if(mfd->UseDnrMultiPass)
			sprintf(str+strlen(str), "(TS-MP %d-%d-%d-%d)",
					mfd->EdgesThreshold,
					mfd->MinThresholdDNR,
					mfd->MaxThresholdDNR,
					mfd->SCD_threshold
					);
		else
			sprintf(str+strlen(str), "(TS %d-%d-%d)",
					mfd->MinThresholdDNR,
					mfd->MaxThresholdDNR,
					mfd->SCD_threshold
					);
	if(mfd->Use_SB)
		if(mfd->SharpenEdges)
			sprintf(str+strlen(str), "(SB-UM %d-%d-%d-%d-%d)",
					mfd->SharpenStrength,
					mfd->radius,
					mfd->MinThresholdSB,
					mfd->MaxThresholdSB,
					mfd->Interlaced
					);
		else
			sprintf(str+strlen(str), "(SB %d-%d-%d-%d)",
					mfd->radius,
					mfd->MinThresholdSB,
					mfd->MaxThresholdSB,
					mfd->Interlaced
					);
	if(mfd->Use_crop)
		sprintf(str+strlen(str), "(C %d-%d-%d-%d-%d-%d)",
				mfd->top,
				mfd->bottom,
				mfd->left,
				mfd->right,
				mfd->xShift,
				mfd->yShift
				);
	if(mfd->Use_gamma)
		sprintf(str+strlen(str), "(G %d-%d-%d)",
				mfd->GammaVal,
				mfd->darkLimit,
				mfd->brightLimit
				);
}
//************************************************************************************************

void nrsScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc)
{
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->MinThresholdDNR=argv[1].asInt()&0xff;
	mfd->MaxThresholdDNR=(argv[1].asInt()>>8)&0xff;
	mfd->SCD_threshold=(argv[1].asInt()>>16)&0xff;
	mfd->EdgesThreshold=(argv[1].asInt()>>24)&0xff;
	mfd->show_blended=(argv[0].asInt()>>5)&1;
	mfd->UseDnrMultiPass=(argv[0].asInt()>>7)&1;
	mfd->Use_DNR=argv[0].asInt()&1;

	mfd->top=argv[2].asInt()&0xffff;
	mfd->bottom=(argv[2].asInt()>>16)&0xffff;
	mfd->left=argv[3].asInt()&0xffff;
	mfd->right=(argv[3].asInt()>>16)&0xffff;
	mfd->xShift=argv[4].asInt()&0xffff;
	mfd->yShift=(argv[4].asInt()>>16)&0xffff;
	mfd->R=argv[5].asInt()&0xff;
	mfd->G=(argv[5].asInt()>>8)&0xff;
	mfd->B=(argv[5].asInt()>>16)&0xff;
	mfd->Use_crop=(argv[0].asInt()>>1)&1;

	mfd->GammaVal=argv[6].asInt()&0xff;
	mfd->darkLimit=(argv[6].asInt()>>8)&0xff;
	mfd->brightLimit=(argv[6].asInt()>>16)&0xff;
	mfd->Use_gamma=(argv[0].asInt()>>2)&1;

	mfd->MinThresholdSB=argv[7].asInt()&0xff;
	mfd->MaxThresholdSB=(argv[7].asInt()>>8)&0xff;
	mfd->radius=(argv[7].asInt()>>16)&0xff;
	mfd->SharpenStrength=(argv[6].asInt()>>24)&0xff;
	mfd->SharpenEdges=(argv[0].asInt()>>6)&1;
	mfd->Use_SB=(argv[0].asInt()>>3)&1;

	mfd->Interlaced=((argv[7].asInt()>>24)&0xff);
	mfd->DisableMMX=(argv[0].asInt()>>4)&1;
}
//************************************************************************************************

bool nrsFssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	_snprintf(buf, buflen, "Config(%d, %d, %d, %d, %d, %d, %d, %d)",
								mfd->Use_DNR|(mfd->Use_crop<<1)|(mfd->Use_gamma<<2)|(mfd->Use_SB<<3)|(mfd->DisableMMX<<4)|(mfd->show_blended<<5)|(mfd->SharpenEdges<<6)|(mfd->UseDnrMultiPass<<7),

								mfd->MinThresholdDNR | (mfd->MaxThresholdDNR<<8) | (mfd->SCD_threshold<<16) | (mfd->EdgesThreshold<<24),

								mfd->top | (mfd->bottom<<16),
								mfd->left | (mfd->right<<16),
								mfd->xShift | (mfd->yShift<<16),
								mfd->R | (mfd->G<<8) |(mfd->B<<16),

								mfd->GammaVal | (mfd->darkLimit<<8) | (mfd->brightLimit<<16) | (mfd->SharpenStrength<<24),

								mfd->MinThresholdSB | (mfd->MaxThresholdSB<<8) | (mfd->radius<<16) | (mfd->Interlaced<<24)
								);

	return true;
}
//************************************************************************************************

#define SET_MFD(field) \
	mfd->field=mfd_sav.field;

int nrsConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd)
{
MyFilterData	*mfd = (MyFilterData *)fa->filter_data,
				mfd_sav;
BYTE			RetVal;

	memcpy(&mfd_sav,mfd,sizeof(MyFilterData));
	RetVal=DialogBoxParam(	fa->filter->module->hInstModule,
				        MAKEINTRESOURCE(IDD_DIALOG_CONFIG),
						hwnd,
						ConfigDlgProc,
						(LPARAM)fa);
	if(RetVal)
	{
		SET_MFD(Use_DNR) SET_MFD(Use_crop) SET_MFD(Use_gamma) SET_MFD(Use_SB) SET_MFD(DisableMMX) SET_MFD(show_blended) SET_MFD(UseDnrMultiPass) SET_MFD(show_edges) SET_MFD(show_DNR_edges)
		SET_MFD(MinThresholdDNR) SET_MFD(MaxThresholdDNR) SET_MFD(SCD_threshold) SET_MFD(EdgesThreshold)
		SET_MFD(top) SET_MFD(bottom) SET_MFD(left) SET_MFD(right)
		SET_MFD(xShift) SET_MFD(yShift)
		SET_MFD(R) SET_MFD(G) SET_MFD(B) 
		SET_MFD(GammaVal) SET_MFD(darkLimit) SET_MFD(brightLimit)
		SET_MFD(MinThresholdSB) SET_MFD(MaxThresholdSB) SET_MFD(radius) SET_MFD(Interlaced)
	}
	return RetVal;
}
