/*
    Decimate Filter for VirtualDub.
    Copyright (C) 1999-2000 Donald A. Graft

    Thanks to Simon Walters for suggesting use of Avisnth scripts.
	Thanks to James Lin for suggesting to delete the most similar
	frame in the set of N frames.
	Thanks to me for the frame comparison. :-)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

	The author can be contacted at:
	Donald Graft
	neuron2@attbi.com.
*/

#include <windows.h>
#include <commctrl.h>
#include <stdio.h>
#include <crtdbg.h>
#include <math.h>
#include <stdlib.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

///////////////////////////////////////////////////////////////////////////

int RunProc(const FilterActivation *fa, const FilterFunctions *ff);
int StartProc(FilterActivation *fa, const FilterFunctions *ff);
int EndProc(FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int InitProc(FilterActivation *fa, const FilterFunctions *ff);
int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

#define MAX_PATTERN 50

typedef struct MyFilterData {
	Pixel32				*pFrame;
	unsigned int		counts[MAX_PATTERN];
	int					index;
	int					deletes;
	char				filename[1024];
	int					decmode;
	FILE				*fp;
	int					debug;
} MyFilterData;

bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

    char save[255];
	int i, j;

	i = j = 0;
	while(mfd->filename[i])
	{
		if (mfd->filename[i] == '\\')
		{
			save[j++] = '\\';
			save[j++] = '\\';
			i++;
		}
		else
		{
			save[j++] = mfd->filename[i++];
		}
	}
	save[j] = 0;

	_snprintf(buf, buflen, "Config(%d, \"%s\", %d)", mfd->decmode, save, mfd->debug);

	return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->decmode = argv[0].asInt();
	strcpy(mfd->filename, *argv[1].asString());
	mfd->debug = !!argv[2].asInt();
}

ScriptFunctionDef func_defs[]={
	{ (ScriptFunctionPtr)ScriptConfig, "Config", "0isi" },
	{ NULL },
};

CScriptObject script_obj={
	NULL, func_defs
};

struct FilterDefinition filterDef_tutorial = {

	NULL, NULL, NULL,		// next, prev, module
	"decimate (1.2)",		// name
	"Decimate 1 of every N frames\nThanks to Simon Walters and James Lin.",
							// desc
	"Donald Graft", 		// maker
	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size

	InitProc,				// initProc
	NULL,					// deinitProc
	RunProc,				// runProc
	NULL,					// paramProc
	ConfigProc, 			// configProc
	StringProc, 			// stringProc
	StartProc,				// startProc
	EndProc,				// endProc

	&script_obj,			// script_obj
	FssProc,				// fssProc

};

int StartProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->pFrame = (Pixel32 *) new int[fa->src.w*fa->src.h];
	return 0;
}

int EndProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	delete[] mfd->pFrame;
	mfd->pFrame = NULL;
	if (mfd->fp) fclose(mfd->fp);
	return 0;
}

///////////////////////////////////////////////////////////////////////////

int RunProc(const FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	const PixDim	width = fa->src.w;
	const PixDim	height = fa->src.h;
	Pixel32 *src, *dst, *p;
	/* The threshold T is designed to give a little noise immunity as
	   well as to ckeep the counts somewhat limited. */
	int T = 7;
	int x, y;
	int frame = fa->pfsi->lCurrentSourceFrame;
	unsigned long count;

    if (frame == 0)
	{
		mfd->index = 0;
		mfd->deletes = 0;
		if (mfd->fp) fclose(mfd->fp);
		if (mfd->filename != NULL)
			mfd->fp = fopen(mfd->filename, "w");
	}

	/* First compare the frame and store the result. */
 	count = 0;
	src = fa->src.data;
    dst = fa->dst.data;
	p = mfd->pFrame;
	for (y = 0; y < height; y++)
	{
		for (x = 0; x < width; x++)
		{
			if (abs(((p[x] & 0xff0000) >> 16) - ((src[x] & 0xff0000) >> 16)) > T)
			{
				count++;
			}
			if (abs(((p[x] & 0xff00) >> 8) - ((src[x] & 0xff00) >> 8)) > T)
			{
				count++;
			}
			if (abs(((p[x] & 0xff) >> 0) - ((src[x] & 0xff) >> 0)) > T)
			{
				count++;
			}
			p[x] = src[x];
			dst[x] = src[x];
		}
		src	= (Pixel *)((char *)src + fa->src.pitch);
		dst	= (Pixel *)((char *)dst + fa->dst.pitch);
		p += width;
	}
	if (frame == 0) count = 0xffffffff;

	if (mfd->debug)
	{
		char buff[255];
		sprintf(buff, "%d: %u\n", frame, count);
		OutputDebugString(buff);

	}

	mfd->counts[mfd->index++] = count;

	/* When we have reached the configured number of frames, decimate the
	   most similar frame of the set of N frames as configured. */
	if (mfd->index == mfd->decmode)
	{
		unsigned int i, lowest, lowest_index;

		lowest = mfd->counts[0];
		lowest_index = 0;
		for (i = 1; i < (unsigned int) mfd->decmode; i++)
		{
			if (mfd->counts[i] < lowest)
			{
				lowest = mfd->counts[i];
				lowest_index = i;
			}
		}
		if (mfd->debug)
		{
			char buff[255];
			sprintf(buff, "delete %2d\n", frame - (mfd->decmode - 1) + lowest_index);
			OutputDebugString(buff);
		}
		if (mfd->fp != NULL)
			fprintf(mfd->fp, "DeleteFrame(%d)\n",
					frame - (mfd->decmode - 1) + lowest_index - mfd->deletes);
		mfd->deletes++;
		mfd->index = 0;
	}

	return 0;
}

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_tutorial;

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm,
															  const FilterFunctions *ff,
															  int& vdfd_ver, int& vdfd_compat)
{
	if (!(fd_tutorial = ff->addFilter(fm, &filterDef_tutorial, sizeof(FilterDefinition))))
		return 1;
	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

	return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff)
{
	ff->removeFilter(fd_tutorial);
}

int InitProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->filename[0] = 0;
	mfd->index = 0;
	mfd->deletes = 0;
	mfd->decmode = 5;
	mfd->debug = 0;
	return 0;
}

BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
	MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

	switch(msg) {
		case WM_INITDIALOG:
			SetWindowLong(hdlg, DWL_USER, lParam);
			mfd = (MyFilterData *)lParam;
			SetWindowText(GetDlgItem(hdlg, IDC_FILENAME), mfd->filename);
			CheckDlgButton(hdlg, IDC_DEBUG, mfd->debug ? BST_CHECKED : BST_UNCHECKED);
 			SetDlgItemInt(hdlg, IDC_MODE, mfd->decmode, FALSE);
			return TRUE;

		case WM_COMMAND:
			switch(LOWORD(wParam)) {
			case IDOK:
				GetWindowText(GetDlgItem(hdlg, IDC_FILENAME), mfd->filename, 255);
 				mfd->decmode = GetDlgItemInt(hdlg, IDC_MODE, &mfd->decmode, FALSE);
				if (mfd->decmode > MAX_PATTERN)
				{
					mfd->decmode = MAX_PATTERN;
 					SetDlgItemInt(hdlg, IDC_MODE, mfd->decmode, FALSE);
				}
				mfd->debug = !!IsDlgButtonChecked(hdlg, IDC_DEBUG);
				EndDialog(hdlg, 0);
				return TRUE;
			case IDHELP:
				{
				char prog[256];
				char path[256];
				LPTSTR ptr;

				GetModuleFileName(NULL, prog, 255);
				GetFullPathName(prog, 255, path, &ptr);
				*ptr = 0;
				strcat(path, "plugins\\Decimate.html");
				ShellExecute(hdlg, "open", path, NULL, NULL, SW_SHOWNORMAL);
				return TRUE;
				}
			case IDCANCEL:
				EndDialog(hdlg, 1);
				return TRUE;
			case IDC_BROWSE:
				{
				OPENFILENAME ofn;
				mfd->filename[0] = NULL;
				ofn.lStructSize = sizeof(OPENFILENAME);
				ofn.hwndOwner = hdlg;
				ofn.hInstance = NULL;
				ofn.lpTemplateName = NULL;
				ofn.lpstrFilter = "Avisynth Script Files\0*.avs\0All Files\0*.*\0\0";
				ofn.lpstrCustomFilter = NULL;
				ofn.nMaxCustFilter = 0;
				ofn.nFilterIndex = 1;
				ofn.lpstrFile = mfd->filename;
				ofn.nMaxFile = 1024;
				ofn.lpstrFileTitle = NULL;
				ofn.nMaxFileTitle = 0;
				ofn.lpstrInitialDir = NULL;
				ofn.lpstrTitle = "Choose Script File";
				ofn.Flags = OFN_CREATEPROMPT;
				ofn.nFileOffset = 0;
				ofn.nFileExtension = 0;
				ofn.lpstrDefExt = NULL;
				ofn.lCustData = 0;
				ofn.lpfnHook = NULL;
				GetOpenFileName(&ofn);
				if (mfd->filename[0] != 0)
				{
					SetWindowText(GetDlgItem(hdlg, IDC_FILENAME), mfd->filename);
				}
				break;
				}
			}
			break;
	}

	return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	if (DialogBoxParam(fa->filter->module->hInstModule, MAKEINTRESOURCE(IDD_FILTER),
		hwnd, ConfigDlgProc, (LPARAM)mfd))
	{
		return TRUE;
	}
    else
	{
		return FALSE;
	}
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	sprintf(str, " (1 in %d", mfd->decmode);
	if (mfd->debug) strcat(str, ", debug)");
	else strcat(str, ")");
}

