/**************************************************************************
*  Image process tool box                                                 *
*     by Yang Yudong                                                      *
*     yangyd@yahoo.com                                                    *
*                                                                         *
***************************************************************************
*  Copyright (C) 1992-1999, Yang Yudong, All rights reserved.             *
*  This file is part of Yang Yudong's image processing software package.  *
*  If you use this software, you agree to the following:                  *
*  This program package is purely experimental, and is licensed "as is".  *
*  Permission is granted to use, modify, and distribute this program      *
*  without charge for any purpose, provided this license/ disclaimer      *
*  notice appears in the copies.  No warranty or maintenance is given,    *
*  either expressed or implied.  In no event shall the author(s) be       *
*  liable to you or a third party for any special, incidental,            *
*  consequential, or other damages, arising out of the use or inability   *
*  to use the program for any purpose (or the loss of data), even if we   *
*  have been advised of such possibilities.  Any public reference or      *
*  advertisement of this source code should refer to it as Yang Yudong's  *
*  orignal.                                                               *
**************************************************************************/
// ****************************************************************
//  Image process tool box
//     by Yang Yudong
//
// File : filebmp.c
// Description: save or load Windows Bitmap image files 
// Create Date: 1996. 9. 25
// Modification(date/where): 1996.11.23 Make the integer byte order system independent
//
// ****************************************************************

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#define _IMG_LIBBUILD_
#include "image.h"
#include "imgfile.h"

#ifndef max
#define  max(a,b) ((a) > (b) ? (a) : (b))
#endif

// Windows data structure

typedef struct tagBITMAPINFOHEADER{
        IDWORD      biSize;
        ILONG       biWidth;
        ILONG       biHeight;
        IWORD       biPlanes;
        IWORD       biBitCount;
        IDWORD      biCompression;
        IDWORD      biSizeImage;
        ILONG       biXPelsPerMeter;
        ILONG       biYPelsPerMeter;
        IDWORD      biClrUsed;
        IDWORD      biClrImportant;
} BITMAPINFOHEADER;

typedef struct tagBITMAPFILEHEADER {
        IWORD    bfType;
        IDWORD   bfSize;
        IWORD    bfReserved1;
        IWORD    bfReserved2;
        IDWORD   bfOffBits;
} BITMAPFILEHEADER;


// save image as a Windows BMP 24 bits image
//  conversion will be added if needed.
// return ITRUE if successful

IBOOL SaveBMP24b(const char * fname, ImageDes img)
 {
  IBYTE * cr;
  IBYTE * cg;
  IBYTE * cb;
  int    offset;
  IBYTE * buf;
  int i, j, k;
  FILE * fp;
  BITMAPFILEHEADER fhead;
  BITMAPINFOHEADER ihead;


  if(!img.load ) return IFALSE;
  if ((buf = (IBYTE *)malloc((img.xsize*3L+3)&0xffffffc)) == NULL)
        return IFALSE;
  fp=fopen(fname, "wb");
  if(fp == NULL) {
    free(buf); return IFALSE;
  }
  fhead.bfType = 0x4d42;       // 'BM'
  fhead.bfSize = 0x36L+((img.xsize*3L+3)&0xffffffc)*img.ysize;
  fhead.bfReserved1 = 0;
  fhead.bfReserved2 = 0;
  fhead.bfOffBits= 0x36L;
  ihead.biSize = 0x28L;
  ihead.biWidth = (ILONG)img.xsize;
  ihead.biHeight = (ILONG)img.ysize;
  ihead.biPlanes = 1;
  ihead.biBitCount = 24;
  ihead.biCompression = 0L;
  ihead.biXPelsPerMeter=2835;  // 72 dpi
  ihead.biYPelsPerMeter=2835;
  ihead.biClrUsed=0;
  ihead.biClrImportant=0;
  ihead.biSizeImage  = ((img.xsize*3L+3)&0xffffffc)*img.ysize;

  // write filehead 
  fputWord(fhead.bfType, fp, ITRUE);
  fputLong(fhead.bfSize, fp, ITRUE);
  fputWord(fhead.bfReserved1, fp, ITRUE);
  fputWord(fhead.bfReserved2, fp, ITRUE);
  fputLong(fhead.bfOffBits, fp, ITRUE);
  
  // write infohead 
  fputLong(ihead.biSize, fp, ITRUE);
  fputLong((IDWORD)ihead.biWidth, fp, ITRUE);
  fputLong((IDWORD)ihead.biHeight, fp, ITRUE);
  fputWord(ihead.biPlanes, fp, ITRUE);
  fputWord(ihead.biBitCount, fp, ITRUE);
  fputLong(ihead.biCompression, fp, ITRUE);
  fputLong(ihead.biSizeImage, fp, ITRUE);
  fputLong((IDWORD)ihead.biXPelsPerMeter, fp, ITRUE);
  fputLong((IDWORD)ihead.biYPelsPerMeter, fp, ITRUE);
  fputLong(ihead.biClrUsed, fp, ITRUE);
  fputLong(ihead.biClrImportant, fp, ITRUE);

  cr = img.r;
  if(img.imagetype == ITrueColor) {
        cg = img.g;
        cb = img.b;
        offset = (img.ysize-1)*img.xsize;
        for(i = 0; i < img.ysize;  i++ ) {
           __Img_Busy(i*100/img.ysize);
           for(j=0, k=0; j<img.xsize; j++, k+=3) { 
             buf[k  ] = *(cb+offset+j);
             buf[k+1] = *(cg+offset+j);
             buf[k+2] = *(cr+offset+j);
           }
           offset -=img.xsize;
           fwrite(buf, (img.xsize*3L+3)&0xffffffc, 1, fp);
        }
   }
  else if(img.imagetype == IGrey) {
        offset = (img.ysize-1)*img.xsize;
        for(i = 0; i < img.ysize;  i++ ) {
           __Img_Busy(i*100/img.ysize);
           for(j=0, k=0; j<img.xsize; j++, k+=3) 
              buf[k  ] = buf[k+1] = buf[k+2] = *(cr+offset+j);
           offset -=img.xsize;
           fwrite(buf, (img.xsize*3L+3)&0xffffffc, 1, fp);
        }
   }
  // just save all indexed color image as color256
  else /*if(img.imagetype == IColor256 || (img.imagetype == IIndexedColor && img.numColors == 256) )*/ {
        offset = (img.ysize-1)*img.xsize;
        for(i = 0; i < img.ysize;  i++ ) {
           __Img_Busy(i*100/img.ysize);
           for(j=0, k=0; j<img.xsize; j++, k+=3) {
              buf[k  ] = img.pal[*(cr+offset+j)].b;
              buf[k+1] = img.pal[*(cr+offset+j)].g;
              buf[k+2] = img.pal[*(cr+offset+j)].r;
           }
           offset -=img.xsize;
           fwrite(buf, (img.xsize*3L+3)&0xffffffc, 1, fp);
        }
   }
            
  fclose(fp);
  free(buf);
  __Img_Busy(0);
  return ITRUE;
}


// save image as a Windows BMP 8 bits image
//  conversion will be added if needed.
// return ITRUE if successful

IBOOL SaveBMP8b(const char * fname, ImageDes img)
 {
  IBYTE * p;
  int    offset;
  IBYTE * buf;
  int i;
  FILE * fp;
  BITMAPFILEHEADER fhead;
  BITMAPINFOHEADER ihead;
  ImageDes timg;

  if(!img.load ) return IFALSE;
  if ((buf = (IBYTE *)malloc(max((img.xsize+3)&0xfffffffc, 1024))) == NULL)
        return IFALSE;
  timg.alloc = IFALSE;
  if(img.imagetype == ITrueColor) {
     if(!img_color24to8(&timg, img, 256, 0)) {
        free(buf);
        return(IFALSE);
     }
     p = timg.r;
  }
  else p = img.r;  
  fp=fopen(fname, "wb");
  if(fp == NULL) {
    FreePicture(&timg); 
    free(buf); 
    return IFALSE;
  }
  ihead.biSize = 0x28L;
  ihead.biWidth = (ILONG)img.xsize;
  ihead.biHeight = (ILONG)img.ysize;
  ihead.biPlanes = 1;
  ihead.biBitCount = 8;
  ihead.biCompression = 0L;
  ihead.biXPelsPerMeter=2835;  // 72 dpi
  ihead.biYPelsPerMeter=2835;
  ihead.biClrUsed=256;
  ihead.biClrImportant=0;
  ihead.biSizeImage  = (((long)img.xsize+3)&0xfffffffc)*img.ysize;

  fhead.bfType = 0x4d42;       // 'BM'
  fhead.bfReserved1 = 0;
  fhead.bfReserved2 = 0;
  fhead.bfSize = 0x36L+ihead.biSizeImage+ihead.biClrUsed*4;
  fhead.bfOffBits= 0x36L+ihead.biClrUsed*4;
  
  // write filehead 
  fputWord(fhead.bfType, fp, ITRUE);
  fputLong(fhead.bfSize, fp, ITRUE);
  fputWord(fhead.bfReserved1, fp, ITRUE);
  fputWord(fhead.bfReserved2, fp, ITRUE);
  fputLong(fhead.bfOffBits, fp, ITRUE);
  
  // write infohead 
  fputLong(ihead.biSize, fp, ITRUE);
  fputLong((IDWORD)ihead.biWidth, fp, ITRUE);
  fputLong((IDWORD)ihead.biHeight, fp, ITRUE);
  fputWord(ihead.biPlanes, fp, ITRUE);
  fputWord(ihead.biBitCount, fp, ITRUE);
  fputLong(ihead.biCompression, fp, ITRUE);
  fputLong(ihead.biSizeImage, fp, ITRUE);
  fputLong((IDWORD)ihead.biXPelsPerMeter, fp, ITRUE);
  fputLong((IDWORD)ihead.biYPelsPerMeter, fp, ITRUE);
  fputLong(ihead.biClrUsed, fp, ITRUE);
  fputLong(ihead.biClrImportant, fp, ITRUE);

  if(img.imagetype == ITrueColor) 
    for(i=0; i<256; i++) {
       buf[i*4+3] = 0;  
       buf[i*4+2] = timg.pal[i].r;
       buf[i*4+1] = timg.pal[i].g;
       buf[i*4+0] = timg.pal[i].b;
    }
  else if(img.imagetype == IColor256 || img.imagetype == IIndexedColor)
    for(i=0; i<256; i++) {
       buf[i*4+3] = 0;  
       buf[i*4+2] = img.pal[i].r;
       buf[i*4+1] = img.pal[i].g;
       buf[i*4+0] = img.pal[i].b;
    }
  else 
    for(i=0; i<256; i++) {
       buf[i*4+3] = 0;  
       buf[i*4+2] = (IBYTE)i;
       buf[i*4+1] = (IBYTE)i;
       buf[i*4+0] = (IBYTE)i;
    }
  fwrite(buf, 1024, 1, fp);        // write palette
  offset = (img.ysize-1)*img.xsize;
  for(i = 0; i < img.ysize;  i++, offset -=img.xsize) {
     __Img_Busy(i*100/img.ysize);
     memcpy(buf, p+offset, img.xsize);
     fwrite(buf, (img.xsize+3)&0xfffffffc, 1, fp);
  }          
  fclose(fp);
  FreePicture(&timg); 
  free(buf);
  __Img_Busy(0);
  return ITRUE;
}




// load Windows BMP 24 bits files
// Return ITRUE if successful

IBOOL LoadBMP24b(const char * fname, ImageDes *img, IBOOL ping)
 {
  IBYTE * cr;
  IBYTE * cg;
  IBYTE * cb;
  int  offset;
  IBYTE *buf;
  int  i, j, k, bpl;
  FILE *fp;
  BITMAPFILEHEADER fhead;
  BITMAPINFOHEADER ihead;
  IBOOL reverse=IFALSE;
  
  fp=fopen(fname, "rb");
  if(fp == NULL)
      return IFALSE; 

  // read filehead 
  fhead.bfType = fgetWord(fp, ITRUE);
  fhead.bfSize = fgetLong(fp, ITRUE);
  fhead.bfReserved1 = fgetWord(fp, ITRUE);
  fhead.bfReserved2 = fgetWord(fp, ITRUE);
  fhead.bfOffBits = fgetLong(fp, ITRUE);
  
  // read infohead 
  ihead.biSize = fgetLong(fp, ITRUE);
  ihead.biWidth = (ILONG)fgetLong(fp, ITRUE);
  ihead.biHeight = (ILONG)fgetLong(fp, ITRUE);
  ihead.biPlanes = fgetWord(fp, ITRUE);
  ihead.biBitCount = fgetWord(fp, ITRUE);
  ihead.biCompression = fgetLong(fp, ITRUE);
  ihead.biSizeImage = fgetLong(fp, ITRUE);
  ihead.biXPelsPerMeter = (ILONG)fgetLong(fp, ITRUE);
  ihead.biYPelsPerMeter = (ILONG)fgetLong(fp, ITRUE);
  ihead.biClrUsed = fgetLong(fp, ITRUE);
  ihead.biClrImportant = fgetLong(fp, ITRUE);

   if (fhead.bfType != 0x4d42        // 'BM'
      || ihead.biPlanes != 1
      || ihead.biBitCount != 24
      || ihead.biCompression != 0L)  // not a valid 24bits BMP file
           {fclose(fp); return IFALSE;}
   
   if(ihead.biHeight < 0) {ihead.biHeight = -ihead.biHeight; reverse = ITRUE; }
   if(ping) 
   {
	   fclose(fp); 
	   return InitPicture(img, ihead.biWidth, ihead.biHeight, ITrueColor, IFALSE, 0);
   }
   if(!AllocPicture(img, ihead.biWidth, ihead.biHeight, ITrueColor, IFALSE, 0)){
       fclose(fp); return IFALSE;  // can not allocate enough memory
   }

   cr = img->r;
   cg = img->g;
   cb = img->b;
   bpl = (img->xsize*3+3)&0xfffffffc;
   if((buf = (IBYTE *)malloc(bpl)) == NULL) {
        fclose(fp);
        FreePicture(img);
        return IFALSE;
   }   // can not allocate buffer


   if(reverse) {
      offset = 0;
      for(i = 0; i<ihead.biHeight;  i++) {
         __Img_Busy(i*100/ihead.biHeight);
         fread(buf, bpl, 1, fp);
         for(j=0, k=0; j<ihead.biWidth; j++, k+=3) { 
               *(cb+offset+j) = buf[k  ];
               *(cg+offset+j) = buf[k+1];
               *(cr+offset+j) = buf[k+2];
         }
         offset +=ihead.biWidth;
      }
   }
   else {
      offset = (ihead.biHeight-1)*ihead.biWidth;
      for(i = 0; i<ihead.biHeight;  i++) {
         __Img_Busy(i*100/ihead.biHeight);
         fread(buf, bpl, 1, fp);
         for(j=0, k=0; j<ihead.biWidth; j++, k+=3) { 
               *(cb+offset+j) = buf[k  ];
               *(cg+offset+j) = buf[k+1];
               *(cr+offset+j) = buf[k+2];
         }
         offset -=ihead.biWidth;
      }
   }
   fclose(fp);
   free(buf);
   img->load = ITRUE;
   __Img_Busy(0);
   return ITRUE;
 }



// load Windows BMP 8 bits files
// Return ITRUE if successful

IBOOL LoadBMP8b(const char * fname, ImageDes *img, IBOOL ping)
 {
  IBYTE * cr;
  int  offset;
  IBYTE *buf;
  int  i, bpl, tail;
  FILE *fp;
  BITMAPFILEHEADER fhead;
  BITMAPINFOHEADER ihead;
  IBOOL reverse=IFALSE;

  fp=fopen(fname, "rb");
  if(fp == NULL)
     return IFALSE; 

  // read filehead 
  fhead.bfType = fgetWord(fp, ITRUE);
  fhead.bfSize = fgetLong(fp, ITRUE);
  fhead.bfReserved1 = fgetWord(fp, ITRUE);
  fhead.bfReserved2 = fgetWord(fp, ITRUE);
  fhead.bfOffBits = fgetLong(fp, ITRUE);
  
  // read infohead 
  ihead.biSize = fgetLong(fp, ITRUE);
  ihead.biWidth = (ILONG)fgetLong(fp, ITRUE);
  ihead.biHeight = (ILONG)fgetLong(fp, ITRUE);
  ihead.biPlanes = fgetWord(fp, ITRUE);
  ihead.biBitCount = fgetWord(fp, ITRUE);
  ihead.biCompression = fgetLong(fp, ITRUE);
  ihead.biSizeImage = fgetLong(fp, ITRUE);
  ihead.biXPelsPerMeter = (ILONG)fgetLong(fp, ITRUE);
  ihead.biYPelsPerMeter = (ILONG)fgetLong(fp, ITRUE);
  ihead.biClrUsed = fgetLong(fp, ITRUE);
  ihead.biClrImportant = fgetLong(fp, ITRUE);

   if (fhead.bfType != 0x4d42        // 'BM'
      || ihead.biPlanes != 1
      || ihead.biBitCount != 8
      || ihead.biCompression != 0L 
      || ihead.biClrUsed > 256     )  // not a valid 8bits BMP file
           {fclose(fp); return IFALSE;}
   
   if(ihead.biHeight < 0) {ihead.biHeight = -ihead.biHeight; reverse = ITRUE; }

   if(ping) 
   {
	   fclose(fp); 
	   return InitPicture(img, ihead.biWidth, ihead.biHeight, IColor256, IFALSE, 256);
   }

   if(!AllocPicture(img, ihead.biWidth, ihead.biHeight, IColor256, IFALSE, 256)){
       fclose(fp); return IFALSE;  // can not allocate enough memory
   }


   if((buf = (IBYTE *)malloc(1024)) == NULL) {
        fclose(fp);
        FreePicture(img);
        return IFALSE;
   }   // can not allocate buffer
   if(ihead.biClrUsed == 0) ihead.biClrUsed = 256;  

   fread(buf, ihead.biClrUsed*4, 1, fp);
   for(i=0; i<(int)ihead.biClrUsed; i++) {
     img->pal[i].r = buf[i*4+2];
     img->pal[i].g = buf[i*4+1];
     img->pal[i].b = buf[i*4+0];
   }
   for(; i<256; i++) {
     img->pal[i].r = 0;
     img->pal[i].g = 0;
     img->pal[i].b = 0;
   }
   cr = img->r;
   bpl = (ihead.biWidth+3)&0xfffc;
   tail = bpl-ihead.biWidth;
   if(reverse) {
      offset = 0;
      for(i = 0; i<ihead.biHeight;  i++) {
         __Img_Busy(i*100/ihead.biHeight);
         fread(cr+offset, ihead.biWidth, 1, fp);
         if(tail) fread(buf, tail, 1, fp);
         offset +=ihead.biWidth;
      }
   }
   else {
      offset = (ihead.biHeight-1)*ihead.biWidth;
      for(i = 0; i<ihead.biHeight;  i++) {
         __Img_Busy(i*100/ihead.biHeight);
         fread(cr+offset, ihead.biWidth, 1, fp);
         if(tail) fread(buf, tail, 1, fp);
         offset -=ihead.biWidth;
      }
   }
   fclose(fp);
   free(buf);
   img->load = ITRUE;
   __Img_Busy(0);
   return ITRUE;
 }


// Load BMP24 bits or BMP 8 bits files
// Return ITRUE if successful

IBOOL LoadBMP(const char * fname, ImageDes *img, IBOOL ping)
{
  FILE *fp;
  BITMAPFILEHEADER fhead;
  BITMAPINFOHEADER ihead;

   fp=fopen(fname, "rb");
   if(fp == NULL)
      return IFALSE; 

  // read filehead 
  fhead.bfType = fgetWord(fp, ITRUE);
  fhead.bfSize = fgetLong(fp, ITRUE);
  fhead.bfReserved1 = fgetWord(fp, ITRUE);
  fhead.bfReserved2 = fgetWord(fp, ITRUE);
  fhead.bfOffBits = fgetLong(fp, ITRUE);
  
  // read infohead 
  ihead.biSize = fgetLong(fp, ITRUE);
  ihead.biWidth = (ILONG)fgetLong(fp, ITRUE);
  ihead.biHeight = (ILONG)fgetLong(fp, ITRUE);
  ihead.biPlanes = fgetWord(fp, ITRUE);
  ihead.biBitCount = fgetWord(fp, ITRUE);
  ihead.biCompression = fgetLong(fp, ITRUE);
  ihead.biSizeImage = fgetLong(fp, ITRUE);
  ihead.biXPelsPerMeter = (ILONG)fgetLong(fp, ITRUE);
  ihead.biYPelsPerMeter = (ILONG)fgetLong(fp, ITRUE);
  ihead.biClrUsed = fgetLong(fp, ITRUE);
  ihead.biClrImportant = fgetLong(fp, ITRUE);

   fclose(fp);
   
   if (fhead.bfType != 0x4d42        // 'BM'
      || ihead.biPlanes != 1
      || ihead.biCompression != 0L )  // not a supported BMP file
	  return IFALSE;
	
   if(ping) 
   {
	   if(ihead.biBitCount == 24) {
		   return InitPicture(img, ihead.biWidth, ihead.biHeight, ITrueColor, IFALSE, 0);
	   }
	   else if(ihead.biBitCount == 8) {
		   return InitPicture(img, ihead.biWidth, ihead.biHeight, IColor256, IFALSE, 256);
	   }
	   return InitPicture(img, ihead.biWidth, ihead.biHeight, IIndexedColor, IFALSE, 1<<ihead.biBitCount);
   }

   if(ihead.biBitCount == 24) {
     return(LoadBMP24b(fname, img, IFALSE));
   }
   else if(ihead.biBitCount == 8) {
     return(LoadBMP8b(fname, img, IFALSE));
   }
   else return IFALSE;
}

