/**************************************************************************
*  Image process tool box                                                 *
*     by Yang Yudong                                                      *
*     yangyd@yahoo.com                                                    *
*                                                                         *
***************************************************************************
*  Copyright (C) 1992-1999, Yang Yudong, All rights reserved.             *
*  This file is part of Yang Yudong's image processing software package.  *
*  If you use this software, you agree to the following:                  *
*  This program package is purely experimental, and is licensed "as is".  *
*  Permission is granted to use, modify, and distribute this program      *
*  without charge for any purpose, provided this license/ disclaimer      *
*  notice appears in the copies.  No warranty or maintenance is given,    *
*  either expressed or implied.  In no event shall the author(s) be       *
*  liable to you or a third party for any special, incidental,            *
*  consequential, or other damages, arising out of the use or inability   *
*  to use the program for any purpose (or the loss of data), even if we   *
*  have been advised of such possibilities.  Any public reference or      *
*  advertisement of this source code should refer to it as Yang Yudong's  *
*  orignal.                                                               *
**************************************************************************/
// ****************************************************************
//  Image process tool box
//     by Yang Yudong
//
// File : filejpeg.c
// Description: save or load JPEG compressed image files
//  This is a moddification from IJG's example.c
//  This program requires Independent JPEG Group's software library Version 6A.
//  which is Copyright (C) 1994-1996, Thomas G. Lane.
//  
// Create Date: 1997. 1. 23
// Modification(date/where): 
//
// ****************************************************************



#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#define _IMG_LIBBUILD_
#include "image.h"
#include "imgfile.h"

#ifdef IJG_JPEG_READY

#include <setjmp.h>

#include IJG_JPEG_HEADER



struct my_error_mgr {
  struct jpeg_error_mgr pub;	/* "public" fields */
  jmp_buf setjmp_buffer;	/* for return to caller */
};

typedef struct my_error_mgr * my_error_ptr;


METHODDEF(void)  __my_error_exit (j_common_ptr cinfo)
{
  /* cinfo->err really points to a my_error_mgr struct, so coerce pointer */
  my_error_ptr myerr = (my_error_ptr) cinfo->err;

  /* Return control to the setjmp point */
  longjmp(myerr->setjmp_buffer, 1);
}


// save image as a JPEG compressed image
//  conversion will be added if needed, gray scale image is stored as is.
// return ITRUE if successful

IBOOL SaveJPEG(const char * fname, ImageDes img, int quality)
 {
  struct jpeg_compress_struct cinfo;
  struct my_error_mgr __jerr;
  FILE * fp;		/* target file */

  IBYTE * cr;
  IBYTE * cg;
  IBYTE * cb;
  JSAMPROW row_pointer[1];
  int    i, j, k, offset;
  IBYTE * buf;


  if(!img.load || quality > 100 || quality < 0) return IFALSE;
  if (img.imagetype != IGrey) {
     if ((buf = (IBYTE *)malloc((img.xsize*3L+3)&0xffffffc)) == NULL)
        return IFALSE;
  }
  else buf = NULL;
  /* Step 1: allocate and initialize JPEG compression object */
  /* We set up the normal JPEG error routines, then override error_exit. */
  __jerr.pub.error_exit = __my_error_exit;
  cinfo.err = jpeg_std_error(&__jerr.pub);
  /* Establish the setjmp return context for my_error_exit to use. */
  fp = NULL;
  if (setjmp(__jerr.setjmp_buffer)) {
    /* If we get here, the JPEG code has signaled an error.
     * We need to clean up the JPEG object, close the input file, and return.
     */
    jpeg_destroy_compress(&cinfo);
    if(fp) fclose(fp);
    if(buf) free(buf);
    return IFALSE;
  }

  /* Now we can initialize the JPEG compression object. */
  jpeg_create_compress(&cinfo);

  /* Step 2: specify data destination (eg, a file) */
  if ((fp = fopen(fname, "wb")) == NULL) {
    free(buf); return IFALSE;
  }
  jpeg_stdio_dest(&cinfo, fp);

  /* Step 3: set parameters for compression */

  cinfo.image_width = img.xsize; 	/* image width and height, in pixels */
  cinfo.image_height = img.ysize;
  cinfo.input_components = (img.imagetype == IGrey ? 1 : 3);		/* # of color components per pixel */
  cinfo.in_color_space   = (img.imagetype == IGrey ? JCS_GRAYSCALE : JCS_RGB); 	/* colorspace of input image */

  jpeg_set_defaults(&cinfo);

  jpeg_set_quality(&cinfo, quality, (unsigned char)ITRUE /* limit to baseline-JPEG values */);

  /* Step 4: Start compressor */

  /* ITRUE ensures that we will write a complete interchange-JPEG file.
   * Pass ITRUE unless you are very sure of what you're doing.
   */
  jpeg_start_compress(&cinfo, (unsigned char)ITRUE);

  /* Step 5: while (scan lines remain to be written) */
  /*           jpeg_write_scanlines(...); */

  /* Here we use the library's state variable cinfo.next_scanline as the
   * loop counter, so that we don't have to keep track ourselves.
   * To keep things simple, we pass one scanline per call; you can pass
   * more if you wish, though.
   */

  if(img.imagetype == ITrueColor) {
        cr = img.r;  cg = img.g;    cb = img.b;     row_pointer[0] = buf;
        for(i = 0, offset = 0; i < img.ysize;  i++ ) {
           __Img_Busy(i*100/img.ysize);
           for(j=0, k=0; j<img.xsize; j++, k+=3, offset++) { 
             buf[k  ] = *(cr+offset);
             buf[k+1] = *(cg+offset);
             buf[k+2] = *(cb+offset);
           }
           jpeg_write_scanlines(&cinfo, row_pointer, 1);
        }
   }
  else if(img.imagetype == IGrey) {
        cr = img.r; 
        for(i = 0,  offset = 0; i < img.ysize;  i++, offset+=img.xsize ) {
           __Img_Busy(i*100/img.ysize);
           row_pointer[0] = cr+offset;
           jpeg_write_scanlines(&cinfo, row_pointer, 1);
        }
   }
  else {
        cr = img.r;  row_pointer[0] = buf;
        for(i = 0, offset = 0; i < img.ysize;  i++) {
           __Img_Busy(i*100/img.ysize);
           for(j=0, k=0; j<img.xsize; j++, k+=3, offset++) {
              buf[k  ] = img.pal[*(cr+offset)].r;
              buf[k+1] = img.pal[*(cr+offset)].g;
              buf[k+2] = img.pal[*(cr+offset)].b;
           }
           jpeg_write_scanlines(&cinfo, row_pointer, 1);
        }
   }

  /* Step 6: Finish compression */
  jpeg_finish_compress(&cinfo);

  fclose(fp);

  /* Step 7: release JPEG compression object */
  jpeg_destroy_compress(&cinfo);
  if(buf) free(buf);
  return ITRUE;
}




// load a JPEG compressed image
//  gray scale image will be keeped.
// return ITRUE if successful

IBOOL LoadJPEG(const char * fname, ImageDes* img, IBOOL ping)
{
  struct jpeg_decompress_struct cinfo;
  struct my_error_mgr __jerr;
  FILE * fp;		/* source file */

  IBYTE * cr;
  IBYTE * cg;
  IBYTE * cb;
  int    i, j, k, offset;
  IBYTE * buf;
  JSAMPROW row_pointer[1];
  IBOOL isgray;

  /* In this example we want to open the input file before doing anything else,
   * so that the setjmp() error recovery below can assume the file is open.
   * VERY IMPORTANT: use "b" option to fopen() if you are on a machine that
   * requires it in order to read binary files.
   */

  if ((fp = fopen(fname, "rb")) == NULL) {
    return IFALSE;
  }

  /* Step 1: allocate and initialize JPEG decompression object */
  __jerr.pub.error_exit = __my_error_exit;
  cinfo.err = jpeg_std_error(&__jerr.pub);
  /* Establish the setjmp return context for my_error_exit to use. */
  buf = NULL;
  if (setjmp(__jerr.setjmp_buffer)) {
    /* If we get here, the JPEG code has signaled an error.
     * We need to clean up the JPEG object, close the input file, and return.
     */
    jpeg_destroy_decompress(&cinfo);
    fclose(fp);
    if(buf) free(buf);
    return IFALSE;
  }

  /* Now we can initialize the JPEG decompression object. */
  jpeg_create_decompress(&cinfo);

  /* Step 2: specify data source (eg, a file) */

  jpeg_stdio_src(&cinfo, fp);

  /* Step 3: read file parameters with jpeg_read_header() */

  (void) jpeg_read_header(&cinfo, (unsigned char)ITRUE);

   if(ping) 
   {
	   fclose(fp);
	   if(cinfo.num_components == 3 && cinfo.out_color_space == JCS_RGB) {
	       jpeg_destroy_decompress(&cinfo);
		   return InitPicture(img, cinfo.image_width, cinfo.image_height, ITrueColor, IFALSE, 0);
	   }
	   else if(cinfo.num_components == 1 && cinfo.out_color_space == JCS_GRAYSCALE) {
	       jpeg_destroy_decompress(&cinfo);
		   return InitPicture(img, cinfo.image_width, cinfo.image_height, IGrey, IFALSE, 256);
	   }
       jpeg_destroy_decompress(&cinfo);
	   return IFALSE;
  }

  if(cinfo.num_components == 3 && cinfo.out_color_space == JCS_RGB) {
     if(!AllocPicture(img, cinfo.image_width, cinfo.image_height, ITrueColor, IFALSE, 0)){
       jpeg_destroy_decompress(&cinfo);
       fclose(fp); return IFALSE;  // can not allocate enough memory
     }
     isgray = IFALSE;
  }
  else if(cinfo.num_components == 1 && cinfo.out_color_space == JCS_GRAYSCALE) {
     if(!AllocPicture(img, cinfo.image_width, cinfo.image_height, IGrey, IFALSE, 256)){
       jpeg_destroy_decompress(&cinfo);
       fclose(fp); return IFALSE;  // can not allocate enough memory
     }
     isgray = ITRUE;
  }
  else {
    jpeg_destroy_decompress(&cinfo);
    fclose(fp);
    return IFALSE;
  }

  if(!isgray) {
    if ((buf = (IBYTE *)malloc(cinfo.image_width*3)) == NULL) {
        jpeg_destroy_decompress(&cinfo);
        fclose(fp);
        FreePicture(img);
        return IFALSE;
     }   // can not allocate buffer
  }  

  /* Step 4: set parameters for decompression */

  /* Step 5: Start decompressor */

  (void) jpeg_start_decompress(&cinfo);

  /* Step 6: while (scan lines remain to be read) */
  /*           jpeg_read_scanlines(...); */
  if(isgray) {
     cr = img->r; k=0;
     while (cinfo.output_scanline < cinfo.output_height) {
       __Img_Busy(k*100/cinfo.output_height);
       row_pointer[0] = cr+k*cinfo.image_width;  k++;
       (void) jpeg_read_scanlines(&cinfo, row_pointer, 1);
     }
  }
  else {
     cr = img->r; cg = img->g; cb = img->b;
     row_pointer[0] = buf; k=0;
     while (cinfo.output_scanline < cinfo.output_height) {
       __Img_Busy(k*100/cinfo.output_height);
       (void) jpeg_read_scanlines(&cinfo, row_pointer, 1);
       offset = k*cinfo.image_width; k++;
       for(i=0,j=0; i<(int)cinfo.image_width; i++, j+=3, offset++) {
         cr[offset] = buf[j];
         cg[offset] = buf[j+1];
         cb[offset] = buf[j+2];
       }
     }
  }   

  /* Step 7: Finish decompression */
  (void) jpeg_finish_decompress(&cinfo);

  /* Step 8: Release JPEG decompression object */
  jpeg_destroy_decompress(&cinfo);

  if(buf) free(buf);
  img->load = ITRUE;
  fclose(fp);
  return ITRUE;
}



#endif
