/**************************************************************************
*  Image process tool box                                                 *
*     by Yang Yudong                                                      *
*     yangyd@yahoo.com                                                    *
*                                                                         *
***************************************************************************
*  Copyright (C) 1992-1999, Yang Yudong, All rights reserved.             *
*  This file is part of Yang Yudong's image processing software package.  *
*  If you use this software, you agree to the following:                  *
*  This program package is purely experimental, and is licensed "as is".  *
*  Permission is granted to use, modify, and distribute this program      *
*  without charge for any purpose, provided this license/ disclaimer      *
*  notice appears in the copies.  No warranty or maintenance is given,    *
*  either expressed or implied.  In no event shall the author(s) be       *
*  liable to you or a third party for any special, incidental,            *
*  consequential, or other damages, arising out of the use or inability   *
*  to use the program for any purpose (or the loss of data), even if we   *
*  have been advised of such possibilities.  Any public reference or      *
*  advertisement of this source code should refer to it as Yang Yudong's  *
*  orignal.                                                               *
**************************************************************************/
// ****************************************************************
//  Image process tool box
//     by Yang Yudong
//
// File : filepngf.c
// Description: save or load PNG image files and load GIF
//  
// Create Date: 1999. 12. 05
// Modification(date/where): 
//
// ****************************************************************

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#define _IMG_LIBBUILD_
#include "image.h"
#include "imgfile.h"
#include "routines.h"

#ifdef LIBPNG_READY

#include <setjmp.h>
#include LIBPNG_HEADER


// upack packed pixels to multiplane format
static IBYTE **setup_packed_planes(ImageDes *img, IBYTE **buffer)
{
	IBYTE **plines;
	IBYTE *buf;
	IBYTE *p[16];
	int numPlane=0;
	int i, j, k;

	switch(img->imagetype)
	{
	case IIndexedColor:
	case IColor256:
		p[0] = img->r; //index;
		numPlane=1;
		break;
	case IGrey:
		p[0] = img->r; //grey;
		numPlane=1;
		if(img->alpha) 
		{
			p[numPlane++] = img->a;
		}
		break;
	case ITrueColor:
		p[0] = img->r;
		p[1] = img->g;
		p[2] = img->b;
		numPlane = 3;
		if(img->alpha) 
		{
			p[numPlane++] = img->a;
		}
		break;
	}

	plines = malloc(sizeof(IBYTE *)*img->ysize);
	if(!plines) return NULL; // failed
	if(numPlane == 1) // the data is single plane
	{
		*buffer = NULL;
		for(i=0; i<img->ysize; i++)
			plines[i] = p[0]+i*img->xsize;
		return plines;
	}

	buf = malloc(sizeof(IBYTE)*img->xsize*img->ysize*numPlane);
	if(!buf) 
	{
		free(plines);
		return NULL; // failed
	}
	*buffer = buf;

	for(i=0; i<img->ysize; i++)
		plines[i] = buf+i*img->xsize*numPlane;

	for(k=0; k<numPlane; k++)
	{
		register IBYTE *src = p[k];
		for(i=0; i<img->ysize; i++)
		{
			register IBYTE *des = plines[i]+k;
			for(j=0; j<img->xsize; j++, des+=numPlane, src++)
				*des = *src;
		}
	}
	return plines;
}

// assume pixel format is RGBA / ValueA / Value

static void cleanup_packed_planes(ImageDes *img, IBYTE **plines, IBYTE *buffer)
{
	free(plines);
	if(buffer) free(buffer);
	return ;
}


IBOOL SavePNG(const char * fname, ImageDes img, IBOOL interlace)
 {
	png_structp png_ptr;
	png_infop info_ptr;
	int height, width, depth, colortype;
	IBYTE **plines;
	IBYTE *buffer;
	
	FILE * fp;		/* source file */
	
	IBOOL alpha=IFALSE;
	
	if(!img.load || !img.alloc) return IFALSE;

	
    fp = fopen(fname, "wb");
    if (!fp)
    {
        return IFALSE;
    }

    /*fread(header, 1, PNG_BYTES_TO_CHECK, fp);
    if (png_sig_cmp(header, 0, PNG_BYTES_TO_CHECK))
    {
		fclose(fp);
        return IFALSE;
    }
*/	
    png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING, (png_voidp)NULL, (png_error_ptr)NULL, (png_error_ptr)NULL);
    if (!png_ptr)
	{
		fclose(fp);
        return IFALSE;
	}
    info_ptr = png_create_info_struct(png_ptr);
	if(!info_ptr)
	{
		png_destroy_write_struct(&png_ptr, NULL);
		fclose(fp);
        return IFALSE;
	}
	
	if (setjmp(png_ptr->jmpbuf))
	{
		png_destroy_write_struct(&png_ptr, &info_ptr);
		fclose(fp);
		return IFALSE;
	}
	//   end_info_ptr = png_create_info_struct(read_ptr);
	
	png_init_io(png_ptr, fp);
	if(img.numColors > 16) depth = 8;
	else if(img.numColors > 4) depth = 4;
	else if(img.numColors > 2) depth = 2;
	else  depth = 1;

	height = img.ysize;
	width  = img.xsize;
	switch(img.imagetype)
	{
	case ITrueColor:
		colortype = PNG_COLOR_TYPE_RGB;
		break;
	case IColor256:
	case IIndexedColor:
		colortype = PNG_COLOR_TYPE_PALETTE;
		break;
	case IGrey:
		colortype = PNG_COLOR_TYPE_GRAY;
		break;
	}
	if(img.alpha)
		colortype |= PNG_COLOR_MASK_ALPHA;

	png_set_IHDR(png_ptr, info_ptr, width, height, depth, colortype, 
		(interlace? PNG_INTERLACE_ADAM7 : PNG_INTERLACE_NONE), PNG_COMPRESSION_TYPE_DEFAULT, PNG_FILTER_TYPE_DEFAULT);

	// if there is gamma
	if(img.gamma > 0)
	{
		png_set_gAMA(png_ptr, info_ptr, 1/img.gamma);
	}
	
	// if there are color palettes
	if(img.imagetype == IColor256 || img.imagetype == IIndexedColor)
	{
		png_color pal[256];
		int numPal = (1<<depth), i;
		for(i=0; i<numPal; i++)
		{
			pal[i].red = img.pal[i].r;
			pal[i].green = img.pal[i].g;
			pal[i].blue = img.pal[i].b;
		}
		png_set_PLTE(png_ptr, info_ptr, pal,  numPal);
	}

	// if there is a back ground 
	if(img.background)
	{
		png_color_16 bk;
		if(img.imagetype == IIndexedColor)
		{
			bk.index = img.back.Index;
		}
		else if(img.imagetype == IIndexedColor)
		{
			bk.gray = img.back.Grey;
		}
		else 
		{
			bk.red   = img.back.Color.r;
			bk.green = img.back.Color.g;
			bk.blue  = img.back.Color.b;
		}
		png_set_bKGD(png_ptr, info_ptr, &bk);
	}
	// if there are transparent colors
	if(img.transparent)
	{
		png_byte trIdx[256];
		png_color_16 trVal;
		int trNum=0, i;
		if(img.imagetype == IIndexedColor)
		{
			for(i=0; i<img.trans.Index; i++) trIdx[i] = 255;
			trIdx[img.trans.Index] = 0; // only one transparent 
			trVal.index = img.trans.Index;
			trNum = img.trans.Index+1;
		}
		else if(img.imagetype == IIndexedColor)
		{
			trVal.gray = (png_uint_16)img.trans.Grey;
		}
		else 
		{
			trVal.red   = (png_uint_16)img.trans.Color.r;
			trVal.green = (png_uint_16)img.trans.Color.g;
			trVal.blue  = (png_uint_16)img.trans.Color.b;
		}
		png_set_tRNS(png_ptr, info_ptr, trIdx, trNum, &trVal);
	}
	

	// this time we must work for multiplane to packed coversion
	plines = setup_packed_planes(&img, &buffer);
	if(!plines)
	{
		png_destroy_write_struct(&png_ptr, &info_ptr);
		fclose(fp);
		return IFALSE;
	}

#if 0
#ifdef _DEBUG
	{
		FILE *fd;
		int sz;
		char *buf=buffer;
		SaveBMP24b("c:\\temp\\tst.bmp", img);
		if(!buf) buf = img.r;
		fd = fopen("c:\\temp\\tst.raw", "wb");
		sz = (img.imagetype == ITrueColor ? 3 : 1) + (img.alpha ? 1 : 0);
		fwrite(buf, sz*img.xsize*img.ysize, 1, fd);
		fclose(fd);
	}
#endif
#endif

	png_write_info(png_ptr, info_ptr);

	// I'm using unpacked pixels
	// this must be set after the write_info, because it'll reset the value
	if(depth < 8)  
		png_set_packing(png_ptr);

	png_write_image(png_ptr, plines);
    png_write_end(png_ptr, info_ptr);
	png_destroy_write_struct(&png_ptr, &info_ptr);

	cleanup_packed_planes(&img, plines, buffer);

    fclose(fp);
    return ITRUE;
}


// upack packed pixels to multiplane format
static IBYTE **setup_multiple_planes(ImageDes *img, IBYTE **buffer)
{
	IBYTE **plines;
	IBYTE *buf;
	int numPlane=0;
	IBYTE *p[16]; // max 16 planes
	int i, j;

	switch(img->imagetype)
	{
	case IIndexedColor:
	case IColor256:
		p[0] = img->r; //index;
		numPlane=1;
		break;
	case IGrey:
		p[0] = img->r; //grey;
		numPlane=1;
		if(img->alpha) 
		{
			p[numPlane++] = img->a;
		}
		break;
	case ITrueColor:
		p[0] = img->r;
		p[1] = img->g;
		p[2] = img->b;
		numPlane = 3;
		if(img->alpha) 
		{
			p[numPlane++] = img->a;
		}
		break;
	}

	plines = malloc(sizeof(IBYTE *)*img->ysize);
	if(!plines) return NULL; // failed
	if(numPlane == 1) // the data is single plane
	{
		*buffer = NULL;
		for(i=0; i<img->ysize; i++)
			plines[i] = p[0]+i*img->xsize;
		return plines;
	}

	buf = malloc(sizeof(IBYTE)*img->xsize*numPlane*numPlane*2);
	if(!buf) 
	{
		free(plines);
		return NULL; // failed
	}
	*buffer = buf;

	for(i=0; i<numPlane; i++)
	{
		p[i] += img->xsize*img->ysize; // start from the end of image
	}
	
	for(j=0, i=img->ysize-1; j<2*numPlane && i>=0; j++, i--)
	{
		plines[i] = buf + j*img->xsize*numPlane;
	}
	for(j=numPlane-1; i>=0; i--)
	{
		p[j] -= img->xsize*numPlane;
		plines[i] = p[j];
		j--;
		if(j < 0) j = numPlane-1; 
	}
	return plines;
}

// assume pixel format is RGBA / ValueA / Value

static void cleanup_multiple_planes(ImageDes *img, IBYTE **plines, IBYTE *buffer)
{
	IBYTE *p[16]; // max 16 planes
	int i, j, k, idx;
	register IBYTE *src, *des;
	int numPlane=0;

	switch(img->imagetype)
	{
	case IIndexedColor:
	case IColor256:
		p[0] = img->r; //index;
		numPlane=1;
		break;
	case IGrey:
		p[0] = img->r; //grey;
		numPlane=1;
		if(img->alpha) 
		{
			p[numPlane++] = img->a;
		}
		break;
	case ITrueColor:
		p[0] = img->r;
		p[1] = img->g;
		p[2] = img->b;
		numPlane = 3;
		if(img->alpha) 
		{
			p[numPlane++] = img->a;
		}
		break;
	}

	if(numPlane == 1) // the simple case
	{
		for(i=0; i<img->ysize; i++)
			memcpy(p[0]+i*img->xsize, plines[i], img->ysize*sizeof(IBYTE));
		free(plines);
		if(buffer) free(buffer);
		return;
	}

	for(i=0; i<img->ysize; i++)
	{
		src = plines[i];
		for(j=0; j<numPlane; j++)
		{
			des = p[j];
			for(k=0, idx=j; k<img->xsize; k++, idx+=numPlane)
				des[k] = src[idx];
			p[j] += img->xsize; // next line
		}
	}
	free(plines);
	if(buffer) free(buffer);
	return ;
}




// load a PNG image
//  gray scale image will be keeped.
// return ITRUE if successful

#define PNG_BYTES_TO_CHECK 8

IBOOL LoadPNG(const char * fname, ImageDes* img, IBOOL ping)
{
	png_structp png_ptr;
	png_infop info_ptr;
	int height, width, depth, colortype, interlace;
	IBYTE **plines;
	IBYTE *buffer;
	IBYTE *transTbl=NULL; // transparent lookup table for palette images
	IDWORD numTr;
	
	FILE * fp;		/* source file */
	
	int numColor;
	IBOOL alpha=IFALSE;
	IImageType imgtype;
	
	char header[PNG_BYTES_TO_CHECK];
	
	
    fp = fopen(fname, "rb");
    if (!fp)
    {
        return IFALSE;
    }
    fread(header, 1, PNG_BYTES_TO_CHECK, fp);
    if (png_sig_cmp(header, 0, PNG_BYTES_TO_CHECK))
    {
		fclose(fp);
        return IFALSE;
    }
	
    png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING, (png_voidp)NULL, (png_error_ptr)NULL, (png_error_ptr)NULL);
    if (!png_ptr)
	{
		fclose(fp);
        return IFALSE;
	}
    info_ptr = png_create_info_struct(png_ptr);
	if(!info_ptr)
	{
		png_destroy_read_struct(&png_ptr, NULL, NULL);
		fclose(fp);
        return IFALSE;
	}
	
	if (setjmp(png_ptr->jmpbuf))
	{
		png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
		FreePicture(img);
		fclose(fp);
		return IFALSE;
	}
	//   end_info_ptr = png_create_info_struct(read_ptr);
	
	png_init_io(png_ptr, fp);
	png_set_sig_bytes(png_ptr, PNG_BYTES_TO_CHECK);
	png_set_read_status_fn(png_ptr, NULL);
	png_read_info(png_ptr, info_ptr);
	png_get_IHDR(png_ptr, info_ptr, &width, &height, &depth, &colortype, &interlace, NULL, NULL);
	
	switch(colortype)
	{
	case PNG_COLOR_TYPE_GRAY_ALPHA:
		alpha=ITRUE;
	case PNG_COLOR_TYPE_GRAY:
		imgtype = IGrey; 
		numColor = depth > 8 ? 8 : depth;
		break;
		
	case PNG_COLOR_TYPE_PALETTE:
		imgtype = IIndexedColor; 
		numColor = depth;
		break;
		
	case PNG_COLOR_TYPE_RGB_ALPHA:
		alpha=ITRUE;
	case PNG_COLOR_TYPE_RGB:
		imgtype = ITrueColor;  
		numColor = depth > 8 ? 8 : depth;
		break;
	default:  // not allowed
		png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
		fclose(fp);
		return IFALSE;
	}

	numColor = 1 << numColor;

	if(ping)
	{
		png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
		fclose(fp);
		return InitPicture(img, width, height, imgtype, alpha, numColor);
		
	}

	if(!AllocPicture(img, width, height, imgtype, alpha, numColor))
	{
		png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
		fclose(fp);
		return IFALSE;
	}
	
    // I only accept 8 bits and less 
	if (depth == 16)
        png_set_strip_16(png_ptr);
	
	// I'm expecting unpacked pixels
	png_set_packing(png_ptr);
	
	// test if there is gamma
	{
		double gamma=1.0;
		if (png_get_gAMA(png_ptr, info_ptr, &gamma))
		{
			img->gamma = 1/gamma;
		}
	}
	// test if there are color palettes
	{
		png_colorp pal;
		int numPal, i;
		// we must have a palette
		if(imgtype == IIndexedColor)
		{
			if (png_get_PLTE(png_ptr, info_ptr, &pal,  &numPal))
			{
				for(i=0; i<numPal&&i<256; i++)
				{
					img->pal[i].r = pal[i].red;
					img->pal[i].g = pal[i].green;
					img->pal[i].b = pal[i].blue;
				}
			}
			else  // no palette defiend for a indexed color image
			{
				png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
				FreePicture(img);
				fclose(fp);
				return IFALSE;
			}
		}
	}
	// test if there is a back ground 
	{
		png_color_16p bk;
		
		if (png_get_bKGD(png_ptr, info_ptr, &bk))
		{
			img->background = ITRUE;
			if(imgtype == IIndexedColor)
			{
				img->back.Index = bk->index;
			}
			else if(imgtype == IGrey)
			{
				img->back.Grey = (IBYTE)(depth == 16 ? (bk->gray>>8): bk->gray);
			}
			else 
			{
				img->back.Color.r = (IBYTE)(depth == 16 ? (bk->red>>8): bk->red);
				img->back.Color.g = (IBYTE)(depth == 16 ? (bk->green>>8): bk->green);
				img->back.Color.b = (IBYTE)(depth == 16 ? (bk->blue>>8): bk->blue);
			}
		}
	}
	
	// test if there are transparent colors
	
	// if there is an alpha channel, there should not have transparent index
	if(!(colortype & PNG_COLOR_MASK_ALPHA))
	{
		png_bytep trIdx;
		png_color_16p trVal;
		// test if there are color palettes
		if (png_get_tRNS(png_ptr, info_ptr, &trIdx, &numTr, &trVal))
		{
			if(imgtype == IIndexedColor)
			{
				if(numTr > 0)
				{
					transTbl = malloc(256);
					if(transTbl)
					{
						memset(transTbl, 255, 256);
						memcpy(transTbl, trIdx, numTr);
					}
				}
			}
			else if(imgtype == IGrey)
			{
				img->transparent = ITRUE;
				img->trans.Grey = (IBYTE)(depth == 16 ? (trVal->gray>>8): trVal->gray);
			}
			else
			{
				img->transparent = ITRUE;
				img->trans.Color.r = (IBYTE)(depth == 16 ? (trVal->red>>8): trVal->red);
				img->trans.Color.g = (IBYTE)(depth == 16 ? (trVal->green>>8): trVal->green);
				img->trans.Color.b = (IBYTE)(depth == 16 ? (trVal->blue>>8): trVal->blue);
			}
		}
	}

	plines = setup_multiple_planes(img, &buffer);
	if(!plines)
	{
		png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
		FreePicture(img);
		fclose(fp);
		return IFALSE;
	}

	png_read_image(png_ptr, plines);
	png_read_end(png_ptr, info_ptr);
	png_destroy_read_struct(&png_ptr, &info_ptr, NULL);

	cleanup_multiple_planes(img, plines, buffer);
    img->load = ITRUE;

	// only one transparent index supported in my data structure, 
	// so I'll try to mix them with background or just threshold it
	if(imgtype == IIndexedColor && transTbl)
	{
		IDWORD i, small;
		IBOOL changed = IFALSE;
		IBYTE threshold = 128;

		// first check if the image has background
		if(img->background) // yes
		{
			// mix all transpatent palette with background color
			for(i=0; i<numTr; i++)
				if(transTbl[i] > 0 && transTbl[i] < 255) // mix these palettes
				{
					threshold = 0; // only keep the some transparent ones ?
					img->pal[i].r = (IBYTE)(((ILONG)img->pal[i].r-(ILONG)img->pal[img->back.Index].r)*(ILONG)transTbl[i]/(ILONG)255 + (ILONG)img->pal[img->back.Index].r);
					img->pal[i].g = (IBYTE)(((ILONG)img->pal[i].g-(ILONG)img->pal[img->back.Index].g)*(ILONG)transTbl[i]/(ILONG)255 + (ILONG)img->pal[img->back.Index].g);
					img->pal[i].b = (IBYTE)(((ILONG)img->pal[i].b-(ILONG)img->pal[img->back.Index].b)*(ILONG)transTbl[i]/(ILONG)255 + (ILONG)img->pal[img->back.Index].b);
				}
		}
		// then check if there is real transparency
		small = 0;
		for(i=1; i<numTr; i++)
			if(transTbl[i] < transTbl[small]) small = i;
		if(transTbl[small] < threshold ) // yes
		{
			img->transparent = ITRUE;
			img->trans.Index = (IBYTE)small;
			transTbl[small] = (IBYTE)small;
			for(i=0; i<256; i++)
			{
				if(transTbl[i] < threshold) // change them to small
				{
					transTbl[i] = (IBYTE)small; // looktbl
					if(i != small) changed = ITRUE;
				}
				else transTbl[i] = (IBYTE)i;
			}
		}
		if(changed) // we must do look table on this image to get the trans colors set
		{
			table_mod_(transTbl, img->r /*index*/, img->xsize, img->ysize);
		}
		free(transTbl);
	}
    fclose(fp);
    return ITRUE;
}


#endif
