/**************************************************************************
*  Image process tool box                                                 *
*     by Yang Yudong                                                      *
*     yangyd@yahoo.com                                                    *
*                                                                         *
***************************************************************************
*  Copyright (C) 1992-1999, Yang Yudong, All rights reserved.             *
*  This file is part of Yang Yudong's image processing software package.  *
*  If you use this software, you agree to the following:                  *
*  This program package is purely experimental, and is licensed "as is".  *
*  Permission is granted to use, modify, and distribute this program      *
*  without charge for any purpose, provided this license/ disclaimer      *
*  notice appears in the copies.  No warranty or maintenance is given,    *
*  either expressed or implied.  In no event shall the author(s) be       *
*  liable to you or a third party for any special, incidental,            *
*  consequential, or other damages, arising out of the use or inability   *
*  to use the program for any purpose (or the loss of data), even if we   *
*  have been advised of such possibilities.  Any public reference or      *
*  advertisement of this source code should refer to it as Yang Yudong's  *
*  orignal.                                                               *
**************************************************************************/
// ****************************************************************
//  Image process tool box
//     by Yang Yudong
//
// File : filetiff.c
// Description: TIFF image file I/O
// Create Date: 1996. 10. 1
// Modification(date/where):
//
// ****************************************************************

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#define _IMG_LIBBUILD_
#include "image.h"
#include "imgfile.h"


//TIF_header_struct defined
typedef struct {
	 IBYTE  byte_order[2];
	 IWORD  version;
	 IDWORD offset;
 } TIF_HEAD;
//End of TIF_header_struct define
//Directory struct define
typedef struct {
	  IWORD tag;
	  IWORD type;
	  IDWORD length;
	  IDWORD offset;
 } TIF_ENTRY;
//End of Directory struct define


/*  tiff tag names   */
#define NewSubfile           254
#define SubfileType          255
#define ImageWidth           256
#define ImageLength          257
#define BitsPerSample        258
#define Compression          259
#define StripOffsets         273
#define RowsPerStrip         278
#define StripByteCounts      279
#define SamplesPerPixel      277
#define PlanarConfiguration  284
#define Group3Options        292
#define Group4Options        293
#define FillOrder            266
#define Threshholding        263
#define CellWidth            264
#define CellLength           265
#define MinSampleValue       280
#define MaxSampleValue       281
#define PhotometricInterp    262
#define GrayResponseUnit     290
#define GrayResponseCurve    291
#define ColorResponseUnit    300
#define ColorResponseCurves  301
#define XResolution          282
#define YResolution          283
#define ResolutionUnit       296
#define Orientation          274
#define DocumentName         269
#define PageName             285
#define XPosition            286
#define YPosition            287
#define PageNumber           297
#define ImageDescription     270
#define Make                 271
#define Model                272
#define FreeOffsets          288
#define FreeByteCounts       289
#define Predictor            317
#define tagPALETTE           320

/*  tiff size  */
#define TIFFbyte     1
#define TIFFascii    2
#define TIFFshort    3
#define TIFFlong     4
#define TIFFrational 5

/*  tiff compression types  */
#define COMPnone     1
#define COMPhuff     2
#define COMPfax3     3
#define COMPfax4     4
#define COMPwrd1     0x8003
#define COMPmpnt     0x8005



// load TIFF 24 bit image
// return ITRUE if successful
// **NOTE: only one strip files supported now

IBOOL LoadTIFF24b(const char * fname, ImageDes *img, IBOOL ping)     // Tiff 24 bits
{
  IBYTE * cr;
  IBYTE * cg;
  IBYTE * cb;
  int  offset;
  IBYTE *buf;
  int  i, j, k, entry, nc, nr;
  FILE  * fd;
  TIF_HEAD  tif_head;
  TIF_ENTRY tifen;
  int   bw=0, unreg=0, tmp;
  int offset1,offset2;
  IBOOL intel;

  fd=fopen(fname, "rb");
  if(fd == NULL)
        return IFALSE;

  if((i=fgetWord(fd, ITRUE))==0x4949) intel = ITRUE; //"II"
  else if(i==0x4d4d) intel = IFALSE;  // "MM"
  else {  // Not a TIFF file
    fclose(fd);
    return IFALSE;
  }
  nc = nr = 0; offset1=offset2=0; 
  tif_head.version = fgetWord(fd, intel);   // read filehead
  if(tif_head.version != 42){  // Not a TIFF file
    fclose(fd);
    return IFALSE;
  }
  tif_head.offset  = fgetLong(fd, intel);   // read filehead
  offset1=tif_head.offset;     // get the direction offset 

  fseek(fd,offset1,SEEK_SET);
  entry = fgetWord(fd, intel);
  for (i=0;i<entry;i++) {                       //Deal with Entry
    tifen.tag = fgetWord(fd, intel);
    tifen.type = fgetWord(fd, intel);
    if(tifen.type==TIFFlong)  {
       tifen.length=fgetLong(fd, intel);
       tifen.offset=fgetLong(fd, intel);
    }
    else {
       tifen.length=(unsigned long)fgetWord(fd, intel);
       fgetWord(fd, intel);
       tifen.offset=(unsigned long)fgetWord(fd, intel);
       tmp=fgetWord(fd, intel);
    }
    switch(tifen.tag) {
        case   SubfileType:
        case   Compression:
        case PlanarConfiguration:
           if(tifen.offset != 1) { // TIFF file not supported
              fclose(fd);
              return IFALSE;
           } 
           break;   
        case   ImageWidth:
           nc=(IWORD)tifen.offset;
           break;
        case   ImageLength:
           nr=(IWORD)tifen.offset;
           break;
        case BitsPerSample:
           if(tifen.length != 3) {  // Not a 24 bits TIFF file
             fclose(fd);
             return IFALSE;
           } 
           break;
        case   PhotometricInterp:
           tmp=(IWORD)tifen.offset;
           if ((tmp !=2) && (tmp !=3) && (tmp !=1))
                                unreg=1;
           if (tmp==1) bw=1;     // reversed 0 white 255 black
           break;
        case   StripOffsets:
           offset2=tifen.offset;
           break;
    } // switch
  } // for            
  if(nc == 0 || nr == 0) {
       fclose(fd); return IFALSE;  // not a valid TIFF file
  }

  if(ping) 
   {
		fclose(fd);
		return InitPicture(img, nc, nr, ITrueColor, IFALSE, 0);
   }

  if(!AllocPicture(img, nc, nr, ITrueColor, IFALSE, 0)){
       fclose(fd); return IFALSE;  // can not allocate enough memory
     }

  if((buf=(IBYTE *)malloc(nc*3)) == NULL) {
         FreePicture(img);
         fclose(fd);
         return IFALSE;
  }   // can not allocate buffer

  cr = img->r;
  cg = img->g;
  cb = img->b;
  offset = 0;

  if (unreg==0) {                       //Read image data
     fseek(fd, offset2, SEEK_SET);
     for(i = 0; i <nr;   i++ ){
        __Img_Busy(i*100/img->ysize);
        fread(buf, nc*3, 1, fd);
        for(j=0, k=0; j<nc; j++, k+=3) {
            *(cr+offset+j) = buf[k  ];
            *(cg+offset+j) = buf[k+1];
            *(cb+offset+j) = buf[k+2];
        }
        offset += nc;
     }
  }
  free(buf);
  fclose(fd);
  img->load = ITRUE;
  __Img_Busy(0);
  return ITRUE;
}

void write_TIFF_TAG(IWORD tag,  IWORD type, IDWORD length,  IDWORD offset, FILE *fp, IBOOL intel)
{
  fputWord(tag, fp, intel);
  fputWord(type, fp, intel);
  fputLong(length, fp, intel);
  fputLong(offset, fp, intel);
}

// save image as a TIFF 24 bits color image
// return ITRUE if successful

IBOOL SaveTIFF24b(const char * fname, ImageDes img)     // Tiff 24 bits
 {
  IBYTE * cr;
  IBYTE * cg;
  IBYTE * cb;
  long offset;
  IBYTE *buf;
  int  i, j, k;
  FILE  * fd;

  if(!img.load ) return IFALSE;
  if((buf =(IBYTE *)malloc(img.xsize*3)) == NULL)
	 return IFALSE;
  fd=fopen(fname, "wb");
  if(fd == NULL)
	{ free(buf); return IFALSE; }


 /* TIFF header 49 49 2A 00 08 00 00 00 */
 fputWord(0x4949, fd, ITRUE);
 fputWord(0x2a, fd, ITRUE);
 fputLong(8, fd, ITRUE);
 
 /* number of TAGs */
 fputWord(12, fd, ITRUE);
 
 /* Entry 0 Subfiletype */
 write_TIFF_TAG(SubfileType, TIFFlong, 1, 1, fd, ITRUE);

 /* Entry 1 Imagewidth */
 write_TIFF_TAG(ImageWidth, TIFFlong, 0, img.xsize, fd, ITRUE);

 /* Entry 2 Imgelength */
 write_TIFF_TAG(ImageLength, TIFFlong, 0, img.ysize, fd, ITRUE);

 /* Entry 3 Bitspersample */
 write_TIFF_TAG(BitsPerSample, 3, 3, sizeof(TIF_ENTRY) * 12 + sizeof(TIF_HEAD) + 2 + 4, fd, ITRUE);

 /* Entry 4 Compression */
 write_TIFF_TAG(Compression, 3, 1, 1, fd, ITRUE);

 /* Entry 5 Photometric */
 write_TIFF_TAG(PhotometricInterp, 3, 1, 2, fd, ITRUE);

 /* Entry 6 strip offsets */
 write_TIFF_TAG(StripOffsets, 4, 1, sizeof(TIF_ENTRY)*12 + sizeof(TIF_HEAD) + 2 + 4 + 6 + 8 + 8, fd, ITRUE);

 /* Entry 7 Samplesperpixel */
 write_TIFF_TAG(SamplesPerPixel, 3, 1, 3, fd, ITRUE);

 /* Entry 8 Xres */
 write_TIFF_TAG(XResolution, 5, 1, sizeof(TIF_ENTRY)* 12 + sizeof(TIF_HEAD) + 2 + 4 + 6, fd, ITRUE);

 /* Entry 9 Yres */
 write_TIFF_TAG(YResolution, 5, 1, sizeof(TIF_ENTRY)*12 + sizeof(TIF_HEAD) + 2 + 4 + 6 + 8, fd, ITRUE);

 /* Entry 10 PlanarConfig */
 write_TIFF_TAG(PlanarConfiguration, 3, 1, 1, fd, ITRUE);

 /* Entry 11 Res Unit */
 write_TIFF_TAG(ResolutionUnit, 3, 1, 2, fd, ITRUE);

 // end of dir
 fputLong(0, fd, ITRUE);

 // bitspersample
 fputWord(8, fd, ITRUE);
 fputWord(8, fd, ITRUE);
 fputWord(8, fd, ITRUE);

 // x, y resolution
 fputLong(72, fd, ITRUE);
 fputLong(1, fd, ITRUE);
 fputLong(72, fd, ITRUE);
 fputLong(1, fd, ITRUE);

  cr = img.r;
  if(img.imagetype == ITrueColor) {
	cg = img.g;
	cb = img.b;
	offset = 0;
	for(i = 0; i < img.ysize;  i++ ){
	   __Img_Busy(i*100/img.ysize);
	   for(j=0, k=0; j<img.xsize; j++, k+=3) {
		buf[k  ] = *(cr+offset);
		buf[k+1] = *(cg+offset);
		buf[k+2] = *(cb+offset);
		offset++;
	   }
	   fwrite(buf, img.xsize*3, 1, fd);
	}
  }
  else if(img.imagetype == IGrey) {
    offset = 0;
    for(i = 0; i < img.ysize;  i++ ) {
	__Img_Busy(i*100/img.ysize);
	for(j=0, k=0; j<img.xsize; j++, k+=3) {
		buf[k  ] = buf[k+1] = buf[k+2] = *(cr+offset);
		offset ++;
	}
	fwrite(buf, img.xsize*3, 1, fd);
    }
  }
  else {
    offset = 0;
    for(i = 0; i < img.ysize;  i++ ) {
	__Img_Busy(i*100/img.ysize);
	for(j=0, k=0; j<img.xsize; j++, k+=3) {
	      buf[k  ] = img.pal[*(cr+offset+j)].b;
	      buf[k+1] = img.pal[*(cr+offset+j)].g;
	      buf[k+2] = img.pal[*(cr+offset+j)].r;
	      offset ++;
	}
	fwrite(buf, img.xsize*3, 1, fd);
    }
  }

  fclose(fd);
  free(buf);
  __Img_Busy(0);
  return ITRUE;
 }
