/**************************************************************************
*  Image process tool box                                                 *
*     by Yang Yudong                                                      *
*     yangyd@yahoo.com                                                    *
*                                                                         *
***************************************************************************
*  Copyright (C) 1992-1999, Yang Yudong, All rights reserved.             *
*  This file is part of Yang Yudong's image processing software package.  *
*  If you use this software, you agree to the following:                  *
*  This program package is purely experimental, and is licensed "as is".  *
*  Permission is granted to use, modify, and distribute this program      *
*  without charge for any purpose, provided this license/ disclaimer      *
*  notice appears in the copies.  No warranty or maintenance is given,    *
*  either expressed or implied.  In no event shall the author(s) be       *
*  liable to you or a third party for any special, incidental,            *
*  consequential, or other damages, arising out of the use or inability   *
*  to use the program for any purpose (or the loss of data), even if we   *
*  have been advised of such possibilities.  Any public reference or      *
*  advertisement of this source code should refer to it as Yang Yudong's  *
*  orignal.                                                               *
**************************************************************************/
// ****************************************************************
//  Image process tool box
//     by Yang Yudong
//
// File : iaffine.c
// Description:  do full affine transformation of an image 
// Create Date:  1996. 10. 28
// Modification(date/where): 
//
// ****************************************************************

#include <malloc.h>
#include <stdio.h>
#include <string.h>
#include <float.h>
#include <math.h>
#include <stdlib.h>
#define _IMG_LIBBUILD_
#include "imgproc.h"
#include "routines.h"

#define D2R  (3.1415926535897932385/180.0)
#ifndef DBL_EPSILON
 #define DBL_EPSILON	2.22045e-16
#endif

// des[u; v] = [af[0] af[1]; af[2] af[3]]*src[u; v] + [af[4]; af[5]]
 
static void __affine_size(int ix, int iy, double af[6], int *rx, int *ry)
{
  double x1, y1, x2, y2, x, y;

  x = (double)ix/2.0;  y = (double)iy/2.0;

  x1 = fabs( x*af[0] + y*af[1]+af[4]);
  x2 = fabs(-x*af[0] + y*af[1]+af[4]);
  if(x2 > x1) x1 = x2;
  x2 = fabs(-x*af[0] - y*af[1]+af[4]);
  if(x2 > x1) x1 = x2;
  x2 = fabs( x*af[0] - y*af[1]+af[4]);
  if(x2 > x1) x1 = x2;

  y1 = fabs( x*af[2] + y*af[3]+af[5]);
  y2 = fabs(-x*af[2] + y*af[3]+af[5]);
  if(y2 > y1) y1 = y2;
  y2 = fabs(-x*af[2] - y*af[3]+af[5]);
  if(y2 > y1) y1 = y2;
  y2 = fabs( x*af[2] - y*af[3]+af[5]);
  if(y2 > y1) y1 = y2;

  *rx = (int)ceil(2*x1);
  *ry = (int)ceil(2*y1);
}

// des[u; v] = [af[0] af[1]; af[2] af[3]]*src[u; v] + [af[4]; af[5]]
//  des-center_des  = A*(src - center_src) + T
//    iA = A^-1
//      /* if A = [a b; c d], |A| != 0,  then iA = inv(A) = [d -b; -c a]/(a*d - b*c) */
//   src = iA*(des-center_des-T) + center_src
//   src = iA*des + {-iA(center_des + T) + center_src}
static void __affine_build_table(long *lut, int dx, int dy, int sx, int sy, double af[6])
{ int i;
  double a, b, c, d, s, tu, tv, ux, uy;

  s = af[0]*af[3] - af[1]*af[2];
  if(fabs(s) <= DBL_EPSILON) {
    for(i=0; i<2*(dx+dy); i++) lut[i] = 0;
    return ;
  }
  
  a =  af[3]/s*1024.0;   b = -af[1]/s*1024.0;
  c = -af[2]/s*1024.0;   d =  af[0]/s*1024.0;
  tu = (double)dx/2.0 + af[4];
  tv = (double)dy/2.0 + af[5];
  ux = -tu*a - tv*b + ((double)sx/2.0)*1024.0; 
  uy = -tu*c - tv*d + ((double)sy/2.0)*1024.0; 
  
  for(i=0; i<dx; i++) {
    lut[i   ] =  (long)(a*i+ux);
    lut[i+dx] =  (long)(c*i+uy);
  }
  for(i=0; i<dy; i++) {
    lut[i+2*dx   ] = (long)(i*b);
    lut[i+2*dx+dy] = (long)(i*d);
  }
}  

// Do full affine transformation (image center) of img to get des, des is the same size as img
// Positive tx is translate right,  Positive ty is translate down 
IBOOL  image_full_affine(ImageDes *des, ImageDes img, double af[6], IDWORD fills, IInterpType interpolation)
{
  long *lut;
  
  if(!img.load) return IFALSE;
  if(fabs(af[0]*af[3] - af[1]*af[2]) <= DBL_EPSILON) //   the affine transformation is one dimensional
    return IFALSE;
    
  __Img_Busy(1);
  if(!AllocPicture(des, img.xsize, img.ysize, img.imagetype, img.alpha, img.numColors)) {__Img_Busy(0); return IFALSE;}
  lut = (long *)malloc(sizeof(long)*2*(img.xsize+img.ysize));
  if(lut == NULL) {
     FreePicture(des);
     __Img_Busy(0); 
     return IFALSE;
  } 
  __affine_build_table(lut, des->xsize, des->ysize, img.xsize, img.ysize, af);

  des->load = ITRUE;
  des->transparent = img.transparent;
  des->trans = img.trans;
  des->background = img.background;
  des->back = img.back;
  des->gamma = img.gamma;

  if(img.imagetype == IColor256 || img.imagetype == IIndexedColor) {
	  interpolation = IInterp_fast;
	  memcpy(des->pal, img.pal, sizeof(IMyRGB)*256);
  }

  switch (interpolation) {
    case IInterp_normal:   // or bilinear interpolation
	   if(des->alpha) __lut2d(img.a, img.xsize, img.ysize, des->a, des->xsize, des->ysize, lut, DWORD_COLOR_ALPHA(fills));
       __lut2d(img.r, img.xsize, img.ysize, des->r, des->xsize, des->ysize, lut, DWORD_COLOR_RED(fills));
       __Img_Busy(img.imagetype != ITrueColor ? 95 : 33);
       if(img.imagetype == ITrueColor) {
         __lut2d(img.g, img.xsize, img.ysize, des->g, des->xsize, des->ysize, lut, DWORD_COLOR_GREEN(fills));
         __Img_Busy(66);
         __lut2d(img.b, img.xsize, img.ysize, des->b, des->xsize, des->ysize, lut, DWORD_COLOR_BLUE(fills));
         __Img_Busy(99);
       }
       break;
    case IInterp_fast  :   // or near neighbor interpolation (palette image can only do this)
	   if(des->alpha) __lut2d_fast(img.a, img.xsize, img.ysize, des->a, des->xsize, des->ysize, lut, DWORD_COLOR_ALPHA(fills));
       __lut2d_fast(img.r, img.xsize, img.ysize, des->r, des->xsize, des->ysize, lut, DWORD_COLOR_RED(fills));
       __Img_Busy(img.imagetype != ITrueColor ? 95 : 33);
       if(img.imagetype == ITrueColor) {
         __lut2d_fast(img.g, img.xsize, img.ysize, des->g, des->xsize, des->ysize, lut, DWORD_COLOR_GREEN(fills));
         __Img_Busy(66);
         __lut2d_fast(img.b, img.xsize, img.ysize, des->b, des->xsize, des->ysize, lut, DWORD_COLOR_BLUE(fills));
         __Img_Busy(99);
       }
       break;
    case IInterp_fine  :   // or bicubic interpolation
	   if(des->alpha) __lut2d_fine(img.a, img.xsize, img.ysize, des->a, des->xsize, des->ysize, lut, DWORD_COLOR_ALPHA(fills));
       __lut2d_fine(img.r, img.xsize, img.ysize, des->r, des->xsize, des->ysize, lut, DWORD_COLOR_RED(fills));
       __Img_Busy(img.imagetype != ITrueColor ? 95 : 33);
       if(img.imagetype == ITrueColor) {
         __lut2d_fine(img.g, img.xsize, img.ysize, des->g, des->xsize, des->ysize, lut, DWORD_COLOR_GREEN(fills));
         __Img_Busy(66);
         __lut2d_fine(img.b, img.xsize, img.ysize, des->b, des->xsize, des->ysize, lut, DWORD_COLOR_BLUE(fills));
         __Img_Busy(99);
       }
       break;
  }
  free(lut);
  __Img_Busy(0);
  return ITRUE;
}  




// Do full affine transformation (image center) of img to get des, des contain the full result
// Positive tx is translate right,  Positive ty is translate down 
IBOOL  image_full_affine_size(ImageDes *des, ImageDes img, double af[6], IDWORD fills, IInterpType interpolation)
{
  long *lut;
  int dx, dy;
  
  if(!img.load) return IFALSE;
  if(fabs(af[0]*af[3] - af[1]*af[2]) <= DBL_EPSILON) //   the affine transformation is one dimensional
    return IFALSE;

  __Img_Busy(1);
  __affine_size(img.xsize, img.ysize, af, &dx, &dy);
  if(!AllocPicture(des, dx, dy, img.imagetype, img.alpha, img.numColors)) {__Img_Busy(0); return IFALSE;}
  lut = (long *)malloc(sizeof(long)*2*(dx+dy));
  if(lut == NULL) {
     FreePicture(des);
     __Img_Busy(0); 
     return IFALSE;
  } 
  __affine_build_table(lut, des->xsize, des->ysize, img.xsize, img.ysize, af);

  des->load = ITRUE;
  des->transparent = img.transparent;
  des->trans = img.trans;
  des->background = img.background;
  des->back = img.back;
  des->gamma = img.gamma;

  if(img.imagetype == IColor256 || img.imagetype == IIndexedColor) 
  {
	  interpolation = IInterp_fast;
	  memcpy(des->pal, img.pal, sizeof(IMyRGB)*256);
  }

  switch (interpolation) {
    case IInterp_normal:   // or bilinear interpolation
	   if(des->alpha) __lut2d(img.a, img.xsize, img.ysize, des->a, des->xsize, des->ysize, lut, DWORD_COLOR_ALPHA(fills));
       __lut2d(img.r, img.xsize, img.ysize, des->r, des->xsize, des->ysize, lut, DWORD_COLOR_RED(fills));
       __Img_Busy(img.imagetype != ITrueColor ? 95 : 33);
       if(img.imagetype == ITrueColor) {
         __lut2d(img.g, img.xsize, img.ysize, des->g, des->xsize, des->ysize, lut, DWORD_COLOR_GREEN(fills));
         __Img_Busy(66);
         __lut2d(img.b, img.xsize, img.ysize, des->b, des->xsize, des->ysize, lut, DWORD_COLOR_BLUE(fills));
         __Img_Busy(99);
       }
       break;
    case IInterp_fast  :   // or near neighbor interpolation (palette image can only do this)
	   if(des->alpha) __lut2d_fast(img.a, img.xsize, img.ysize, des->a, des->xsize, des->ysize, lut, DWORD_COLOR_ALPHA(fills));
       __lut2d_fast(img.r, img.xsize, img.ysize, des->r, des->xsize, des->ysize, lut, DWORD_COLOR_RED(fills));
       __Img_Busy(img.imagetype != ITrueColor ? 95 : 33);
       if(img.imagetype == ITrueColor) {
         __lut2d_fast(img.g, img.xsize, img.ysize, des->g, des->xsize, des->ysize, lut, DWORD_COLOR_GREEN(fills));
         __Img_Busy(66);
         __lut2d_fast(img.b, img.xsize, img.ysize, des->b, des->xsize, des->ysize, lut, DWORD_COLOR_BLUE(fills));
         __Img_Busy(99);
       }
       break;
    case IInterp_fine  :   // or bicubic interpolation
	   if(des->alpha) __lut2d_fine(img.a, img.xsize, img.ysize, des->a, des->xsize, des->ysize, lut, DWORD_COLOR_ALPHA(fills));
       __lut2d_fine(img.r, img.xsize, img.ysize, des->r, des->xsize, des->ysize, lut, DWORD_COLOR_RED(fills));
       __Img_Busy(img.imagetype != ITrueColor ? 95 : 33);
       if(img.imagetype == ITrueColor) {
         __lut2d_fine(img.g, img.xsize, img.ysize, des->g, des->xsize, des->ysize, lut, DWORD_COLOR_GREEN(fills));
         __Img_Busy(66);
         __lut2d_fine(img.b, img.xsize, img.ysize, des->b, des->xsize, des->ysize, lut, DWORD_COLOR_BLUE(fills));
         __Img_Busy(99);
       }
       break;
  }
  free(lut);
  __Img_Busy(0);
  return ITRUE;
}  


