/**************************************************************************
*  Image process tool box                                                 *
*     by Yang Yudong                                                      *
*     yangyd@yahoo.com                                                    *
*                                                                         *
***************************************************************************
*  Copyright (C) 1992-1999, Yang Yudong, All rights reserved.             *
*  This file is part of Yang Yudong's image processing software package.  *
*  If you use this software, you agree to the following:                  *
*  This program package is purely experimental, and is licensed "as is".  *
*  Permission is granted to use, modify, and distribute this program      *
*  without charge for any purpose, provided this license/ disclaimer      *
*  notice appears in the copies.  No warranty or maintenance is given,    *
*  either expressed or implied.  In no event shall the author(s) be       *
*  liable to you or a third party for any special, incidental,            *
*  consequential, or other damages, arising out of the use or inability   *
*  to use the program for any purpose (or the loss of data), even if we   *
*  have been advised of such possibilities.  Any public reference or      *
*  advertisement of this source code should refer to it as Yang Yudong's  *
*  orignal.                                                               *
**************************************************************************/
// ****************************************************************
//  Image process tool box
//     by Yang Yudong
//
// File : id_line.c
// Description:  draw line in image
// Create Date:  1996. 12. 18
// Modification(date/where): 
//
// ****************************************************************

#include <stdlib.h>
#define _IMG_LIBBUILD_
#include "image.h"
#include "imgdraw.h"

/****************************************************************************
*
* Function:     img_draw_line
* Parameters:   img         - image to draw the line on
*               x1,y1       - First endpoint of line
*               x2,y2       - Second endpoint of line
* Description:  Scan convert a line segment using the MidPoint Digital
*               Differential Analyser algorithm.
*
****************************************************************************/
void img_draw_line(ImageDes img, int x1,int y1,int x2,int y2)
{
    int     d;                      /* Decision variable                */
    int     dx,dy;                  /* Dx and Dy values for the line    */
    int     Eincr,NEincr;           /* Decision variable increments     */
    int     yincr;                  /* Increment for y values           */
    int     t;                      /* Counters etc.                    */

    if(img.alloc == IFALSE) return;
    dx = abs(x2 - x1);
    dy = abs(y2 - y1);
    if (dy <= dx) {
        /* We have a line with a slope between -1 and 1
         *
         * Ensure that we are always scan converting the line from left to
         * right to ensure that we produce the same line from P1 to P0 as the
         * line from P0 to P1.
         */
        if (x2 < x1) {
            t = x2; x2 = x1; x1 = t;    /* Swap X coordinates           */
            t = y2; y2 = y1; y1 = t;    /* Swap Y coordinates           */
            }

        if (y2 > y1)
            yincr = 1;
        else
            yincr = -1;

        d = 2*dy - dx;              /* Initial decision variable value  */
        Eincr = 2*dy;               /* Increment to move to E pixel     */
        NEincr = 2*(dy - dx);       /* Increment to move to NE pixel    */

        img_draw_pixel(img, x1,y1);      /* Draw the first point at (x1,y1)  */

        /* Incrementally determine the positions of the remaining pixels
         */
        for (x1++; x1 <= x2; x1++) {
            if (d < 0) {
                d += Eincr;         /* Choose the Eastern Pixel         */
                }
            else {
                d += NEincr;        /* Choose the North Eastern Pixel   */
                y1 += yincr;        /* (or SE pixel for dx/dy < 0!)     */
                }
            img_draw_pixel(img, x1,y1); /* Draw the point                   */
            }
        }
    else {
        /* We have a line with a slope between -1 and 1 (ie: includes
         * vertical lines). We must swap our x and y coordinates for this.
         *
         * Ensure that we are always scan converting the line from left to
         * right to ensure that we produce the same line from P1 to P0 as the
         * line from P0 to P1.
         */
        if (y2 < y1) {
            t = x2; x2 = x1; x1 = t;    /* Swap X coordinates           */
            t = y2; y2 = y1; y1 = t;    /* Swap Y coordinates           */
            }

        if (x2 > x1)
            yincr = 1;
        else
            yincr = -1;

        d = 2*dx - dy;              /* Initial decision variable value  */
        Eincr = 2*dx;               /* Increment to move to E pixel     */
        NEincr = 2*(dx - dy);       /* Increment to move to NE pixel    */

        img_draw_pixel(img, x1,y1); /* Draw the first point at (x1,y1)  */

        /* Incrementally determine the positions of the remaining pixels
         */
        for (y1++; y1 <= y2; y1++) {
            if (d < 0) {
                d += Eincr;         /* Choose the Eastern Pixel         */
                }
            else {
                d += NEincr;        /* Choose the North Eastern Pixel   */
                x1 += yincr;        /* (or SE pixel for dx/dy < 0!)     */
                }
            img_draw_pixel(img, x1,y1);  /* Draw the point                   */
            }
        }
}
