/**************************************************************************
*  Image process tool box                                                 *
*     by Yang Yudong                                                      *
*     yangyd@yahoo.com                                                    *
*                                                                         *
***************************************************************************
*  Copyright (C) 1992-1999, Yang Yudong, All rights reserved.             *
*  This file is part of Yang Yudong's image processing software package.  *
*  If you use this software, you agree to the following:                  *
*  This program package is purely experimental, and is licensed "as is".  *
*  Permission is granted to use, modify, and distribute this program      *
*  without charge for any purpose, provided this license/ disclaimer      *
*  notice appears in the copies.  No warranty or maintenance is given,    *
*  either expressed or implied.  In no event shall the author(s) be       *
*  liable to you or a third party for any special, incidental,            *
*  consequential, or other damages, arising out of the use or inability   *
*  to use the program for any purpose (or the loss of data), even if we   *
*  have been advised of such possibilities.  Any public reference or      *
*  advertisement of this source code should refer to it as Yang Yudong's  *
*  orignal.                                                               *
**************************************************************************/
//****************************************************************
//  Image process tool box
//     by Yang Yudong
//
// File : imgc24_8.c
// Description: convert 24 bit true color image to 8 bit indexed color image 
// Create Date: 1996. 9. 25
// Modification(date/where):
//
//****************************************************************
#include <malloc.h>
#include <string.h>
#include <stdio.h>
#include <limits.h>

#define _IMG_LIBBUILD_
#include "image.h"
#include "routines.h"


// convert 24 bits color image to 8 bits pseudo color (256 indexed)
//     where num_of_color is the number of indexed colors required.
//           min_index is the minimal color index number
//     des can be a unallocated descriptor
//     output palettes are sorted by increasing brightness order
// return ITRUE if success, IFALSE if fail

IBOOL img_color24to8(ImageDes* des, ImageDes src, IDWORD num_of_color, IDWORD min_index)
 {
  int   index;
  int   red, grn, blu;
  int   *colors, max, mix1, mix2;

  int  dist,value;
  int  i, j, tmp, istart, iend;
  int  *ind;
  IBYTE *rgb;
  IBYTE *cr;
  IBYTE *cg;
  IBYTE *cb;

  IBYTE *img8;
  IMyRGB  *pal;
  int   offset;

   #ifdef _DEBUG
     fprintf(stderr, "Converting ITrueColor to 256 color...");
   #endif
   if(src.load && 
      src.imagetype == ITrueColor &&
      num_of_color>1 &&
      num_of_color<=256 &&
      num_of_color+min_index <= 256) {
      if ((rgb = (IBYTE *)malloc(4096+sizeof(int)*4096+sizeof(int)*256)) == NULL)
                 return IFALSE;    // can not allocate temp buffer
      colors = (int *)(rgb+4096);
      ind = (colors + 4096);
   }
   else return IFALSE;

   __Img_Busy(1);
   if(AllocPicture(des, src.xsize, src.ysize, IColor256, src.alpha, num_of_color) == IFALSE) {
        free(rgb);
        __Img_Busy(0);
        return IFALSE;                    // can not allocate DIB buffer
   }
   img8 = des->r;
   pal  = des->pal;
   des->load = ITRUE;
   des->transparent = IFALSE;
   des->trans = src.trans;
   des->background = src.background;
   des->back = src.back;
   des->gamma = src.gamma;
   if(src.alpha) memcpy(des->a, src.a, src.xsize*src.ysize);
   #ifdef _DEBUG
     fprintf(stderr, "Count.");
   #endif
   for(i=0; i<4096; i++)  // Initialize the colors  statical table
      colors[i]=0;
   __Img_Busy(5);
   offset = 0; cr = src.r; cg = src.g; cb = src.b;                     // point to start of image
   for(i=0; i<src.ysize; i++ ) {
      for (j=0; j<src.xsize; j++)   {        // Count the colors
             tmp=(((IWORD)cr[offset]&0xf0)<<4)|((IWORD)cg[offset]&0xf8)|(((IWORD)cb[offset]&0xe0)>>5);  // r : g : b = 4:5:3  , 4096 colors selected
             colors[tmp]++;
             offset++;
      }
   }
   __Img_Busy(25);
   #ifdef _DEBUG
     fprintf(stderr, "Sort.");
   #endif
   istart = min_index;
   iend = istart+num_of_color;
   for(i=istart; i<iend; i++)  {       // Select palette colors
      max=0;   index = i;
      for (j=0; j<4096; j++)
         if (colors[j]>max) {
            max=colors[j];
            index=j;
         }
      ind[i] = index;
      colors[index] = -colors[index];                          // negtive it to aviod another selection
      rgb[index] = (IBYTE)i;
      pal[i].r   =(IBYTE)((index&0x0f00)>>4);             // set it's red
      pal[i].g   =(IBYTE)((index&0x0f8));                 // set it's green
      pal[i].b   =(IBYTE)((index&0x07)<<5);       // set it's blue
   }
   __Img_Busy(40);
   #ifdef _DEBUG
     fprintf(stderr, "Map.");
   #endif
   for (i=0; i<4096; i++) {     // Map other colors to palette using distance
      tmp=0;                        // no selection done
      if(colors[i] > 0){        // if it is an selected color or an zero counted one
         dist=65535;                      // maximum distance
         for (j=istart; j<iend; j++) {
            red=(pal[j].r>>3);
            grn=(pal[j].g>>3);
            blu=(pal[j].b>>3);
            red-=((i&0x0f00)>>7);
            grn-=((i&0x00f8)>>3);
            blu-=((i&0x0007)<<2);
            value=(IWORD)(red*red+grn*grn+blu*blu);   // calculate the distance
            if (value<dist){
              dist=value;   // an better match
              tmp=j;
            }
         }
      }
      if(colors[i] >= 0) rgb[i]=(IBYTE)tmp;      // store the matching relation to that palette
      else if(colors[i] > 0) {
         mix1 = -colors[ind[tmp]];
         mix2 = colors[i];
         // mix red
         red =   (int)(( (IDWORD)pal[tmp].r*mix1
                +(IDWORD)((i&0x0f00)>>4)*mix2)    // mix this two colors by a weighted average
                 /(mix1+mix2));
         if(red > 255) red = 255;                // tructuate
         pal[tmp].r =(IBYTE)red;
         // mix green
         grn =  (int)(( (IDWORD)pal[tmp].g*mix1
               +(IDWORD)(i&0x0f8)*mix2)   // mix this two colors by a weighted average
               /(mix1+mix2));
         if(grn > 255) grn = 255;                // tructuate
         pal[tmp].g =(IBYTE)grn;
         // mix blue
         blu =   (int)(( (IDWORD)pal[tmp].b*mix1
                +(IDWORD)((i&0x07)<<5)*mix2)      // mix this two colors by a weighted average
                /(mix1+mix2));
         if(blu > 255) blu = 255;                // tructuate
         pal[tmp].b =(IBYTE)blu;
         // calculate mixed numbers
         colors[ind[tmp]] -= mix2;       // mixed numbers (Negtive)
      }
   }

   __Img_Busy(80);
   // look up table for 8 bit color data
   offset = 0;  // point to start of image
   for(i=0; i<src.ysize; i++ )
      for (j=0; j<src.xsize; j++) {          // set the colors
         tmp=((cr[offset]&0xf0)<<4)|(cg[offset]&0xf8)|((cb[offset]&0xe0)>>5);  // r : g : b = 4:5:3  , 4096 colors selected
         img8[offset]=rgb[tmp];
         offset++;
      }
   __Img_Busy(95);
   for(i=0; i<istart; i++) {
       pal[i].r = 0;
       pal[i].g = 0;
       pal[i].b = 0;
   }
   for(i=istart; i<iend; i++) {
       pal[i].r = pal[i].r ? (pal[i].r |0x0f) : 0;
       pal[i].g = pal[i].g ? (pal[i].g |0x07) : 0;
       pal[i].b = pal[i].b ? (pal[i].b |0x1f) : 0;
   }
   for(; i<256; i++) {
       pal[i].r = 255;
       pal[i].g = 255;
       pal[i].b = 255;
   }
   #ifdef _DEBUG
     fprintf(stderr, "Done.\n");
   #endif
   free(rgb);
   __Img_Busy(0);
   return ITRUE;
 }


IBOOL img_colorquantization(ImageDes* des, ImageDes src, IDWORD num_of_color, IDWORD min_index)
{
	register IBYTE *sbuff = src.r;
	IDWORD histo[256];
	IDWORD table[256];
	IDWORD *jumpTable;
	IDWORD i,j, k, m, swp, dist, currdist, usedColor;
	IDWORD transIdx=0;
	IBOOL trans=IFALSE;
    IMyRGB  *pal;

    if( !src.load ||
        num_of_color<1 ||
        num_of_color>256 ||
        num_of_color+min_index > 256) return IFALSE; 
	if(src.imagetype == ITrueColor) 
		return img_color24to8(des, src, num_of_color, min_index);
	if(/*(src.imagetype == IColor256 || src.imagetype == IIndexedColor) && */src.numColors <= num_of_color)
	{
		return img_newcopy(des, src);
	}

    if(AllocPicture(des, src.xsize, src.ysize, IIndexedColor, src.alpha, num_of_color) == IFALSE) 
	{
        return IFALSE;                    // can not allocate DIB buffer
    }

    if(src.alpha) memcpy(des->a, src.a, src.xsize*src.ysize);
    pal  = des->pal;
	for(i=0; i<min_index; i++) 
	{
		pal[i].r = 0;
		pal[i].g = 0;
		pal[i].b = 0;
	}
	for(i=min_index+num_of_color; i<256; i++) 
	{
		pal[i].r = 0;
		pal[i].g = 0;
		pal[i].b = 0;
	}
	pal += min_index;

	memset(histo, 0, sizeof(IDWORD)*256);  // Zero bins array
	for (i=0; i<(IDWORD)src.xsize; i++)
		for (j=0; j<(IDWORD)src.ysize; j++) 
		{
			histo[*sbuff]++;
			sbuff++;
		}
	if(src.transparent)
	{
		if(histo[src.trans.Index] == 0) // no trans at all
		{
			trans = IFALSE;
		}
		else
		{
			trans = ITRUE;
		}
	}

	for (i=0; i<src.numColors; i++) table[i] = i;

	// dirty selection sort for the small number
	for (i=0; i<num_of_color; i++)
	{
		m=i;
		for (j=i+1; j<src.numColors; j++) 
		{
			if(histo[m] < histo[j])
			{
				m = j;
			}
		}
		if(m > i) 
		{
			swp = histo[m];
			histo[m] = histo[i];
			histo[i] = swp;
			swp = table[m];
			table[m] = table[i];
			table[i] = swp;
		}
	}
	usedColor = num_of_color; 
	if(trans)  // has an transparent color
	{
		for(i=0; i<usedColor; i++)
			if(table[i] == src.trans.Index) break;
		if(i < usedColor) 
		{
			swp = table[usedColor-1];
			table[usedColor-1] = table[i];
			table[i] = swp;
		}
		else
		{
			swp = table[usedColor-1];
			table[usedColor-1] = src.trans.Index;
			table[src.trans.Index] = swp;
		}
		//
		usedColor--; 
		transIdx = usedColor;
/*		if(src.imagetype == IGrey) 
		{
			pal[transIdx].r = pal[transIdx].g = pal[transIdx].b = src.transIndex;
		}
		else pal[transIdx] = src.pal[src.transIndex];
*/	}

    // calculate the remap table
	jumpTable = histo;

	if(trans) for(i=0; i<256; i++) jumpTable[i] = transIdx;
	else  for(i=0; i<256; i++) jumpTable[i] = num_of_color-1;

	if(src.imagetype == IColor256 || src.imagetype == IIndexedColor)
	{
		for(i=0; i<num_of_color; i++) 
		{
			jumpTable[table[i]] = i+min_index;; // the color of table[i] is mapped to i;
			pal[i] = src.pal[table[i]];
		}
		// remap other colors
		for (i=num_of_color; i<src.numColors; i++)
		{
			m = 0;
			k = table[i];  // index k has been changed to posi i
			dist = INT_MAX;
			for(j=0; j<usedColor; j++)
			{
				currdist =  ((int)src.pal[k].r - (int)pal[j].r)*((int)src.pal[k].r - (int)pal[j].r)
					       +((int)src.pal[k].g - (int)pal[j].g)*((int)src.pal[k].g - (int)pal[j].g)
					       +((int)src.pal[k].b - (int)pal[j].b)*((int)src.pal[k].b - (int)pal[j].b);
				if(currdist < dist)
				{
					dist = currdist;
					m = j;
				}
			}
			// the lut value
			jumpTable[k] = m+min_index;
		}
	}
	else
	{
		for(i=0; i<num_of_color; i++) 
		{
			pal[i+min_index].r = (IBYTE)table[i];
			pal[i+min_index].g = (IBYTE)table[i];
			pal[i+min_index].b = (IBYTE)table[i];
			jumpTable[table[i]] = i+min_index;; // the color of table[i] is mapped to i;
		}
		// remap other colors
		for (i=num_of_color; i<src.numColors; i++)
		{
			m = 0;
			k = table[i];
			dist = INT_MAX;
			for(j=0; j<usedColor; j++)
			{
				if(k > table[j]) currdist =  k - table[j];
				else currdist =  table[j] - k;
				if(currdist < dist)
				{
					dist = currdist;
					m = j;
				}
			}
			// the lut value
			jumpTable[k] = m+min_index;
		}
	}
    memcpy(des->r /*index*/, src.r /*index*/, src.xsize*src.ysize);
    table_mod_int((int *)jumpTable, des->r, des->xsize, des->ysize);
    des->load = ITRUE;
    des->transparent = trans;
    des->trans.Index = (IBYTE)transIdx;
    des->gamma = src.gamma;
	return ITRUE;
}
