/**************************************************************************
*  Image process tool box                                                 *
*     by Yang Yudong                                                      *
*     yangyd@yahoo.com                                                    *
*                                                                         *
***************************************************************************
*  Copyright (C) 1992-1999, Yang Yudong, All rights reserved.             *
*  This file is part of Yang Yudong's image processing software package.  *
*  If you use this software, you agree to the following:                  *
*  This program package is purely experimental, and is licensed "as is".  *
*  Permission is granted to use, modify, and distribute this program      *
*  without charge for any purpose, provided this license/ disclaimer      *
*  notice appears in the copies.  No warranty or maintenance is given,    *
*  either expressed or implied.  In no event shall the author(s) be       *
*  liable to you or a third party for any special, incidental,            *
*  consequential, or other damages, arising out of the use or inability   *
*  to use the program for any purpose (or the loss of data), even if we   *
*  have been advised of such possibilities.  Any public reference or      *
*  advertisement of this source code should refer to it as Yang Yudong's  *
*  orignal.                                                               *
**************************************************************************/
// ****************************************************************
//  Image process tool box
//     by Yang Yudong
//
// File : imgflat.c
// Description:  flatten an image with transparent, alpha and background 
// Create Date:  2000. 1. 5
// Modification(date/where): 
//
// ****************************************************************

#include <malloc.h>
#include <string.h>
#include <stdio.h>

#define _IMG_LIBBUILD_
#include "image.h"
#include "routines.h"


IBOOL img_flatten(ImageDes *img, IDWORD bkcolor)
{
	IBYTE bkr, bkg, bkb;
	IDWORD imgsize;

	if(!img->load || !img->alloc) return IFALSE;
	if(!img->alpha && !img->transparent) return ITRUE;  // no transparent and no alpha at all
	if(img->background) 
	{
		switch(img->imagetype)
		{
			case ITrueColor:
				bkr = img->back.Color.r;
				bkg = img->back.Color.g;
				bkb = img->back.Color.b;
				break;
			case IGrey:
				bkr = bkg = bkb = img->back.Grey;
				break;
			case IIndexedColor:
			case IColor256:
				bkr = bkg = bkb = img->back.Index;
				break;
		}
	}
	else
	{
		bkr = DWORD_COLOR_RED(bkcolor);
		bkg = DWORD_COLOR_GREEN(bkcolor);
		bkb = DWORD_COLOR_BLUE(bkcolor);
	}

	img->background = IFALSE; // clear the background

	imgsize = img->xsize*img->ysize;

	if(img->alpha)  // this image has an alpha channel
	{
		switch(img->imagetype)
		{
			case ITrueColor:
				{
					register IDWORD i;
					if(img->transparent)
					{
						register IBYTE *a;
						register IBYTE *r, *g, *b;
						IBYTE tr = img->trans.Color.r;
						IBYTE tg = img->trans.Color.g;
						IBYTE tb = img->trans.Color.b;
						a = img->a;
						r = img->r;
						g = img->g;
						b = img->b;

						for(i=imgsize; i; i--, a++, r++, g++, b++)
						{
							if(*r == tr && *g == tg && *b == tb) 
							{
								*r = bkr; *g = bkg; *b = bkb;
							}
							else
							{
								*r = (IBYTE)(((ILONG)*r-(ILONG)bkr)*(ILONG)*a/(ILONG)255 + (ILONG)bkr);
								*g = (IBYTE)(((ILONG)*g-(ILONG)bkg)*(ILONG)*a/(ILONG)255 + (ILONG)bkg);
								*b = (IBYTE)(((ILONG)*b-(ILONG)bkb)*(ILONG)*a/(ILONG)255 + (ILONG)bkb);
							}
						}
					}
					else
					{
						register IBYTE *a;
						register IBYTE *p;
						a = img->a;
						p = img->r;
						for(i=imgsize; i; i--, p++, a++)
						{
							*p = (IBYTE)(((ILONG)*p-(ILONG)bkr)*(ILONG)*a/(ILONG)255 + (ILONG)bkr);
						}
						a = img->a;
						p = img->g;
						for(i=imgsize; i; i--, p++, a++)
						{
							*p = (IBYTE)(((ILONG)*p-(ILONG)bkg)*(ILONG)*a/(ILONG)255 + (ILONG)bkg);
						}
						a = img->a;
						p = img->b;
						for(i=imgsize; i; i--, p++, a++)
						{
							*p = (IBYTE)(((ILONG)*p-(ILONG)bkb)*(ILONG)*a/(ILONG)255 + (ILONG)bkb);
						}
					}
				}
				break;
			case IGrey:
				{
					register IBYTE *p, *a;
					register IDWORD i;
					p = img->r; //grey;
					a = img->a;
					if(img->transparent)
					{
						IBYTE tr = img->trans.Grey;
						for(i=imgsize; i; i--, p++, a++)
						{
							if(*p == tr) *p = bkr;
							else *p = (IBYTE)(((ILONG)*p-(ILONG)bkr)*(ILONG)*a/(ILONG)255 + (ILONG)bkr);
						}
					}
					else
					{
						for(i=imgsize; i; i--, p++, a++)
						{
							*p = (IBYTE)(((ILONG)*p-(ILONG)bkr)*(ILONG)*a/(ILONG)255 + (ILONG)bkr);
						}
					}
				}
				break;
			case IIndexedColor:
			case IColor256:
				{
					register IBYTE *p, *a;
					register IDWORD i;
					p = img->r; //index;
					a = img->a;
					if(img->transparent)
					{
						IBYTE tr = img->trans.Index;
						for(i=imgsize; i; i--, p++, a++)
						{
							if(*a < 128 || *p == tr) *p = bkr;
						}
					}
					else
					{
						for(i=imgsize; i; i--, p++, a++)
						{
							if(*a < 128) *p = bkr;
						}
					}
				}
				break;
		}
		free(img->a);
		img->a = NULL;
		img->alpha = IFALSE;	// clear the alpha channel
	}
	else  // this image has only an transparent color
	{
		switch(img->imagetype)
		{
			case ITrueColor:
				{
					register IDWORD i;
					register IBYTE *r, *g, *b;
					IBYTE tr = img->trans.Color.r;
					IBYTE tg = img->trans.Color.g;
					IBYTE tb = img->trans.Color.b;
					r = img->r;
					g = img->g;
					b = img->b;
					
					for(i=imgsize; i; i--, r++, g++, b++)
					{
						if(*r == tr && *g == tg && *b == tb) 
						{
							*r = bkr; *g = bkg; *b = bkb;
						}
					}
				}
				break;
			case IGrey:
				{
					register IBYTE *p;
					register IDWORD i;
					register IBYTE tr = img->trans.Grey;
					p = img->r; //grey;
					for(i=imgsize; i; i--, p++)
					{
						if(*p == tr) *p = bkr;
					}
				}
				break;
			case IIndexedColor:
			case IColor256:
				{
					register IBYTE *p;
					register IDWORD i;
					register IBYTE tr = img->trans.Index;
					p = img->r; //index;
					for(i=imgsize; i; i--, p++)
					{
						if(*p == tr) *p = bkr;
					}
				}
				break;
		}
	}

	img->transparent = IFALSE; // clear the transparent color

	return ITRUE;
}


IBOOL img_optimize_palette(ImageDes *img)
{
	register IBYTE *p;
	register IDWORD *histo;
	register IDWORD i;
	IBYTE *tbl;
	IDWORD imgsize;
	IDWORD nump;
	IBOOL moved;

	if(!img->load || !img->alloc) return IFALSE;
	if(!(img->imagetype == IIndexedColor || img->imagetype == IColor256)) return IFALSE;

	histo = malloc((sizeof(IDWORD)+sizeof(IBYTE))*256);
	if(!histo) return IFALSE;
	tbl = (IBYTE *)(histo+256);


	// firstly count the palette
	memset(histo, 0, sizeof(IDWORD)*256);
	p = img->r; //index;
	imgsize = img->xsize * img->ysize;
	for(i=imgsize; i; i--, p++) histo[*p]++;  

	// then we try to compress the palette
	for(i=0, nump=0, moved=IFALSE; i<256; i++)
	{
		tbl[i] = (IBYTE)i;
		if(histo[i]) // no empty
		{
			if(i > nump) 
			{
				moved = ITRUE;
			}
			tbl[i] = (IBYTE)nump; // remap the color to current empty one
			nump++;
		}
	}
	img->numColors = nump;
	if(moved)
	{
		p = img->r; //index;
		table_mod_(tbl, img->r /*index*/, img->xsize, img->ysize);
		for(i=0; i<256; i++) img->pal[tbl[i]] = img->pal[i];
		if(img->background) img->back.Index = (IBYTE)tbl[img->back.Index];
		if(img->transparent) img->trans.Index = (IBYTE)tbl[img->trans.Index];
	}
	free(histo);
	return ITRUE;
}
