/**************************************************************************
*  Image process tool box                                                 *
*     by Yang Yudong                                                      *
*     yangyd@yahoo.com                                                    *
*                                                                         *
***************************************************************************
*  Copyright (C) 1992-1999, Yang Yudong, All rights reserved.             *
*  This file is part of Yang Yudong's image processing software package.  *
*  If you use this software, you agree to the following:                  *
*  This program package is purely experimental, and is licensed "as is".  *
*  Permission is granted to use, modify, and distribute this program      *
*  without charge for any purpose, provided this license/ disclaimer      *
*  notice appears in the copies.  No warranty or maintenance is given,    *
*  either expressed or implied.  In no event shall the author(s) be       *
*  liable to you or a third party for any special, incidental,            *
*  consequential, or other damages, arising out of the use or inability   *
*  to use the program for any purpose (or the loss of data), even if we   *
*  have been advised of such possibilities.  Any public reference or      *
*  advertisement of this source code should refer to it as Yang Yudong's  *
*  orignal.                                                               *
**************************************************************************/
// ****************************************************************
//  Image process tool box
//     by Yang Yudong
//
// File : irotate.c
// Description:  rotate an image by any degree, positive is ClockWise
// Create Date:  1996. 10. 7
// Modification(date/where): 
//
// ****************************************************************

#include <malloc.h>
#include <stdio.h>
#include <string.h>
#include <float.h>
#include <math.h>
#include <stdlib.h>
#define _IMG_LIBBUILD_
#include "imgproc.h"
#include "routines.h"

#ifndef max
#define  max(a,b) ((a) > (b) ? (a) : (b))
#endif

#define D2R  (3.1415926535897932385/180.0)

static void __rotsize(int ix, int iy, double ang, int *rx, int *ry)
{
  double x1, y1, x2, y2, x, y;
  ang *= D2R;
  x = ix/2.0;  y = iy/2.0;
  x1 = fabs(x*cos(ang) - y*sin(ang));
  y1 = fabs(x*sin(ang) + y*cos(ang));
  x2 = fabs(x*cos(ang) + y*sin(ang));
  y2 = fabs(x*sin(ang) - y*cos(ang));
  *rx = (int)ceil(2*max(x1, x2));
  *ry = (int)ceil(2*max(y1, y2));
}

// des-center_des = R*(src - center_src)
//   R = (c -s,  s  c),   negR = (c  s,  -s c)
//   src = negR*(des-center_des) + center_src
//   src = negR*des - negR*center_des + center_src
static void __rbuild_table(long *lut, int rx, int ry, int ix, int iy, double ang)
{
  int i;
  double sa, ca, dx, dy;
  ang *= D2R;
  sa = sin(ang)*1024.0; ca = cos(ang)*1024.0;
  dx =  (double)rx/2.0*ca + (double)ry/2.0*sa - (double)ix/2.0*1024.0; 
  dy = -(double)rx/2.0*sa + (double)ry/2.0*ca - (double)iy/2.0*1024.0; 
  for(i=0; i<rx; i++) {
    lut[i   ] = (long)( (double)i*ca - dx);
    lut[i+rx] = (long)(-(double)i*sa - dy);
  }
  for(i=0; i<ry; i++) {
    lut[i+2*rx   ] = (long)((double)i*sa);
    lut[i+2*rx+ry] = (long)((double)i*ca);
  }
}
 

// Rotate (image center), des is enlarged as require
// Positive ang is CW rotation
IBOOL  image_rotate(ImageDes *des, ImageDes img, double ang, IDWORD fills, IInterpType interpolation)
{
  int rx, ry;
  long *lut;
  
  if(!img.load) return IFALSE;
  __Img_Busy(1);
  __rotsize(img.xsize, img.ysize, ang, &rx, &ry);
  if(!AllocPicture(des, rx, ry, img.imagetype, img.alpha, img.numColors)) {__Img_Busy(0); return IFALSE;}
  lut = (long *)malloc(sizeof(long)*2*(rx+ry));
  if(lut == NULL) {
     FreePicture(des);
     __Img_Busy(0); 
     return IFALSE;
  } 
  __rbuild_table(lut, rx, ry, img.xsize, img.ysize, ang);

  des->load = ITRUE;
  des->transparent = img.transparent;
  des->trans = img.trans;
  des->background = img.background;
  des->back = img.back;
  des->gamma = img.gamma;
  if(img.imagetype == IColor256 || img.imagetype == IIndexedColor) {
	  interpolation = IInterp_fast;
	  memcpy(des->pal, img.pal, 768);
  }

  switch (interpolation) {
    case IInterp_normal:   // or bilinear interpolation
	   if(des->alpha) __lut2d(img.a, img.xsize, img.ysize, des->a, des->xsize, des->ysize, lut, DWORD_COLOR_ALPHA(fills));
       __lut2d(img.r, img.xsize, img.ysize, des->r, des->xsize, des->ysize, lut, DWORD_COLOR_RED(fills));
       __Img_Busy(img.imagetype != ITrueColor ? 95 : 33);
       if(img.imagetype == ITrueColor) {
         __lut2d(img.g, img.xsize, img.ysize, des->g, des->xsize, des->ysize, lut, DWORD_COLOR_GREEN(fills));
         __Img_Busy(66);
         __lut2d(img.b, img.xsize, img.ysize, des->b, des->xsize, des->ysize, lut, DWORD_COLOR_BLUE(fills));
         __Img_Busy(99);
       }
       break;
    case IInterp_fast  :   // or near neighbor interpolation (palette image can only do this)
	   if(des->alpha) __lut2d_fast(img.a, img.xsize, img.ysize, des->a, des->xsize, des->ysize, lut, DWORD_COLOR_ALPHA(fills));
       __lut2d_fast(img.r, img.xsize, img.ysize, des->r, des->xsize, des->ysize, lut, DWORD_COLOR_RED(fills));
       __Img_Busy(img.imagetype != ITrueColor ? 95 : 33);
       if(img.imagetype == ITrueColor) {
         __lut2d_fast(img.g, img.xsize, img.ysize, des->g, des->xsize, des->ysize, lut, DWORD_COLOR_GREEN(fills));
         __Img_Busy(66);
         __lut2d_fast(img.b, img.xsize, img.ysize, des->b, des->xsize, des->ysize, lut, DWORD_COLOR_BLUE(fills));
         __Img_Busy(99);
       }
       break;
    case IInterp_fine  :   // or bicubic interpolation
	   if(des->alpha) __lut2d_fine(img.a, img.xsize, img.ysize, des->a, des->xsize, des->ysize, lut, DWORD_COLOR_ALPHA(fills));
       __lut2d_fine(img.r, img.xsize, img.ysize, des->r, des->xsize, des->ysize, lut, DWORD_COLOR_RED(fills));
       __Img_Busy(img.imagetype != ITrueColor ? 95 : 33);
       if(img.imagetype == ITrueColor) {
         __lut2d_fine(img.g, img.xsize, img.ysize, des->g, des->xsize, des->ysize, lut, DWORD_COLOR_GREEN(fills));
         __Img_Busy(66);
         __lut2d_fine(img.b, img.xsize, img.ysize, des->b, des->xsize, des->ysize, lut, DWORD_COLOR_BLUE(fills));
         __Img_Busy(99);
       }
       break;
  }
  free(lut);
  __Img_Busy(0);
  return ITRUE;
}  


