/**************************************************************************
*  Image process tool box                                                 *
*     by Yang Yudong                                                      *
*     yangyd@yahoo.com                                                    *
*                                                                         *
***************************************************************************
*  Copyright (C) 1992-1999, Yang Yudong, All rights reserved.             *
*  This file is part of Yang Yudong's image processing software package.  *
*  If you use this software, you agree to the following:                  *
*  This program package is purely experimental, and is licensed "as is".  *
*  Permission is granted to use, modify, and distribute this program      *
*  without charge for any purpose, provided this license/ disclaimer      *
*  notice appears in the copies.  No warranty or maintenance is given,    *
*  either expressed or implied.  In no event shall the author(s) be       *
*  liable to you or a third party for any special, incidental,            *
*  consequential, or other damages, arising out of the use or inability   *
*  to use the program for any purpose (or the loss of data), even if we   *
*  have been advised of such possibilities.  Any public reference or      *
*  advertisement of this source code should refer to it as Yang Yudong's  *
*  orignal.                                                               *
**************************************************************************/
// ****************************************************************
//  Image process tool box
//     by Yang Yudong
//
// File : m_qsort.c
// Description:  quick sort of double, float, int, byte
// Create Date:  1997. 5. 2
// ****************************************************************

//mathutil.h

#include <malloc.h>
#include <stdlib.h>

/*-----------------------------------------------------------------------*

Background

The Quicker Sort algorithm was first described by C.A.R.Hoare in the
Computer Journal, No. 5 (1962), pp.10..15, and in addition is frequently
described in computing literature, notably in D. Knuth's Sorting and
Searching.  The method used here includes a number of refinements:

- The median-of-three technique described by Singleton (Communications
  of the A.C.M., No 12 (1969) pp 185..187) is used, where the median
  operation is also the special case sort for 3 elements.  This slightly
  improves the average speed, especially when comparisons are slower
  than exchanges, but more importantly it prevents worst-case behavior
  on partly sorted files.  If a simplistic quicker-sort is run on a file
  which is only slightly disordered (a common need in some applications)
  then it is as slow as a bubble-sort.  The median technique prevents
  this.

  Another serious problem with the plain algorithm is that worst-case
  behavior causes very deep recursion (almost one level per table
  element !), so again it is best to use the median technique.

- The values of width and compar are copied to static storage and a help
  function with a minimum of parameters is used to reduce the recursion
  overheads.  Recursion is used both for simplicity and because there
  is no practical gain from conversion to loops: the extra housekeeping
  needed for loops needs registers for speed, but the 8086 family has not
  enough registers.  Juggling registers takes just as much time as calling
  subroutines.

*------------------------------------------------------------------------*/

/*
  Optimize pointer comparisons knowing segments are identical,
  except in HUGE model.
*/


/*-----------------------------------------------------------------------*

Name            qSortHelp - performs the quicker sort

Usage           static void qSortHelp (void *pivotP,   size_t nElem);

Description     performs the quicker sort on the nElem element array
		pointed to by pivotP.

Return value    Nothing

*------------------------------------------------------------------------*/

#define Exchange(a,b)  {exchg = *a; *a = *b; *b = exchg; }


static void  r_qSortHelp (double *pivotP, int nElem)
{
    double     *leftP, *rightP, *pivotEnd, *pivotTemp, *leftTemp;
    static  double exchg;
    int  lNum;


tailRecursion:
    if (nElem <= 2)
	{
	if (nElem == 2)
	    {
	    if (*pivotP > *(rightP=pivotP+1))
		Exchange (pivotP, rightP)
	    }
	return;
	}

    rightP = (nElem - 1) + pivotP;
    leftP  = (nElem >> 1) + pivotP;

/*  sort the pivot, left, and right elements for "median of 3" */

    if (*leftP > *rightP)
	Exchange (leftP, rightP)
    if (*leftP > *pivotP)
	Exchange (leftP, pivotP)
    else if (*pivotP > *rightP)
	Exchange (pivotP, rightP)

    if (nElem == 3)
	{
	Exchange (pivotP, leftP)
	return;
	}

/*  now for the classic Hoare algorithm */

    leftP = pivotEnd = pivotP + 1;

    do
	{
	while (*leftP <= *pivotP)
	    {
	    if (*leftP == *pivotP)
		{
		Exchange(leftP, pivotEnd)
		pivotEnd ++;
		}
	    if (leftP < rightP)
		leftP ++;
	    else
		goto qBreak;
	    }

	while (leftP < rightP)
	    {
	    if (*pivotP < *rightP)
		rightP --;
	    else
		{
		Exchange (leftP, rightP)
		if (*pivotP > *rightP)
		    {
		    leftP ++;
		    rightP --;
		    }
		break;
		}
	    }
	}   while (leftP < rightP);

qBreak:

    if (*leftP <= *pivotP)
	leftP ++;

    leftTemp = leftP - 1;

    pivotTemp = pivotP;

    while ((pivotTemp < pivotEnd) && (leftTemp >= pivotEnd))
	{
	Exchange(pivotTemp, leftTemp)
	pivotTemp ++;
	leftTemp --;
	}

    lNum = leftP - pivotEnd;
    nElem += pivotP - leftP;

    // Sort smaller partition first to reduce stack usage
    if (nElem < lNum)
	{
	r_qSortHelp (leftP, nElem);
	nElem = lNum;
	}
    else
	{
	r_qSortHelp (pivotP, lNum);
	pivotP = leftP;
	}

    goto tailRecursion;
}

//------------------------------------------------------------
static void  f_qSortHelp (float *pivotP, int nElem)
{
    float     *leftP, *rightP, *pivotEnd, *pivotTemp, *leftTemp;
    static  float exchg;
    int  lNum;


tailRecursion:
    if (nElem <= 2)
	{
	if (nElem == 2)
	    {
	    if (*pivotP > *(rightP=pivotP+1))
		Exchange (pivotP, rightP)
	    }
	return;
	}

    rightP = (nElem - 1) + pivotP;
    leftP  = (nElem >> 1) + pivotP;

/*  sort the pivot, left, and right elements for "median of 3" */

    if (*leftP > *rightP)
	Exchange (leftP, rightP)
    if (*leftP > *pivotP)
	Exchange (leftP, pivotP)
    else if (*pivotP > *rightP)
	Exchange (pivotP, rightP)

    if (nElem == 3)
	{
	Exchange (pivotP, leftP)
	return;
	}

/*  now for the classic Hoare algorithm */

    leftP = pivotEnd = pivotP + 1;

    do
	{
	while (*leftP <= *pivotP)
	    {
	    if (*leftP == *pivotP)
		{
		Exchange(leftP, pivotEnd)
		pivotEnd ++;
		}
	    if (leftP < rightP)
		leftP ++;
	    else
		goto qBreak;
	    }

	while (leftP < rightP)
	    {
	    if (*pivotP < *rightP)
		rightP --;
	    else
		{
		Exchange (leftP, rightP)
		if (*pivotP > *rightP)
		    {
		    leftP ++;
		    rightP --;
		    }
		break;
		}
	    }
	}   while (leftP < rightP);

qBreak:

    if (*leftP <= *pivotP)
	leftP ++;

    leftTemp = leftP - 1;

    pivotTemp = pivotP;

    while ((pivotTemp < pivotEnd) && (leftTemp >= pivotEnd))
	{
	Exchange(pivotTemp, leftTemp)
	pivotTemp ++;
	leftTemp --;
	}

    lNum = leftP - pivotEnd;
    nElem += pivotP - leftP;

    // Sort smaller partition first to reduce stack usage
    if (nElem < lNum)
	{
	f_qSortHelp (leftP, nElem);
	nElem = lNum;
	}
    else
	{
	f_qSortHelp (pivotP, lNum);
	pivotP = leftP;
	}

    goto tailRecursion;
}


//------------------------------------------------------------
static void  i_qSortHelp (int *pivotP, int nElem)
{
    int     *leftP, *rightP, *pivotEnd, *pivotTemp, *leftTemp;
    static  int exchg;
    int lNum;


tailRecursion:
    if (nElem <= 2)
	{
	if (nElem == 2)
	    {
	    if (*pivotP > *(rightP=pivotP+1))
		Exchange (pivotP, rightP)
	    }
	return;
	}

    rightP = (nElem - 1) + pivotP;
    leftP  = (nElem >> 1) + pivotP;

/*  sort the pivot, left, and right elements for "median of 3" */

    if (*leftP > *rightP)
	Exchange (leftP, rightP)
    if (*leftP > *pivotP)
	Exchange (leftP, pivotP)
    else if (*pivotP > *rightP)
	Exchange (pivotP, rightP)

    if (nElem == 3)
	{
	Exchange (pivotP, leftP)
	return;
	}

/*  now for the classic Hoare algorithm */

    leftP = pivotEnd = pivotP + 1;

    do
	{
	while (*leftP <= *pivotP)
	    {
	    if (*leftP == *pivotP)
		{
		Exchange(leftP, pivotEnd)
		pivotEnd ++;
		}
	    if (leftP < rightP)
		leftP ++;
	    else
		goto qBreak;
	    }

	while (leftP < rightP)
	    {
	    if (*pivotP < *rightP)
		rightP --;
	    else
		{
		Exchange (leftP, rightP)
		if (*pivotP > *rightP)
		    {
		    leftP ++;
		    rightP --;
		    }
		break;
		}
	    }
	}   while (leftP < rightP);

qBreak:

    if (*leftP <= *pivotP)
	leftP ++;

    leftTemp = leftP - 1;

    pivotTemp = pivotP;

    while ((pivotTemp < pivotEnd) && (leftTemp >= pivotEnd))
	{
	Exchange(pivotTemp, leftTemp)
	pivotTemp ++;
	leftTemp --;
	}

    lNum = leftP - pivotEnd;
    nElem += pivotP - leftP;

    // Sort smaller partition first to reduce stack usage
    if (nElem < lNum)
	{
	i_qSortHelp (leftP, nElem);
	nElem = lNum;
	}
    else
	{
	i_qSortHelp (pivotP, lNum);
	pivotP = leftP;
	}

    goto tailRecursion;
}

//------------------------------------------------------------
static void  b_qSortHelp (unsigned char *pivotP, int nElem)
{
    unsigned char     *leftP, *rightP, *pivotEnd, *pivotTemp, *leftTemp;
    static  unsigned char exchg;
    int  lNum;


tailRecursion:
    if (nElem <= 2)
	{
	if (nElem == 2)
	    {
	    if (*pivotP > *(rightP=pivotP+1))
		Exchange (pivotP, rightP)
	    }
	return;
	}

    rightP = (nElem - 1) + pivotP;
    leftP  = (nElem >> 1) + pivotP;

/*  sort the pivot, left, and right elements for "median of 3" */

    if (*leftP > *rightP)
	Exchange (leftP, rightP)
    if (*leftP > *pivotP)
	Exchange (leftP, pivotP)
    else if (*pivotP > *rightP)
	Exchange (pivotP, rightP)

    if (nElem == 3)
	{
	Exchange (pivotP, leftP)
	return;
	}

/*  now for the classic Hoare algorithm */

    leftP = pivotEnd = pivotP + 1;

    do
	{
	while (*leftP <= *pivotP)
	    {
	    if (*leftP == *pivotP)
		{
		Exchange(leftP, pivotEnd)
		pivotEnd ++;
		}
	    if (leftP < rightP)
		leftP ++;
	    else
		goto qBreak;
	    }

	while (leftP < rightP)
	    {
	    if (*pivotP < *rightP)
		rightP --;
	    else
		{
		Exchange (leftP, rightP)
		if (*pivotP > *rightP)
		    {
		    leftP ++;
		    rightP --;
		    }
		break;
		}
	    }
	}   while (leftP < rightP);

qBreak:

    if (*leftP <= *pivotP)
	leftP ++;

    leftTemp = leftP - 1;

    pivotTemp = pivotP;

    while ((pivotTemp < pivotEnd) && (leftTemp >= pivotEnd))
	{
	Exchange(pivotTemp, leftTemp)
	pivotTemp ++;
	leftTemp --;
	}

    lNum = leftP - pivotEnd;
    nElem += pivotP - leftP;

    // Sort smaller partition first to reduce stack usage
    if (nElem < lNum)
	{
	b_qSortHelp (leftP, nElem);
	nElem = lNum;
	}
    else
	{
	b_qSortHelp (pivotP, lNum);
	pivotP = leftP;
	}

    goto tailRecursion;
}

/*-----------------------------------------------------------------------*

Name            qsort - sorts using the quick sort routine

Usage           void qsort(int nelem, void *base);


Description     qsort is an implementation of the "median of three"
		variant of the quicksort algorithm. qsort sorts the entries
		in a table into order by repeatedly calling the user-defined
		comparison function pointed to by fcmp.

		base points to the base (0th element) of the table to be sorted.

		nelem is the number of entries in the table.

		width is the size of each entry in the table, in bytes.

		*fcmp, the comparison function, accepts two arguments, elem1
		and elem2, each a pointer to an entry in the table. The
		comparison function compares each of the pointed-to items
		(*elem1 and *elem2), and returns an integer based on the result
		of the comparison.

			If the items            fcmp returns

			*elem1 <  *elem2         an integer < 0
			*elem1 == *elem2         0
			*elem1 >  *elem2         an integer > 0

		In the comparison, the less than symbol (<) means that the left
		element should appear before the right element in the final,
		sorted sequence. Similarly, the greater than (>) symbol
		means that the left element should appear after the right
		element in the final, sorted sequence.

Return value    qsort does not return a value.

*------------------------------------------------------------------------*/

void  r_qsort (int nElem, double * baseP)

{
    if (nElem <= 1)
	return;
    r_qSortHelp (baseP, nElem);
}

void  f_qsort (int nElem, float * baseP)

{
    if (nElem <= 1)
	return;
    f_qSortHelp (baseP, nElem);
}

void  i_qsort (int nElem, int * baseP)

{
    if (nElem <= 1)
	return;
    i_qSortHelp (baseP, nElem);
}

void  b_qsort (int nElem, unsigned char * baseP)

{
    if (nElem <= 1)
	return;
    b_qSortHelp (baseP, nElem);
}
