// VirtualDub filter: deinterlace - smooth
// by Gunnar Thalin (guth@home.se)

#include <windows.h>
#include <stdio.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

///////////////////////////////////////////////////////////////////////////

int InitProc(FilterActivation *fa, const FilterFunctions *ff);
int StartProc(FilterActivation *fa, const FilterFunctions *ff);
int EndProc(FilterActivation *fa, const FilterFunctions *ff);
int RunProc(const FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

typedef struct MyFilterData {
	bool    bShowDeinterlacedAreaOnly;
	bool    bBlend;
	bool    bAltOrder;
	int     iThreshold;
	int     iEdgeDetect;
	bool    bLog;
	float   fLogInterlacePercent;
	int     iInterFrameLeaveThreshold;
	int     iInterFrameAverage;
	HANDLE  hLog;
	Pixel32 *pframe, *pframeprev;
	int     *piFrameDiffs;
	int     iFrameNo;
} MyFilterData;


bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	_snprintf(buf, buflen, "Config(%i, %i, %i, %i, %i, %i, \"%g\", %i, %i)",
		mfd->bShowDeinterlacedAreaOnly,
		mfd->bBlend,
		mfd->bAltOrder,
		mfd->iThreshold,
		mfd->iEdgeDetect,
		mfd->bLog,
		mfd->fLogInterlacePercent,
		mfd->iInterFrameLeaveThreshold,
		mfd->iInterFrameAverage);

	return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->bShowDeinterlacedAreaOnly = !!argv[0].asInt();
	mfd->bBlend = !!argv[1].asInt();
	mfd->bAltOrder = !!argv[2].asInt();
	mfd->iThreshold = argv[3].asInt();
	mfd->iEdgeDetect = argv[4].asInt();
	mfd->bLog = !!argv[5].asInt();
	mfd->fLogInterlacePercent = (float)atof(*(argv[6].asString()));
}

void ScriptConfig2(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->bShowDeinterlacedAreaOnly = !!argv[0].asInt();
	mfd->bBlend = !!argv[1].asInt();
	mfd->bAltOrder = !!argv[2].asInt();
	mfd->iThreshold = argv[3].asInt();
	mfd->iEdgeDetect = argv[4].asInt();
	mfd->bLog = !!argv[5].asInt();
	mfd->fLogInterlacePercent = (float)atof(*(argv[6].asString()));
	mfd->iInterFrameLeaveThreshold = argv[7].asInt();
	mfd->iInterFrameAverage = argv[8].asInt();
}

ScriptFunctionDef func_defs[]={
	{ (ScriptFunctionPtr)ScriptConfig, "Config", "0iiiiiis" },
	{ (ScriptFunctionPtr)ScriptConfig2, "Config", "0iiiiiisii" },
	{ NULL },
};

CScriptObject script_obj={
	NULL, func_defs
};


struct FilterDefinition filterDef = {

	NULL, NULL, NULL,	// next, prev, module
	"deinterlace - smooth v1.1",	// name
	"Generates high resolution, non-interlaced 50/60fps video from interlaced material that has been splitted into separate fields.",	// desc
	"Gunnar Thalin",		// maker
	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size

	InitProc,		// initProc
	NULL,			// deinitProc
	RunProc,		// runProc
	ParamProc,		// paramProc
	ConfigProc,		// configProc
	StringProc,		// stringProc
	StartProc,			// startProc
	EndProc,			// endProc

	&script_obj,	// script_obj
	FssProc,		// fssProc

};


///////////////////////////////////////////////////////////////////////////

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd;

///////////////////////////////////////////////////////////////////////////

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
    if (!(fd = ff->addFilter(fm, &filterDef, sizeof(FilterDefinition))))
        return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

    return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
    ff->removeFilter(fd);
}

///////////////////////////////////////////////////////////////////////////

int InitProc(FilterActivation *fa, const FilterFunctions *ff) {
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->bShowDeinterlacedAreaOnly = false;
	mfd->bBlend = false;
	mfd->bAltOrder = false;
	mfd->iThreshold = 24;
	mfd->iEdgeDetect = 20;
	mfd->bLog = false;
	mfd->fLogInterlacePercent = 0.0f;
	mfd->iInterFrameLeaveThreshold = 35;
	mfd->iInterFrameAverage = 80;

    return 0;
}

int StartProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->pframe = new Pixel32[fa->src.pitch * 2 * fa->src.h];
	mfd->pframeprev = new Pixel32[fa->src.pitch * 2 * fa->src.h];
	mfd->piFrameDiffs = new int[fa->src.w * 2 * fa->src.h];
	int iInitDiff = mfd->iInterFrameLeaveThreshold * mfd->iInterFrameLeaveThreshold + 1;
	for(int i = fa->src.w * 2 * fa->src.h - 1; i >= 0; i--)
		mfd->piFrameDiffs[i] = iInitDiff;
	mfd->iFrameNo = 0;

	if(mfd->bLog)
	{
		mfd->hLog = CreateFile("DeinterlaceSmooth.log", GENERIC_WRITE, FILE_SHARE_READ, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
		if(mfd->hLog != NULL)
		{
			DWORD dw;
			WriteFile(mfd->hLog, "Frame\tInterlaced area (%)\r\n", 27, &dw, NULL);
		}
	}

	return 0;
}

int EndProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	delete[] mfd->pframe;
	mfd->pframe = NULL;
	delete[] mfd->pframeprev;
	mfd->pframeprev = NULL;
	delete[] mfd->piFrameDiffs;
	mfd->piFrameDiffs = NULL;

	if(mfd->bLog && mfd->hLog != NULL)
		CloseHandle(mfd->hLog);

	return 0;
}

int RunProc(const FilterActivation *fa, const FilterFunctions *ff) {
   MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	int iThreshold = mfd->iThreshold;
	iThreshold = iThreshold * iThreshold * 4;
	int iEdgeDetect = mfd->iEdgeDetect;
	if(iEdgeDetect > 180)
		iEdgeDetect = 180;	// We don't want an integer overflow in the interlace calculation.
	iEdgeDetect = iEdgeDetect * iEdgeDetect;
	bool bShowDeinterlacedAreaOnly = mfd->bShowDeinterlacedAreaOnly;
	bool bBlend = mfd->bBlend;
	int iInterFrameLeaveThreshold = mfd->iInterFrameLeaveThreshold * mfd->iInterFrameLeaveThreshold;
	int iInterFrameAverage = mfd->iInterFrameAverage * 1024 / 100;
	if(iInterFrameAverage < 0)
		iInterFrameAverage = 0;
	if(iInterFrameAverage > 1023)
		iInterFrameAverage = 1023;
	int iInterFrameAverageRest = 1024 - iInterFrameAverage;

	int iR0, iG0, iB0, iR1, iG1, iB1, iR2, iG2, iB2, iR3, iG3, iB3;
	int iR2p, iG2p, iB2p;
	Pixel32 *pframe1, *pframe2, *pframe3, *pdst1, *pframe2p;
	int *piFrameDiffs2p;
	int iInterlaceValue0, iInterlaceValue1, iInterlaceValue2;
	int iFrameDiffValue0, iFrameDiffValue1, iFrameDiffValue2;
	int iDeinterlacedPixels = 0;
	iR1 = iG1 = iB1 = 0;	// Avoid compiler warning. The value is not used.

	Pixel32 *psrc = fa->src.data;
	Pixel32 *pdst = fa->dst.data;
	Pixel32 *pframe = mfd->pframe;
	Pixel32 *pframeprev = mfd->pframeprev;
	int *piFrameDiffs = mfd->piFrameDiffs;

	int iOddEven = mfd->bAltOrder && (fa->pfsi->lCurrentSourceFrame % 2) == 1 ||
						!mfd->bAltOrder && (fa->pfsi->lCurrentSourceFrame % 2) == 0 ? 0 : 1;

	// Combine this and previous frame (or rather fields) into a full frame.
	for(int y = 0; y < fa->dst.h; y++)
	{
		if((y % 2) == iOddEven)
		{	 // Let line from previous field remain in pframe. So do nothing unless there is no previous field.
			if(mfd->iFrameNo == 0)
			{	// There is no previous field. Interpolate from line above and below instead.
				int ysrc = y >> 1;
				Pixel32 *pprevrow = (ysrc - 1 >= 0 ? (Pixel32 *)((char *)psrc - fa->src.pitch) : psrc);
				Pixel32 *pnextrow = psrc;
				Pixel32 prevrow, nextrow;
				for(int x = 0; x < fa->src.w; x++)
				{
					prevrow = *pprevrow++;
					nextrow = *pnextrow++;
					*(pframe + x) = ((prevrow & 0xff) + (nextrow & 0xff)) / 2 +
					(((((prevrow >> 8) & 0xff) + ((nextrow >> 8) & 0xff)) / 2) << 8) + 
					(((((prevrow >> 16) & 0xff) + ((nextrow >> 16) & 0xff)) / 2) << 16);
				}
			}
		}
		else
		{
			// Take line from this field.
			memcpy(pframe, psrc, fa->src.w * sizeof(Pixel32));
			psrc = (Pixel32 *)((char *)psrc + fa->src.pitch);
		}

		pframe = (Pixel32 *)((char *)pframe + fa->src.pitch);

	}

	psrc = (Pixel32 *)fa->src.data;
	pframe = (Pixel32 *)mfd->pframe;

	// Do some area based deinterlacing on the combined frame.
	for(int x = 0; x < fa->dst.w; x++)
	{
		pframe3 = pframe + x;
		iR3 = (*pframe3 >> 16) & 0xff;
		iG3 = (*pframe3 >> 8) & 0xff;
		iB3 = *pframe3 & 0xff;
		pframe2 = (Pixel32 *)((char *)pframe3 + fa->dst.pitch);
		iR2 = (*pframe2 >> 16) & 0xff;
		iG2 = (*pframe2 >> 8) & 0xff;
		iB2 = *pframe2 & 0xff;
		
		pframe2p = pframeprev + x;
		piFrameDiffs2p = piFrameDiffs + x;

		pdst1 = pdst + x;
		iInterlaceValue1 = iInterlaceValue2 = 0;
		iFrameDiffValue1 = iFrameDiffValue2 = 0;

		// In this loop, output is delayed one loop (or row) from input. Current output row has variables with digit 1 and current input row has digit 2.
		for(y = 0; y <= fa->dst.h; y++)
		{
			pframe1 = pframe2;
			pframe2 = pframe3;
			pframe3 = (Pixel32 *)((char *)pframe3 + fa->dst.pitch);
			iR0 = iR1;
			iG0 = iG1;
			iB0 = iB1;
			iR1 = iR2;
			iG1 = iG2;
			iB1 = iB2;
			iR2 = iR3;
			iG2 = iG3;
			iB2 = iB3;
			if(y < fa->dst.h - 1)
			{
				iR3 = (*pframe3 >> 16) & 0xff;
				iG3 = (*pframe3 >> 8) & 0xff;
				iB3 = *pframe3 & 0xff;

				iR2p = (*pframe2p >> 16) & 0xff;
				iG2p = (*pframe2p >> 8) & 0xff;
				iB2p = *pframe2p & 0xff;
			}
			else
			{
				iR3 = iR1;
				iG3 = iG1;
				iB3 = iB1;
			}
			pframe2p = (Pixel32 *)((char *)pframe2p + fa->dst.pitch);

			iInterlaceValue0 = iInterlaceValue1;
			iInterlaceValue1 = iInterlaceValue2;
			iFrameDiffValue0 = iFrameDiffValue1;
			iFrameDiffValue1 = iFrameDiffValue2;
			if(y < fa->dst.h)
			{
				// Calculate the interlace value by checking if the pixel color on previous row differs much
				// from this row and next row differs much (with same sign) from this row.
				// Detect edges by checking so that pixel color on previous row doesn't differ too much from next row.
				// If it does, it's probably just an ordinary edge.
				// 3, 6 and 1 are approximate values for converting R, G, B to intensity.
				iInterlaceValue2 = (3 * ((iR1 - iR2) * (iR3 - iR2) - ((iEdgeDetect * (iR1 - iR3) * (iR1 - iR3)) >> 12)) +
									6 * ((iG1 - iG2) * (iG3 - iG2) - ((iEdgeDetect * (iG1 - iG3) * (iG1 - iG3)) >> 12)) +
									(iB1 - iB2) * (iB3 - iB2) - ((iEdgeDetect * (iB1 - iB3) * (iB1 - iB3)) >> 12));
				iFrameDiffValue2 = (iInterFrameAverage * *piFrameDiffs2p +
											iInterFrameAverageRest * (mfd->iFrameNo > 1 ?
												3 * (iR2 - iR2p) * (iR2 - iR2p) + 6 * (iG2 - iG2p) * (iG2 - iG2p) + (iB2 - iB2p) * (iB2 - iB2p) :
												*piFrameDiffs2p)) >> 10;
				*piFrameDiffs2p = iFrameDiffValue2;
				piFrameDiffs2p += fa->dst.w;
			}
			else
			{
				iInterlaceValue2 = 0;
				iFrameDiffValue2 = 0;
			}

			if(y > 0)
			{	// Get mean interlace value of 3 rows (i.e. 5 pixels are examined).
				// Middle row has twice the weight.
				if(iInterlaceValue0 + 2 * iInterlaceValue1 + iInterlaceValue2 > iThreshold &&
					iFrameDiffValue0 + 2 * iFrameDiffValue1 + iFrameDiffValue2 >= iInterFrameLeaveThreshold)
				{
					if(bShowDeinterlacedAreaOnly)
					{
						*pdst1 = 0x00903030 + (bBlend ? (((iR0 + 2 * iR1 + iR2) >> 4) << 16) + (((iG0 + 2 * iG1 + iG2) >> 4) << 8) + ((iB0 + 2 * iB1 + iB2) >> 4)
																:
																  (y % 2 == iOddEven ? (iR1 >> 2 << 16) + (iG1 >> 2 << 8) + (iB1 >> 2) : (((iR0 + iR2) >> 3) << 16) + (((iG0 + iG2) >> 3) << 8) + ((iB0 + iB2) >> 3)));
					}
					else
					{
						// Blend: Get mean value of previous and next row (weight 0.25) and this row (weight 0.5).
						// Interpolate: Odd lines: Copy from source. Even lines: Get mean value of previous and next row
						*pdst1 = bBlend ? (((iR0 + 2 * iR1 + iR2) >> 2) << 16) + (((iG0 + 2 * iG1 + iG2) >> 2) << 8) + ((iB0 + 2 * iB1 + iB2) >> 2)
										  :
										  (y % 2 == iOddEven ? *pframe1 : (((iR0 + iR2) >> 1) << 16) + (((iG0 + iG2) >> 1) << 8) + ((iB0 + iB2) >> 1));

					}
					iDeinterlacedPixels++;
				}
				else
				{
					if(bShowDeinterlacedAreaOnly)
					{
						*pdst1 = (iR1 >> 2 << 16) + (iG1 >> 2 << 8) + (iB1 >> 2);
						if(iInterlaceValue0 + 2 * iInterlaceValue1 + iInterlaceValue2 > iThreshold)
							*pdst1 += 0x00309030;
						else if(iFrameDiffValue0 + 2 * iFrameDiffValue1 + iFrameDiffValue2 >= iInterFrameLeaveThreshold)
							*pdst1 += 0x00303090;
						else
							*pdst1 += 0x00303030;
					}
					else
						*pdst1 = *pframe1;
				}
				pdst1 = (Pixel32 *)((char *)pdst1 + fa->dst.pitch);
			}
		}
	}

	memcpy(mfd->pframeprev, pframe, 2 * fa->src.h * fa->src.pitch);
	mfd->iFrameNo++;

	if(mfd->bLog)
	{
		float fDeinterlacedPercent = 100.0f * iDeinterlacedPixels / fa->src.w / fa->src.h;
		if(fDeinterlacedPercent > mfd->fLogInterlacePercent && mfd->hLog != NULL)
		{
			char szLog[25];
			sprintf(szLog, "%li\t%5.1f\r\n", fa->pfsi->lCurrentSourceFrame, fDeinterlacedPercent);
			DWORD dw;
			WriteFile(mfd->hLog, szLog, strlen(szLog), &dw, NULL);
		}
	}

	return 0;
}

long ParamProc(FilterActivation *fa, const FilterFunctions *ff) {
    fa->dst.h *= 2;
    return FILTERPARAM_SWAP_BUFFERS;
}


BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
    MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

    switch(msg) {
        case WM_INITDIALOG:
            SetWindowLong(hdlg, DWL_USER, lParam);
            mfd = (MyFilterData *)lParam;

            CheckDlgButton(hdlg, IDC_SHOW_DEINTERLACED_AREAS_ONLY, mfd->bShowDeinterlacedAreaOnly ? BST_CHECKED:BST_UNCHECKED);
            CheckDlgButton(hdlg, IDC_BLEND, mfd->bBlend ? BST_CHECKED:BST_UNCHECKED);
            CheckDlgButton(hdlg, IDC_ALT_ORDER, mfd->bAltOrder ? BST_CHECKED:BST_UNCHECKED);
            char str[10000];
            sprintf(str, "%i", mfd->iThreshold);
            SetDlgItemText(hdlg, IDC_THRESHOLD, str);
            sprintf(str, "%i", mfd->iEdgeDetect);
            SetDlgItemText(hdlg, IDC_EDGE_DETECT, str);
            CheckDlgButton(hdlg, IDC_LOG, mfd->bLog ? BST_CHECKED:BST_UNCHECKED);
            sprintf(str, "%g", mfd->fLogInterlacePercent);
            SetDlgItemText(hdlg, IDC_INTERLACE_PERCENT, str);
            sprintf(str, "%i", mfd->iInterFrameLeaveThreshold);
            SetDlgItemText(hdlg, IDC_INTER_FRAME_LEAVE, str);
            sprintf(str, "%i", mfd->iInterFrameAverage);
            SetDlgItemText(hdlg, IDC_INTER_FRAME_AVERAGE, str);

				strcpy(str, "Colorcode:\r\nAreas are colored differently to help you find suitable parameter values.\r\nRed - Deinterlaced areas.\r\nBlue - Non-static areas that would be deinterlaced if interlace patterns were found.\r\nGreen - Static areas that do contain interlace patterns but are still left untouched.\r\nGrey - Static areas without interlace patterns\r\n\r\n");
				strcat(str, "Blend:\r\nBlends this and previous field in interlaced areas. (You should probably avoid using this as it just blurs the video.)\r\n\r\n");
				strcat(str, "Interpolate:\r\nInterpolate the field to full frame size in interlaced areas.\r\n\r\n");
				strcat(str, "Alternate field order:\r\nPut even or odd fields on first line of output video. The correct setting depends on your video clip. If static video jumps up and down slightly when you step through it, change this setting.\r\n\r\n");
				strcat(str, "Interlace threshold:\r\nControls the detection of interlace patterns. Lower values deinterlaces more.\r\n\r\n");
				strcat(str, "Edge detect:\r\nIt's difficult to distinguish between interlace lines and real edges (which should not be deinterlaced). This value controls this decision. Higher value leaves more edges intact.\r\n\r\n");
				strcat(str, "Static threshold:\r\nThe filter tries to detect static areas to avoid deinterlacing fine details which could result in flickering. This value controls how much a pixel can variate and still be called static. Use as low value as possible to avoid leaving interlace patterns. Values above 50 (or so) are not recommended. Good quality video can use lower values. If you don't have any text or logos that may flicker I suggest using very low values. 0 makes it work like version 1.0.\r\n\r\n");
				strcat(str, "Static averaging:\r\nControls how long history to look at when determining if areas are static or not. Low values (= short history) find static fast (but maybe incorrectly, leaving interlace patterns). High values means static details may flicker for a longer time before \"converging\". Also, it can be slower to react when areas go from static to non-static. Valid range is 0-100. A good rule is to set <static averaging> = 2 * <static threshold>, or higher (but don't get too close to 100).\r\n\r\n");
				strcat(str, "Log to file:\r\nThe frames that have an interlaced area exceeding the given number of percent can be logged to a file which will be placed in the current directory. It's called DeinterlaceSmooth.log and contains frame numbers and the interlaced area (in %) for each frame.");
            SetDlgItemText(hdlg, ED_HELP, str);

            return TRUE;

        case WM_COMMAND:
            switch(LOWORD(wParam)) {
            case IDOK:
                mfd->bShowDeinterlacedAreaOnly = !!IsDlgButtonChecked(hdlg, IDC_SHOW_DEINTERLACED_AREAS_ONLY);
                mfd->bBlend = !!IsDlgButtonChecked(hdlg, IDC_BLEND);
                mfd->bAltOrder = !!IsDlgButtonChecked(hdlg, IDC_ALT_ORDER);
                char str[20];
                GetDlgItemText(hdlg, IDC_THRESHOLD, str, 20);
                mfd->iThreshold = atoi(str);
                GetDlgItemText(hdlg, IDC_EDGE_DETECT, str, 20);
                mfd->iEdgeDetect = atoi(str);
                mfd->bLog = !!IsDlgButtonChecked(hdlg, IDC_LOG);
                GetDlgItemText(hdlg, IDC_INTERLACE_PERCENT, str, 20);
                mfd->fLogInterlacePercent = (float)atof(str);
                GetDlgItemText(hdlg, IDC_INTER_FRAME_LEAVE, str, 20);
                mfd->iInterFrameLeaveThreshold = atoi(str);
                GetDlgItemText(hdlg, IDC_INTER_FRAME_AVERAGE, str, 20);
                mfd->iInterFrameAverage = atoi(str);
                EndDialog(hdlg, 0);
                return TRUE;
            case IDCANCEL:
                EndDialog(hdlg, 1);
                return FALSE;
            }
            break;
    }

    return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd) {
    return DialogBoxParam(fa->filter->module->hInstModule,
            MAKEINTRESOURCE(IDD_FILTER_DEINTERLACE_AREA_BASED), hwnd,
            (DLGPROC)ConfigDlgProc, (LPARAM)fa->filter_data);
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str) {
	sprintf(str, " (thresh %i, edge %i, %s%s)",
		((MyFilterData *)(fa->filter_data))->iThreshold,
		((MyFilterData *)(fa->filter_data))->iEdgeDetect,
		((MyFilterData *)(fa->filter_data))->bBlend ? "blend" : "interp",
		((MyFilterData *)(fa->filter_data))->bShowDeinterlacedAreaOnly ? ", show areas" : "");
}
