#include <windows.h>
#include <stdio.h>
#include <commctrl.h>

#include "resource.h"
#include "filter.h"

///////////////////////////////////////////////////////////////////////////

int dnrRunProc(const FilterActivation *fa, const FilterFunctions *ff);
int dnrStartProc(FilterActivation *fa, const FilterFunctions *ff);
int dnrEndProc(FilterActivation *fa, const FilterFunctions *ff);
int dnrInitProc(FilterActivation *fa, const FilterFunctions *ff);
int dnrConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void dnrStringProc(const FilterActivation *fa, const FilterFunctions *ff, char *buf); 

///////////////////////////////////////////////////////////////////////////


typedef struct MyFilterData {
    unsigned char   *old_data;
    long            size, level;
} MyFilterData;

struct FilterDefinition filterDef_dnr = {

    NULL, NULL, NULL,						// next, prev, module
    "dynamic noise reduction (original)",	// name
    "Reduces changes across frames by eliminating dynamic noise without blurring.",
											// desc
	"Steven Don",							// maker
    NULL,									// private_data
    sizeof(MyFilterData),					// inst_data_size

    dnrInitProc,							// initProc
    NULL,									// deinitProc
    dnrRunProc,								// runProc
    NULL,									// paramProc
    dnrConfigProc,							// configProc
    dnrStringProc,							// stringProc
    dnrStartProc,							// startProc
    dnrEndProc,								// endProc

    NULL,									// script_obj
    NULL,									// fssProc

};

static unsigned char divTable [766];

///////////////////////////////////////////////////////////////////////////

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_dnr;

///////////////////////////////////////////////////////////////////////////

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
    if (!(fd_dnr = ff->addFilter(fm, &filterDef_dnr, sizeof(FilterDefinition))))
        return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

    return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
    ff->removeFilter(fd_dnr);
}

///////////////////////////////////////////////////////////////////////////

int dnrStartProc(FilterActivation *fa, const FilterFunctions *ff) {
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

    mfd->size = fa->src.Size ();
    
    //Make buffer and clear it
    if (!(mfd->old_data = new unsigned char [mfd->size]))
        return 1;
    
    memset (mfd->old_data, 0, mfd->size);
    return 0;
}

int dnrRunProc(const FilterActivation *fa, const FilterFunctions *ff) {
    long diff, ofs;
    PixDim w, h;
    Pixel32 *src, *dst;
    Pixel32 a, b, c;
    unsigned char ap, bp;

    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

    /*
	//Old, really unoptimised version
	if (mfd->level) {
        ofs = mfd->size;
        src = (unsigned char *)fa->src.data;
        dst = (unsigned char *)mfd->old_data;
        
        do {
            diff = abs (*src - *dst);
            if (diff < (mfd->level)) {
                if (diff > (mfd->level >> 1)) *dst = divTable [*src + *src + *dst];
            } else {
                *dst = *src;
            }
            src++; dst++;
        } while (--ofs);
    }*/

    if (mfd->level) {
        ofs = mfd->size;
        src = (Pixel32 *)fa->src.data;
        dst = (Pixel32 *)mfd->old_data;

        h = fa->src.h;
        do {
            w = fa->src.w;

            do {
                a = *src;
                b = *dst;

                ap = (unsigned char)(a >> 16);
                bp = (unsigned char)(b >> 16);
                diff = abs (ap - bp);
                if (diff < (mfd->level)) {
                    if (diff > (mfd->level >> 1)) ap = divTable [ap + ap + bp];
                    else ap = bp;
                }
                c = ap << 8;
                ap = (unsigned char)(a >> 8);
                bp = (unsigned char)(b >> 8);
                diff = abs (ap - bp);
                if (diff < (mfd->level)) {
                    if (diff > (mfd->level >> 1)) ap = divTable [ap + ap + bp];
                    else ap = bp;
                }
                c |= ap;
                c <<= 8;
                ap = (unsigned char)a;
                bp = (unsigned char)b;
                diff = abs (ap - bp);
                if (diff < (mfd->level)) {
                    if (diff > (mfd->level >> 1)) ap = divTable [ap + ap + bp];
                    else ap = bp;
                }
                c |= ap;

                *dst = c;

                ++src, ++dst;
            } while(--w);

            src = (Pixel32 *)((char *)src + fa->src.modulo);
            dst = (Pixel32 *)((char *)dst + fa->dst.modulo);
        } while(--h);
        memcpy (fa->dst.data, mfd->old_data, mfd->size);
    }

    return 0;
}

int dnrEndProc(FilterActivation *fa, const FilterFunctions *ff) {
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;
    delete[] mfd->old_data;
    mfd->old_data = NULL;
    return 0;
}

int dnrInitProc(FilterActivation *fa, const FilterFunctions *ff) {
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;
    mfd->level = 12;

    for (int index = 0; index < 766; index++) divTable [index] = index / 3;

    return 0;
}

BOOL CALLBACK dnrConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
    MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

    switch(msg) {
        case WM_INITDIALOG:
            {
                SetWindowLong(hdlg, DWL_USER, lParam);
                mfd = (MyFilterData *)lParam;
                SendDlgItemMessage (hdlg, IDC_SLIDER_LEVEL, TBM_SETTICFREQ, 4, 0);
                SendDlgItemMessage (hdlg, IDC_SLIDER_LEVEL, TBM_SETRANGE, false, MAKELONG (0, 31));
                SendDlgItemMessage (hdlg, IDC_SLIDER_LEVEL, TBM_SETPOS, true, mfd->level);
                char leveltext [4];
                sprintf (leveltext, "%d", mfd->level);
                SetDlgItemText (hdlg, IDC_LEVEL, leveltext);
            }

            return TRUE;

        case WM_NOTIFY:
            {
                long level = SendDlgItemMessage (hdlg, IDC_SLIDER_LEVEL, TBM_GETPOS, 0, 0);
                char leveltext [4];
                sprintf (leveltext, "%d", level);
                SetDlgItemText (hdlg, IDC_LEVEL, leveltext);
            }

        case WM_COMMAND:
            switch(LOWORD(wParam)) {
                case IDOK:
                    mfd->level = SendDlgItemMessage (hdlg, IDC_SLIDER_LEVEL, TBM_GETPOS, 0, 0);
                    EndDialog(hdlg, 0);
                    return TRUE;
                case IDCANCEL:
                    EndDialog(hdlg, 1);
                    return FALSE;
             }
        break;
    }

    return FALSE;
}

int dnrConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd) {
    return DialogBoxParam(fa->filter->module->hInstModule,
        MAKEINTRESOURCE(IDD_DIALOG_CONFIG), hwnd,
        dnrConfigDlgProc, (LPARAM)fa->filter_data);
}

void dnrStringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str) {
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;
    sprintf(str, " (%d)", mfd->level);
}