/*
    Lum Band Filter for VirtualDub -- Seperately displace the hue of areas
      of differing luminescence. Copyright (C) 2001 Tom Ford

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    The author can be contacted at:
      Tom Ford
      flend@compsoc.net

    This work includes code from:
      Colorize Filter for VirtualDub -- Change a video to a monochrome video
        with selectable base color. Copyright (C) 1999-2000 Donald A. Graft

      By 
        Donald Graft
        neuron2@home.com.
*/

#include <windows.h>
#include <commctrl.h>
#include <stdio.h>
#include <crtdbg.h>
#include <math.h>
#include <stdlib.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

#define NUM_BANDS_MAX 50

int RunProc(const FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int InitProc(FilterActivation *fa, const FilterFunctions *ff);
int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);
void RGBtoHSL(Pixel32 src, double *H, double *S, double *L);
void HLStoRGB(double H, double L, double S, int *r, int *g, int *b);

typedef struct FilterDataStruct {
  IFilterPreview *ifp;
  int hue_displacement;
  int num_bands;
  int saturation_adjust;
  int extremity_clip;
} FilterData;


bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
  FilterData *filter_data = (FilterData *)fa->filter_data;

  _snprintf(buf, buflen, "Config(%d, %d, %d, %d)", filter_data->num_bands, filter_data->hue_displacement, filter_data->saturation_adjust, filter_data->extremity_clip);
  return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {

  FilterActivation *fa = (FilterActivation *)lpVoid;
  FilterData *filter_data = (FilterData *)fa->filter_data;

  filter_data->num_bands = argv[0].asInt();
  filter_data->hue_displacement = argv[1].asInt();
  filter_data->saturation_adjust = argv[2].asInt();
  filter_data->extremity_clip = argv[3].asInt();
}

ScriptFunctionDef func_defs[] = {
  { (ScriptFunctionPtr)ScriptConfig, "Config", "0iiii" },
  { NULL },
};

CScriptObject script_obj = {
  NULL, func_defs
};

struct FilterDefinition filterdef = {

  NULL, NULL, NULL,     // next, prev, module
  "luminescence band",  // name
  "Displaces the hue of a pixel proportional to its luminescence.",        // desc
  "Tom Ford",           // maker
  NULL,                 // private_data
  sizeof(FilterData),   // inst_data_size

  InitProc,             // initProc
  NULL,                 // deinitProc
  RunProc,              // runProc
  NULL,                 // paramProc
  ConfigProc,           // configProc
  StringProc,           // stringProc
  NULL,                 // startProc
  NULL,                 // endProc

  &script_obj,          // script_obj
  FssProc,              // fssProc
};

int RunProc(const FilterActivation *fa, const FilterFunctions *ff) {

  FilterData *filter_data = (FilterData *)fa->filter_data;
  FilterStateInfo *info = fa->pfsi;
  const PixDim width = fa->src.w;
  const PixDim height = fa->src.h;
  Pixel32 *src, *dst;
  int x, y;
  int r, g, b;
  double H, S, L;
  int hue;
  double output_hue, output_sat;
  double band_width, sat_adj, ext_low, ext_high;
  int i;

  src = fa->src.data;
  dst = fa->dst.data;

  band_width = 1.0 / filter_data->num_bands;
  sat_adj = filter_data->saturation_adjust / 100.0;
  ext_low = filter_data->extremity_clip / 100.0;
  ext_high = 1.0 - ext_low;

  for (y = 0; y < height; y++) {
    for (x = 0; x < width; x++) {

      //Generate HLS of input pixel
      RGBtoHSL(src[x], &H, &S, &L);

      //Bias luminescence strongly away from blacks and slightly
      //away from strong whites
      if(L < 0.5)
        L = ext_low + (L * ext_high);
      else
        L = L * ext_high;

      //Perform luminescence banding
      double band_start = 0;

      for(i=0; i < filter_data->num_bands; i++) {
        if(L > band_start && L <= band_start + band_width) {
          //Displace hue band_num x hue_displacement
          hue = (int)(H * 239);
          hue += filter_data->hue_displacement * (i+1);
          hue %= 240;
          break;
        }
        band_start += band_width;
      }

      //Write output pixel
      output_sat = S + ((1.0 - S) * sat_adj);
      output_hue = hue / 240.0;
      HLStoRGB(output_hue, L, output_sat, &r, &g, &b);
      dst[x] = (r << 16) | (g << 8) | (b);                   
    }

    src = (Pixel *)((char *)src + fa->src.pitch);
    dst = (Pixel *)((char *)dst + fa->dst.pitch);
  }
  return 0;
}

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_lumband;

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
  if (!(fd_lumband = ff->addFilter(fm, &filterdef, sizeof(FilterDefinition))))
    return 1;

  vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
  vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

  return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
  ff->removeFilter(fd_lumband);
}

int InitProc(FilterActivation *fa, const FilterFunctions *ff)
{
  FilterData *filter_data = (FilterData *)fa->filter_data;

  filter_data->num_bands = 10;
  filter_data->hue_displacement = 40;

  return 0;
}

BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
  FilterData *filter_data = (FilterData *)GetWindowLong(hdlg, DWL_USER);

  switch(msg) {
    HWND hWnd;

  case WM_PAINT:
    return 0;

  case WM_INITDIALOG:
    SetWindowLong(hdlg, DWL_USER, lParam);
    filter_data = (FilterData *)lParam;

    hWnd = GetDlgItem(hdlg, IDC_BANDS);
    SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(1, NUM_BANDS_MAX));
    SendMessage(hWnd, TBM_SETTICFREQ, 10 , 0);
    SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, filter_data->num_bands);
    SetDlgItemInt(hdlg, IDC_EBANDS, filter_data->num_bands, FALSE);
    hWnd = GetDlgItem(hdlg, IDC_DISP);
    SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 239));
    SendMessage(hWnd, TBM_SETTICFREQ, 10 , 0);
    SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, filter_data->hue_displacement);
    SetDlgItemInt(hdlg, IDC_EDISP, filter_data->hue_displacement, FALSE);
    hWnd = GetDlgItem(hdlg, IDC_SAT);
    SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 100));
    SendMessage(hWnd, TBM_SETTICFREQ, 10 , 0);
    SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, filter_data->saturation_adjust);
    SetDlgItemInt(hdlg, IDC_ESAT, filter_data->saturation_adjust, FALSE);
    hWnd = GetDlgItem(hdlg, IDC_LUM);
    SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 100));
    SendMessage(hWnd, TBM_SETTICFREQ, 10 , 0);
    SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, filter_data->saturation_adjust);
    SetDlgItemInt(hdlg, IDC_ELUM, filter_data->extremity_clip, FALSE);

    filter_data->ifp->InitButton(GetDlgItem(hdlg, IDPREVIEW));
    return TRUE;

  case WM_HSCROLL:
    {
      int cycle, disp, sat, lum;
      HWND hwnd = GetDlgItem(hdlg, IDC_BANDS);
      cycle = SendMessage(hwnd, TBM_GETPOS, 0, 0);
      SetDlgItemInt(hdlg, IDC_EBANDS, cycle, FALSE);
      hwnd = GetDlgItem(hdlg, IDC_DISP);
      disp = SendMessage(hwnd, TBM_GETPOS, 0, 0);
      SetDlgItemInt(hdlg, IDC_EDISP, disp, FALSE);
      hwnd = GetDlgItem(hdlg, IDC_SAT);
      sat = SendMessage(hwnd, TBM_GETPOS, 0, 0);
      SetDlgItemInt(hdlg, IDC_ESAT, sat, FALSE);
      hwnd = GetDlgItem(hdlg, IDC_LUM);
      lum = SendMessage(hwnd, TBM_GETPOS, 0, 0);
      SetDlgItemInt(hdlg, IDC_ELUM, lum, FALSE);

      if (cycle != filter_data->num_bands || disp != filter_data->hue_displacement || sat != filter_data->saturation_adjust || lum != filter_data->extremity_clip) {
        filter_data->num_bands = cycle;
        filter_data->hue_displacement = disp;
        filter_data->saturation_adjust = sat;
        filter_data->extremity_clip = lum;
      }

      filter_data->ifp->RedoFrame();
      break;
    }

  case WM_COMMAND:
    switch(LOWORD(wParam)) {
    case IDPREVIEW:
      filter_data->ifp->Toggle(hdlg);
      break;
    case IDOK:
      EndDialog(hdlg, 0);
      return TRUE;
    case IDHELP:
      {
        MessageBox(NULL, "This filter should be pretty straightforward.", "LuminescenceBand 'help'.", MB_OK | MB_ICONINFORMATION);
        /*char prog[256];
          char path[256];
          LPTSTR ptr;

          GetModuleFileName(NULL, prog, 255);
          GetFullPathName(prog, 255, path, &ptr);
          *ptr = 0;
          strcat(path, "plugins\\Colorize.html");
          ShellExecute(hdlg, "open", path, NULL, NULL, SW_SHOWNORMAL);*/
        return TRUE;
      }
    case IDCANCEL:
      EndDialog(hdlg, 1);
      return TRUE;

    case IDC_BANDSPLUS:
   
      if (filter_data->num_bands < NUM_BANDS_MAX) {

        filter_data->num_bands++;
        SetDlgItemInt(hdlg, IDC_EBANDS, filter_data->num_bands, FALSE);
        SendMessage(GetDlgItem(hdlg, IDC_BANDS), TBM_SETPOS, (WPARAM)TRUE, filter_data->num_bands);

        filter_data->ifp->RedoFrame();
      }
      break;
    case IDC_BANDSMINUS:

      if (filter_data->num_bands > 1) {

        filter_data->num_bands--;
        SetDlgItemInt(hdlg, IDC_EBANDS, filter_data->num_bands, FALSE);
        SendMessage(GetDlgItem(hdlg, IDC_BANDS), TBM_SETPOS, (WPARAM)TRUE, filter_data->num_bands);

        filter_data->ifp->RedoFrame();
      }
      break;

    case IDC_DISPPLUS:
      if (filter_data->hue_displacement < 239) {

        filter_data->hue_displacement++;
        SetDlgItemInt(hdlg, IDC_EDISP, filter_data->hue_displacement, FALSE);
        SendMessage(GetDlgItem(hdlg, IDC_DISP), TBM_SETPOS, (WPARAM)TRUE, filter_data->hue_displacement);

        filter_data->ifp->RedoFrame();
      }
      break;
    case IDC_DISPMINUS:
      if (filter_data->hue_displacement > 0) {

        filter_data->hue_displacement--;
        SetDlgItemInt(hdlg, IDC_EDISP, filter_data->hue_displacement, FALSE);
        SendMessage(GetDlgItem(hdlg, IDC_DISP), TBM_SETPOS, (WPARAM)TRUE, filter_data->hue_displacement);

        filter_data->ifp->RedoFrame();
      }
      break;

    case IDC_SATPLUS:
      if (filter_data->saturation_adjust < 100) {

        filter_data->saturation_adjust++;
        SetDlgItemInt(hdlg, IDC_ESAT, filter_data->saturation_adjust, FALSE);
        SendMessage(GetDlgItem(hdlg, IDC_SAT), TBM_SETPOS, (WPARAM)TRUE, filter_data->saturation_adjust);

        filter_data->ifp->RedoFrame();
      }
      break;
    case IDC_SATMINUS:
      if (filter_data->saturation_adjust > 0) {

        filter_data->saturation_adjust--;
        SetDlgItemInt(hdlg, IDC_ESAT, filter_data->saturation_adjust, FALSE);
        SendMessage(GetDlgItem(hdlg, IDC_SAT), TBM_SETPOS, (WPARAM)TRUE, filter_data->saturation_adjust);

        filter_data->ifp->RedoFrame();
      }
      break;

    case IDC_LUMPLUS:
      if (filter_data->extremity_clip < 100) {

        filter_data->extremity_clip++;
        SetDlgItemInt(hdlg, IDC_ELUM, filter_data->extremity_clip, FALSE);
        SendMessage(GetDlgItem(hdlg, IDC_LUM), TBM_SETPOS, (WPARAM)TRUE, filter_data->extremity_clip);

        filter_data->ifp->RedoFrame();
      }
      break;
    case IDC_LUMMINUS:
      if (filter_data->extremity_clip > 0) {

        filter_data->extremity_clip--;
        SetDlgItemInt(hdlg, IDC_ELUM, filter_data->extremity_clip, FALSE);
        SendMessage(GetDlgItem(hdlg, IDC_LUM), TBM_SETPOS, (WPARAM)TRUE, filter_data->extremity_clip);

        filter_data->ifp->RedoFrame();
      }
      break;

    }
    break;
  }

  return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd)
{
  FilterData *filter_data = (FilterData *)fa->filter_data;
  FilterData filter_data_old = *filter_data;
  int ret;

  filter_data->ifp = fa->ifp;

  ret = DialogBoxParam(fa->filter->module->hInstModule, MAKEINTRESOURCE(IDD_FILTER),
                       hwnd, ConfigDlgProc, (LPARAM) filter_data);
  if (ret)
    {
      *filter_data = filter_data_old;
      ret = TRUE;
    }
  else
    {
      ret = FALSE;
    }
  return(ret);
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str)
{
  FilterData *filter_data = (FilterData *)fa->filter_data;

  sprintf(str, " (cframes %d, disp %d, sat %d, clip %d)", filter_data->num_bands, filter_data->hue_displacement, filter_data->saturation_adjust, filter_data->extremity_clip);
}

void RGBtoHSL(Pixel32 src, double *H, double *S, double *L)
{
  double delta;
  double r = (double)((src & 0xff0000) >> 16)/255;
  double g = (double)((src & 0xff00) >> 8)/255;
  double b = (double)(src & 0xff)/255;
  double cmax = max(r,max(g,b));
  double cmin = min(r,min(g,b));

  *L=(cmax+cmin)/2.0;
  if(cmax==cmin) {
    *S = 0;
    *H = 0; // it's really undefined
  } else {
    if(*L < 0.5) 
      *S = (cmax-cmin)/(cmax+cmin);
    else
      *S = (cmax-cmin)/(2.0-cmax-cmin);
    delta = cmax - cmin;
    if(r==cmax)
      *H = (g-b)/delta;
    else if(g==cmax)
      *H = 2.0 +(b-r)/delta;
    else
      *H=4.0+(r-g)/delta;
    *H /= 6.0;
    if(*H < 0.0)
      *H += 1;
  }
}

int HuetoRGB(double m1, double m2, double h)
{
  if( h < 0 ) h += 1.0;
  if( h > 1 ) h -= 1.0;
  if( 6.0*h < 1 )
    return ((int) (255 * (m1+(m2-m1)*h*6.0)));
  if( 2.0*h < 1 )
    return ((int) (255 * m2));
  if( 3.0*h < 2.0 )
    return ((int) (255 * (m1+(m2-m1)*((2.0/3.0)-h)*6.0)));
  return ((int) (255 * m1));
}

void HLStoRGB(double H, double L, double S, int *r, int *g, int *b)
{
  double m1, m2;

  if(S==0) {
    *r = *g = *b = (int) (255 * L);
  } else {
    if(L <=0.5)
      m2 = L*(1.0+S);
    else
      m2 = L+S-L*S;
    m1 = 2.0*L-m2;
    *r = HuetoRGB(m1,m2,H+1.0/3.0);
    *g = HuetoRGB(m1,m2,H);
    *b = HuetoRGB(m1,m2,H-1.0/3.0);
  }
}


