/*//////////////////////////////////////////////////////////////////////////

	flaXen's De-Ghosting Tool for VirtualDub
	(c) Copyright 2000 Dan Flower (flaXen)

	An attempt to reduce or eliminate "ghosts" from analog video sources

*//////////////////////////////////////////////////////////////////////////

#include <windows.h>
#include <stdio.h>
#include <math.h>

#include "resource.h"
#include "filter.h"
#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#define	PI		3.14159265358979

typedef struct MyFilterData {
	Pixel32	*TmpBuff;
	long	*ytab, *itab, *qtab;
	long	*rtab, *gtab, *btab;
	long	*cytf, *Sig, mmx;
	int		DarkBase, DarkBias, LightBase, LightBias;
	int		SigStren, SigMax, CounterStren, CounterSens;
	int		SigFalloff, Reverse;
	int		YCCin, YCCout, UseYIQ;
} MyFilterData;

///////////////////////////////////////////////////////////////////////////

int		fxGhostRunProc(const FilterActivation *fa, const FilterFunctions *ff);
void	fxGhost_DeGhost(Pixel32 *, Pixel32 *, long, long, long, MyFilterData *);
void	fxGhost_RGB2YCC(Pixel32 *, Pixel32 *, long, long, long, MyFilterData *);
void	fxGhost_YCC2RGB(Pixel32 *, Pixel32 *, long, long, long, MyFilterData *);
int		fxGhostStartProc(FilterActivation *fa, const FilterFunctions *ff);
int		fxGhostEndProc(FilterActivation *fa, const FilterFunctions *ff); 
int		fxGhostInitProc(FilterActivation *fa, const FilterFunctions *ff);
int		fxGhostConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void	fxGhostStringProc(const FilterActivation *fa, const FilterFunctions *ff, char *buf);
void	fxGhostScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool	fxGhostFssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

ScriptFunctionDef fxGhost_func_defs[]={
    { (ScriptFunctionPtr)fxGhostScriptConfig, "Config", "0iiiiiiiiiiiii" },
    { NULL },
};

CScriptObject fxGhostobj={
    NULL, fxGhost_func_defs
};

struct FilterDefinition filterDef_fxGhost = {
	NULL, NULL, NULL,		// next, prev, module

	"De-Ghosting",		// name
	"flaXen's De-Ghosting Tool",	// desc
	"flaXen",				// maker

	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size
	fxGhostInitProc,			// initProc
	NULL,					// deinitProc
	fxGhostRunProc,			// runProc
	NULL,					// paramProc
	fxGhostConfigProc,		// configProc
	fxGhostStringProc,		// stringProc
	fxGhostStartProc,			// startProc
	fxGhostEndProc,			// endProc
	&fxGhostobj,				// script_obj
	fxGhostFssProc,			// fssProc
};

///////////////////////////////////////////////////////////////////////////

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_fxGhost;

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
	if (!(fd_fxGhost = ff->addFilter(fm, &filterDef_fxGhost, sizeof(FilterDefinition))))
		return 1;

	vdfd_ver    = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;
	return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
	ff->removeFilter(fd_fxGhost);
}

///////////////////////////////////////////////////////////////////////////

int fxGhostRunProc(const FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData	*mfd = (MyFilterData *)fa->filter_data;
	Pixel32	*isrc, *idst, *itmp;
	const long	swid = fa->src.w, shei = fa->src.h;
	const long	spit = fa->src.pitch >> 2;
	const long	ssize = fa->src.pitch * fa->src.h;

	isrc = (Pixel32 *)fa->src.data;
	idst = (Pixel32 *)fa->dst.data;
	itmp = (Pixel32 *)mfd->TmpBuff;

	if (mfd->YCCin == BST_CHECKED) {
		// YUV/YIQ colorspace input
		if (mfd->YCCout == BST_UNCHECKED) {
			// RGB colorspace output
			memcpy(itmp, isrc, ssize);
			fxGhost_DeGhost(isrc, itmp, swid, shei, spit, mfd);
			fxGhost_YCC2RGB(itmp, idst, swid, shei, spit, mfd);
		} else {
			// YIQ/YUV colorspace output
			memcpy(idst, isrc, ssize);
			fxGhost_DeGhost(isrc, idst, swid, shei, spit, mfd);
		}
	} else {
		// RGB colorspace input
		if (mfd->YCCout == BST_UNCHECKED) {
			// RGB colorspace output
			fxGhost_RGB2YCC(isrc, itmp, swid, shei, spit, mfd);
			fxGhost_DeGhost(itmp, itmp, swid, shei, spit, mfd);
			fxGhost_YCC2RGB(itmp, idst, swid, shei, spit, mfd);
		} else {
			// YIQ colorspace output
			fxGhost_RGB2YCC(isrc, itmp, swid, shei, spit, mfd);
			fxGhost_DeGhost(itmp, itmp, swid, shei, spit, mfd);
			memcpy(idst, itmp, ssize);
		}
	}

	return 0;
}

///////////////////////////////////////////////////////////////////////////

void	fxGhost_DeGhost(Pixel32 *isrc, Pixel32 *idst, long swid, long shei, long spit, MyFilterData *mfd) {
	Pixel32	*src, *dst;
	long	x, y, l, c, si, sc;
	float	ss, sf, cs, sig;

	ss = (float)((float)mfd->SigStren / 100.0);
	cs = (float)((float)mfd->CounterStren / 100.0);
	sf = (float)((float)mfd->SigFalloff / 10.0);
	for (y = 0; y < shei; y++) {
		src = isrc + y * spit;
		dst = idst + y * spit;
		sig = 0; sc = 0;
		/*if (y < 4) {
			for (x = 0; x < swid; x++) {
				if (x < 256)
					*dst++ = (mfd->Sig[x] << 16) | 0x008080L;
			}
		} else {	*/
		for (x = 0; x < swid; x++) {
			c = *src++;
			l = (c >> 16) & 255;
			si = mfd->Sig[l];
			if (si) sc++;
			sig += (float)si * ss;
			if (sig > mfd->SigMax) sig = (float)mfd->SigMax;
			if (sc > mfd->CounterSens && !si) {
				//if (sc > mfd->CounterSens && sig > mfd->CounterSens && si < 1) {
				//l = 255;
				l += (long)(sig * cs);
				if (l > 255) l = 255;
			}
			//l = (long)(sig * cs * 4);
			sig -= sf;
			if (sig < 0) sig = 0;

			//l = (((c >> 16) & 255) >> 2) + (sig >> 16);
			//*dst++ = c & 0xFF000000L | (l << 16) | 0x00008080L;
			*dst++ = c & 0xFF00FFFFL | (l << 16);
		//}
		}
	}
}

///////////////////////////////////////////////////////////////////////////

void	fxGhost_RGB2YCC(Pixel32 *isrc, Pixel32 *idst, long swid, long shei, long spit, MyFilterData *mfd) {
	Pixel32	*src, *dst;
	long	y, c;
	long	*ytab = (long *)mfd->ytab;
	long	*itab = (long *)mfd->itab;
	long	*qtab = (long *)mfd->qtab;

	if (mfd->UseYIQ == BST_CHECKED) {
		// Convert RGB -> YIQ
		for (y = 0; y < shei; y++) {
			src = isrc + y * spit;
			dst = idst + y * spit;
			__asm {
				push	esi
				push	edi
				mov		ecx, [swid]

cty10:			mov		esi, [src]
				add		[src], 4
				mov		edx, [esi]
				mov		ebx, edx
				shr		ebx, 6
				and		ebx, 0x0003FCL
				mov		eax, edx
				shr		eax, 14
				and		eax, 0x0003FCL
				shl		edx, 2
				and		edx, 0x0003FCL
				add		ebx, 1024
				add		edx, 2048

				mov		esi, [ytab]
				mov		edi, [esi + eax]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cty20
				test	edi, 0xFF000000L
				jz		cty21
				mov		edi, 0x00FF0000L
				jmp		cty21
cty20:			xor		edi, edi
cty21:			mov		[c], edi

				mov		esi, [itab]
				mov		edi, [esi + eax]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cty30
				test	edi, 0xFF000000L
				jz		cty31
				mov		edi, 0x00FF0000L
				jmp		cty31
cty30:			xor		edi, edi
cty31:			shr		edi, 8
				or		[c], edi

				mov		esi, [qtab]
				mov		edi, [esi + eax]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cty40
				test	edi, 0xFF000000L
				jz		cty41
				mov		edi, 0x00FF0000L
cty40:			xor		edi, edi
cty41:			shr		edi, 16
				or		[c], edi

				mov		edi, [dst]
				add		[dst], 4
				mov		eax, [c]
				mov		[edi], eax

				dec		ecx
				jnz		cty10
				pop		edi
				pop		esi
			}
		}
	} else {
		// Convert RGB -> YUV
		for (y = 0; y < shei; y++) {
			src = isrc + y * spit;
			dst = idst + y * spit;
			__asm {
				push	esi
				push	edi
				mov		ecx, [swid]

ctu10:			mov		esi, [src]
				add		[src], 4
				mov		edx, [esi]
				mov		ebx, edx
				shr		ebx, 6
				and		ebx, 0x0003FCL
				mov		eax, edx
				shr		eax, 14
				and		eax, 0x0003FCL
				shl		edx, 2
				and		edx, 0x0003FCL
				add		ebx, 1024
				add		edx, 2048

				mov		esi, [ytab]
				mov		edi, [esi + eax]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		ctu20
				test	edi, 0xFF000000L
				jz		ctu21
				mov		edi, 0x00FF0000L
				jmp		ctu21
ctu20:			xor		edi, edi
ctu21:			mov		[c], edi

				mov		esi, [itab]
				mov		edi, [esi + eax]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		ctu30
				test	edi, 0xFF000000L
				jz		ctu31
				mov		edi, 0x00FF0000L
				jmp		ctu31
ctu30:			xor		edi, edi
ctu31:			shr		edi, 8
				or		[c], edi

				mov		esi, [qtab]
				mov		edi, [esi + eax]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		ctu40
				test	edi, 0xFF000000L
				jz		ctu41
				mov		edi, 0x00FF0000L
				jmp		ctu41
ctu40:			xor		edi, edi
ctu41:			shr		edi, 16
				or		[c], edi

				mov		edi, [dst]
				add		[dst], 4
				mov		eax, [c]
				mov		[edi], eax

				dec		ecx
				jnz		ctu10
				pop		edi
				pop		esi
			}
		}
	}
}

///////////////////////////////////////////////////////////////////////////

void	fxGhost_YCC2RGB(Pixel32 *isrc, Pixel32 *idst, long swid, long shei, long spit, MyFilterData *mfd) {
	Pixel32	*src, *dst;
	long	y, c;
	long	*rtab = (long *)mfd->rtab;
	long	*gtab = (long *)mfd->gtab;
	long	*btab = (long *)mfd->btab;
	long	*cytf = (long *)mfd->cytf;

	if (mfd->UseYIQ == BST_CHECKED) {
		// Convert YIQ -> RGB
		for (y = 0; y < shei; y++) {
			src = isrc + y * spit;
			dst = idst + y * spit;
			__asm {
				push	esi
				push	edi
				mov		ecx, [swid]

cfy10:			mov		esi, [src]
				add		[src], 4
				mov		edx, [esi]
				mov		ebx, edx
				shr		ebx, 6
				and		ebx, 0x0003FCL
				mov		eax, edx
				and		eax, 0x00FF0000L
				shl		edx, 2
				and		edx, 0x0003FCL
				add		edx, 1024

				mov		edi, eax
				mov		esi, [rtab]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cfy20
				test	edi, 0xFF000000L
				jz		cfy21
				mov		edi, 0x00FF0000L
				jmp		cfy21
cfy20:			xor		edi, edi
cfy21:			mov		[c], edi

				mov		edi, eax
				mov		esi, [gtab]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cfy30
				test	edi, 0xFF000000L
				jz		cfy31
				mov		edi, 0x00FF0000L
				jmp		cfy31
cfy30:			xor		edi, edi
cfy31:			shr		edi, 8
				or		[c], edi

				mov		edi, eax
				mov		esi, [btab]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cfy40
				test	edi, 0xFF000000L
				jz		cfy41
				mov		edi, 0x00FF0000L
				jmp		cfy41
cfy40:			xor		edi, edi
cfy41:			shr		edi, 16
				or		[c], edi

				mov		edi, [dst]
				add		[dst], 4
				mov		eax, [c]
				mov		[edi], eax

				dec		ecx
				jnz		cfy10
				pop		edi
				pop		esi
			}
		}
	} else {
		// Convert YUV -> RGB
		for (y = 0; y < shei; y++) {
			src = isrc + y * spit;
			dst = idst + y * spit;
			__asm {
				push	esi
				push	edi
				mov		ecx, [swid]

cfu10:			mov		esi, [src]
				add		[src], 4
				mov		edx, [esi]
				mov		ebx, edx
				shr		ebx, 6
				and		ebx, 0x0003FCL
				mov		eax, edx
				shr		eax, 14
				and		eax, 0x0003FCL
				shl		edx, 2
				and		edx, 0x0003FCL
				add		edx, 1024

				mov		esi, [cytf]
				mov		eax, [esi + eax]

				mov		edi, eax
				mov		esi, [rtab]
				add		edi, [esi + ebx]
				and		edi, 0xFFFF0000L
				js		cfu20
				test	edi, 0xFF000000L
				jz		cfu21
				mov		edi, 0x00FF0000L
				jmp		cfu21
cfu20:			xor		edi, edi
cfu21:			mov		[c], edi

				mov		edi, eax
				mov		esi, [gtab]
				add		edi, [esi + ebx]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cfu30
				test	edi, 0xFF000000L
				jz		cfu31
				mov		edi, 0x00FF0000L
				jmp		cfu31
cfu30:			xor		edi, edi
cfu31:			shr		edi, 8
				or		[c], edi

				mov		edi, eax
				mov		esi, [btab]
				add		edi, [esi + edx]
				and		edi, 0xFFFF0000L
				js		cfu40
				test	edi, 0xFF000000L
				jz		cfu41
				mov		edi, 0x00FF0000L
				jmp		cfu41
cfu40:			xor		edi, edi
cfu41:			shr		edi, 16
				or		[c], edi

				mov		edi, [dst]
				add		[dst], 4
				mov		eax, [c]
				mov		[edi], eax

				dec		ecx
				jnz		cfu10
				pop		edi
				pop		esi
			}
		}
	}
}

///////////////////////////////////////////////////////////////////////////

int fxGhostStartProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	long	d;

	// Detect MMX and enable if possible
	mfd->mmx = ff->isMMXEnabled();
	if (fa->src.w & 1) mfd->mmx = 0;

	// Temporary frame buffer
	d = fa->src.pitch * fa->src.h;
	if (!(mfd->TmpBuff = new Pixel32[d])) return 1;

	// Biased signal strength look-up table
	if (!(mfd->Sig = new long[256])) return 1;

	// Y transform for YUV -> RGB conversion
	if (!(mfd->cytf = new long[256])) return 1;

	// RGB -> Ycc conversion tables
	if (!(mfd->ytab = new long[768])) return 1;
	if (!(mfd->itab = new long[768])) return 1;
	if (!(mfd->qtab = new long[768])) return 1;

	// Ycc -> RGB conversion tables
	if (!(mfd->rtab = new long[512])) return 1;
	if (!(mfd->gtab = new long[512])) return 1;
	if (!(mfd->btab = new long[512])) return 1;

	// Calculate data for RGB -> Ycc conversion tables
	for (d = 0; d < 256; d++) {
		if (mfd->UseYIQ == BST_CHECKED) {
			// RGB -> YIQ transformation matrix
			mfd->ytab[d      ] = (long)(+0.299 * d * 65536 + 0.5);
			mfd->ytab[d + 256] = (long)(+0.587 * d * 65536 + 0.5);
			mfd->ytab[d + 512] = (long)(+0.114 * d * 65536 + 0.5);
			mfd->itab[d      ] = (long)(+0.596 * d * 65536 + 0.5) + 8388608;
			mfd->itab[d + 256] = (long)(-0.275 * d * 65536 + 0.5);
			mfd->itab[d + 512] = (long)(-0.321 * d * 65536 + 0.5);
			mfd->qtab[d      ] = (long)(+0.212 * d * 65536 + 0.5) + 8388608;
			mfd->qtab[d + 256] = (long)(-0.523 * d * 65536 + 0.5);
			mfd->qtab[d + 512] = (long)(+0.311 * d * 65536 + 0.5);
			// YIQ -> RGB transformation matrix
			mfd->rtab[d      ] = (long)(+0.956 * (d-128) * 65536 + 0.5);
			mfd->rtab[d + 256] = (long)(+0.621 * (d-128) * 65536 + 0.5);
			mfd->gtab[d      ] = (long)(-0.272 * (d-128) * 65536 + 0.5);
			mfd->gtab[d + 256] = (long)(-0.647 * (d-128) * 65536 + 0.5);
			mfd->btab[d      ] = (long)(-1.105 * (d-128) * 65536 + 0.5);
			mfd->btab[d + 256] = (long)(+1.702 * (d-128) * 65536 + 0.5);
		} else {
			// RGB -> YUV transformation matrix
			mfd->ytab[d      ] = (long)(+0.257 * d * 65536 + 0.5) + 1048576;
			mfd->ytab[d + 256] = (long)(+0.504 * d * 65536 + 0.5);
			mfd->ytab[d + 512] = (long)(+0.098 * d * 65536 + 0.5);
			mfd->itab[d      ] = (long)(+0.439 * d * 65536 + 0.5) + 8388608;
			mfd->itab[d + 256] = (long)(-0.368 * d * 65536 + 0.5);
			mfd->itab[d + 512] = (long)(-0.071 * d * 65536 + 0.5);
			mfd->qtab[d      ] = (long)(-0.148 * d * 65536 + 0.5) + 8388608;
			mfd->qtab[d + 256] = (long)(-0.291 * d * 65536 + 0.5);
			mfd->qtab[d + 512] = (long)(+0.439 * d * 65536 + 0.5);
			// YUV -> RGB transformation matrix
			mfd->rtab[d      ] = (long)(+1.596 * (d-128) * 65536 + 0.5);
			mfd->rtab[d + 256] = 0;
			mfd->gtab[d      ] = (long)(-0.813 * (d-128) * 65536 + 0.5);
			mfd->gtab[d + 256] = (long)(-0.391 * (d-128) * 65536 + 0.5);
			mfd->btab[d      ] = 0;
			mfd->btab[d + 256] = (long)(+2.018 * (d-128) * 65536 + 0.5);
			mfd->cytf[d] = (long)(((d - 16) * 1.164) * 65536.0 + 0.5);	// Y transform
		}
	}

	for (d = 0; d < 256; d++)
		if (d < mfd->DarkBase)
			mfd->Sig[d] = 255;
		else
			mfd->Sig[d] = 0;

	for (d = 0; d < mfd->DarkBias; d++) {
		mfd->Sig[d + mfd->DarkBase] = (long)(255.0 - (cos((float)((mfd->DarkBias - d) * (PI / mfd->DarkBias / 2))) * 255.0));
	}

	return 0;
}

///////////////////////////////////////////////////////////////////////////

int fxGhostEndProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	// Delete temporary frame buffer
	delete[] mfd->TmpBuff; mfd->TmpBuff = NULL;

	// Delete signal table
	delete[] mfd->Sig; mfd->Sig = NULL;

	// Delete the Y transformation table
	delete[] mfd->cytf; mfd->cytf = NULL;

	// Delete RGB -> YIQ/YUV conversion tables
	delete[] mfd->ytab; mfd->ytab = NULL;
	delete[] mfd->itab; mfd->itab = NULL;
	delete[] mfd->qtab; mfd->qtab = NULL;
	
	delete[] mfd->rtab; mfd->rtab = NULL;
	delete[] mfd->gtab; mfd->gtab = NULL;
	delete[] mfd->btab; mfd->btab = NULL;

	return 0;
}

///////////////////////////////////////////////////////////////////////////

int fxGhostInitProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->DarkBase = 10;
	mfd->DarkBias = 90;
	mfd->LightBase = 10;
	mfd->LightBias = 90;
	mfd->SigStren = 30;
	mfd->SigMax = 80;
	mfd->SigFalloff = 30;
	mfd->CounterSens = 1;
	mfd->CounterStren = 15;
	mfd->Reverse = BST_UNCHECKED;
	mfd->YCCin	= BST_UNCHECKED;
	mfd->YCCout	= BST_UNCHECKED;
	mfd->UseYIQ = BST_UNCHECKED;

	return 0;
}

///////////////////////////////////////////////////////////////////////////

BOOL CALLBACK fxGhostConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
	MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

	switch(msg) {
	case WM_INITDIALOG:
		SetWindowLong(hdlg, DWL_USER, lParam);
		mfd = (MyFilterData *)lParam;
		SetDlgItemInt(hdlg, IDC_DG_DBASE, mfd->DarkBase, FALSE);
		SetDlgItemInt(hdlg, IDC_DG_DBIAS, mfd->DarkBias, FALSE);
		SetDlgItemInt(hdlg, IDC_DG_LBASE, mfd->LightBase, FALSE);
		SetDlgItemInt(hdlg, IDC_DG_LBIAS, mfd->LightBias, FALSE);
		SetDlgItemInt(hdlg, IDC_DG_SSTREN, mfd->SigStren, FALSE);
		SetDlgItemInt(hdlg, IDC_DG_SMAX, mfd->SigMax, FALSE);
		SetDlgItemInt(hdlg, IDC_DG_SFALLOFF, mfd->SigFalloff, FALSE);
		SetDlgItemInt(hdlg, IDC_DG_CSENS, mfd->CounterSens, FALSE);
		SetDlgItemInt(hdlg, IDC_DG_CSTREN, mfd->CounterStren, FALSE);
		CheckDlgButton(hdlg, IDC_DG_REVERSE, mfd->Reverse ? BST_CHECKED : BST_UNCHECKED);
		CheckDlgButton(hdlg, IDC_DG_YCCIN, mfd->YCCin ? BST_CHECKED : BST_UNCHECKED);
		CheckDlgButton(hdlg, IDC_DG_YCCOUT, mfd->YCCout ? BST_CHECKED : BST_UNCHECKED);
		CheckDlgButton(hdlg, IDC_DG_YIQ, mfd->UseYIQ ? BST_CHECKED : BST_UNCHECKED);
		return TRUE;

	case WM_COMMAND:
		switch(LOWORD(wParam)) {
		case IDOK:
			mfd->DarkBase = GetDlgItemInt(hdlg, IDC_DG_DBASE, &mfd->DarkBase, FALSE);
			mfd->DarkBias = GetDlgItemInt(hdlg, IDC_DG_DBIAS, &mfd->DarkBias, FALSE);
			mfd->LightBase = GetDlgItemInt(hdlg, IDC_DG_LBASE, &mfd->LightBase, FALSE);
			mfd->LightBias = GetDlgItemInt(hdlg, IDC_DG_LBIAS, &mfd->LightBias, FALSE);
			mfd->SigStren = GetDlgItemInt(hdlg, IDC_DG_SSTREN, &mfd->SigStren, FALSE);
			mfd->SigMax = GetDlgItemInt(hdlg, IDC_DG_SMAX, &mfd->SigMax, FALSE);
			mfd->SigFalloff = GetDlgItemInt(hdlg, IDC_DG_SFALLOFF, &mfd->SigFalloff, FALSE);
			mfd->CounterSens = GetDlgItemInt(hdlg, IDC_DG_CSENS, &mfd->CounterSens, FALSE);
			mfd->CounterStren = GetDlgItemInt(hdlg, IDC_DG_CSTREN, &mfd->CounterStren, FALSE);
			mfd->Reverse = !!IsDlgButtonChecked(hdlg, IDC_DG_REVERSE);
			mfd->YCCin = !!IsDlgButtonChecked(hdlg, IDC_DG_YCCIN);
			mfd->YCCout = !!IsDlgButtonChecked(hdlg, IDC_DG_YCCOUT);
			mfd->UseYIQ = !!IsDlgButtonChecked(hdlg, IDC_DG_YIQ);

			if (mfd->DarkBase > 255) mfd->DarkBase = 255;
			if (mfd->DarkBias > 255) mfd->DarkBias = 255;
			if (mfd->LightBase > 255) mfd->LightBase = 255;
			if (mfd->LightBias > 255) mfd->LightBias = 255;
			if (mfd->SigStren > 200) mfd->SigStren = 200;
			if (mfd->SigMax > 255) mfd->SigMax = 255;
			if (mfd->SigFalloff > 255) mfd->SigMax = 255;
			if (mfd->CounterSens > 255) mfd->CounterSens = 255;
			if (mfd->CounterStren > 200) mfd->CounterStren = 200;

			EndDialog(hdlg, 0);

			return TRUE;
		case IDCANCEL:
			EndDialog(hdlg, 1);
			return FALSE;
		}
		break;
	}

	return FALSE;
}

///////////////////////////////////////////////////////////////////////////

int fxGhostConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd) {
	return DialogBoxParam(fa->filter->module->hInstModule,
	       MAKEINTRESOURCE(IDD_FXGHOST_CONFIG), hwnd,
		   fxGhostConfigDlgProc, (LPARAM)fa->filter_data);
}

///////////////////////////////////////////////////////////////////////////

void fxGhostStringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	char	fstr[10];

	if (mfd->UseYIQ == BST_CHECKED)
		sprintf(fstr, "YIQ");
	else
		sprintf(fstr, "YUV");

	sprintf(str, " %d, %d, %d, %d - %d, %d, %d, %d, %d (%s -> %s) %s", mfd->DarkBase, mfd->DarkBias,
		mfd->LightBase, mfd->LightBias, mfd->SigStren, mfd->SigMax, mfd->SigFalloff, mfd->CounterSens,
		mfd->CounterStren, mfd->YCCin == BST_CHECKED ? fstr : "RGB", mfd->YCCout == BST_CHECKED ? fstr : "RGB",
		mfd->Reverse == BST_CHECKED ? "Reversed" : "");
}

///////////////////////////////////////////////////////////////////////////

void fxGhostScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
    FilterActivation *fa = (FilterActivation *)lpVoid;
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->DarkBase = argv[0].asInt();
	mfd->DarkBias = argv[1].asInt();
	mfd->LightBase = argv[2].asInt();
	mfd->LightBias = argv[3].asInt();
	mfd->SigStren = argv[4].asInt();
	mfd->SigMax = argv[5].asInt();
	mfd->SigFalloff = argv[6].asInt();
	mfd->CounterSens = argv[7].asInt();
	mfd->CounterStren = argv[8].asInt();
	mfd->Reverse = argv[9].asInt() == 1 ? BST_CHECKED : BST_UNCHECKED;
	mfd->YCCin = argv[10].asInt() == 1 ? BST_CHECKED : BST_UNCHECKED;
	mfd->YCCout = argv[11].asInt() == 1 ? BST_CHECKED : BST_UNCHECKED;
	mfd->UseYIQ = argv[12].asInt() == 1 ? BST_CHECKED : BST_UNCHECKED;
}

///////////////////////////////////////////////////////////////////////////

bool fxGhostFssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

    _snprintf(buf, buflen, "Config(%d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d)", mfd->DarkBase, mfd->DarkBias,
		mfd->LightBase, mfd->LightBias, mfd->SigStren, mfd->SigMax, mfd->SigFalloff, mfd->CounterSens, mfd->CounterStren,
		mfd->Reverse == BST_CHECKED ? 1 : 0, mfd->YCCin == BST_CHECKED ? 1 : 0, mfd->YCCout == BST_CHECKED ? 1 : 0,
		mfd->UseYIQ == BST_CHECKED ? 1 : 0);

    return true;
}