/*
    Xsharpen Filter for VirtualDub -- sharpen by mapping pixels
    to the closest of window max or min. Copyright (C) 1999-2000 Donald A. Graft

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

	The author can be contacted at:
	Donald Graft
	neuron2@home.com.
*/

#include <windows.h>
#include <commctrl.h>
#include <stdio.h>
#include <crtdbg.h>
#include <math.h>
#include <stdlib.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

///////////////////////////////////////////////////////////////////////////

int RunProc(const FilterActivation *fa, const FilterFunctions *ff);
int StartProc(FilterActivation *fa, const FilterFunctions *ff);
int EndProc(FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int InitProc(FilterActivation *fa, const FilterFunctions *ff);
int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

typedef struct MyFilterData {
	IFilterPreview		*ifp;
	int					strength, strengthInv, threshold;
} MyFilterData;

bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	_snprintf(buf, buflen, "Config(%d, %d)",
		mfd->threshold,
		mfd->strength);

	return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->threshold		= argv[0].asInt();
	mfd->strength		= argv[1].asInt();
	mfd->strengthInv	= 255 - mfd->strength;
}

ScriptFunctionDef func_defs[]={
	{ (ScriptFunctionPtr)ScriptConfig, "Config", "0ii" },
	{ NULL },
};

CScriptObject script_obj={
	NULL, func_defs
};

struct FilterDefinition filterDef_tutorial = {

	NULL, NULL, NULL,		// next, prev, module
	"xsharpen (1.0 beta 2)",	// name
	"Sharpen by mapping pixels to the closest of window max or min.",
							// desc
	"Donald Graft", 		// maker
	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size

	InitProc,				// initProc
	NULL,					// deinitProc
	RunProc,				// runProc
	NULL,					// paramProc
	ConfigProc, 			// configProc
	StringProc, 			// stringProc
	NULL,				// startProc
	NULL,				// endProc

	&script_obj,			// script_obj
	FssProc,				// fssProc
};

///////////////////////////////////////////////////////////////////////////

int RunProc(const FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	const PixDim	width = fa->src.w;
	const PixDim	height = fa->src.h;
	Pixel32 *src, *dst;
	int x, y;
	int r, g, b, R, G, B;
    Pixel32 p, min, max;
	int luma, lumac, lumamax, lumamin, mindiff, maxdiff;

	/* First copy through the four border lines. */
	src	= fa->src.data;
	dst	= fa->dst.data;
	for (x = 0; x < width; x++)
	{
		dst[x] = src[x];
	}
	src	= (Pixel *)((char *)fa->src.data + (height - 1) * fa->src.pitch);
	dst	= (Pixel *)((char *)fa->dst.data + (height - 1) * fa->dst.pitch);
	for (x = 0; x < width; x++)
	{
		dst[x] = src[x];
	}
	src	= fa->src.data;
	dst	= fa->dst.data;
	for (y = 0; y < height; y++)
	{
		dst[0] = src[0];
		dst[width-1] = src[width-1];
		src	= (Pixel *)((char *)src + fa->src.pitch);
		dst	= (Pixel *)((char *)dst + fa->dst.pitch);
	}

	/* Now calculate and store the pixel luminances for the remaining pixels. */
	src	= fa->src.data;
	for (y = 0; y < height; y++)
	{
		for (x = 0; x < width; x++)
		{
			r = (src[x] >> 16) & 0xff;
			g = (src[x] >> 8) & 0xff;
			b = src[x] & 0xff;
			luma = (55 * r + 182 * g + 19 * b) >> 8;
			src[x] &= 0x00ffffff;
			src[x] |= (luma << 24);
		}
		src	= (Pixel *)((char *)src + fa->src.pitch);
	}

	/* Finally run the 3x3 rank-order sharpening kernel over the pixels. */
	src	= (Pixel *)((char *)fa->src.data + fa->src.pitch);
	dst	= (Pixel *)((char *)fa->dst.data + fa->dst.pitch);
	for (y = 1; y < height - 1; y++)
	{
		for (x = 1; x < width - 1; x++)
		{
			/* Find the brightest and dimmest pixels in the 3x3 window
			   surrounding the current pixel. */
			lumamax = -1;
			lumamin = 1000;

			p = ((Pixel32 *)((char *)src - fa->src.pitch))[x-1];
			luma = p >> 24;
			if (luma > lumamax)
			{
				lumamax = luma;
				max = p;
			}
			if (luma < lumamin)
			{
				lumamin = luma;
				min = p;
			}
			
			p = ((Pixel32 *)((char *)src - fa->src.pitch))[x];
			luma = p >> 24;
			if (luma > lumamax)
			{
				lumamax = luma;
				max = p;
			}
			if (luma < lumamin)
			{
				lumamin = luma;
				min = p;
			}
			
			p = ((Pixel32 *)((char *)src - fa->src.pitch))[x+1];
			luma = p >> 24;
			if (luma > lumamax)
			{
				lumamax = luma;
				max = p;
			}
			if (luma < lumamin)
			{
				lumamin = luma;
				min = p;
			}
			
			p = src[x-1];
			luma = p >> 24;
			if (luma > lumamax)
			{
				lumamax = luma;
				max = p;
			}
			if (luma < lumamin)
			{
				lumamin = luma;
				min = p;
			}
			
			p = src[x];
			lumac = luma = p >> 24;
			if (luma > lumamax)
			{
				lumamax = luma;
				max = p;
			}
			if (luma < lumamin)
			{
				lumamin = luma;
				min = p;
			}
			
			p = src[x+1];
			luma = p >> 24;
			if (luma > lumamax)
			{
				lumamax = luma;
				max = p;
			}
			if (luma < lumamin)
			{
				lumamin = luma;
				min = p;
			}
			
			p = ((Pixel32 *)((char *)src + fa->src.pitch))[x-1];
			luma = p >> 24;
			if (luma > lumamax)
			{
				lumamax = luma;
				max = p;
			}
			if (luma < lumamin)
			{
				lumamin = luma;
				min = p;
			}
			
			p = ((Pixel32 *)((char *)src + fa->src.pitch))[x];
			luma = p >> 24;
			if (luma > lumamax)
			{
				lumamax = luma;
				max = p;
			}
			if (luma < lumamin)
			{
				lumamin = luma;
				min = p;
			}
			
			p = ((Pixel32 *)((char *)src + fa->src.pitch))[x+1];
			luma = p >> 24;
			if (luma > lumamax)
			{
				lumamax = luma;
				max = p;
			}
			if (luma < lumamin)
			{
				lumamin = luma;
				min = p;
			}

			/* Determine whether the current pixel is closer to the
			   brightest or the dimmest pixel. Then compare the current
			   pixel to that closest pixel. If the difference is within
			   threshold, map the current pixel to the closest pixel;
			   otherwise pass it through. */
			p = -1;
			if (mfd->strength != 0)
			{
				mindiff = lumac - lumamin;
				maxdiff = lumamax - lumac;
				if (mindiff > maxdiff)
				{
					if (maxdiff < mfd->threshold)
					{
						p = max;
					}
				}
				else
				{
					if (mindiff < mfd->threshold)
					{
						p = min;
					}
				}
			}
			if (p == -1)
			{
				dst[x] = src[x];
			}
			else
			{
				R = (src[x] >> 16) & 0xff;
				G = (src[x] >> 8) & 0xff;
				B = src[x] & 0xff;
				r = (p >> 16) & 0xff;
				g = (p >> 8) & 0xff;
				b = p & 0xff;
				r = (mfd->strength * r + mfd->strengthInv * R) / 255;
				g = (mfd->strength * g + mfd->strengthInv * G) / 255;
				b = (mfd->strength * b + mfd->strengthInv * B) / 255;
				dst[x] = (r << 16) | (g << 8) | b;
			}
		}
		src	= (Pixel *)((char *)src + fa->src.pitch);
		dst	= (Pixel *)((char *)dst + fa->dst.pitch);
	}
	return 0;
}

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_tutorial;

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
	if (!(fd_tutorial = ff->addFilter(fm, &filterDef_tutorial, sizeof(FilterDefinition))))
		return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

	return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
	ff->removeFilter(fd_tutorial);
}

int InitProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->strength = 255;
	mfd->strengthInv = 255 - mfd->strength;
	mfd->threshold = 255;
	return 0;
}

BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
	MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

	switch(msg) {
		HWND hWnd;

		case WM_INITDIALOG:
			SetWindowLong(hdlg, DWL_USER, lParam);
			mfd = (MyFilterData *)lParam;

			hWnd = GetDlgItem(hdlg, IDC_THRESHOLD);
			SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 255));
			SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->threshold);
			SetDlgItemInt(hdlg, IDC_ETHRESHOLD, mfd->threshold, FALSE);
			hWnd = GetDlgItem(hdlg, IDC_STRENGTH);
			SendMessage(hWnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 255));
			SendMessage(hWnd, TBM_SETPOS, (WPARAM)TRUE, mfd->strength);
			SetDlgItemInt(hdlg, IDC_ESTRENGTH, mfd->strength, FALSE);

			mfd->ifp->InitButton(GetDlgItem(hdlg, IDPREVIEW));
			return TRUE;

		case WM_HSCROLL:
			{
				int threshold, strength;
				HWND hwnd = GetDlgItem(hdlg, IDC_THRESHOLD);
				threshold = SendMessage(hwnd, TBM_GETPOS, 0, 0);
				SetDlgItemInt(hdlg, IDC_ETHRESHOLD, mfd->threshold, FALSE);
				hwnd = GetDlgItem(hdlg, IDC_STRENGTH);
				strength = SendMessage(hwnd, TBM_GETPOS, 0, 0);
				SetDlgItemInt(hdlg, IDC_ESTRENGTH, mfd->strength, FALSE);
				if (threshold != mfd->threshold || strength != mfd->strength)
				{
					mfd->threshold = threshold;
					mfd->strength = strength;
					mfd->strengthInv = 255 - mfd->strength;
					mfd->ifp->RedoFrame();
				}
				break;
			}

		case WM_COMMAND:
			switch(LOWORD(wParam)) {
			case IDPREVIEW:
				mfd->ifp->Toggle(hdlg);
				break;
			case IDOK:
				EndDialog(hdlg, 0);
				return TRUE;
 			case IDHELP:
				{
				char prog[256];
				char path[256];
				LPTSTR ptr;

				GetModuleFileName(NULL, prog, 255);
				GetFullPathName(prog, 255, path, &ptr);
				*ptr = 0;
				strcat(path, "plugins\\Xsharpen.html");
				ShellExecute(hdlg, "open", path, NULL, NULL, SW_SHOWNORMAL);
				return TRUE;
				}
			case IDCANCEL:
				EndDialog(hdlg, 1);
				return TRUE;
			case IDC_THRESHOLDPLUS:
				if (mfd->threshold < 255)
				{
					mfd->threshold += 1;
					SetDlgItemInt(hdlg, IDC_ETHRESHOLD, mfd->threshold, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_THRESHOLD), TBM_SETPOS, (WPARAM)TRUE, mfd->threshold);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_THRESHOLDMINUS:
				if (mfd->threshold > 0)
				{
					mfd->threshold -= 1;
					SetDlgItemInt(hdlg, IDC_ETHRESHOLD, mfd->threshold, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_THRESHOLD), TBM_SETPOS, (WPARAM)TRUE, mfd->threshold);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_STRENGTHPLUS:
				if (mfd->strength < 239)
				{
					mfd->strength += 1;
					mfd->strengthInv = 255 - mfd->strength;
					SetDlgItemInt(hdlg, IDC_ESTRENGTH, mfd->strength, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_STRENGTH), TBM_SETPOS, (WPARAM)TRUE, mfd->strength);
					mfd->ifp->RedoFrame();
				}
				break;
			case IDC_STRENGTHMINUS:
				if (mfd->strength > 0)
				{
					mfd->strength -= 1;
					mfd->strengthInv = 255 - mfd->strength;
					SetDlgItemInt(hdlg, IDC_ESTRENGTH, mfd->strength, FALSE);
					SendMessage(GetDlgItem(hdlg, IDC_STRENGTH), TBM_SETPOS, (WPARAM)TRUE, mfd->strength);
					mfd->ifp->RedoFrame();
				}
				break;
			}
	}

	return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	MyFilterData mfd_old = *mfd;
	int ret;

	mfd->ifp = fa->ifp;

	ret = DialogBoxParam(fa->filter->module->hInstModule, MAKEINTRESOURCE(IDD_FILTER),
						 hwnd, ConfigDlgProc, (LPARAM) mfd);
	if (ret)
	{
		*mfd = mfd_old;
		ret = TRUE;
	}
    else
	{
		ret = FALSE;
	}
	return(ret);
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	sprintf(str, " (strength %d, threshold %d)", mfd->strength, mfd->threshold);
}
