/*

	Deinterlace MAP - Plugin Filter for VirtualDub
    Copyright (C) 2001  Shaun Faulds

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

	Acknowledgments:

	Donald Graft
	Some of this code in this filter comes from the Smart Deinterlacer Filter
	by Donald Graft http://sauron.mordor.net/dgraft/smart.html
	I have used this code to smooth out the motion map before applying
	the Deinterlace algorithm.

	Gunnar Thalin
	I got the general idea about how to implement the interlace detection
	algorithm from Gunnar Thalin's Deinterlace Area Based filter, I wrote my
	own version using absolute values as I found it easier to understand what
	was going on.

	Avery Lee
	A lot of this code comes from the VirtualDub Filter SDK Tutorial written
	by Avery Lee.

*/

#include <windows.h>
#include <stdio.h>
#include <commctrl.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

///////////////////////////////////////////////////////////////////////////

int InitProc(FilterActivation *fa, const FilterFunctions *ff);
int RunProc(const FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char*str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue*argv,int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf,int buflen);
int StartProc(FilterActivation *fa, const FilterFunctions *ff);
int EndProc(FilterActivation *fa, const FilterFunctions *ff);

///////////////////////////////////////////////////////////////////////////

typedef struct FilterData {
	bool show;
	bool smoothMotionMap;
	bool useMotionMap;
	int deinterlaceThreshold;
	int lineDetect;
	int motionThreshold;

	int *prevFrameLuma;
	unsigned char *pixelbackUp;
	unsigned char *pixelChanges;
} FilterData;


bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf,int buflen)
{
	FilterData *fd = (FilterData *)fa->filter_data;

	_snprintf(buf, buflen, "Config(%i, %i, %i, %i, %i, %i)",
		fd->deinterlaceThreshold,
		fd->motionThreshold,
		fd->lineDetect,
		fd->useMotionMap,
		fd->smoothMotionMap,
		fd->show
		);

	return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue*argv,int argc)
{
	FilterActivation *fa = (FilterActivation *)lpVoid;
	FilterData *fd = (FilterData *)fa->filter_data;

	fd->deinterlaceThreshold = argv[0].asInt();
	fd->motionThreshold = argv[1].asInt();
	fd->lineDetect = argv[2].asInt();
	fd->useMotionMap = !!argv[3].asInt();
	fd->smoothMotionMap = !!argv[4].asInt();
	fd->show = !!argv[5].asInt();
}

ScriptFunctionDef func_defs[]={
 { (ScriptFunctionPtr)ScriptConfig, "Config", "0iiiiii" },
 { NULL },
};

CScriptObject script_obj={
	NULL, func_defs
};


struct FilterDefinition filterDef = {

	NULL, NULL, NULL, // next, prev, module
	"Deinterlace MAP", // name
	"Deinterlace MAP (Motion And Pixel)\nVersion 1.0\nInterlace detection and removal based on motion and pixel analysis.\n", // desc
	"Shaun Faulds",  // maker
	NULL,     // private_data
	sizeof(FilterData), // inst_data_size

	InitProc,  // initProc
	NULL,   // deinitProc
	RunProc,  // runProc
	ParamProc,  // paramProc
	ConfigProc,  // configProc
	StringProc,  // stringProc
	StartProc,   // startProc
	EndProc,   // endProc

	&script_obj, // script_obj
	FssProc,  // fssProc

};


///////////////////////////////////////////////////////////////////////////

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff,int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions*ff);

static FilterDefinition *fd;

///////////////////////////////////////////////////////////////////////////

int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat)
{
	if (!(fd = ff->addFilter(fm, &filterDef, sizeof(FilterDefinition))))
		return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

	return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff)
{
	ff->removeFilter(fd);
}

///////////////////////////////////////////////////////////////////////////

int InitProc(FilterActivation *fa, const FilterFunctions *ff)
{
    FilterData *fd = (FilterData *)fa->filter_data;

	fd->show = false;
	fd->deinterlaceThreshold = 110;
	fd->lineDetect = 11;
	fd->smoothMotionMap = true;
	fd->useMotionMap = true;
	fd->motionThreshold = 85;

    return 0;
}

int StartProc(FilterActivation *fa, const FilterFunctions *ff)
{

	FilterData *fd = (FilterData *)fa->filter_data;
	fd->prevFrameLuma = new int[fa->src.w * fa->src.h];
	fd->pixelbackUp = new unsigned char[fa->src.w * fa->src.h];
	fd->pixelChanges = new unsigned char[fa->src.w * fa->src.h];
	return 0;
}

int EndProc(FilterActivation *fa, const FilterFunctions *ff)
{
	FilterData *fd = (FilterData *)fa->filter_data;
	delete[] fd->prevFrameLuma;
	delete[] fd->pixelbackUp;
	delete[] fd->pixelChanges;

	return 0;
}

int RunProc(const FilterActivation *fa, const FilterFunctions *ff)
{

	FilterData *fd = (FilterData *)fa->filter_data;
	fa->dst.data = fa->src.data;

	int Hless1 = fa->src.h-1;
	int Wless1 = fa->src.w-1;
	int w = fa->src.w;
	int h = fa->src.h;
	int pitch = fa->dst.pitch;
	int pitchX2 = pitch * 2;
	int wTimes4 = w * 4;
	int wTimes2 = w * 2;

	int aboveRED, aboveGREEN, aboveBLUE;
	int currentRED, currentGREEN, currentBLUE;
	int belowRED, belowGREEN, belowBLUE;

	int interlaceTotal = 0;

	int gradientRED = 0;
	int gradientGREEN = 0;
	int gradientBLUE = 0;
	int gradientTOTAL = 0;

	int edgeRED = 0;
	int edgeGREEN = 0;
	int edgeBLUE = 0;
	int edgeTOTAL = 0;

	Pixel32 *above, *current, *below;

	int luma;
	int pixelDiff;

	Pixel32 *pixelDATA = fa->dst.data;
	unsigned char *motionMapTEMP = fd->pixelChanges;
	int *prevFrameLumaTEMP = fd->prevFrameLuma;

	///////////////////////////////////////////////////////////////
	//Calculate the motion map

	if(fd->useMotionMap)
	{
		for(int y = 0; y < h; y++)
		{
			for(int x = 0; x < w; x++)
			{
				currentRED = (pixelDATA[x] & 0X00FF0000) >> 16;
				currentGREEN = (pixelDATA[x] & 0X0000FF00) >> 8;
				currentBLUE = (pixelDATA[x] & 0X000000FF);

				luma = ((3 * currentRED) + (6 * currentGREEN) + currentBLUE);

				pixelDiff = abs(prevFrameLumaTEMP[x] - luma);

				if(pixelDiff > fd->motionThreshold)
					motionMapTEMP[x] = 1;
				else
					motionMapTEMP[x] = 0;

				prevFrameLumaTEMP[x] = luma;
			}

			motionMapTEMP = motionMapTEMP + w;
			prevFrameLumaTEMP = prevFrameLumaTEMP + w;
			pixelDATA = (Pixel32*)((char*)pixelDATA + pitch);
		}

		///////////////////////////////////////////////////////////////
		//Smooth motion map from Smart Deinterlacer Filter by Donald Graft
		if(fd->smoothMotionMap)
		{
			int xlo, xhi, ylo, yhi;
			int u, v;
			int N = 5;
			int Nover2 = N/2;
			int sum;
			unsigned char *m;


			// Erode.
			for (int ystuff = 0; ystuff < h; ystuff++)
			{
				for (int xstuff = 0; xstuff < w; xstuff++)
				{
					xlo = xstuff - Nover2; if (xlo < 0) xlo = 0;
					xhi = xstuff + Nover2; if (xhi >= w) xhi = Wless1;
					ylo = ystuff - Nover2; if (ylo < 0) ylo = 0;
					yhi = ystuff + Nover2; if (yhi >= h) yhi = Hless1;
					m = fd->pixelChanges + (ylo * w);
					sum = 0;
					for (u = ylo; u <= yhi; u++)
					{
						for (v = xlo; v <= xhi; v++)
						{
							sum += m[v];
						}
						m += w;
					}
					if (sum > 9)
						fd->pixelbackUp[xstuff + (ystuff*w)] = 1;
					else
						fd->pixelbackUp[xstuff + (ystuff*w)] = 0;
				}
			}

			// Dilate.
			N = 5;
			Nover2 = N/2;

			for (int y20 = 0; y20 < h; y20++)
			{
				for (int x = 0; x < w; x++)
				{
					if(fd->pixelbackUp[x + (y20 * w)] == 0)
					{
						fd->pixelChanges[x + (y20 * w)] = 0;
						continue;
					}
					xlo = x - Nover2; if (xlo < 0) xlo = 0;
					xhi = x + Nover2; if (xhi >= w) xhi = Wless1;
					ylo = y20 - Nover2; if (ylo < 0) ylo = 0;
					yhi = y20 + Nover2; if (yhi >= h) yhi = Hless1;
					m = fd->pixelChanges + (ylo * w);
					for (u = ylo; u <= yhi; u++)
					{
						for (v = xlo; v <= xhi; v++)
						{
							m[v] = 1;
						}
						m += w;
					}
				}
			}
		}
		//End of smooth motion map
	}
	//End of Calculate the motion map

	///////////////////////////////////////////////////////////////
	//Do the deinterlaceing	

	pixelDATA = fa->dst.data;
	motionMapTEMP = fd->pixelChanges;

	pixelDATA = (Pixel32*)((char*)pixelDATA + pitch);
	motionMapTEMP = motionMapTEMP + w;

	for(int y3 = 1; y3 < Hless1-1; y3 += 2)
	{
		for(int x3 = 0; x3 < w; x3++)
		{
			if(!fd->useMotionMap || motionMapTEMP[x3] == 1)
			{
				above = (Pixel32*)((char*)pixelDATA + (x3*4) - pitch);
				current = pixelDATA + x3;
				below = (Pixel32*)((char*)pixelDATA + (x3*4) + pitch);

				aboveRED = (*above & 0X00FF0000) >> 16;
				aboveGREEN = (*above & 0X0000FF00) >> 8;
				aboveBLUE = (*above & 0X000000FF);

				currentRED = (*current & 0X00FF0000) >> 16;
				currentGREEN = (*current & 0X0000FF00) >> 8;
				currentBLUE = (*current & 0X000000FF);

				belowRED = (*below & 0X00FF0000) >> 16;
				belowGREEN = (*below & 0X0000FF00) >> 8;
				belowBLUE = (*below & 0X000000FF);

				gradientRED = 3 * abs((aboveRED - currentRED) + (belowRED - currentRED));
				gradientGREEN = 6 * abs((aboveGREEN - currentGREEN) + (belowGREEN - currentGREEN));
				gradientBLUE = abs((aboveBLUE - currentBLUE) + (belowBLUE - currentBLUE));
				gradientTOTAL = (gradientRED + gradientGREEN + gradientBLUE);

				edgeRED = (3 * abs(aboveRED - belowRED) * fd->lineDetect) / 10;
				edgeGREEN = (6 * abs(aboveGREEN - belowGREEN) * fd->lineDetect) / 10;
				edgeBLUE = (abs(aboveBLUE - belowBLUE) * fd->lineDetect) / 10;
				edgeTOTAL = (edgeRED + edgeGREEN + edgeBLUE);

				interlaceTotal = gradientTOTAL - edgeTOTAL;

				if(fd->useMotionMap && fd->show)
					*current = 0XFFFFFFFF;

				if(interlaceTotal > fd->deinterlaceThreshold)
				{
					if(fd->show)
						*current = 0XFFFF0000;
					else
					{
						int newRED = (aboveRED + belowRED) >> 1;
						int newGREEN = (aboveGREEN + belowGREEN) >> 1;
						int newBLUE = (aboveBLUE + belowBLUE) >> 1;
						*current = newRED << 16 | newGREEN << 8 | newBLUE;
					}
				}

			}
		}
		pixelDATA = (Pixel32*)((char*)pixelDATA + pitchX2);
		motionMapTEMP = motionMapTEMP + wTimes2;
	}

	return 0;
}

long ParamProc(FilterActivation *fa, const FilterFunctions *ff)
{
	fa->dst.offset = fa->src.offset;
	return 0;
}


BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
	FilterData *fd = (FilterData *)GetWindowLong(hdlg, DWL_USER);
	HWND hwndCtl;

    switch(msg)
	{
        case WM_INITDIALOG:
			SetWindowLong(hdlg, DWL_USER, lParam);
			fd = (FilterData *)lParam;

			CheckDlgButton(hdlg, IDC_SHOW, fd->show ? BST_CHECKED:BST_UNCHECKED);
			CheckDlgButton(hdlg, IDC_USE_MAP, fd->useMotionMap ? BST_CHECKED:BST_UNCHECKED);
			CheckDlgButton(hdlg, IDC_SMOOTH_MAP, fd->smoothMotionMap ? BST_CHECKED:BST_UNCHECKED);

			char str[10];
			sprintf(str, "%i", fd->deinterlaceThreshold);
			SetDlgItemText(hdlg, IDC_DEINTERLACE_THRESHOLD, str);
			sprintf(str, "%i", fd->lineDetect);
			SetDlgItemText(hdlg, IDC_EDGE_DETECT, str);
			sprintf(str, "%i", fd->motionThreshold);
			SetDlgItemText(hdlg, IDC_MOTION, str);

			hwndCtl = GetDlgItem(hdlg, IDC_MOTION_SPIN);
			SendMessage(hwndCtl, UDM_SETRANGE, 0, MAKELPARAM(300, 1));
			hwndCtl = GetDlgItem(hdlg, IDC_DEINTERLACE_THRESHOLD_SPIN);
			SendMessage(hwndCtl, UDM_SETRANGE, 0, MAKELPARAM(500, 1));
			hwndCtl = GetDlgItem(hdlg, IDC_EDGE_DETECT_SPIN);
			SendMessage(hwndCtl, UDM_SETRANGE, 0, MAKELPARAM(100, 1));

            return TRUE;

        case WM_COMMAND:
            switch(LOWORD(wParam))
			{
				case IDOK:
					fd->show = !!IsDlgButtonChecked(hdlg, IDC_SHOW);
					fd->useMotionMap = !!IsDlgButtonChecked(hdlg, IDC_USE_MAP);
					fd->smoothMotionMap = !!IsDlgButtonChecked(hdlg, IDC_SMOOTH_MAP);

					char str[10];
					GetDlgItemText(hdlg, IDC_DEINTERLACE_THRESHOLD, str, 10);
					fd->deinterlaceThreshold = atoi(str);
					GetDlgItemText(hdlg, IDC_EDGE_DETECT, str, 10);
					fd->lineDetect = atoi(str);
					GetDlgItemText(hdlg, IDC_MOTION, str, 10);
					fd->motionThreshold = atoi(str);
					EndDialog(hdlg, 0);
					return TRUE;

				case IDCANCEL:
					EndDialog(hdlg, 1);
					return FALSE;
			}
			break;
	}

    return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd)
{
	return DialogBoxParam(fa->filter->module->hInstModule,
		MAKEINTRESOURCE(IDD_DEINTERLACEMAP_PROP), hwnd,
		(DLGPROC)ConfigDlgProc, (LPARAM)fa->filter_data);
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff,char *str)
{
	FilterData *fd = (FilterData *)fa->filter_data;
//	sprintf(str, " (limit %i, show %s)",
//		fd->colourGradient,
//		fd->show?"ON":"OFF");
}

