// --  main.cpp  -------------------------------------------------------------

/*
 * --- Area proximity locking of pixels, for Virtual Dub ---
 *
 * Copyright (C) 2001 Chir 
 *
 * DESCRIPTION: this filter locks pixels that are surrounded by 
 *              pixels already locked by some previous filter.
 *
 * THIS PROGRAM IS FREE SOFTWARE; YOU CAN REDISTRIBUTE IT AND/OR MODIFY
 * IT UNDER THE TERMS OF THE GNU GENERAL PUBLIC LICENSE AS PUBLISHED BY
 * THE FREE SOFTWARE FOUNDATION; EITHER VERSION 2 OF THE LICENSE, OR
 * (AT YOUR OPTION) ANY LATER VERSION.
 *
 * THIS PROGRAM IS DISTRIBUTED IN THE HOPE THAT IT WILL BE USEFUL,
 * BUT WITHOUT ANY WARRANTY; WITHOUT EVEN THE IMPLIED WARRANTY OF
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  SEE THE
 * GNU GENERAL PUBLIC LICENSE FOR MORE DETAILS.
 *
 * YOU SHOULD HAVE RECEIVED A COPY OF THE GNU GENERAL PUBLIC LICENSE
 * ALONG WITH THIS PROGRAM; IF NOT, WRITE TO THE FREE SOFTWARE
 * FOUNDATION, INC., 59 TEMPLE PLACE, SUITE 330, BOSTON, MA 02111-1307 USA
 */

#include <stdio.h>
#include <assert.h>
#include <windows.h>
#include <commctrl.h>
#include "resource.h"
#include "../dub/include/ScriptValue.h"
#include "../dub/include/filter.h"

// ---------------------------------------------------------------------------
/*
inline Pixel32* getAddress( VFBitmap& b, int x, int y )
{
    return (Pixel32*)((char*)b.data + x*(sizeof(Pixel32)*b.w+b.modulo)) + y;
}
*/
#define getAddress3(b,x,y)  (Pixel32*)((Pixel32*)((char*)b.data + x*(sizeof(Pixel32)*b.w+b.modulo)) + y)

/*
inline Pixel32* getAddress( Pixel32* data, int w, int h, int modulo, int x, int y )
{
    return (Pixel32*)((char*)data + x*(sizeof(Pixel32)*w+modulo)) + y;
}
*/
#define getAddress6(data,w,h,modulo,x,y)  (Pixel32*)((Pixel32*)((char*)data + x*(sizeof(Pixel32)*w+modulo)) + y)

inline bool isLockable( Pixel32* old_src, Pixel32* dst, int w, int w_bytes, int threshold )
{
    Pixel32* dst2 = (Pixel32*)((char*)dst + w_bytes) - 1;
    Pixel32* old_src2 = (Pixel32*)((char*)old_src + w_bytes) - 1;
    int l = 0;                                      // Number of locked pixels around this one.
    int i, j;

    // Count the number of pixels already locked, around this one.
    i = 3;
    do
    {
        j = 3;
        do
        {
            if( *dst2++ == *old_src2++ )
            {
                if( ++l > threshold )  return true;   // Success.
            }
        } while( --j );
        dst2 = (Pixel32*)((char*)dst2 - w_bytes) - 3;
        old_src2 = (Pixel32*)((char*)old_src2 - w_bytes) - 3;
    } while( --i );
    return false;
}

// ---------------------------------------------------------------------------

int  myFilterRunProc( const FilterActivation* fa, const FilterFunctions* ff );
int  myFilterStartProc( FilterActivation* fa, const FilterFunctions* ff );
int  myFilterEndProc( FilterActivation* fa, const FilterFunctions* ff );
long myFilterParamProc( FilterActivation* fa, const FilterFunctions* ff );
int  myFilterConfigProc( FilterActivation* fa, const FilterFunctions* ff, HWND hwnd );
void myFilterStringProc( const FilterActivation* fa, const FilterFunctions* ff, char* str );
void myFilterScriptConfig( IScriptInterpreter* isi, void* lpVoid, CScriptValue* argv, int argc );
bool myFilterFssProc( FilterActivation* fa, const FilterFunctions* ff, char* buf, int buflen );

class MyFilterData
{
    unsigned char*  m_buffer;       // To keep the previous frame.
    int             m_threshold;    // Locking level.
    bool            m_highlighted;  // Indicates if modified pixels have to be highlighted.
  public:
    MyFilterData()  { m_buffer = NULL;  m_threshold = 5; m_highlighted = false; }
    ~MyFilterData()  { setBuffer(NULL); }
    inline void setHighlighting( bool h )  { m_highlighted = h; }
    inline bool isHighlighting()  { return m_highlighted; }
    void setBuffer( unsigned char* b )
    {
        if( m_buffer != NULL )  delete[] m_buffer;
        m_buffer = b;
    }
    unsigned char* getBuffer()  { return m_buffer; }
    void setThreshold( int t )  { m_threshold = t; }
    inline int getThreshold()  { return m_threshold; }
};

ScriptFunctionDef myFilter_func_defs[] =
{
    { (ScriptFunctionPtr)myFilterScriptConfig, "Config", "0ii" },
    { NULL },
};

CScriptObject myFilter_obj =
{
    NULL, myFilter_func_defs
};

struct FilterDefinition filterDef_myFilter =
{
    NULL, NULL, NULL,           // next, prev, module.
    "proximity locker v0.3",
    "Locks pixels that are near already-locked pixels.",
    "Chir",                     // maker.
    NULL,                       // private_data.
    sizeof(MyFilterData),       // inst_data_size.

    NULL,                       // initProc.
    NULL,                       // deinitProc.
    myFilterRunProc,            // runProc.
    myFilterParamProc,          // paramProc.
    myFilterConfigProc,         // configProc.
    myFilterStringProc,         // stringProc.
    myFilterStartProc,          // startProc.
    myFilterEndProc,            // endProc.

    &myFilter_obj,              // script_obj.
    myFilterFssProc,            // fssProc.
};

// ---------------------------------------------------------------------------
extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition* fd_myFilter;

// ---------------------------------------------------------------------------
int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule* fm, const FilterFunctions* ff, int& vdfd_ver, int& vdfd_compat)
{
    if( !(fd_myFilter = ff->addFilter(fm, &filterDef_myFilter, sizeof(FilterDefinition))) )  return 1;

    vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
    vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;
    return 0;
}

void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule* fm, const FilterFunctions* ff)
{
    ff->removeFilter(fd_myFilter);
}

// ---------------------------------------------------------------------------
int myFilterStartProc( FilterActivation* fa, const FilterFunctions* ff )
{
    MyFilterData* mfd = (MyFilterData*)fa->filter_data;

    return 0;   // Everything went ok.
}

int myFilterRunProc(const FilterActivation* fa, const FilterFunctions* ff)
{
    MyFilterData* mfd = (MyFilterData*)fa->filter_data;
    int w = fa->src.w , h = fa->src.h;
    int w_bytes = w*sizeof(Pixel32) + fa->src.modulo;

    if( mfd->getBuffer() == NULL )
    {
        // Make buffer and clear it.
        mfd->setBuffer( new unsigned char[fa->src.size] );
        memcpy( mfd->getBuffer(), fa->src.data, fa->src.size );
    }

    // Copy the current frame into the destination buffer.
    int temp = w*sizeof(Pixel32);
	for( int x=fa->src.h-1; x >= 0; --x )
	{
        memcpy( getAddress3( fa->dst, x, 0 ), getAddress3( fa->src, x, 0 ) , temp );
    }

    if( mfd->isHighlighting() )
    {
        for( int x=h-1; x >= 0; --x )
        {
            for( int y=w-1; y >= 0; --y )
            {
                Pixel32* src = getAddress3( fa->src, x, y );
                Pixel32* old_src = getAddress6( (Pixel32*)mfd->getBuffer(), w, h, fa->src.modulo, x, y );
                Pixel32* dst = getAddress3( fa->dst, x, y );

                if( *src == *old_src )
                {
                    *dst = *old_src = 0x007F7F;             // Locked pixels appear as dark cyan.
                }
            }
        }
    }

    // Run the actual filtering.
    bool b_modified;

    if( mfd->isHighlighting() )
    {
        do
        {
            Pixel32* src = (Pixel32*)fa->src.data;
            Pixel32* old_src = (Pixel32*)mfd->getBuffer();
            Pixel32* dst = (Pixel32*)fa->dst.data;

            b_modified = false;
            for( int x=h-1; x >= 0; --x )
            {
                for( int y=w-1; y >= 0; --y )
                {
                    if( *dst != *old_src )
                    {
                        // Count the number of pixels already locked, around this one.
                        int k = 0;                                      // Number of locked pixels around this one.
                        int n = 0;                                      // Number of existing pixels around this one.

                        Pixel32* old_src2 = (Pixel32*)((char*)old_src + w_bytes) - 1;;  // getAddress( (Pixel32*)mfd->getBuffer(), w, h, fa->src.modulo, x-1, y-1 );
                        Pixel32* dst2 = (Pixel32*)((char*)dst + w_bytes) - 1;  // getAddress( fa->dst, x-1, y-1 );
                        for( int i=-1; i < 2; ++i )
                        {
                            for( int j=-1; j < 2; ++j )
                            {
                                if( (i != 0  ||  j != 0)  &&  x+i >= 0  &&  x+i < h  &&  y+j >= 0  &&  y+j < w )
                                {
                                    if( *dst2 == *old_src2 )  ++k;
                                    ++n;                                // Count an additional valid pixel.
                                }
                                ++dst2;  ++old_src2;
                            }
                            old_src2 = (Pixel32*)((char*)old_src2 - w_bytes) - 3;
                            dst2 = (Pixel32*)((char*)dst2 - w_bytes) - 3;
                        }
                        if( (8*k)/n > mfd->getThreshold() )
                        {
                            // Lock this pixel.
                            *dst = *old_src = *src = 0xFFFFFF;          // Newly locked pixels appear as white.
                            b_modified = true;                          // Run the filter loop again.
                        }
                    }
                    ++dst;  ++src;  ++old_src;
                }
                src = (Pixel32*)((char*)src + fa->src.modulo);
                old_src = (Pixel32*)((char*)old_src + fa->src.modulo);
                dst = (Pixel32*)((char*)dst + fa->dst.modulo);
            }
        } while( b_modified );
    }
    else
    {
        int w_3 = w - 3  ,  h_3 = h - 3;
        int th = mfd->getThreshold();
        Pixel32 *src , *old_src, *dst;

        do
        {
            // Critical loop (processes the central area of the frame).
            do
            {
                // Skip the first line, and the first column of pixels.
                src = (Pixel32*)((char*)fa->src.data + w_bytes) +1;
                old_src = (Pixel32*)((char*)mfd->getBuffer() + w_bytes) +1;
                dst = (Pixel32*)((char*)fa->dst.data + w_bytes) +1;

                b_modified = false;
                int x = h_3;
                do
                {
                    int y = w_3;
                    do
                    {
                        if( *dst != *old_src  &&  isLockable( old_src, dst, w, w_bytes, th ) )
                        {
                            // Lock this pixel.
                            *dst = *src = *old_src;                     // Newly locked pixel.
                            b_modified = true;                          // Run the filter loop again.
                        }
                        ++dst;  ++src;  ++old_src;
                    } while( --y );
                    src = (Pixel32*)((char*)src + fa->src.modulo) + 2;  // Skip the last column of pixels.
                    old_src = (Pixel32*)((char*)old_src + fa->src.modulo) + 2;
                    dst = (Pixel32*)((char*)dst + fa->dst.modulo) + 2;
                } while( --x );
            } while( b_modified );

            // Secondary loop (processes the frame border).
            src = (Pixel32*)fa->src.data;
            old_src = (Pixel32*)mfd->getBuffer();
            dst = (Pixel32*)fa->dst.data;

            assert( getAddress(dst,w,h,fa->dst.modulo,0,0) == dst );
            assert( getAddress(src,w,h,fa->src.modulo,0,0) == src );
            assert( getAddress(old_src,w,h,fa->src.modulo,0,0) == old_src );

            for( int x=0; x < h; ++x )
            {
                for( int y=0; y < w; ++y )
                {
                    assert( getAddress((Pixel32*)fa->dst.data,w,h,fa->dst.modulo,x,y) == dst );
                    assert( getAddress((Pixel32*)fa->src.data,w,h,fa->src.modulo,x,y) == src );
                    assert( getAddress((Pixel32*)mfd->getBuffer(),w,h,fa->src.modulo,x,y) == old_src );

                    if( x != 0  &&  x != h-1  &&  y != 0  &&  y != w-1 )
                    {
                        // Skip most of this loop.
                        assert( y == 1 );
                        src += w_3;  old_src += w_3;  dst += w_3;
                        y += w_3;   // y = w - 2;
                    }
                    else
                    {
                        if( *dst != *old_src )
                        {
                            // Count the number of pixels already locked, around this one.
                            int k = 0;                                      // Number of locked pixels around this one.
                            int n = 0;                                      // Number of existing pixels around this one.
                            Pixel32* old_src2 = (Pixel32*)((char*)old_src + w_bytes) - 1;
                            Pixel32* dst2 = (Pixel32*)((char*)dst + w_bytes) - 1;

                            for( int i=1; i >= -1; --i )
                            {
                                for( int j=-1; j <= 1; ++j )
                                {
                                    if( (i != 0  ||  j != 0)  &&  x+i >= 0  &&  x+i < h  &&  y+j >= 0  &&  y+j < w )
                                    {
                                        if( *dst2 == *old_src2 )  ++k;
                                        ++n;                                // Count an additional valid pixel.
                                    }
                                    ++dst2;  ++old_src2;
                                }
                                old_src2 = (Pixel32*)((char*)old_src2 - w_bytes) - 3;
                                dst2 = (Pixel32*)((char*)dst2 - w_bytes) - 3;
                            }

                            if( (8*k)/n > th )
                            {
                                // Lock this pixel.
                                *dst = *src = *old_src;                     // Newly locked pixel.
                                b_modified = true;                          // Run the filter loop again.
                            }
                        }
                    }
                    ++dst;  ++src;  ++old_src;
                }
                src = (Pixel32*)((char*)src + fa->src.modulo);
                old_src = (Pixel32*)((char*)old_src + fa->src.modulo);
                dst = (Pixel32*)((char*)dst + fa->dst.modulo);
            }
        } while( b_modified );
    }
    // Memorize the corrected frame.
    memcpy( mfd->getBuffer(), fa->src.data, fa->src.size );

    return 0;
}

int myFilterEndProc( FilterActivation* fa, const FilterFunctions* ff )
{
    MyFilterData* mfd = (MyFilterData*)fa->filter_data;

    mfd->setBuffer(NULL);   // delete buffer.
    return 0;
}

long myFilterParamProc( FilterActivation* fa, const FilterFunctions* ff )
{
    MyFilterData* mfd = (MyFilterData*)fa->filter_data;

    fa->dst.pitch = (fa->dst.w*4 + 7) & -8;
    return FILTERPARAM_SWAP_BUFFERS;
}


BOOL CALLBACK myFilterConfigDlgProc( HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam )
{
    MyFilterData* mfd = (MyFilterData*)GetWindowLong(hdlg, DWL_USER);

    switch(msg)
    {
        case WM_INITDIALOG:
            SetWindowLong( hdlg, DWL_USER, lParam );
            mfd = (MyFilterData *)lParam;

            CheckDlgButton( hdlg, IDC_EXPAND, mfd->isHighlighting() ? BST_CHECKED : BST_UNCHECKED );
                            HWND hwnd;
                            hwnd = GetDlgItem( hdlg, IDC_SLIDER1 );
                            SendMessage( hwnd, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(0, 4) );
                            SendMessage( hwnd, TBM_SETTICFREQ, 1 , 0 );
                            SendMessage( hwnd, TBM_SETPOS, (WPARAM)TRUE, 8 - mfd->getThreshold()      );
            return true;
        case WM_COMMAND:
            switch( LOWORD(wParam) )
            {
                case IDOK:
                    HWND hwnd;
                    hwnd = GetDlgItem( hdlg, IDC_SLIDER1 );
                    int val;
                    val = SendMessage( hwnd, TBM_GETPOS, 0, 0 );
                    mfd->setHighlighting( IsDlgButtonChecked(hdlg, IDC_EXPAND) == 1 );
                    mfd->setThreshold( 8 - val );
                    EndDialog( hdlg, 0 );
                    return true;
                case IDCANCEL:
                    EndDialog( hdlg, 1 );
                    return false;
            }
            break;
    }
    return false;
}

int myFilterConfigProc( FilterActivation* fa, const FilterFunctions* ff, HWND hwnd )
{
    return DialogBoxParam( fa->filter->module->hInstModule,
                           MAKEINTRESOURCE(IDD_MY_FILTER), hwnd,
                           myFilterConfigDlgProc, (LPARAM)fa->filter_data );
}

void myFilterStringProc( const FilterActivation* fa, const FilterFunctions* ff, char* str )
{
    MyFilterData* mfd = (MyFilterData*)fa->filter_data;

    strcpy( str, mfd->isHighlighting()  ?  " (show changes)"  :  " (normal)" );
}

void myFilterScriptConfig( IScriptInterpreter*isi, void*lpVoid, CScriptValue*argv, int argc )
{
    FilterActivation* fa = (FilterActivation*)lpVoid;
    MyFilterData* mfd = (MyFilterData*)fa->filter_data;

    mfd->setHighlighting( argv[0].asInt() == 1 );
    mfd->setThreshold( argv[1].asInt() );
}

bool myFilterFssProc( FilterActivation* fa, const FilterFunctions* ff, char* buf, int buflen )
{
    MyFilterData* mfd = (MyFilterData *)fa->filter_data;

    _snprintf( buf, buflen, "Config(%d,%d)", mfd->isHighlighting(), mfd->getThreshold() );
    return true;
}
// ---------------------------------------------------------------------------
