/*
    DeLogo Filter for VirtualDub -- Remove a still logo/watermark from video
	Copyright (C) 2001 Karel Suhajda

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful, 
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

	The author can be contacted at:
	Karel Suhajda
	kasuha@post.cz
*/

#include <stdio.h>
#include <windows.h>
#include <commctrl.h>
#include <math.h>
#include <float.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

// if logging is defined, all logs are written to C:\delogo.log
//#define LOGGING
//#define LOGLEVEL 3

///////////////////////////////////////////////////////////////////////////

#ifdef LOGGING

#define log(x) {logtofile x;}

#define logl(x, y) { if ((x) <= LOGLEVEL) logtofile y; }

#else

#define log(x) {;}

#define logl(x, y) {;}

#endif // LOGGING

///////////////////////////////////////////////////////////////////////////

#ifdef LOGGING

void logtofile(char *string, ...) 
{
	FILE *fout;
	va_list argptr;

	fout=fopen("c:\\delogo.log", "a+t");
	if (fout!=NULL) 
	{
		va_start(argptr, string);
		vfprintf(fout, string, argptr);
		fprintf(fout, "\n");
		va_end(argptr);
		fclose(fout);
	}
}

#endif // LOGGING

///////////////////////////////////////////////////////////////////////////

int  InitProc(FilterActivation *fa, const FilterFunctions *ff);
void DeinitProc(FilterActivation *fa, const FilterFunctions *ff);
int  RunProc(const FilterActivation *fa, const FilterFunctions *ff);
int  StartProc(FilterActivation *fa, const FilterFunctions *ff);
int  EndProc(FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int  ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
BOOL CALLBACK ConfigDlgProc (HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

#define MAXSTR 260

#define ALPHATOP 200

#define MINDEPTH		10
#define MAXDEPTH		80
#define DEFAULTDEPTH	15
#define TICKDEPTH		10
#define PRECDEPTH		1

#define MINASPECT		-120
#define MAXASPECT		120
#define DEFAULTASPECT	0
#define TICKASPECT		30

#define MINPOWER		0
#define MAXPOWER		160
#define DEFAULTPOWER	40
#define TICKPOWER		10
#define	PRECPOWER		1
#define MAXREALPOWER	16

#define MINFALLOFF		0
#define MAXFALLOFF		64
#define DEFAULTFALLOFF	16
#define TICKFALLOFF 	4
#define PRECFALLOFF		0

#define MINALPHA		0
#define MAXALPHA		255
#define DEFAULTALPHA	160
#define TICKALPHA		32
#define	PRECALPHA		0

#define MINRADIUS		0
#define MAXRADIUS		80
#define DEFAULTRADIUS	0
#define TICKRADIUS		10
#define PRECRADIUS		1

#define MINSHIFT		-100
#define	MAXSHIFT		100
#define	DEFAULTSHIFT	0
#define TICKSHIFT		20
#define PRECSHIFT		2

#define FALLOFFCOEF		(0x3FFF/MAXFALLOFF)

typedef unsigned long ulong;

typedef struct MyBitmap {
	int 	w, h;
	Pixel32 *bitmap;
	char	path[MAXSTR];
} MyBitmap;

typedef struct Interval {
	long	first, last;
} Interval;

typedef struct IntControl {
	char	text[MAXSTR];
	Interval *interval;
	int count;
} IntControl;

typedef struct TargetPixel {
	int		x, y;
} TargetPixel;			// pixels to be repaired

typedef struct SourcePixel {
	int		x, y;
	long	offset;
} SourcePixel;			// pixels to be used in repair process

class TargetPixelArray {
public:
	TargetPixel *pixels;
	long count;
	long pcount;

	TargetPixelArray(void) 
	{
		logl(5, ("TargetPixelArray::TargetPixelArray()"));
		pixels=NULL;
		count=0;
		pcount=0;
	}

	~TargetPixelArray(void) 
	{
		logl(5, ("TargetPixelArray::~TargetPixelArray()"));
		logl(3, ("Delete TargetPixel(%d)", pixels));
		delete [] pixels;
		pixels=NULL;
		count=0;
		pcount=0;
	}

	bool ok(void) 
	{
		logl(5, ("TargetPixelArray::ok()"));
		return (count==0)==(pixels==NULL);
	}

	bool init(long cnt) 
	{
		logl(5, ("TargetPixelArray::init()"));
		logl(3, ("Delete TargetPixel(%d)", pixels));
		delete [] pixels;

		pcount=0;
		if (cnt>0) 
		{
			count = cnt;
			pixels = new TargetPixel[count];
			logl(3, ("New TargetPixel = %d", pixels));
		}
		else 
		{
			count = 0;
			pixels = NULL;
		}
		return ok();
	}

	void addPixel(long x, long y) 
	{
		logl(9, ("TargetPixelArray::addPixel(%d, %d)", x, y));
		if (pcount < count) 
		{
			pixels[pcount].x=x;
			pixels[pcount].y=y;
			++pcount;
		}
	}
};

class SourcePixelArray {
public:
	SourcePixel *pixels;
	long count;
	long pixelNo;

	SourcePixelArray(void) 
	{
		logl(5, ("SourcePixelArray::SourcePixelArray()"));
		pixels=NULL;
		count=0;
		pixelNo=0;
	}

	~SourcePixelArray(void) 
	{
		logl(5, ("SourcePixelArray::~SourcePixelArray()"));
		logl(3, ("Delete SourcePixel(%d)", pixels));
		delete [] pixels;
		pixels=NULL;
		count=0;
		pixelNo=0;
	}

	bool ok(void) 
	{
		logl(5, ("SourcePixelArray::ok()"));
		return (count==0)==(pixels==NULL);
	}

	bool init(long cnt) 
	{
		logl(5, ("SourcePixelArray::init()"));

		logl(3, ("Delete SourcePixel(%d)", pixels));
		delete [] pixels;
		pixelNo=0;
		if (cnt>0) 
		{
			count=cnt;
			pixels=new SourcePixel[count];
			logl(3, ("New SourcePixel = %d", pixels));
		} 
		else 
		{
			count=0;
			pixels=NULL;
		}
		return ok();
	}

	void addPixel(long x, long y, long offset) 
	{
		logl(9, ("SourcePixelArray::addPixel(%d, %d, %d)", x, y, offset));
		pixels[pixelNo].x=x;
		pixels[pixelNo].y=y;
		pixels[pixelNo].offset=offset;
		pixelNo++;
	}

};

class RepairSet {
public:
	SourcePixelArray source;
	TargetPixelArray target;
	long *repre;
	bool repreSet;
	long count;

	RepairSet(void) 
	{
		logl(5, ("RepairSet::RepairSet()"));
		repre=0;
		repreSet=false;
		count=0;
	}

	bool initSource(long count) 
	{
		logl(5, ("RepairSet::init()"));
		return source.init(count);
	}

	bool initTarget(long count) 
	{
		logl(5, ("RepairSet::initTarget()"));
		return target.init(count);
	}

	void maxDists(long &dx, long &dy) 
	{
		logl(5, ("RepairSet::maxDists() ->"));
		long i, sx1, sx2, sy1, sy2, tx1, tx2, ty1, ty2;
		SourcePixel *sptr;
		TargetPixel *tptr;
		dx=0;
		dy=0;
		if (source.ok() 
			&& target.ok() 
			&& (source.count>0) 
			&& (target.count>0)) 
		{
			i=source.count;
			sptr=source.pixels;
			sx1=sx2=sptr->x;
			sy1=sy2=sptr->y;
			while (--i) 
			{
				sptr++;
				if (sptr->x < sx1) 
				{
					sx1=sptr->x; 
				}
				else if (sptr->x > sx2) 
				{
					sx2=sptr->x;
				}

				if (sptr->y < sy1) 
				{
					sy1=sptr->y; 
				}
				else if (sptr->y > sy2) 
				{
					sy2=sptr->y;
				}
			}

			i=target.count;
			tptr=target.pixels;
			tx1=tx2=tptr->x;
			ty1=ty2=tptr->y;
			while (--i) 
			{
				tptr++;
				if (tptr->x < tx1) 
				{
					tx1=tptr->x; 
				}
				else if (tptr->x > tx2) 
				{
					tx2=tptr->x;
				}

				if (tptr->y < ty1) 
				{
					ty1=tptr->y; 
				}
				else if (tptr->y > ty2) 
				{
					ty2=tptr->y;
				}
			}

			if ((dx = sx2 - tx1) < (i = tx2 - sx1)) 
			{
				dx=i;
			}

			if ((dy = sy2 - ty1) < (i = ty2 - sy1)) 
			{
				dy=i;
			}
		}

		logl(5, ("<- RepairSet::maxDists()"));
	}
};

class RepairSetArray {
public:
	RepairSet *set;
	long count;

	RepairSetArray(void) 
	{
		logl(5, ("RepairSetArray::RepairSetArray()"));
		set=NULL;
		count=0;
	}

	~RepairSetArray(void) 
	{
		logl(5, ("RepairSetArray::~RepairSetArray()"));
		logl(3, ("Delete RepairSet(%d)", set));
		delete [] set;
		set=NULL;
		count=0;
	}

	bool ok(void) 
	{
		logl(5, ("RepairSetArray::ok()"));
		return (count == 0) == (set == NULL);
	}

	bool init(long cnt) 
	{
		logl(5, ("RepairSetArray::init()"));
		logl(3, ("Delete RepairSet(%d)", set));
		delete [] set;
		if (cnt>0) 
		{
			count=cnt;
			set = new RepairSet[count];
			logl(3, ("New RepairSet = %d", set));
		} 
		else 
		{
			count=0;
			set=NULL;
		}
		return ok();
	}

	void deinit(void) 
	{
		logl(5, ("RepairSetArray::deinit()"));
		logl(3, ("Delete RepairSet(%d)", set));
		delete [] set;
		set=NULL;
		count=0;
	}

	inline void addOneToRepre(long *repre) 
	{
		logl(9, ("RepairSetArray::addOneToRepre()"));
		RepairSet *sptr = set;
		long c = count;
		while (c--) 
		{
			if (sptr->repreSet) 
			{
				if (sptr->repre == repre) 
				{
					sptr->count++;
					return;
				}
			} 
			else 
			{
				sptr->repreSet = true;
				sptr->repre = repre;
				sptr->count = 1;
				return;
			}
			++sptr;
		}
	}

	inline bool initTargets(void) 
	{
		logl(5, ("RepairSetArray::initTargets()"));
		RepairSet *sptr = set;
		long c = count;
		bool ok = true;
		while (c--) 
		{
			if (!sptr->target.init(sptr->count))
			{
				ok=false;
			}
			++sptr;
		}
		return ok;
	}

	inline void addTargetPixel(long *repre, long x, long y) 
	{
		logl(9, ("RepairSetArray::addTargetPixel(*, %d, %d)", x, y));
		RepairSet *sptr = set;
		long c = count;
		while (c--) 
		{
			if (sptr->repreSet && (sptr->repre == repre)) 
			{
				sptr->target.addPixel(x, y);
				return;
			}
			++sptr;
		}
	}
};

typedef enum {	
	Standard, 
	DeBlend, 
	Alpha, 
	Color, 
	Repair, 
	Analyse
}	ePreview;

typedef struct MyFilterData {
	bool	func_deblend;	// DeBlend on/off
	bool	func_repair;	// Repair on/off
	bool	func_onframes;	// Filter active on frames/always
	IntControl	func_int;	// OnFrames intervals
	ePreview	func_preview;	// processing mode; clears on exit from config dialog

	MyBitmap	mask_deblend;	// masks
	MyBitmap	mask_alpha;
	MyBitmap	mask_color;
	MyBitmap	mask_repair;
	MyBitmap	mask_analyse;

	IntControl	anal_int;	// Analyse intervals
	ulong	*anal_sum;		// sum array
	ulong	*anal_sqrsum;	// square sum array
	ulong	anal_iter;		// number of iterations done
	long	anal_remain;	// estimate number of iterations that can be done; -1=no further iterations

	int 	repair_depth;	// repair depth, 10=1; >=1
	int 	repair_power;	// repair power, >=100
	bool	repair_interlaced;	// repair interlaced?
	int		repair_aspect;	// pixel aspect ratio - (100 = h=w, 10 = h=w/10, 1000 = h=w*10)

	RepairSetArray	repair_array;
	float	*repair_powertable; // power table
	long	repair_powerwidth;	// width of the power table

	int 	param_falloff;	// Falloff value, 0..32
	int 	param_alpha;	// Alpha to repair, 0..255
	int 	param_radius;	// Repair radius, 1..8
	int		param_shift;	// DeBlend shift, -100..100

	FilterActivation *fa;	
	HWND	hdlg;
	bool	samplesave; // if true, sampled image has to be saved instead of analysed
	bool	sampleall; // if true, all sampled images have to be analysed, not just those specified
	bool	unsample; // subtract instead of adding
	MyBitmap	sampledimage; // transfer space for saving frame
	bool	previewstate;	// true if preview window is open
} MyFilterData;

ScriptFunctionDef func_defs[]=
{
	{ (ScriptFunctionPtr)ScriptConfig, "Config", "0isssssiiii" }, 
    { NULL }, 
};

CScriptObject script_obj={
    NULL, func_defs
};

struct FilterDefinition filterDef = {

	NULL, NULL, NULL, 		// next, prev, module
	"DeLogo 1.3.2", 			// name
	"Removes opaque and alpha-blended logos", 
							// desc
	"Karel Suhajda", 		// maker
	NULL, 					// private_data
	sizeof(MyFilterData), 	// inst_data_size

	InitProc, 				// initProc
	DeinitProc, 				// deinitProc
	RunProc, 				// runProc
	ParamProc, 				// paramProc
	ConfigProc, 			// configProc
	StringProc, 				// stringProc
	StartProc, 				// startProc
	EndProc, 				// endProc
	&script_obj, 			// script_obj
	FssProc, 				// fssProc
};

void RedoRepair(MyFilterData *mfd, const FilterActivation *fa);
void CarveDeblend(MyFilterData *mfd, const FilterActivation *fa);

///////////////////////////////////////////////////////////////////////////

extern "C" int __declspec(dllexport) __cdecl VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" void __declspec(dllexport) __cdecl VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_ref;

///////////////////////////////////////////////////////////////////////////

int 
__declspec(dllexport) 
__cdecl 
VirtualdubFilterModuleInit2(
	FilterModule *fm, 
	const FilterFunctions *ff, 
	int& vdfd_ver, 
	int& vdfd_compat) 
{
	if (!(fd_ref = ff->addFilter(fm, &filterDef, sizeof(FilterDefinition))))
	{
		return 1;
	}

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

	return 0;
}

void 
__declspec(dllexport) 
__cdecl 
VirtualdubFilterModuleDeinit(
	FilterModule *fm, 
	const FilterFunctions *ff) 
{
	ff->removeFilter(fd_ref);
}

///////////////////////////////////////////////////////////////////////////

// Clear interval control

void DeinitInt(IntControl *ptr) 
{
	logl(5, ("DeinitInt() ->"));
	logl(3, ("Delete Interval(%d)", ptr->interval));
	delete[] ptr->interval;
	ptr->interval=NULL;
	ptr->count=0;
	logl(5, ("<- DeinitInt()"));
}

// Clear bitmap

void DeinitBitmap(MyBitmap *ptr) 
{
	logl(5, ("DeinitBitmap() ->"));
	logl(3, ("Delete Bitmap(%d)", ptr->bitmap));
	delete[] ptr->bitmap;
	ptr->bitmap=NULL;
	ptr->w=0;
	ptr->h=0;
	logl(5, ("<- DeinitBitmap()"));
}

// Clear repair queue

inline void DeinitRepair(MyFilterData *mfd) 
{
	logl(5, ("DeinitRepair() ->"));

	mfd->repair_array.deinit();

	logl(5, ("<- DeinitRepair()"));
}

// Clear statistics arrays

void DeinitStatistics(MyFilterData *mfd) 
{
	logl(5, ("DeinitStatistics() ->"));

	logl(3, ("Delete anal_sqrsum(%d)", mfd->anal_sqrsum));
	delete[] mfd->anal_sqrsum;
	mfd->anal_sqrsum=NULL;
	logl(3, ("Delete anal_sum(%d)", mfd->anal_sum));
	delete[] mfd->anal_sum;
	mfd->anal_sum=NULL;
	mfd->anal_iter=0;
	mfd->anal_remain=0;

	logl(5, ("<- DeinitStatistics()"));
}

// Removes/resets all variables used in config dialog only

void DeinitDialog(MyFilterData *mfd) 
{
	logl(5, ("DeinitDialog() ->"));

	DeinitStatistics(mfd);
	DeinitBitmap(&mfd->mask_analyse);
	DeinitBitmap(&mfd->sampledimage);
	DeinitInt(&mfd->anal_int);
	mfd->func_preview = Standard;

	logl(5, ("<- DeinitDialog()"));
}

///////////////////////////////////////////////////////////////////////////

int BrowseLoad(HWND hdlg, char *filename) 
{
	logl(5, ("BrowseLoad() ->"));

	OPENFILENAME ofn;

	ofn.lStructSize=sizeof(OPENFILENAME);
	ofn.hwndOwner=hdlg;
	ofn.lpstrFilter="Bitmap files (*.bmp)\0*.bmp\0All files (*.*)\0*.*\0\0";
	ofn.lpstrCustomFilter=NULL;
	ofn.nFilterIndex=1;
	ofn.lpstrFile=filename;
	ofn.nMaxFile=MAXSTR;
	ofn.lpstrInitialDir=NULL;
	ofn.lpstrTitle="Open bitmap";
	ofn.Flags=OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST | OFN_HIDEREADONLY;
	ofn.lpstrDefExt="bmp";
	ofn.lpstrFileTitle=NULL;

	logl(5, ("<- BrowseLoad()"));
	return GetOpenFileName(&ofn);
}

///////////////////////////////////////////////////////////////////////////

void LoadBitmapFile(MyBitmap *bitmap, HWND hdlg, BOOL warnings) 
{
	logl(5, ("LoadBitmapFile() ->"));

	BITMAPFILEHEADER bmfh;
	BITMAPINFOHEADER bmih;
	FILE *fin = NULL; // jfp 20/05/02
	Pixel32 *bufuk;
	long y, x;
	byte *bmpline, *bmpc;
	byte r, g, b;

	enum restype { 
		BMP_OK, // OK
		BMP_NONAME, // No name given
		BMP_NOFILE, // File can not be open
		BMP_WRONG, // File seems not be BMP file
		BMP_ERROR, // Error reading contents of file
		BMP_BADFMT, // Bad format of file - must be uncompressed 24-bit RGB
		BMP_MEMERR // Not enough memory to load image
	} result;

	DeinitBitmap(bitmap);

	result=BMP_OK;
	if (bitmap->path[0]==0) 
	{
		result=BMP_NONAME;
	}
	else if ((fin=fopen(bitmap->path, "rb"))==NULL) 
	{
		result=BMP_NOFILE;
	}
	else if (fread(&bmfh, 1, sizeof(bmfh), fin)!=sizeof(bmfh)) 
	{
		result=BMP_ERROR;
	}
	else if (fread(&bmih, 1, sizeof(bmih), fin)!=sizeof(bmih)) 
	{
		result=BMP_ERROR;
	}
	else if ((bmfh.bfType!=0x4D42) || (bmfh.bfReserved1!=0) || (bmfh.bfReserved2!=0)) 
	{
		result=BMP_WRONG;
	}
	else if ((bmih.biSize<sizeof(bmih)) || (bmih.biHeight==0) || (bmih.biWidth==0)) 
	{
		result=BMP_WRONG;
	}
	else if ((bmih.biBitCount!=24) || (bmih.biCompression!=BI_RGB) || (bmih.biPlanes!=1) || (bmih.biHeight<0)) 
	{
		result=BMP_BADFMT;
	}
	else if (fseek(fin, 0, SEEK_END)) 
	{
		result=BMP_ERROR;
// I just hope nobody will create a bitmap bigger than 2 GB ;-)
	}
	else if (ftell(fin)-bmfh.bfOffBits<(ulong)(3*bmih.biHeight*bmih.biWidth)) 
	{
		result=BMP_ERROR;
	}
	else if (fseek(fin, bmfh.bfOffBits, SEEK_SET)) 
	{
		result=BMP_ERROR;
	}
	else if (NULL == (bitmap->bitmap = new Pixel32[bmih.biHeight*bmih.biWidth])) 
	{
		result=BMP_MEMERR;
	}
	else if (!(bmpline = new byte[bmih.biWidth*3])) 
	{
		result=BMP_MEMERR;
	}
	else 
	{
		logl(3, ("New bitmap = %d", bitmap->bitmap));
		bitmap->w=bmih.biWidth;
		bitmap->h=bmih.biHeight;

		bufuk=bitmap->bitmap;
		y=bmih.biHeight;
		do 
		{
			if (fread(bmpline, (bmih.biWidth*3+3)&~3, 1, fin)!=1) 
			{
				result=BMP_WRONG;
				y=1; // to get out of the loop
			}
			else
			{
				bmpc=bmpline;
				x=bmih.biWidth;
				do 
				{
					r=*bmpc++;
					g=*bmpc++;
					b=*bmpc++;
					*bufuk++=(((long)b)<<16) | (((long)g)<<8) | ((long)r);
				} while (--x);
			}
		} while (--y);
		logl(3, ("Delete bmpline(%d)", bmpline));
		delete [] bmpline;
	}
	
	if (fin!=NULL) fclose(fin);

	if (result!=BMP_OK) 
	{
		DeinitBitmap(bitmap);
		if (warnings) 
		{
			switch (result) 
			{
			case BMP_NONAME:
				MessageBox(hdlg, "Missing bitmap file name", "Bitmap load error", MB_OK);
				break;
				 
			case BMP_NOFILE:
				MessageBox(hdlg, "Bitmap file not found", "Bitmap load error", MB_OK);
				break;

			case BMP_ERROR:
				MessageBox(hdlg, "Error reading bitmap file", "Bitmap load error", MB_OK);
				break;

			case BMP_WRONG:
				MessageBox(hdlg, "Selected file is corrupted or it is not a bitmap file", "Bitmap load error", MB_OK);
				break;

			case BMP_BADFMT:
				MessageBox(hdlg, "Only uncompressed 24-bit RGB bottom-up bitmaps are supported", "Bitmap load error", MB_OK);
				break;

			case BMP_MEMERR:
				MessageBox(hdlg, "Not enough memory to load bitmap file", "Bitmap load error", MB_OK);
				break;
			}
		}
	}
	logl(5, ("<- LoadBitmapFile()"));
}

///////////////////////////////////////////////////////////////////////////

void SaveBitmap(MyFilterData *mfd, MyBitmap *bmpptr) 
{
	logl(5, ("SaveBitmapFile() ->"));

	OPENFILENAME ofn;
	BITMAPFILEHEADER bmfh;
	BITMAPINFOHEADER bmih;
	FILE *fout;
	Pixel32 *bufuk;
	long y, x;
	byte *bmpline, *bmpc;
	bool result;

	if (bmpptr->bitmap==NULL) 
	{
		return;
	}

	bmfh.bfType=0x4D42;
	bmfh.bfSize=sizeof(bmfh)+sizeof(bmih)+bmpptr->w*bmpptr->h*3;
	bmfh.bfReserved1=0;
	bmfh.bfReserved2=0;
	bmfh.bfOffBits=sizeof(bmfh)+sizeof(bmih);

	bmih.biSize=sizeof(bmih);
	bmih.biWidth=bmpptr->w;
	bmih.biHeight=bmpptr->h;
	bmih.biPlanes=1;
	bmih.biBitCount=24;
	bmih.biCompression=BI_RGB;
	bmih.biSizeImage=0;
	bmih.biXPelsPerMeter=3780;
	bmih.biYPelsPerMeter=3780;
	bmih.biClrUsed=0;
	bmih.biClrImportant=0;


	ofn.lStructSize=sizeof(OPENFILENAME);
	ofn.hwndOwner=mfd->hdlg;
	ofn.lpstrFilter="Bitmap files (*.bmp)\0*.bmp\0\0";
	ofn.lpstrCustomFilter=NULL;
	ofn.nFilterIndex=1;
	ofn.lpstrFile=bmpptr->path;
	ofn.nMaxFile=MAXSTR;
	ofn.lpstrInitialDir=NULL;
	ofn.lpstrTitle="Save bitmap";
	ofn.Flags=OFN_PATHMUSTEXIST | OFN_HIDEREADONLY | OFN_OVERWRITEPROMPT;
	ofn.lpstrDefExt="bmp";
	ofn.lpstrFileTitle=NULL;

	if (NULL == (bmpline = new byte[(bmpptr->w*3+3)&~3])) 
	{
		MessageBox(mfd->hdlg, "Not enough memory to save bitmap!", "Bitmap save error", MB_OK);
	}
	else 
	{
		logl(3, ("New bmpline = %d", bmpline));
		if (GetSaveFileName(&ofn)) 
		{
			fout=fopen(ofn.lpstrFile, "wb");
			if (fout==NULL) 
			{
				MessageBox(mfd->hdlg, "Can't create file!", "Bitmap save error", MB_OK);
			}
			else 
			{
				result=(fwrite(&bmfh, sizeof(bmfh), 1, fout)>0);

				if (result) 
				{
					result=(fwrite(&bmih, 1, sizeof(bmih), fout)>0);
				}

				if (result) 
				{
					bufuk=bmpptr->bitmap;
					y=bmpptr->h;
					do 
					{
						x=bmpptr->w;
						bmpc=bmpline;
						do 
						{
							*bmpc++=(byte)(*bufuk & 255);
							*bmpc++=(byte)((*bufuk >> 8) & 255);
							*bmpc++=(byte)((*bufuk >> 16) & 255);
							bufuk++;
						} while (--x);
						if (fwrite(bmpline, (bmpptr->w*3+3)&~3, 1, fout)<1) 
						{
							result=false;
							y=1;
						}
					} while (--y);
				}
				if (!result) 
				{
					MessageBox(mfd->hdlg, "Error writing bitmap file", "Bitmap save error", MB_OK);
				}
				fclose(fout);
			}
		}
		logl(3, ("Delete bmpline(%d)", bmpline));
		delete [] bmpline;
	}
	logl(5, ("<- SaveBitmapFile()"));
}

///////////////////////////////////////////////////////////////////////////

// Checks if mask exists and has proper dimensions

bool MaskOK(const FilterActivation *fa, MyBitmap *bmpptr) 
{
	logl(5, ("<MaskOK()>"));
	return ((bmpptr->bitmap!=NULL)
			&& (bmpptr->h==fa->src.h)
			&& (bmpptr->w==fa->src.w));
}
///////////////////////////////////////////////////////////////////////////

// Goes through interval list, returns TRUE if value is in some interval

bool SearchIntervals(IntControl *ints, long value) 
{
	logl(5, ("SearchIntervals() ->"));

	Interval *intptr=ints->interval;

	long c=ints->count;

	while (c--) 
	{
		if (intptr->first<=value) 
		{
			if (intptr->last>=value) 
			{
				logl(5, ("<- SearchIntervals(): True"));
				return true;
			}
		} 
		else 
		{
			logl(5, ("<- SearchIntervals(): False"));
			return false;
		}
		intptr++;
	}

	logl(5, ("<- SearchIntervals(): False (2)"));
	return false;
}

///////////////////////////////////////////////////////////////////////////

// This function analyses intervals given in analstring
// if the intarray parameter is not NULL, puts intervals into the array and optimises them
// in both cases returns number of intervals found

int ParseIntervals(char *analstring, Interval *intarray) 
{
	logl(5, ("ParseIntervals() ->"));

	int noints=0;
	int strpos=0;
	int scanned;
	long k, l;

	while (analstring[strpos] == ' ') 
	{
		strpos++;
	}

	if (analstring[strpos] == '\0') 
	{
		if (intarray != NULL) 
		{
			intarray[0].first=0;
			intarray[0].last=0x7FFFFFFF;
		}
		noints = 1;
	}
	else
	{
		while (analstring[strpos] != '\0') 
		{
			logl(8,("ParseIntervals(): String to parse: %s", analstring[strpos]));
			if (analstring[strpos]!='-') 
			{
				if (sscanf(&analstring[strpos], "%ld%n", &k, &scanned)!=1) 
				{
					goto failed;
				}
				if (k<0) 
				{
					goto failed;
				}
				strpos+=scanned;
			} 
			else 
			{
				k=0;
			}
			logl(8,("ParseIntervals(): Starting number = %d", k));

			while (analstring[strpos] == ' ') 
			{
				strpos++;
			}

			if (analstring[strpos]!='-') 
			{
				l=k;
			} 
			else 
			{
				do
				{
					strpos++;
				} while (analstring[strpos] == ' ') ;

				if ((analstring[strpos]=='\0') || (analstring[strpos]==',')) 
				{
					l=0x7FFFFFFF;
				} 
				else 
				{
					if (sscanf(&analstring[strpos], "%ld%n", &l, &scanned)!=1) 
					{
						goto failed;
					}
					if (k>l) 
					{
						goto failed;
					}
					strpos+=scanned;
				}
			}

			while (analstring[strpos] == ' ') 
			{
				strpos++;
			}


			logl(8,("ParseIntervals(): Ending number = %d", l));

			if (analstring[strpos]==',') 
			{
				do 
				{
					strpos++;
				} while (analstring[strpos] == ' ');

				if (analstring[strpos]=='\0') 
				{
					goto failed;
				}
			} 
			else if (analstring[strpos]!='\0') 
			{
				goto failed;
			}

			if (intarray != NULL) 
			{
				intarray[noints].first=k;
				intarray[noints].last=l;
			}
			noints++;
		}

	// If array exists, lets sort it and merge intervals where possible
		if (intarray != NULL) 
		{
			int i=1;
			while (i < noints) 
			{
				if (i == 0) 
				{
					i++;
				} 
				else 
				{
					if (intarray[i].first<intarray[i-1].first) 
					{
						k=intarray[i].first;
						l=intarray[i].last;
						intarray[i].first=intarray[i-1].first;
						intarray[i].last=intarray[i-1].last;
						intarray[i-1].first=k;
						intarray[i-1].last=l;
						i--;
					} 
					else 
					{
	// these two are sorted. They may still overlap or connect though
						if (intarray[i].first<=intarray[i-1].last+1) 
						{
							if (intarray[i].last>intarray[i-1].last) 
							{
								intarray[i-1].last=intarray[i].last;
							}

							noints--;
							intarray[i].first=intarray[noints].first;
							intarray[i].last=intarray[noints].last;
							i--;
						} 
						else 
						{
							i++;
						}
					}
				}
			}
		}
	}
	logl(5, ("<- ParseIntervals(): %d", noints));
	return noints;

failed:
	logl(5, ("<- ParseIntervals(): Failure"));
	return 0;
}

///////////////////////////////////////////////////////////////////////////

// Re-create interval list from string

void CreateIntervals(IntControl *intctrl) 
{
	logl(5, ("CreateIntervals() ->"));

	DeinitInt(intctrl);
	intctrl->count=ParseIntervals(intctrl->text, NULL);
	if (intctrl->count>0) 
	{
		intctrl->interval=new Interval[intctrl->count];
		logl(3, ("New Interval = %d", intctrl->interval));
		if (intctrl->interval!=NULL) 
		{
			intctrl->count=ParseIntervals(intctrl->text, intctrl->interval);
		} 
		else 
		{
			intctrl->count=0;
		}
	}

	logl(5, ("<- CreateIntervals()"));
}

///////////////////////////////////////////////////////////////////////////

double AspectWidth(MyFilterData *mfd) 
{
	if (mfd->repair_aspect>=0) 
	{
		return 1;
	}
	else 
	{
		return pow(2, -(double)mfd->repair_aspect/30);
	}
}

double AspectHeight(MyFilterData *mfd) 
{
	if (mfd->repair_aspect<=0) 
	{
		return 1;
	}
	else 
	{
		return pow(2, (double)mfd->repair_aspect/30);
	}
}

///////////////////////////////////////////////////////////////////////////

int InitProc(FilterActivation *fa, const FilterFunctions *ff) 
{
	logl(5, ("InitProc()"));
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->func_deblend = true;
	mfd->func_repair = true;
	mfd->func_onframes = true;
	mfd->func_preview = Standard;
	mfd->repair_depth = DEFAULTDEPTH;
	mfd->repair_interlaced = false;
	mfd->repair_aspect = DEFAULTASPECT;
	mfd->repair_power = DEFAULTPOWER;
	mfd->param_falloff = DEFAULTFALLOFF;
	mfd->param_alpha = DEFAULTALPHA;
	mfd->param_radius = DEFAULTRADIUS;
	mfd->param_shift = DEFAULTSHIFT;
	return 0;
}

///////////////////////////////////////////////////////////////////////////

void DeinitProc(FilterActivation *fa, const FilterFunctions *ff) 
{
	logl(5, ("DeinitProc()"));
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	DeinitDialog(mfd);

	DeinitBitmap(&mfd->mask_alpha);
	DeinitBitmap(&mfd->mask_deblend);
	DeinitBitmap(&mfd->mask_repair);
	DeinitBitmap(&mfd->mask_color);

	DeinitInt(&mfd->func_int);

	DeinitRepair(mfd);
}

///////////////////////////////////////////////////////////////////////////

void Slashes(char *string) 
{
	logl(5, ("Slashes()"));
	while (*string != '\0') 
	{
		if (*string=='\\') 
		{
			*string='/';
		}
		string++;
	}
}

///////////////////////////////////////////////////////////////////////////

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) 
{
	logl(5, ("ScriptConfig()"));
    FilterActivation *fa = (FilterActivation *)lpVoid;
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	DeinitBitmap(&mfd->mask_deblend);
	DeinitBitmap(&mfd->mask_alpha);
	DeinitBitmap(&mfd->mask_color);
	DeinitBitmap(&mfd->mask_repair);

	mfd->func_deblend=true;
	mfd->func_repair=true;

	mfd->func_onframes = !!argv[0].asInt();

	strncpy(mfd->func_int.text, *argv[1].asString(), MAXSTR);
	CreateIntervals(&mfd->func_int);

	strncpy(mfd->mask_deblend.path, *argv[2].asString(), MAXSTR);
	LoadBitmapFile(&mfd->mask_deblend, 0, false);
	if ((mfd->mask_deblend.path[0]) && (mfd->mask_deblend.bitmap==NULL)) 
	{
		EXT_SCRIPT_ERROR(FCALL_UNKNOWN_STR);
	}

	strncpy(mfd->mask_alpha.path, *argv[3].asString(), MAXSTR);
	LoadBitmapFile(&mfd->mask_alpha, 0, false);
	if ((mfd->mask_alpha.path[0]) && (mfd->mask_alpha.bitmap==NULL)) 
	{
		EXT_SCRIPT_ERROR(FCALL_UNKNOWN_STR);
	}

	strncpy(mfd->mask_color.path, *argv[4].asString(), MAXSTR);
	LoadBitmapFile(&mfd->mask_color, 0, false);
	if ((mfd->mask_color.path[0]) && (mfd->mask_color.bitmap==NULL)) 
	{
		EXT_SCRIPT_ERROR(FCALL_UNKNOWN_STR);
	}

	strncpy(mfd->mask_repair.path, *argv[5].asString(), MAXSTR);
	LoadBitmapFile(&mfd->mask_repair, 0, false);
	if ((mfd->mask_repair.path[0]) && (mfd->mask_repair.bitmap==NULL)) 
	{
		EXT_SCRIPT_ERROR(FCALL_UNKNOWN_STR);
	}

	mfd->repair_depth=argv[6].asInt();
	mfd->repair_power=argv[7].asInt();
	mfd->repair_interlaced=!!argv[8].asInt();
	mfd->repair_aspect=argv[9].asInt();

	if ((mfd->repair_depth<MINDEPTH) 
		|| (mfd->repair_depth>MAXDEPTH)
		|| (mfd->repair_power<MINPOWER) 
		|| (mfd->repair_power>MAXPOWER)
		|| (mfd->repair_aspect<MINASPECT)
		|| (mfd->repair_aspect>MAXASPECT)) 
	{
		EXT_SCRIPT_ERROR(FCALL_OUT_OF_RANGE);
	}
}

///////////////////////////////////////////////////////////////////////////

bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) 
{
	logl(5, ("FssProc()"));
	MyFilterData *mfd=(MyFilterData *)fa->filter_data;

	// Produce warning about unsaved masks
	bool dorepair = 
		(mfd->func_repair
		&& MaskOK(fa, &mfd->mask_repair));

	bool badrepair =
		(dorepair
		&& (mfd->mask_repair.path[0] == '\0'));

	bool dodeblend = 
		(mfd->func_deblend
		&& MaskOK(fa, &mfd->mask_alpha)
		&& MaskOK(fa, &mfd->mask_color)
		&& MaskOK(fa, &mfd->mask_deblend));

	bool baddeblend =
		(dodeblend
		&& ((mfd->mask_alpha.path[0] == '\0')
			|| (mfd->mask_color.path[0] == '\0')
			|| (mfd->mask_deblend.path[0] == '\0')));

	if (badrepair && baddeblend) 
	{
		MessageBox(NULL, "Masks for the Repair and DeBlend functions are not saved; the DeLogo filter will have no function.", "DeLogo Batch Warning", MB_OK);
	}
	else if (badrepair) 
	{
		if (dodeblend) 
		{
			MessageBox(NULL, "The Repair mask is not saved; the DeLogo filter Repair function will be suppressed.", "DeLogo Batch Warning", MB_OK);
		}
		else 
		{
			MessageBox(NULL, "The Repair mask is not saved; the DeLogo filter will have no function.", "DeLogo Batch Warning", MB_OK);
		}
	}
	else if (baddeblend) 
	{
		if (dorepair) 
		{
			MessageBox(NULL, "The DeBlend function masks are not saved; the DeLogo filter DeBlend function will be suppressed.", "DeLogo Batch Warning", MB_OK);
		}
		else 
		{
			MessageBox(NULL, "The DeBlend function masks are not saved; the DeLogo filter will have no function.", "DeLogo Batch Warning", MB_OK);
		}
	}

	Slashes(mfd->mask_deblend.path);
	Slashes(mfd->mask_alpha.path);
	Slashes(mfd->mask_color.path);
	Slashes(mfd->mask_repair.path);

	if (dodeblend && !baddeblend)
	{
		// DeBlend OK
		if (dorepair && !badrepair)
		{
			// Deblend and Repair
			return(0<_snprintf(buf, buflen, "Config(%d, \"%s\", \"%s\", \"%s\", \"%s\", \"%s\", %d, %d, %d, %d)", 
				mfd->func_onframes, mfd->func_int.text, 
				mfd->mask_deblend.path, mfd->mask_alpha.path, mfd->mask_color.path, 
				mfd->mask_repair.path, mfd->repair_depth, mfd->repair_power, !!mfd->repair_interlaced, mfd->repair_aspect));
		}
		else 
		{
			// Deblend only
			return(0<_snprintf(buf, buflen, "Config(%d, \"%s\", \"%s\", \"%s\", \"%s\", \"%s\", %d, %d, %d, %d)", 
				mfd->func_onframes, mfd->func_int.text, 
				mfd->mask_deblend.path, mfd->mask_alpha.path, mfd->mask_color.path, 
				"", mfd->repair_depth, mfd->repair_power, !!mfd->repair_interlaced, mfd->repair_aspect));
		}
	}
	else 
	{
		if (dorepair && !badrepair)
		{
			// Repair only
			return(0<_snprintf(buf, buflen, "Config(%d, \"%s\", \"%s\", \"%s\", \"%s\", \"%s\", %d, %d, %d, %d)", 
				mfd->func_onframes, mfd->func_int.text, 
				"", "", "", 
				mfd->mask_repair.path, mfd->repair_depth, mfd->repair_power, !!mfd->repair_interlaced, mfd->repair_aspect));
		}
		else 
		{
			// no function
			return(false);
		}
	}
}

///////////////////////////////////////////////////////////////////////////

int StartProc(FilterActivation *fa, const FilterFunctions *ff) 
{
	logl(5, ("StartProc()"));
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	RedoRepair(mfd, fa);
	CarveDeblend(mfd, fa);

	return 0;
}

///////////////////////////////////////////////////////////////////////////

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str) 
{
	logl(5, ("StringProc()"));
    MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	if (MaskOK(fa, &mfd->mask_alpha) 
		&& MaskOK(fa, &mfd->mask_color) 
		&& MaskOK(fa, &mfd->mask_deblend) 
		&& mfd->func_deblend) 
	{
		if (MaskOK(fa, &mfd->mask_repair) && mfd->func_repair) 
		{
			strcpy(str, " (DeBlend & Repair)");
		}
		else 
		{
			strcpy(str, " (DeBlend)");
		}
	}
	else 
	{
		if (MaskOK(fa, &mfd->mask_repair) && mfd->func_repair) 
		{
			strcpy(str, " (Repair)");
		}
		else 
		{
			strcpy(str, " (no function)");
		}
	}
}

///////////////////////////////////////////////////////////////////////////

void DisplayMask(const FilterActivation *fa, MyBitmap *mask) 
{
	logl(5, ("DisplayMask()"));
	if (MaskOK(fa, mask)) 
	{
		Pixel32 *src=mask->bitmap;
		Pixel32 *dst=(Pixel32 *)fa->dst.data;
		long h = fa->src.h;
		do 
		{
			long w = fa->src.w;
			do 
			{
				*dst++ = *src++;
			} while (--w);
			dst = (Pixel32 *)((char *)dst + fa->dst.modulo);
		} while (--h);
	}
}

int RunProc(const FilterActivation *fa, const FilterFunctions *ff) 
{
	logl(5, ("RunProc()"));
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	PixDim w, h;
	Pixel32 *src, *srca, *srcc, *srcd, *dst;
	long ra, ga, ba, rc, gc, bc, rx, gx, bx, ry, gy, by, d, p;

	switch(mfd->func_preview) 
	{
	case DeBlend:
		DisplayMask(fa, &mfd->mask_deblend);
		break;

	case Alpha:
		DisplayMask(fa, &mfd->mask_alpha);
		break;

	case Color:
		DisplayMask(fa, &mfd->mask_color);
		break;

	case Repair:
		DisplayMask(fa, &mfd->mask_repair);
		break;

	case Analyse:
		if (MaskOK(fa, &mfd->mask_analyse)) 
		{
			src=mfd->mask_analyse.bitmap;
			dst=(Pixel32 *)fa->dst.data;
			h = fa->src.h;
			do 
			{
				w = fa->src.w;
				do 
				{
					d=*src++;
					if ((d & 3)==1) 
					{
						*dst++ = 0;
					}
					else 
					{
						*dst++ = d;
					}
				} while (--w);
				dst = (Pixel32 *)((char *)dst + fa->dst.modulo);
			} while (--h);
		}
		break;

	case Standard:
		if ((mfd->func_int.count==0) // process, if specified intervals are invalid
			|| !mfd->func_onframes // process, if intervals are switched off
			|| SearchIntervals(&mfd->func_int, fa->pfsi->lCurrentSourceFrame)) // process, if current frame is inside an interval
		{
			// Phase I: deblend
			// DeBlend must be on and we must have correct masks
			if (mfd->func_deblend && MaskOK(fa, &mfd->mask_deblend)
				&& MaskOK(fa, &mfd->mask_alpha)
				&& MaskOK(fa, &mfd->mask_color))
			{
				src=(Pixel32 *)fa->src.data;
				dst=(Pixel32 *)fa->dst.data;
				srca=mfd->mask_alpha.bitmap;
				srcc=mfd->mask_color.bitmap;
				srcd=mfd->mask_deblend.bitmap;
				h = fa->src.h;
				do 
				{
					w = fa->src.w;
					do 
					{
						if (d=(*srcd++ & 0xff0000)) 
						{
							d>>=16;
							p=*srca++;
							ra=(p & 0xff0000)>>16;
							ga=(p & 0x00ff00)>>8;
							ba=(p & 0x0000ff);
							p=*srcc++;
							rc=(p & 0xff0000)>>16;
							gc=(p & 0x00ff00)>>8;
							bc=(p & 0x0000ff);
							p=*src++;
							rx=(p & 0xff0000)>>16;
							gx=(p & 0x00ff00)>>8;
							bx=(p & 0x0000ff);
							
							if (ra) 
							{
								ry=rc+((rx-rc)*ALPHATOP)/ra;
								if (ry>255) 
								{
									ry=255;
								}
								else if (ry<0) 
								{
									ry=0;
								}
							}
							ry=(ry*d+rx*(255-d))/255;
							
							if (ga) 
							{
								gy=gc+((gx-gc)*ALPHATOP)/ga;
								if (gy>255) 
								{
									gy=255;
								}
								else if (gy<0) 
								{
									gy=0;
								}
							}
							gy=(gy*d+gx*(255-d))/255;
							
							if (ba) 
							{
								by=bc+((bx-bc)*ALPHATOP)/ba;
								if (by>255) 
								{
									by=255;
								}
								else if (by<0) 
								{
									by=0;
								}
							}
							by=(by*d+bx*(255-d))/255;
							
							*dst++=(ry<<16)|(gy<<8)|by;
						}
						else 
						{
							srca++;
							srcc++;
							src++;
							dst++;
						}
					} while (--w);
					dst = (Pixel32 *)((char *)dst + fa->dst.modulo);
					src = (Pixel32 *)((char *)src + fa->src.modulo);
				} while (--h);
			}
			// Phase II: Repair
			// Must be on and mask ready and processed
			logl(5, ("repair_begin"));
			if (mfd->func_repair
				&& MaskOK(fa, &mfd->mask_repair)
				&& (mfd->repair_array.ok()))
			{
				RepairSet *set=mfd->repair_array.set;
				long i=mfd->repair_array.count;
				// for each repair set
				while (i--) 
				{
					logl(5, ("repair: begin set"));
					long t=set->target.count;
					if ((t>0) && (set->source.count>0)) 
					{
						logl(5, ("repair: begin set 2"));
						TargetPixel *tptr = set->target.pixels;
						while (t--) 
						{

							float dr = 0;
							float dg = 0;
							float db = 0;
							float ww = 0;

							long s=set->source.count;
							SourcePixel *sptr = set->source.pixels;
							
							while (s--) 
							{
								float wg=mfd->repair_powertable[abs(tptr->y-sptr->y)*mfd->repair_powerwidth+abs(tptr->x-sptr->x)];

								p=*((Pixel32 *)(((char *)fa->src.data)+sptr->offset));

								ww+=wg;
								dr+=((p & 0xff0000)>>16)*wg;
								dg+=((p & 0x00ff00)>>8)*wg;
								db+=(p & 0x0000ff)*wg;
								
								sptr++;
							}

							*((Pixel32 *)(((char *)fa->src.data)+tptr->x*sizeof(Pixel32)+tptr->y*fa->src.pitch))
								= (((long)(dr/ww)) << 16) | (((long)(dg/ww)) << 8) | ((long)(db/ww));
							tptr++;

						}
					}
					set++;
				}
			}
		}
		break;
	}	
	return 0;
}

///////////////////////////////////////////////////////////////////////////

int EndProc(FilterActivation *fa, const FilterFunctions *ff) 
{
	logl(5, ("EndProc()"));
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	return 0;
}

///////////////////////////////////////////////////////////////////////////

long ParamProc(FilterActivation *fa, const FilterFunctions *ff) 
{
	logl(5, ("ParamProc()"));
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	fa->dst.offset  = fa->src.offset;
	fa->dst.modulo  = fa->src.modulo;
	fa->dst.pitch   = fa->src.pitch;

	return 0;
}

///////////////////////////////////////////////////////////////////////////

void ShowOneMask(HWND hdlg, MyBitmap *bmpptr, int NameControlId, int DimControlId) 
{
	logl(8, ("ShowOneMask() ->"));
	char string[MAXSTR];

	if (bmpptr->bitmap == NULL) 
	{
		SetDlgItemText(hdlg, NameControlId, "- none -");
		SetDlgItemText(hdlg, DimControlId, "");
	} 
	else 
	{
		sprintf(string, "%u x %u", bmpptr->w, bmpptr->h);
		SetDlgItemText(hdlg, DimControlId, string);
		if (bmpptr->path[0] == 0) 
		{
			SetDlgItemText(hdlg, NameControlId, "- generated -");
		} 
		else 
		{
			SetDlgItemText(hdlg, NameControlId, bmpptr->path);
		}
	}
	logl(8, ("<- ShowOneMask()"));
}

///////////////////////////////////////////////////////////////////////////

// Displays all masks properties

void ShowMaskProperties(MyFilterData *mfd) 
{
	logl(5, ("ShowMaskProperties()"));
	ShowOneMask(mfd->hdlg, &mfd->mask_deblend, IDC_FILE_DEBLEND, IDC_DIM_DEBLEND); 
	ShowOneMask(mfd->hdlg, &mfd->mask_alpha, IDC_FILE_ALPHA, IDC_DIM_ALPHA);	
	ShowOneMask(mfd->hdlg, &mfd->mask_color, IDC_FILE_COLOR, IDC_DIM_COLOR);	
	ShowOneMask(mfd->hdlg, &mfd->mask_repair, IDC_FILE_REPAIR, IDC_DIM_REPAIR);	
	ShowOneMask(mfd->hdlg, &mfd->mask_analyse, IDC_FILE_ANALYSE, IDC_DIM_ANALYSE); 
}

///////////////////////////////////////////////////////////////////////////

// Load of a mask

void LoadMask(HWND hdlg, MyBitmap *bmpptr, bool warnings) 
{
	logl(5, ("LoadMask()"));
	if (BrowseLoad(hdlg, bmpptr->path)) 
	{
		LoadBitmapFile(bmpptr, hdlg, warnings);
	}
}

void LoadDeblend(MyFilterData *mfd, bool warnings) 
{
	logl(5, ("LoadDeblend()"));
	LoadMask(mfd->hdlg, &mfd->mask_deblend, warnings);
}

void LoadAlpha(MyFilterData *mfd, bool warnings) 
{
	logl(5, ("LoadAlpha()"));
	LoadMask(mfd->hdlg, &mfd->mask_alpha, warnings);
}

void LoadColor(MyFilterData *mfd, bool warnings) 
{
	logl(5, ("LoadColor()"));
	LoadMask(mfd->hdlg, &mfd->mask_color, warnings);
}

void LoadRepair(MyFilterData *mfd, bool warnings) 
{
	logl(5, ("LoadRepair()"));
	LoadMask(mfd->hdlg, &mfd->mask_repair, warnings);
}

///////////////////////////////////////////////////////////////////////////

// Analyse mask preprocess
// omit all red, green and blue pixels
// for all other pixels set value to 00XXXXXX XXXXXXXX YYYYYYYY YYYYYY01
// where X...X is 0..MAXFALLOFF-1 distance to the closest red pixel (0 if greater than MAXFALLOFF)
// and Y...Y is 0..MAXFALLOFF-1 distance to the closest blue pixel (0 if greater than MAXFALLOFF)

void PreprocessAnalyse(MyFilterData *mfd) 
{
	logl(5, ("PreprocessAnalyse()"));
	int x, y, xl, xh, yl, yh, x1, y1;
	Pixel32 *ptr, *ptr1, *ptr2;
	Pixel32 p, p2;
	long r, b;

	// pass 1 - change all non-red/green/blue pixels to default value
	ptr=mfd->mask_analyse.bitmap;
	for (y=0;y<mfd->mask_analyse.h;y++) 
	{
		for (x=0;x<mfd->mask_analyse.w;x++) 
		{
			switch(*ptr) 
			{
			case 0xff0000:
			case 0x00ff00:
			case 0x0000ff:
				break;

			default:
				*ptr=0x3ffffffd; //00111111 11111111 11111111 11111101
			}
			ptr++;
		}
	}

	// pass 2 - calculate minimum distances
	ptr=mfd->mask_analyse.bitmap;
	for (y=0;y<mfd->mask_analyse.h;y++) 
	{
		for (x=0;x<mfd->mask_analyse.w;x++) 
		{
			switch (p = *ptr) 
			{
			case 0xff0000:
			case 0x0000ff:
				// found red or blue pixel
				// process it only if there is a neighbor of another color
				if (((x>0) && (*(ptr-1)!=p))
					|| ((x<mfd->mask_analyse.w-1) && (*(ptr+1)!=p))
					|| ((y>0) && (*(ptr-mfd->mask_analyse.w)!=p))
					|| ((y<mfd->mask_analyse.h-1) && (*(ptr+mfd->mask_analyse.w)!=p))) 
				{
					// calculate appropriate diameter
					xl=(x>MAXFALLOFF)?-MAXFALLOFF:-x;
					xh=(x+MAXFALLOFF<mfd->mask_analyse.w)?MAXFALLOFF:mfd->mask_analyse.w-x-1;
					yl=(y>MAXFALLOFF)?-MAXFALLOFF:-y;
					yh=(y+MAXFALLOFF<mfd->mask_analyse.h)?MAXFALLOFF:mfd->mask_analyse.h-y-1;

					ptr1=ptr+xl+yl*mfd->mask_analyse.w;
					for (y1=yl;y1<=yh;y1++) 
					{
						ptr2=ptr1;
						for (x1=xl;x1<=xh;x1++) 
						{
							switch(p2=*ptr2) 
							{
							case 0xff0000:
							case 0x00ff00:
							case 0x0000ff:
								// don't change red, green or blue
								break;

							default:
								// calculate the distance
								r=x1*x1+y1*y1;
								if (p == 0xff0000) 
								{
									b = (p2 & 0x3FFF0000) >> 16;
									if (b>r) 
									{
										*ptr2 = r<<16 | (p2 & 0xc000ffff);
									}
								}
								else 
								{
									b = (p2 & 0x0000fffc) >> 2;
									if (b>r) 
									{
										*ptr2 = r<<2 | (p2 & 0xffff0003);
									}
								}
							}
							ptr2++;
						}
						ptr1+=mfd->mask_analyse.w;
					}
				}
			}
			ptr++;
		}
	}

	// pass 3 - calculate square roots and replace by zero where too big
	ptr=mfd->mask_analyse.bitmap;
	for (y=0;y<mfd->mask_analyse.h;y++) 
	{
		for (x=0;x<mfd->mask_analyse.w;x++) 
		{
			switch(p=*ptr) 
			{
			case 0xff0000:
			case 0x00ff00:
			case 0x0000ff:
				break;

			default:
				r = (p & 0x3fff0000) >> 16;
				b = (p & 0x0000fffc) >> 2;
				if (r>=MAXFALLOFF*MAXFALLOFF) 
				{
					r=0;
				}
				else 
				{
					r=(long)(FALLOFFCOEF*sqrt(((double)r)));
				}

				if (b>=MAXFALLOFF*MAXFALLOFF) 
				{
					b=0;
				}
				else 
				{
					b=(long)(FALLOFFCOEF*sqrt(((double)b)));
				}
				*ptr=(r<<16) | (b<<2) | 1;
			}
			ptr++;
		}
	}
}

///////////////////////////////////////////////////////////////////////////

// Load analysis mask

bool LoadAnalyse(MyFilterData *mfd) 
{
	logl(5, ("LoadAnalyse() ->"));
	bool ok;

	ok=false;
	if (BrowseLoad(mfd->hdlg, mfd->mask_analyse.path)) 
	{
		LoadBitmapFile(&mfd->mask_analyse, mfd->hdlg, true);
		if (mfd->mask_analyse.bitmap!=NULL) 
		{
			if ((mfd->mask_analyse.w!=mfd->fa->src.w) || (mfd->mask_analyse.h!=mfd->fa->src.h)) 
			{
				char string[MAXSTR];
				sprintf(string, "Mask dimensions (%d x %d) don't correspond to video (%d x %d)", 
					mfd->mask_analyse.w, mfd->mask_analyse.h, mfd->fa->src.w, mfd->fa->src.h); 					
				DeinitBitmap(&mfd->mask_analyse);
				MessageBox(mfd->hdlg, string, "Mask load error", MB_OK);
			}
			else 
			{
				// Check if mask contains at least one repair point and at least one reference point
				long i;
				long count=mfd->mask_analyse.w*mfd->mask_analyse.h;
				long repcnt=0;
				long refcnt=0;

				for (i=0;(i<count) && ((repcnt==0) || (refcnt==0));i++) 
				{
					switch (mfd->mask_analyse.bitmap[i]) 
					{
					case 0xff0000:
						repcnt++;
						break;

					case 0x0000ff:
					case 0x00ff00:
						break;

					default:
						refcnt++;
						break;
					}
				}
				if ((repcnt==0) || (refcnt==0)) 
				{
					DeinitBitmap(&mfd->mask_analyse);
					if (repcnt==0) 
					{
						MessageBox(mfd->hdlg, "Mask must contain at least one repair pixel (RED)", "Mask load error", MB_OK);
					}
					else 
					{
						MessageBox(mfd->hdlg, "Mask must contain at least one reference pixel (not RED/GREEN/BLUE)", "Mask load error", MB_OK);
					}
				}
				else 
				{
					ok=true;
				}
			}
		}
	}

	if (ok) 
	{
		PreprocessAnalyse(mfd);
	}

	logl(5, ("<- LoadAnalyse(): %s", ok ? "True" : "False"));
	return ok;
}

///////////////////////////////////////////////////////////////////////////

// Refresh slider value window

bool RefreshSlider(HWND hdlg, int SliderControlId, int ValueControlId, int *ValuePtr, bool ForceRefresh, int power) 
{
	logl(5, ("RefreshSlider() ->"));

	bool result = false;

	int NewValue = SendMessage(GetDlgItem(hdlg, SliderControlId), TBM_GETPOS, 0, 0);

	if (ForceRefresh || (NewValue != *ValuePtr)) 
	{
		char string[MAXSTR];
		*ValuePtr=NewValue;
		sprintf(string, "%.*f", power, NewValue/pow(10, power));
		SetDlgItemText(hdlg, ValueControlId, string);
		result = true;
	}
	
	logl(5, ("<- RefreshSlider(): %s", result ? "True" : "False"));
	return result;
}

///////////////////////////////////////////////////////////////////////////

bool RefreshDepth(MyFilterData *mfd, bool forcerefresh) 
{
	logl(8, ("<RefreshDepth()>"));
	return RefreshSlider(
		mfd->hdlg, 
		IDC_REPAIR_DEPTH, 
		IDC_VAL_DEPTH, 
		&mfd->repair_depth, 
		forcerefresh, 
		PRECDEPTH);
}

bool RefreshPower(MyFilterData *mfd, bool forcerefresh) 
{
	logl(8, ("<RefreshPower()>"));
	return RefreshSlider(
		mfd->hdlg, 
		IDC_REPAIR_POWER, 
		IDC_VAL_POWER, 
		&mfd->repair_power, 
		forcerefresh, 
		PRECPOWER);
}

bool RefreshFalloff(MyFilterData *mfd, bool forcerefresh) 
{
	logl(8, ("<RefreshFalloff()>"));
	return RefreshSlider(
		mfd->hdlg, 
		IDC_PARAM_FALLOFF, 
		IDC_VAL_FALLOFF, 
		&mfd->param_falloff, 
		forcerefresh, 
		PRECFALLOFF);
}

bool RefreshAlpha(MyFilterData *mfd, bool forcerefresh) 
{
	logl(8, ("<RefreshAlpha()>"));
	return RefreshSlider(
		mfd->hdlg, 
		IDC_PARAM_ALPHA, 
		IDC_VAL_ALPHA, 
		&mfd->param_alpha, 
		forcerefresh, 
		PRECALPHA);
}

bool RefreshRadius(MyFilterData *mfd, bool forcerefresh) 
{
	logl(8, ("<RefreshRadius()>"));
	return RefreshSlider(
		mfd->hdlg, 
		IDC_PARAM_RADIUS, 
		IDC_VAL_RADIUS, 
		&mfd->param_radius, 
		forcerefresh, 
		PRECRADIUS);
}

bool RefreshShift(MyFilterData *mfd, bool forcerefresh) 
{
	logl(8, ("<RefreshShift()>"));
	return RefreshSlider(
		mfd->hdlg, 
		IDC_PARAM_SHIFT, 
		IDC_VAL_SHIFT, 
		&mfd->param_shift, 
		forcerefresh, 
		PRECSHIFT);
}

bool RefreshAspect(MyFilterData *mfd, bool forcerefresh) 
{
	logl(8, ("RefreshAspect() ->"));
	int NewValue=SendMessage(GetDlgItem(mfd->hdlg, IDC_REPAIR_ASPECT), TBM_GETPOS, 0, 0);

	if (forcerefresh || (NewValue != mfd->repair_aspect)) 
	{
		char string[MAXSTR];
		mfd->repair_aspect=NewValue;
		sprintf(string, "%.4g : %.4g", AspectWidth(mfd), AspectHeight(mfd));
		SetDlgItemText(mfd->hdlg, IDC_VAL_ASPECT, string);
		logl(8, ("<- RefreshAspect(): True"));
		return true;
	}

	logl(8, ("<- RefreshAspect(): False"));
	return false;
}

///////////////////////////////////////////////////////////////////////////

// Prepare statistics arrays
// If not enough memory, delete them
// Don't touch anything if they already exist

void PrepareStatistics(MyFilterData *mfd) 
{
	logl(5, ("PrepareStatistics()"));
	long i, size;
	size=mfd->fa->src.w*mfd->fa->src.h*3;

	if (mfd->anal_sum==NULL) 
	{
		mfd->anal_sum = new ulong[size];
		logl(3, ("New anal_sum = %d", mfd->anal_sum));
		mfd->anal_iter=0;
	}

	if (mfd->anal_sqrsum==NULL) 
	{
		mfd->anal_sqrsum = new ulong[size];
		logl(3, ("New anal_sqrsum = %d", mfd->anal_sqrsum));
		mfd->anal_iter=0;
	}

	if (mfd->anal_iter==0) 
	{
		if ((mfd->anal_sum!=NULL) && (mfd->anal_sqrsum!=NULL)) 
		{
			mfd->anal_remain=0;
			for (i=0;i<size;i++) 
			{
				mfd->anal_sum[i]=0;
			}

			for (i=0;i<size;i++) 
			{
				mfd->anal_sqrsum[i]=0;
			}
		}
		else 
		{
			DeinitStatistics(mfd);
		}
	}
}

///////////////////////////////////////////////////////////////////////////

// Create empty bitmap

bool CreateEmptyMask(MyFilterData *mfd, MyBitmap *bmpptr) 
{
	logl(5, ("CreateEmptyMask()"));
	if ((bmpptr->bitmap!=NULL) && ((bmpptr->w!=mfd->fa->src.w) || (bmpptr->h!=mfd->fa->src.h))) 
	{
		DeinitBitmap(bmpptr);
	}
	bmpptr->w=mfd->fa->src.w;
	bmpptr->h=mfd->fa->src.h;

	if (bmpptr->bitmap==NULL) 
	{
		bmpptr->bitmap=new Pixel32[bmpptr->w*bmpptr->h];
		logl(3, ("New bitmap = %d", bmpptr->bitmap));
	}
	bmpptr->path[0]=0;
	return (bmpptr->bitmap!=NULL);
}

///////////////////////////////////////////////////////////////////////////

// This function analyses frame
// It has two purposes:
// - prepare frame for save if "Save frame" button was pressed
// - add frame values to statistics arrays if "Sample frame" or "Sample video"

void SampleCallback(VFBitmap *bitmap, long lFrame, long lCount, void *pData) 
{
	logl(5, ("SampleCallback() ->"));
	MyFilterData *mfd=(MyFilterData *)pData;
	int h, w;
	Pixel32 *pixelptr, *src;
	ulong *dst, *dstsqr;
	Pixel32 pixel;
	ulong r, g, b;
	long m, max;

	if (mfd->samplesave) 
	{
		if (CreateEmptyMask(mfd, &mfd->sampledimage)) 
		{
			pixelptr=mfd->sampledimage.bitmap;

			src=bitmap->data;
			h = bitmap->h;
			do 
			{
				w = bitmap->w;
				do 
				{
					*pixelptr++=*src++;
				} while (--w);
		    src = (Pixel32 *)((char *)src + bitmap->modulo);
		    } while (--h);
		}
	}
	else 
	{
		if ((mfd->anal_sum != NULL) 
			&& (mfd->anal_sqrsum != NULL)
		    && (mfd->anal_remain >= 0) 
			&& (mfd->sampleall || SearchIntervals(&mfd->anal_int, lFrame))) 
		{
			logl(3, ("SampleCallback(): Sampling frame %d", lFrame));
			if (mfd->unsample) 
			{
				dst=mfd->anal_sum;
				dstsqr=mfd->anal_sqrsum;
				src=bitmap->data;
				h = bitmap->h;
				do 
				{
					w = bitmap->w;
					do 
					{
						pixel=*src++;
						r=(pixel & 0xff0000)>>16;
						g=(pixel & 0x00ff00)>>8;
						b=(pixel & 0x0000ff);						
						if (*dst > r) 
						{
							*dst++ -= r;
						}
						else 
						{
							*dst++ = 0;
						}

						if (*dst > g) 
						{
							*dst++ -= g;
						}
						else 
						{
							*dst++ = 0;
						}

						if (*dst > b) 
						{
							*dst++ -= b;
						}
						else 
						{
							*dst++ = 0;
						}

						r *= r;
						g *= g;
						b *= b;

						if (*dstsqr > r) 
						{
							*dstsqr++ -= r;
						}
						else 
						{
							*dstsqr++ = 0;
						}

						if (*dstsqr > g) 
						{
							*dstsqr++ -= g;
						}
						else 
						{
							*dstsqr++ = 0;
						}

						if (*dstsqr > b) 
						{
							*dstsqr++ -= b;
						}
						else 
						{
							*dstsqr++ = 0;
						}
					} while (--w);
					src = (Pixel32 *)((char *)src + bitmap->modulo);
				} while (--h);
			}
			else 
			{
				if (mfd->anal_remain==0) 
				{
					max=0; // search for maximum value in sqr sums
					dstsqr=mfd->anal_sqrsum; 
					h=bitmap->h*bitmap->w;
					do 
					{
						if ((m=*dstsqr++)>max) 
						{
							max=m;
						}
					} while (--h);				
					mfd->anal_remain=(0x7fffffff-max)/(255*255); // calculate how many additions possible in worst case
				}
				mfd->anal_remain--; //decrement number of expected additions
				if (mfd->anal_remain>=0) 
				{ 
					mfd->anal_iter++;
					dst=mfd->anal_sum;
					dstsqr=mfd->anal_sqrsum;
					src=bitmap->data;
					h = bitmap->h;
					do 
					{
						w = bitmap->w;
						do 
						{
							pixel=*src++;
							*(dst++)+=(r=(pixel & 0xff0000)>>16);
							*(dst++)+=(g=(pixel & 0x00ff00)>>8);
							*(dst++)+=(b=(pixel & 0x0000ff));
							*(dstsqr++)+=r*r;
							*(dstsqr++)+=g*g;
							*(dstsqr++)+=b*b;
						} while (--w);
						src = (Pixel32 *)((char *)src + bitmap->modulo);
					} while (--h);
				}
			}
		}
		else
		{
			logl(3, ("SampleCallback(): Skipping frame %d", lFrame));
		}

	}
	logl(5, ("<- SampleCallback()"));
}

///////////////////////////////////////////////////////////////////////////

// Recalculate deblend mask according to loaded analysis mask and falloff slider

void RecalcDeblend(MyFilterData *mfd) 
{
	logl(5, ("RecalcDeblend()"));
	int w, h;
	Pixel32 *src, *dst;
	Pixel32 pixel;
	long r, b, d, c;

	if (MaskOK(mfd->fa, &mfd->mask_analyse)) 
	{
		if (CreateEmptyMask(mfd, &mfd->mask_deblend)) 
		{
			src=mfd->mask_analyse.bitmap;
			dst=mfd->mask_deblend.bitmap;
			for (h=0;h<mfd->mask_analyse.h;h++) 
			{
				for (w=0;w<mfd->mask_analyse.w;w++) 
				{
					switch (pixel=*src) 
					{
					case 0xff0000:
					case 0x00ff00:
					case 0x0000ff:
						*(dst++)=pixel;
						break;

					default:
						r = (pixel & 0x3FFF0000) >> 16; // this is FALLOFFCOEF*distance to red
						b = (pixel & 0x0000FFFC) >> 2; // this is FALLOFFCOEF*distance to blue
						d = FALLOFFCOEF*mfd->param_falloff;
						if ((r==0) || (r>=d)) 
						{
							c = 0;
						}
						else if ((b==0) || (b+r>=d)) 
						{
							c=(255*(d-r))/d;
						}
						else 
						{
							c=(255*b)/(b+r);
						}
						*dst++ = c << 16;
						break;
					}
					src++;
				}
			}
		}
	}
}

// Remove unwanted colors from just loaded DeBlend mask

void PrepareDeblend(MyFilterData *mfd) 
{
	logl(5, ("PrepareDeblend() ->"));

	if (MaskOK(mfd->fa, &mfd->mask_deblend)) 
	{
		logl(6, ("PrepareDeblend(): Mask OK"));

		Pixel32 *src=mfd->mask_deblend.bitmap;
		long i=mfd->mask_deblend.w*mfd->mask_deblend.h;

		do 
		{
			if ((*src & 0xff0000) != 0) 
			{
				*src &= 0xff0000;
			} 
			else if ((*src != 0x00ff00) && (*src != 0x0000ff)) 
			{
				*src = 0;
			}
			++src;
		} while (--i);
	}
	logl(5, ("<- PrepareDeblend()"));
}

// Return the carved part of Deblend mask

void UnCarveDeblend(MyFilterData *mfd, const FilterActivation *fa) 
{
	logl(5, ("UnCarveDeblend()"));
	Pixel32 *src;
	Pixel32 p;
	long i;

	if (MaskOK(fa, &mfd->mask_deblend)) 
	{
		src=mfd->mask_deblend.bitmap;
		i=mfd->mask_deblend.w*mfd->mask_deblend.h;
		do 
		{
			p = *src;
			if (((p & 0xff0000) == 0)
				&& (((p & 0x00ff00) >> 8) == (p & 0x0000ff))) 
			{

				*src = (p << 8) & 0xff0000;
			}
			++src;
		} while (--i);
	}
}

// Remove Repair mask from Deblend to speed up processing

void CarveDeblend(MyFilterData *mfd, const FilterActivation *fa) 
{
	logl(5, ("CarveDeblend()"));
	Pixel32 *src1, *src2;
	long i;
	
	UnCarveDeblend(mfd, fa);

	// Then turn all repaired areas to cyan
	if (MaskOK(fa, &mfd->mask_deblend)
		&& MaskOK(fa, &mfd->mask_repair)) 
	{
		src1=mfd->mask_deblend.bitmap;
		src2=mfd->mask_repair.bitmap;
		i=mfd->mask_deblend.w*mfd->mask_deblend.h;
		do 
		{
			if ((*src2 == 0xff0000) && ((*src1 & 0xff0000) != 0)) 
			{
				*src1 = (*src1 >> 16) | (*src1 >> 8);
			}
			++src1;
			++src2;
		} while (--i);
	}
}

// Recalculate Repair Power table

void RedoRepairTable(MyFilterData *mfd) 
{
	logl(5, ("RedoRepairTable() ->"));

	float *pwptr;
	long xmax, ymax, x, y, maxcount;
	float d;
	
	xmax=0;
	ymax=0;
	maxcount=0;

	logl(3, ("Delete Repair_powertable(%d)", mfd->repair_powertable));
	delete [] mfd->repair_powertable;
	mfd->repair_powertable=NULL;

	if (mfd->repair_array.ok()) 
	{
		long s = mfd->repair_array.count;

		RepairSet *set=mfd->repair_array.set;

		while (s--) 
		{
			set->maxDists(x, y);

			if (x>xmax) 
			{
				xmax=x;
			}

			if (y>ymax) 
			{
				ymax=y;
			}

			if (set->source.count > maxcount) 
			{
				maxcount = set->source.count;
			}

			set++;
		}

		if (mfd->repair_interlaced && ((xmax & 1) == 0)) 
		{
			++xmax;
		}

		mfd->repair_powertable = new float[(xmax+1)*(ymax+1)];
		logl(3, ("New repair_powertable = %d", mfd->repair_powertable));

		if ((mfd->repair_powertable == NULL) || (maxcount == 0))
		{
			logl(3, ("Delete Repair_powertable(%d)", mfd->repair_powertable));
			delete [] mfd->repair_powertable;
			mfd->repair_powertable = NULL;
			DeinitRepair(mfd);
		} 
		else 
		{
			if (mfd->repair_interlaced) 
			{
				mfd->repair_powerwidth=(xmax+1)/2;
			} 
			else 
			{
				mfd->repair_powerwidth=xmax+1;
			}
				
			// fill the table with coefficients

			pwptr=mfd->repair_powertable;

			double coef1 = FLT_MAX/(256*maxcount);
			double coef2 = mfd->repair_power*((float)MAXREALPOWER/(2*MAXPOWER));
			double coefx = AspectWidth(mfd);
			coefx *= coefx;
			double coefy = AspectHeight(mfd);
			coefy *= coefy;

			for (y=0; y<=ymax; y+=mfd->repair_interlaced?2:1) 
			{
				for (x=0; x<=xmax; x++) 
				{
					if ((y != 0) || (x != 0)) 
					{
						d=(float)(coef1/pow((x*x*coefx+y*y*coefy), coef2));
						if (d<FLT_EPSILON) 
						{
							d=FLT_EPSILON;
						}
					} 
					else 
					{
						d=0;
					}
					*pwptr++=d;
				}
			}
		}
	}
	logl(5, ("<- RedoRepairTable()"));
}

#define SetMask 0xfffffffe
#define SetBlack 0x80000000
#define SetGreen 0x80000001

#define NotSet(src) (((src) & SetMask) == SetBlack)
#define IsSet(src) (((src) & SetMask) != SetBlack)

long *Representative(long *src) 
{
	logl(9, ("<Representative()>"));
	// Return if it is border
	if (NotSet(*src)) 
	{
		return src;
	}
	// Step 1 - find the representative
	long *s2 = src;
	while (*s2 != 0) 
	{
		s2 += *s2;
	}
	// Step 2 - make shortcuts along the way
	while (*src != 0) 
	{
		long l = *src;
		*src = s2 - src;
		src += l;
	}
	return src;
}

void Union(long *src1, long *src2) 
{
	logl(9, ("<Union()>"));

	src1 = Representative(src1);
	src2 = Representative(src2);

	if ((*src1 == 0) && (*src2 == 0)) 
	{
		*src2 = src1 - src2;
	}
}

// Recalculate Repair pixel list according to repair mask and repair depth slider

void RedoRepair(MyFilterData *mfd, const FilterActivation *fa) 
{
// prepare repair mask
	logl(5, ("RedoRepair() ->"));

	long *src, *src2;
	long c, sn, count;
	long x, y;
	bool ok;

	DeinitRepair(mfd);

	if (mfd->mask_repair.bitmap!=NULL) 
	{

		// Let's implement a simple set union algorithm
		// Every pixel will contain a relative offset to its representative
		// non-repair pixels will contain 0x80000000
		// At first stage, every repair pixel will point to itself
		// thus it will contain 0

		src=(long *)mfd->mask_repair.bitmap;
		c=mfd->mask_repair.w*mfd->mask_repair.h;
		do 
		{
			switch (*src) 
			{
			case 0xff0000: 
				*src=0; 
				break;

			case 0x00ff00: 
				*src=SetGreen; 
				break;

			default: 
				*src=SetBlack; 
				break;
			}
			src++;
		} while (--c);

		// Pass 2 - every pixel will be to a set with its direct neighbors
		// this will create that many sets as there are continuous repair areas
		// don't forget about interlacing - fields must go to differetn sets

		src=(long *)mfd->mask_repair.bitmap;

		long dy = mfd->repair_interlaced ? 2 : 1;
		long dyw = dy * mfd->mask_repair.w;

		for (y = 0; y < mfd->mask_repair.h; y++) 
		{
			for (x = 0; x < mfd->mask_repair.w ; x++) 
			{
				if (IsSet(*src)) 
				{
					// No need to test contents of neighbors - it will be done in the Union
					if (x > 0) 
					{
						Union((long *)src - 1, (long *)src);
					}

					if (x + 1 < mfd->mask_repair.w) 
					{
						Union((long *)src, (long *)src + 1);
					}

					if (y >= dy) 
					{
						Union((long *)src - dyw, (long *)src);
					}

					if (y + dy < mfd->mask_repair.h) 
					{
						Union((long *)src, (long *)src + dyw);
					}
				}
				src++;
			}
		}

		// Pass 3 - lets count the sets
		// There is only one representative for each set
		// so there is exactly that many zero pixels

		src = (long *)mfd->mask_repair.bitmap;
		c = mfd->mask_repair.w * mfd->mask_repair.h;
		count = 0;

		do 
		{
			if (*src == 0) 
			{
				count++;
			}
			++src;
		} while (--c);

		ok = mfd->repair_array.init(count);

		if (ok) 
		{
			// Pass 4 - count pixels in each set
			src = (long *)mfd->mask_repair.bitmap;
			c = mfd->mask_repair.w * mfd->mask_repair.h;
			do 
			{
				if (IsSet(*src)) 
				{
					src2 = Representative(src);
					mfd->repair_array.addOneToRepre(src2);
				}
				src++;
			} while (--c);

			// Now create target fields
			ok = mfd->repair_array.initTargets();
		}

		if (ok) 
		{
			// Pass 5 - put all pixels to appropriate sets
			src=(long *)mfd->mask_repair.bitmap;
			for (y=0;y<mfd->mask_repair.h;y++) 
			{
				for (x=0;x<mfd->mask_repair.w;x++) 
				{
					if (IsSet(*src)) 
					{
						src2 = Representative(src);
						mfd->repair_array.addTargetPixel(src2, x, y);
					}
					src++;
				}
			}
			logl(5, ("RedoRepair(): Pass 5 finished"));

			// Pass 6 - lets convert the bitmap back to a red/black bitmap
			c = mfd->mask_repair.w * mfd->mask_repair.h;
			src = (long *)mfd->mask_repair.bitmap;
			do 
			{
				switch (*src) 
				{
				case SetBlack: 
					*src=0; 
					break;

				case SetGreen: 
					*src=0x00ff00; 
					break;

				default: 
					*src=0xff0000; 
					break;
				}
				src++;
			} while (--c);

			logl(5, ("RedoRepair(): Pass 6 finished"));

			// Now find appropriate border for each repair set 
			double aw = AspectWidth(mfd);
			double ah = AspectHeight(mfd);
			long drx = (long)ceil(mfd->repair_depth*aw/10)+1;
			long dry = (long)ceil(mfd->repair_depth*ah/10)+1;
			// when interlaced, dry must be even
			if ((mfd->repair_interlaced) && (dry&1)) 
			{
				++dry;
			}

			for (sn=0;sn<mfd->repair_array.count;sn++) 
			{
				count=0;
				TargetPixel *tptr = mfd->repair_array.set[sn].target.pixels;
				c = mfd->repair_array.set[sn].target.count;
				while (c--) 
				{
					x=tptr->x;
					y=tptr->y;
					src = ((long *)mfd->mask_repair.bitmap)+(x+mfd->mask_repair.w*y);
					// there must be a non-red neighbor in order to proceed with this pixel
					if (((x>0) && (*(src-1) != 0xff0000))
						|| ((x<mfd->mask_repair.w-1) && (*(src+1) != 0xff0000))
						|| ((y>dy-1) && (*(src-dyw) != 0xff0000))
						|| ((y<mfd->mask_repair.h-dy) && (*(src+dyw) != 0xff0000)))
					{
						long xlo=x-drx;
						long xhi=x+drx;
						long ylo=y-dry;
						long yhi=y+dry;

						double maxd = (double)(mfd->repair_depth*mfd->repair_depth)/100;

						if (xlo<0) 
						{
							xlo=0;
						}

						if (xhi>mfd->mask_repair.w-1) 
						{
							xhi=mfd->mask_repair.w-1;
						}

						while (ylo<0) 
						{
							ylo+=dy;
						}

						if (yhi>mfd->mask_repair.h-1) 
						{
							yhi=mfd->mask_repair.h-1;
						}

						src2=src+(xlo-x)+(ylo-y)*mfd->mask_repair.w;
						long yc = ylo;

						while (yc<=yhi) 
						{
							long *src3 = src2;
							long xc=xlo;
							while (xc<=xhi) 
							{
								if ((*src3 != 0xff0000) && (*src3 != 0x00ff00) && (*src3 != 0xffff01)) 
								{
									double dbx=(xc-x)/aw;
									double dby=(yc-y)/ah;
									if ((dbx*dbx+dby*dby<=maxd) || ((x==xc) && (abs(y-yc)==dy)) || ((y==yc) && (abs(x-xc)==1))) 
									{
										*src3 = 0xffff01;
										++count;
									}
								}
								++src3;
								++xc;
							}
							src2+=dyw;
							yc+=dy;
						}
					}
					++tptr;
				}
				if (ok) 
				{
					ok=mfd->repair_array.set[sn].source.init(count);
				}
				if (ok) 
				{
					// Pass 7 - get all marked pixels, put them to source pixels
					// and make them perfect yellow
					src=(long *)mfd->mask_repair.bitmap;
					for (y=0;y<mfd->mask_repair.h;y++) 
					{
						for (x=0;x<mfd->mask_repair.w;x++) 
						{
							if (*src == 0xffff01) 
							{
								mfd->repair_array.set[sn].source.addPixel(x, y, y*fa->src.pitch+x*sizeof(Pixel32));
								*src=0xffff00;
							}
							src++;
						}
					}
				}
			}
		}
			
		if (!ok) 
		{
			DeinitRepair(mfd);
		}
		else 
		{
			RedoRepairTable(mfd);
		}
	}
	logl(5, ("<- RedoRepair()"));
}

// Recalculate Repair mask according to loaded analysis mask, analysis done and maxalpha slider

void RecalcRepair(MyFilterData *mfd) 
{
	logl(5, ("RecalcRepair() ->"));

	Pixel32 *src1, *src2, *dst, *dst2;
	Pixel32 pixel;
	ulong r, g, b;
	long x, y, x1, y1;

	if (MaskOK(mfd->fa, &mfd->mask_analyse)
		&& MaskOK(mfd->fa, &mfd->mask_alpha))
	{
		if (CreateEmptyMask(mfd, &mfd->mask_repair)) 
		{
			src1=mfd->mask_analyse.bitmap;
			dst=mfd->mask_repair.bitmap;
			y=mfd->mask_analyse.w*mfd->mask_analyse.h;
			do 
			{
				switch (*src1++) 
				{
				case 0xff0000:
					*dst++ = 0;	// red: black, may change to red
					break;

				case 0x00ff00:
					*dst++ = 0x00ff00; // green: copy
					break;

				default:
					*dst++ = 0x0000ff; // other: blue (cannot change to red)
				}
			} while (--y);

			src1=mfd->mask_analyse.bitmap;
			src2=mfd->mask_alpha.bitmap;
			dst=mfd->mask_repair.bitmap;
			r=mfd->param_alpha<<16;
			g=mfd->param_alpha<<8;
			b=mfd->param_alpha;
			for (y=0;y<mfd->mask_analyse.h;y++) 
			{
				for (x=0;x<mfd->mask_analyse.w;x++) 
				{
					if (*src1++ != 0xff0000) 
					{ // don't touch if not red
						src2++;
					}
					else 
					{ // look if enough alpha
						pixel = *src2++;
						if (((pixel & 0xff0000)<=r) && ((pixel & 0x00ff00)<=g) && ((pixel & 0x0000ff)<=b)) 
						{
							// create red circle
							for (y1=-mfd->param_radius/10;y1<=mfd->param_radius/10+1;y1++) 
							{
								for (x1=-mfd->param_radius/10;x1<=mfd->param_radius/10+1;x1++) 
								{
									if ((x1*x1+y1*y1)*100<=mfd->param_radius*mfd->param_radius) 
									{
										if ((x+x1>=0) && (y+y1>=0) && (x+x1<mfd->mask_analyse.w) && (y+y1<mfd->mask_analyse.h)) 
										{
											dst2=dst+(x1+y1*mfd->mask_analyse.w);
											if (*dst2==0) 
											{
												*dst2=0xff0000;
											}
										}
									}
								}
							}
						}
					}
					dst++;
				}
			}
		}
	}
	RedoRepair(mfd, mfd->fa);
	CarveDeblend(mfd, mfd->fa);
	logl(5, ("<- RecalcRepair()"));
}

// Recalculate alpha and color masks according to loaded analysis mask and analysis done

void RecalcAlphaColor(MyFilterData *mfd) 
{
	logl(5, ("RecalcAlphaColor() ->"));

	Pixel32 *src1, *dst1, *dst2;
	ulong *src2, *src3;
	ulong alpha, color;
	int i;
	long count, n;
	double avgp[3], avgr[3], p, r, a, c;

// analysis arrays must exist
// some iterations must have been done
// analysis mask must exist
	if ((mfd->anal_sum!=NULL) && (mfd->anal_sqrsum!=NULL) && (mfd->anal_iter>0) 
		&& MaskOK(mfd->fa, &mfd->mask_analyse))
	{
		if (CreateEmptyMask(mfd, &mfd->mask_alpha) && CreateEmptyMask(mfd, &mfd->mask_color)) 
		{
// First pass - get averages
			src1=mfd->mask_analyse.bitmap;
			src2=mfd->anal_sum;
			src3=mfd->anal_sqrsum;
			count=mfd->mask_analyse.w*mfd->mask_analyse.h;
			n=0;
			avgp[0]=0;
			avgp[1]=0;
			avgp[2]=0;
			avgr[0]=0;
			avgr[1]=0;
			avgr[2]=0;
			do 
			{
				switch (*src1++) 
				{
				case 0xff0000:
				case 0x00ff00:
				case 0x0000ff:
					src2+=3;
					src3+=3;
					break;

				default:
					n++;
					i=3;
					while (i--) 
					{
						p=((double)(*src2++))/mfd->anal_iter;
						avgp[i]+=p;
						avgr[i]+=sqrt(((double)(*src3++))/mfd->anal_iter-p*p);
					}
				}
			} while (--count);

			for (i=0;i<3;i++) 
			{
				avgp[i]=((mfd->param_shift+300)*avgp[i]/(n*300));
				avgr[i]=((mfd->param_shift+300)*avgr[i]/(n*300));
			}
// Okay, we have the averages
// Now let's create alpha and color masks
			src1=mfd->mask_analyse.bitmap;
			src2=mfd->anal_sum;
			src3=mfd->anal_sqrsum;
			dst1=mfd->mask_alpha.bitmap;
			dst2=mfd->mask_color.bitmap;
			count=mfd->mask_analyse.w*mfd->mask_analyse.h;
			do 
			{
				i=3;
				alpha=0;
				color=0;
				while (i--) 
				{
					p=((double)(*src2++))/mfd->anal_iter;
					r=sqrt(((double)(*src3++))/mfd->anal_iter-p*p);

					if (avgr[i]==0) 
					{
						a=1;
					}
					else 
					{
						a=r/avgr[i];
						if (ALPHATOP*a>255) 
						{
							a=255/(double)ALPHATOP;
						}
					}
					if (a!=1) 
					{
						c=((avgp[i]-avgr[i])*a-(p-r))/(a-1);
						if (c<0) 
						{
							c=0;
						}

						if (c>255) 
						{
							c=255;
						}
					}
					else 
					{
						c=255;
					}
					
					alpha=(alpha<<8) | (long)(a*ALPHATOP);
					color=(color<<8) | (long)c;
				};
				*dst1++ = alpha;
				*dst2++ = color;
			} while (--count);
		}
	}
	RecalcRepair(mfd);
	logl(5, ("<- RecalcAlphaColor()"));
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd) 
{
	logl(5, ("<ConfigProc()>"));

	((MyFilterData *)fa->filter_data)->fa = fa;

    return DialogBoxParam(fa->filter->module->hInstModule, 
            MAKEINTRESOURCE(IDD_CONFIG_MAIN), hwnd, 
            ConfigDlgProc, (LPARAM)fa->filter_data);
}

void EnableDlgItem(HWND hdlg, int DlgItemID, bool Enable) 
{
	EnableWindow(GetDlgItem(hdlg, DlgItemID), Enable);
}

void EnableRadioButton(MyFilterData *mfd, int ControlId, bool Enabled) 
{
	if (!Enabled && IsDlgButtonChecked(mfd->hdlg, ControlId)) 
	{
		CheckDlgButton(mfd->hdlg, IDC_SHOW_STANDARD, BST_CHECKED);
		CheckDlgButton(mfd->hdlg, ControlId, BST_UNCHECKED);
		mfd->func_preview = Standard;
	}
	EnableDlgItem(mfd->hdlg, ControlId, Enabled);
}

// Purpose of this function is to enable/disable dialog box controls according to processing state
// return true, if preview has to be redrawn

void CheckControls(MyFilterData *mfd) 
{
	logl(5, ("<CheckControls()>"));

	EnableDlgItem(mfd->hdlg, IDC_FUNCTION_DEBLEND, MaskOK(mfd->fa, &mfd->mask_deblend) && MaskOK(mfd->fa, &mfd->mask_alpha) && MaskOK(mfd->fa, &mfd->mask_color));
	EnableDlgItem(mfd->hdlg, IDC_FUNCTION_REPAIR, MaskOK(mfd->fa, &mfd->mask_repair));
	EnableDlgItem(mfd->hdlg, IDC_FUNCTION_ONFRAMES, mfd->func_int.count>0);

	EnableRadioButton(mfd, IDC_SHOW_DEBLEND, MaskOK(mfd->fa, &mfd->mask_deblend));
	EnableRadioButton(mfd, IDC_SHOW_ALPHA, MaskOK(mfd->fa, &mfd->mask_alpha));
	EnableRadioButton(mfd, IDC_SHOW_COLOR, MaskOK(mfd->fa, &mfd->mask_color));
	EnableRadioButton(mfd, IDC_SHOW_REPAIR, MaskOK(mfd->fa, &mfd->mask_repair));
	EnableRadioButton(mfd, IDC_SHOW_ANALYSE, MaskOK(mfd->fa, &mfd->mask_analyse));

	EnableDlgItem(mfd->hdlg, IDC_SAVE_DEBLEND, MaskOK(mfd->fa, &mfd->mask_deblend));
	EnableDlgItem(mfd->hdlg, IDC_SAVE_ALPHA, MaskOK(mfd->fa, &mfd->mask_alpha));
	EnableDlgItem(mfd->hdlg, IDC_SAVE_COLOR, MaskOK(mfd->fa, &mfd->mask_color));
	EnableDlgItem(mfd->hdlg, IDC_SAVE_REPAIR, MaskOK(mfd->fa, &mfd->mask_repair));

	EnableDlgItem(mfd->hdlg, IDC_SAVE_ANALYSE, mfd->previewstate);

	EnableDlgItem(mfd->hdlg, IDC_ANAL_SAMPLEONE, mfd->previewstate);
	EnableDlgItem(mfd->hdlg, IDC_ANAL_UNSAMPLEONE, mfd->previewstate);
	EnableDlgItem(mfd->hdlg, IDC_ANAL_SAMPLEMANY, mfd->previewstate && (mfd->anal_int.count>0));

	EnableDlgItem(mfd->hdlg, IDC_PARAM_ALPHA, MaskOK(mfd->fa, &mfd->mask_analyse) && (mfd->anal_iter>0));
	EnableDlgItem(mfd->hdlg, IDC_PARAM_RADIUS, MaskOK(mfd->fa, &mfd->mask_analyse) && (mfd->anal_iter>0));
	EnableDlgItem(mfd->hdlg, IDC_PARAM_SHIFT, MaskOK(mfd->fa, &mfd->mask_analyse) && (mfd->anal_iter>0));
	EnableDlgItem(mfd->hdlg, IDC_PARAM_FALLOFF, MaskOK(mfd->fa, &mfd->mask_analyse));
}

static void PreviewButtonCallback(bool newstate, void *pvdata) 
{
	logl(5, ("PreviewButtonCallback() ->"));
	MyFilterData *mfd=(MyFilterData *)pvdata;

	mfd->previewstate=newstate;
	CheckControls(mfd);
	logl(5, ("<- PreviewButtonCallback()"));
}

void TogglePreview(MyFilterData *mfd) 
{
	logl(5, ("TogglePreview()"));
	mfd->fa->ifp->Toggle(GetParent(mfd->hdlg));
}

void ClosePreview(MyFilterData *mfd) 
{
	logl(5, ("ClosePreview()"));
	if (mfd->previewstate) 
	{
		TogglePreview(mfd);
	}
}

void SetupSlider(HWND hdlg, int ControlId, int TickFreq, int MinValue, int MaxValue, int CurrentValue) 
{
	logl(5, ("SetupSlider()"));
	SendDlgItemMessage(hdlg, ControlId, TBM_SETTICFREQ, TickFreq, 0);
	SendDlgItemMessage(hdlg, ControlId, TBM_SETRANGE, (WPARAM)TRUE, MAKELONG(MinValue, MaxValue));
	SendDlgItemMessage(hdlg, ControlId, TBM_SETPOS, (WPARAM)TRUE, CurrentValue);
}

BOOL CALLBACK ConfigDlgProc (HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) 
{
    MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

	switch(msg) 
	{
    case WM_INITDIALOG:
        SetWindowLong(hdlg, DWL_USER, lParam);
        mfd = (MyFilterData *)lParam;

		mfd->fa->ifp->InitButton(GetDlgItem(hdlg, IDC_PREVIEW));
		mfd->hdlg=hdlg;
		mfd->previewstate=false;
		mfd->fa->ifp->SetButtonCallback(PreviewButtonCallback, mfd);
		mfd->fa->ifp->SetSampleCallback(SampleCallback, mfd);

		if (mfd->func_deblend) 
		{
			CheckDlgButton(hdlg, IDC_FUNCTION_DEBLEND, BST_CHECKED);
		}

		if (mfd->func_repair) 
		{
			CheckDlgButton(hdlg, IDC_FUNCTION_REPAIR, BST_CHECKED);
		}

		if (mfd->func_onframes) 
		{
			CheckDlgButton(hdlg, IDC_FUNCTION_ONFRAMES, BST_CHECKED);
		}

		if (mfd->repair_interlaced) 
		{
			CheckDlgButton(hdlg, IDC_REPAIR_INTERLACED, BST_CHECKED);
		}

		CheckDlgButton(hdlg, IDC_SHOW_STANDARD, BST_CHECKED);

		SetDlgItemText(hdlg, IDC_FUNCTION_FRAMES, mfd->func_int.text);
		CreateIntervals(&mfd->func_int);

		SetDlgItemText(hdlg, IDC_ANAL_FRAMES, mfd->anal_int.text);
		CreateIntervals(&mfd->anal_int);

		ShowMaskProperties(mfd);

		SetupSlider(mfd->hdlg, IDC_REPAIR_DEPTH, TICKDEPTH, MINDEPTH, MAXDEPTH, mfd->repair_depth);
		SetupSlider(mfd->hdlg, IDC_REPAIR_POWER, TICKPOWER, MINPOWER, MAXPOWER, mfd->repair_power);
		SetupSlider(mfd->hdlg, IDC_REPAIR_ASPECT, TICKASPECT, MINASPECT, MAXASPECT, mfd->repair_aspect);
		SetupSlider(mfd->hdlg, IDC_PARAM_FALLOFF, TICKFALLOFF, MINFALLOFF, MAXFALLOFF, mfd->param_falloff);
		SetupSlider(mfd->hdlg, IDC_PARAM_ALPHA, TICKALPHA, MINALPHA, MAXALPHA, mfd->param_alpha);
		SetupSlider(mfd->hdlg, IDC_PARAM_RADIUS, TICKRADIUS, MINRADIUS, MAXRADIUS, mfd->param_radius);
		SetupSlider(mfd->hdlg, IDC_PARAM_SHIFT, TICKSHIFT, MINSHIFT, MAXSHIFT, mfd->param_shift);

		RefreshDepth(mfd, true);
		RefreshPower(mfd, true);
		RefreshAspect(mfd, true);
		RefreshFalloff(mfd, true);
		RefreshAlpha(mfd, true);
		RefreshRadius(mfd, true);
		RefreshShift(mfd, true);

		CheckControls(mfd);

        return TRUE;
    case WM_COMMAND:
		switch (HIWORD(wParam)) 
		{
		case BN_CLICKED:
			switch(LOWORD(wParam)) 
			{
			case IDOK:
				DeinitDialog(mfd);
				EndDialog(hdlg, 0);
				return TRUE;

			case IDCANCEL:
				DeinitDialog(mfd);
	            EndDialog(hdlg, 1);
		        return FALSE;

			case IDC_FUNCTION_DEBLEND:
				mfd->func_deblend=!!IsDlgButtonChecked(mfd->hdlg, IDC_FUNCTION_DEBLEND);
				mfd->fa->ifp->RedoFrame();
				break;

			case IDC_FUNCTION_REPAIR:
				mfd->func_repair=!!IsDlgButtonChecked(mfd->hdlg, IDC_FUNCTION_REPAIR);
				mfd->fa->ifp->RedoFrame();
				break;

			case IDC_FUNCTION_ONFRAMES:
				mfd->func_onframes=!!IsDlgButtonChecked(mfd->hdlg, IDC_FUNCTION_ONFRAMES);
				mfd->fa->ifp->RedoFrame();
				break;

			case IDC_SHOW_STANDARD:
				mfd->func_preview = Standard;
				mfd->fa->ifp->RedoFrame();
				break;

			case IDC_SHOW_DEBLEND:
				mfd->func_preview = DeBlend;
				mfd->fa->ifp->RedoFrame();
				break;

			case IDC_SHOW_ALPHA:
				mfd->func_preview = Alpha;
				mfd->fa->ifp->RedoFrame();
				break;

			case IDC_SHOW_COLOR:
				mfd->func_preview = Color;
				mfd->fa->ifp->RedoFrame();
				break;

			case IDC_SHOW_REPAIR:
				mfd->func_preview = Repair;
				mfd->fa->ifp->RedoFrame();
				break;

			case IDC_SHOW_ANALYSE:
				mfd->func_preview = Analyse;
				mfd->fa->ifp->RedoFrame();
				break;

			case IDC_SAVE_DEBLEND:
				ClosePreview(mfd);
				CheckControls(mfd);
				UnCarveDeblend(mfd, mfd->fa);
				SaveBitmap(mfd, &mfd->mask_deblend);
				CarveDeblend(mfd, mfd->fa);
				ShowMaskProperties(mfd);
				break;

			case IDC_SAVE_ALPHA:
				ClosePreview(mfd);
				CheckControls(mfd);
				SaveBitmap(mfd, &mfd->mask_alpha);
				ShowMaskProperties(mfd);
				break;

			case IDC_SAVE_COLOR:
				ClosePreview(mfd);
				CheckControls(mfd);
				SaveBitmap(mfd, &mfd->mask_color);
				ShowMaskProperties(mfd);
				break;

			case IDC_SAVE_REPAIR:
				ClosePreview(mfd);
				CheckControls(mfd);
				SaveBitmap(mfd, &mfd->mask_repair);
				ShowMaskProperties(mfd);
				break;

			case IDC_SAVE_ANALYSE:
				DeinitBitmap(&mfd->sampledimage);
				mfd->samplesave=true;
				mfd->fa->ifp->SampleCurrentFrame();
				if (mfd->sampledimage.bitmap!=NULL) 
				{
					ClosePreview(mfd);
					CheckControls(mfd);
					SaveBitmap(mfd, &mfd->sampledimage);
				}
				DeinitBitmap(&mfd->sampledimage);
				break;

			case IDC_LOAD_DEBLEND:
				ClosePreview(mfd);
				CheckControls(mfd);
				LoadDeblend(mfd, true);
				PrepareDeblend(mfd);
				CarveDeblend(mfd, mfd->fa);
				ShowMaskProperties(mfd);
				CheckControls(mfd);
				mfd->fa->ifp->RedoFrame();
				break;

			case IDC_LOAD_ALPHA:
				ClosePreview(mfd);
				CheckControls(mfd);
				LoadAlpha(mfd, true);
				ShowMaskProperties(mfd);
				CheckControls(mfd);
				mfd->fa->ifp->RedoFrame();
				break;

			case IDC_LOAD_COLOR:
				ClosePreview(mfd);
				CheckControls(mfd);
				LoadColor(mfd, true);
				ShowMaskProperties(mfd);
				CheckControls(mfd);
				mfd->fa->ifp->RedoFrame();
				break;

			case IDC_LOAD_REPAIR:
				ClosePreview(mfd);
				CheckControls(mfd);
				LoadRepair(mfd, true);
				RedoRepair(mfd, mfd->fa);
				CarveDeblend(mfd, mfd->fa);
				ShowMaskProperties(mfd);
				CheckControls(mfd);
				mfd->fa->ifp->RedoFrame();
				break;

			case IDC_LOAD_ANALYSE:
				ClosePreview(mfd);
				CheckControls(mfd);
				if (LoadAnalyse(mfd)) 
				{
					RecalcDeblend(mfd);
					RecalcAlphaColor(mfd);
					ShowMaskProperties(mfd);
				}
				CheckControls(mfd);
				mfd->fa->ifp->RedoFrame();
				break;

			case IDC_REPAIR_INTERLACED:
				mfd->repair_interlaced=!!IsDlgButtonChecked(mfd->hdlg, IDC_REPAIR_INTERLACED);
				RedoRepair(mfd, mfd->fa);
				mfd->fa->ifp->RedoFrame();
				break;

			case IDC_ANAL_RESET:
				DeinitStatistics(mfd);
				CheckControls(mfd);
				break;

			case IDC_PREVIEW:
				TogglePreview(mfd);
				CheckControls(mfd);
				break;

			case IDC_ANAL_SAMPLEONE:
				PrepareStatistics(mfd);
				mfd->samplesave=false; // don't save it
				mfd->sampleall=true; // analyse even if it is not in list
				mfd->unsample=false;
				mfd->fa->ifp->SampleCurrentFrame();
				RecalcDeblend(mfd);
				RecalcAlphaColor(mfd);
				ShowMaskProperties(mfd);
				CheckControls(mfd);
				mfd->fa->ifp->RedoFrame();
				break;

			case IDC_ANAL_UNSAMPLEONE:
				PrepareStatistics(mfd);
				mfd->samplesave=false; // don't save it
				mfd->sampleall=true; // analyse even if it is not in list
				mfd->unsample=true;
				mfd->fa->ifp->SampleCurrentFrame();
				RecalcDeblend(mfd);
				RecalcAlphaColor(mfd);
				ShowMaskProperties(mfd);
				CheckControls(mfd);
				mfd->fa->ifp->RedoFrame();
				break;

			case IDC_ANAL_SAMPLEMANY:
				DeinitStatistics(mfd);
				PrepareStatistics(mfd);
				mfd->samplesave=false; // don't save it
				mfd->sampleall=false; // analyse only specified
				mfd->unsample=false;
				mfd->fa->ifp->SampleFrames();
				RecalcDeblend(mfd);
				RecalcAlphaColor(mfd);
				ShowMaskProperties(mfd);
				CheckControls(mfd);
				mfd->fa->ifp->RedoFrame();
				break;
			}
			break;

		case EN_KILLFOCUS:
			switch(LOWORD(wParam)) 
			{
			case IDC_FUNCTION_FRAMES:
				GetDlgItemText(hdlg, IDC_FUNCTION_FRAMES, mfd->func_int.text, MAXSTR);
				CreateIntervals(&mfd->func_int);
				CheckControls(mfd);
				mfd->fa->ifp->RedoFrame();
				break;

			case IDC_ANAL_FRAMES:
				GetDlgItemText(hdlg, IDC_ANAL_FRAMES, mfd->anal_int.text, MAXSTR);				
				CreateIntervals(&mfd->anal_int);
				CheckControls(mfd);
				break;
			}
			break;
		}
		break;
	case WM_NOTIFY:
		switch (((LPNMHDR)lParam)->code) 
		{
		case NM_CUSTOMDRAW:
			if (((LPNMCUSTOMDRAW)lParam)->dwDrawStage==CDDS_PREPAINT) 
			{
				switch (((NMHDR *)lParam)->idFrom) 
				{
				case IDC_REPAIR_DEPTH:
					if (RefreshDepth(mfd, false)) 
					{
						RedoRepair(mfd, mfd->fa);
						mfd->fa->ifp->RedoFrame();
					}
					break;

				case IDC_REPAIR_POWER:
					if (RefreshPower(mfd, false)) 
					{
						RedoRepairTable(mfd);
						mfd->fa->ifp->RedoFrame();
					}
					break;

				case IDC_REPAIR_ASPECT:
					if (RefreshAspect(mfd, false)) 
					{
						RedoRepair(mfd, mfd->fa);
						mfd->fa->ifp->RedoFrame();
					}
					break;

				case IDC_PARAM_FALLOFF:
					if (RefreshFalloff(mfd, false)) 
					{
						RecalcDeblend(mfd);
						CarveDeblend(mfd, mfd->fa);
						mfd->fa->ifp->RedoFrame();
					}
					break;

				case IDC_PARAM_ALPHA:
					logl(5, ("*** 0"));
					if (RefreshAlpha(mfd, false)) 
					{
						logl(5, ("*** 1"));
						RecalcRepair(mfd);
						logl(5, ("*** 2"));
						mfd->fa->ifp->RedoFrame();
						logl(5, ("*** 3"));
					}
					logl(5, ("*** 4"));
					break;

				case IDC_PARAM_RADIUS:
					if (RefreshRadius(mfd, false)) 
					{
						RecalcRepair(mfd);
						mfd->fa->ifp->RedoFrame();
					}
					break;

				case IDC_PARAM_SHIFT:
					if (RefreshShift(mfd, false)) 
					{
						RecalcAlphaColor(mfd);
						mfd->fa->ifp->RedoFrame();
					}
					break;
				}
			}
			break;
		}
		break;
	}
	return FALSE;
}
